program recwav;

(*  RECWAV.PAS (version 1.1)

    This is a program to record .wav files on the Tandy 1000SL/TL/RL and
    later models with the proprietary Tandy DAC chip.  In accordance with
    the chip's capabilities, .wav's are recorded in 8-bit mono.

    Syntax:
      RECWAV [/S] [/T] [/N] [/nnnn] <filename>

    Where:
      /S, if specified, means that Recwav is running on a 1000SL.  Recwav
    will attempt to detect an SL; this switch is an override.  The recor-
    ding speed must be set differently on the SL than on the TL.
      /T, if specified, means that Recwav is running on a 1000TL.  This is
    the default.  I think most machines will use the TL's recording speed.
      /N, if specified, tells Recwav to bypass its search for the best
    sampling rate.  The Tandy DAC can record at a certain discrete set of
    rates.  Recwav, by default, will search for an available rate close to
    the specified one, use the rate it finds, and adjust the .wav header
    accordingly.  With /N, Recwav will take a stab at finding a rate close
    to the specified rate, but it will not check to see whether the rate
    selected is the best one, and it will not adjust the .wav header.  The
    program will start up faster with /N, but sounds will play back at a
    slightly different pitch than they record.
      /nnnn is the optional sampling rate in Hz; for example, /11025 or
    /22050.  A rate between 875 and 65535 may be specified, with the default
    being 11000.  The DAC can record at rates as low as 88Hz, but there's no
    point in it.  If /N is not specified, the specified rate will be ad-
    justed to match one of the DAC's available recording rates (see above).
      filename is the name of the output .wav file, including drive and path
    if desired.  If the extension is omitted, it defaults to .wav.  To
    specify a file without an extension, end the name with a period.

    Recwav prompts for a keystroke to begin recording, then another key-
    stroke to stop recording.  This version has been modified to record
    sounds of arbitrary length (not just 32kB blocks).

    This program is based on information contained in the _Tandy 1000TL
    Technical Reference Manual_; the TLSLSND package by Bruce M. Terry,
    Jr.; and the RIFF WAVE file format specification excerpted by Rob Ryan
    from the official Microsoft "Multimedia Programming Interface and Data
    Specification, v.1.0."
*)

  (* make sure there's heap space for the DMA buffers *)
{$M 16384,70000,655360}

uses dos;

(*********************************************************************)
(**************************** types **********************************)
(*********************************************************************)

type
  ascarray =                 (* array type for ASCIIZ string *)
    array[1..256] of char;
  reccode =                  (* error code from do_rec *)
    (recok, diskerr, fulldisk, overflow, openfail);
  bool2 =                    (* buffer flag type *)
    array[0..1] of boolean;

(*********************************************************************)
(************************ global variables ***************************)
(*********************************************************************)

var
  buffer0,                   (* first sound DMA buffer address *)
  buffer1,                   (* second sound DMA buffer address *)
  nextexit,                  (* pointer to next exit procedure in chain *)
  int1bvec:                  (* default <control>-<break> vector *)
    pointer;
  divider:                   (* computed divider for sound input *)
    word;
  strfile:                   (* output file name, as Pascal string *)
    string;
  is_SL,                     (* true if running on a 1000SL *)
  do_check:                  (* true if sampling rate should be adjusted *)
    boolean;
  inrate:                    (* input sampling rate in Hz *)
    word;
  drive:                     (* drive number for output file *)
    byte;

(*********************************************************************)
(******************** interrupt service routines *********************)
(*********************************************************************)

procedure int1b; interrupt;
  (*  This procedure, which is null, disables <control>-<break> when
      hooked.  *)

begin (* int1b *)
end; (* int1b *)

(*********************************************************************)

procedure int23; interrupt;
  (*  This procedure, which is null, disables <control>-C when hooked.  *)

begin (* int23 *)
end; (* int23 *)

(*********************************************************************)

procedure int24( flags, CS, IP, AX, BX, CX, DX, SI, DI, DS, ES, BP: word );
    interrupt;
  (*  This replacement critical error handler fails the system call, which
      keeps the program from being aborted in case of disk errors.  *)

begin (* int24 *)
  AX := AX AND $FF00;
  AX := AX OR 3;
end; (* int24 *)

(*********************************************************************)
(************************** subroutines ******************************)
(*********************************************************************)

function dac_found:
    boolean;
  (*  This function returns true if a Tandy DAC is detected.  *)

var
  regs:                      (* registers for Int 1Ah *)
    registers;

begin (* dac_found *)
  regs.AX := $8003;
  regs.CX := 0;
  intr( $1A, regs );
  if regs.CX = $5353 then
    begin
    dac_found := false;
    exit;
    end;
  regs.AX := $8100;
  intr( $1A, regs );
  dac_found := regs.AX <> $8100;
end; (* dac_found *)

(*********************************************************************)

{$L DAC_INIT}
procedure dac_init; external;
  (*  This external assembly routine initializes the DAC chip.  *)

(*********************************************************************)

procedure allocate_buffers(
  var buffer0,               (* pointers to 2 32k DMA buffers returned *)
      buffer1:
        pointer );
  (*  This routine allocates DMA buffers from the heap.  $M directive
      above ensures that there will be enough.  *)

begin (* allocate_buffers *)
  getmem( buffer0, 32768 );
  getmem( buffer1, 32768 );
end; (* allocate_buffers *)

(*********************************************************************)

procedure process_command_line(
  var inrate:                (* input sampling rate in Hz, returned *)
    word;
  var do_check,              (* returned true if sampling rate should be *)
                             (*   adjusted                               *)
      is_SL:                 (* returned true if running on an SL *)
    boolean;
  var strfile:               (* output filename, returned *)
    string );
  (*  This routine processes the command line parameters, setting the four
      variables above.  *)

var
  i,                         (* loop variable (general purpose) *)
  code,                      (* return code for val() procedure *)
  filept:                    (* number of parameter for filename *)
    integer;
  ratefound,                 (* true if /nnnn found on command line *)
  saidSL,                    (* true if /S or /T specified *)
  filefound:                 (* true if filename found on command line *)
    boolean;
  parm:                      (* miscellaneous command-line parameter *)
    string;
  ch:                        (* single-letter command-line option *)
    char;
  longrate:                  (* rate as long *)
    longint;

  (*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*)

  function is_opt( st: string ):
      boolean;
    (*  This function returns true if st is a command-line option, i.e.,
        if it consists of a '/' followed by at least one character.  *)

  begin (* is_opt *)
    if length( st ) < 2 then
      is_opt := false
    else if st[1] <> '/' then
      is_opt := false
    else
      is_opt := true;
  end; (* is_opt *)

  (*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*)

  function detect_SL:
      boolean;
    (*  This function returns true if a 1000SL is detected. *)

  var
    regs:                    (* registers for Int 15h *)
      registers;
    wordptr:                 (* for peeking at memory *)
      ^word;

  begin (* detect_SL *)
      (* if no 1000-series ID bytes, not an SL *)
    if mem[$FFFF:$E] <> $FF then
      detect_SL := false
    else if mem[$FC00:0] <> $21 then
      detect_SL := false

      (* It's a 1000, at least - check for 1000SL *)
    else
      begin
        (* do system ID call *)
      regs.AH := $C0;
      intr( $15, regs );
        (* if call not supported, not an SL *)
      if odd( regs.flags ) then
        detect_SL := false
        (* call supported, check model and submodel *)
      else
        begin
        wordptr := ptr( regs.ES, regs.BX+2 );
        detect_SL := wordptr^ = $FF;
        end; (* else if system ID call supported *)
      end; (* else if 1000-series *)
  end; (* detect_SL *)

  (*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*)

begin (* process_command_line *)
    (* if not at least one command-line parameter, halt with message *)
  if paramcount = 0 then
    begin
    writeln;
    writeln( 'Recwav - .wav recorder for Tandy' );
    writeln( 'Syntax:' );
    writeln( '  RECWAV [/S] [/T] [/N] [/nnnn] <filename>' );
    halt;
    end; (* if paramcount = 0 *)

    (* look for the filename *)
  filefound := false;
  i := 0;
  while not filefound and (i < paramcount) do
    begin
    i := i + 1;
    if not is_opt( paramstr( i ) ) then
      begin
      filefound := true;
      filept := i;
      end; (* if *)
    end; (* while not filefound *)

    (* if the file was not found, halt with message *)
  if not filefound then
    begin
    writeln( 'You forgot to specify an output file.' );
    halt;
    end;

    (* return the filename *)
  strfile := paramstr( filept );

    (* check for /S, /T, and /N parameters *)
  saidSL := false;
  do_check := true;
  for i := 1 to (filept-1) do
    if length( paramstr( i ) ) = 2 then
      begin
      parm := paramstr( i );
      ch := upcase( parm[2] );
      if ch = 'S' then
        begin
        saidSL := true;
        is_SL := true;
        end
      else if ch = 'T' then
        begin
        saidSL := true;
        is_SL := false;
        end
      else if ch = 'N' then
        do_check := false;
      end; (* if length( paramstr( i ) ) = 2 *)

    (* if the user didn't specify SL or TL on the command line, try to
       detect an SL *)
  if not saidSL then
    is_SL := detect_SL;

    (* check for sampling rate on command line *)
  ratefound := false;
  i := 1;
  while (i <= filept-1) and not ratefound do
    begin
    if length( paramstr( i ) ) >= 4 then
      begin
      parm := paramstr( i );           (* get the string *)
      delete( parm, 1, 1 );            (* delete the '/' *)
      val( parm, longrate, code );     (* convert to number *)
      if (code = 0) and (longrate >= 875) and (longrate < 65536) then
        begin
        ratefound := true;
        inrate := longrate;
        end; (* if a valid rate *)
      end; (* if parameter at least 4 chars *)
    i := i + 1;
    end; (* while *)

    (* if rate not specified, set to default *)
  if not ratefound then
    inrate := 11000;
end; (* process_command_line *)

(*********************************************************************)

function get_drive(
  strfile:                   (* name of output file *)
    string ): byte;
  (*  This function examines the output filename and returns the drive
      number:  0=default, 1=A:, 2=B:, etc.  *)

begin (* get_drive *)
  if pos( ':', strfile ) <> 2 then
    get_drive := 0
  else
    get_drive := ord( upcase( strfile[1] ) ) - ord( 'A' ) + 1;
end; (* get_drive *)

(*********************************************************************)

procedure set_extension(
  var strfile:               (* name of output file *)
    string );
  (*  This procedure sets the extension of the output file to .wav if no
      extension was specified.  *)

var
  dotplace:                  (* position of '.' in filename *)
    integer;

begin (* set_extension *)
  dotplace := pos( '.', strfile );
  if dotplace = 0 then
    strfile := strfile + '.wav';
end; (* set_extension *)

(*********************************************************************)

{$L TIME_DIV}
function time_div(
  buffer:                    (* buffer to record into, 65535 bytes *)
    pointer;
  divider:                   (* divider to use *)
    word ):
      word; external;
  (*  This external assembler subroutine records sound into a buffer for
      8 timer ticks (about 0.4394 seconds) using the divider passed to it
      and returns the number of bytes of sound recorded in that time.  The
      buffer must be 65535 bytes in size to ensure that there will be a
      32k segment that does not cross a 64k boundary.  *)

(*********************************************************************)

procedure writehex(
  var f:                     (* file to write to *)
    text;
  l:                         (* longint to write *)
    longint );
  (*  This procedure writes out a longint in hex.  (For debugging.)  *)

var
  i,                         (* loop count *)
  shiftcount:                (* number to shift *)
    integer;
  outbyte:                   (* byte to write out *)
    byte;
  outchar:                   (* char to write out *)
    char;

begin (* writehex *)
  shiftcount := 32;
  for i := 1 to 8 do
    begin
    shiftcount := shiftcount - 4;
    outbyte := (l SHR shiftcount) AND $0F;
    if outbyte < 10 then
      outchar := chr( ord( '0' ) + outbyte )
    else
      outchar := chr( ord( 'A' ) + outbyte - 10 );
    write( f, outchar );
    end; (* for *)
end; (* writehex *)

(*********************************************************************)

procedure compute_divider(
  var inrate:                (* sampling rate desired on entry, best *)
    word;                    (*   available rate on exit             *)
  is_SL,                     (* true if running on a 1000SL *)
  do_check:                  (* true if sampling rate should be adjusted *)
    boolean;
  var divider:               (* DAC divider value computed *)
    word );
  (*  This routine takes the desired sampling rate and uses a fixed
      formula to determine an approximate divider for that rate.  It
      then times the actual recording rate for dividers near the chosen
      one, selecting the nearest to the desired rate.  The divider and
      corresponding rate chosen are returned.  The initial estimate for
      the recording divider is based on the following experimental
      formula from the 1000TL (see Divider.doc):

                           3.579865E+05
      recording_divider := ------------- - 1.276329
                           sampling_rate

      If the machine is an SL, the initial divider estimate must then be
      divided by 1.15 before rounding to an integer.  If do_check is false,
      the input sampling rate is not adjusted, and the initial estimate
      is used for the divider.  *)

  (*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*)

  function count2rate(
    count:                   (* count of samples recorded *)
      word ):
        real;
    (*  This function takes the number of samples recorded in 8 system timer
        ticks and returns the effective sampling rate.  *)

  begin (* count2rate *)
    count2rate := count * 2.275813709;
  end; (* count2rate *)

  (*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*)

var
  real_divider,              (* divider estimate, as real *)
  current_rate,              (* current sampling rate *)
  last_rate:                 (* previous sampling rate *)
    real;
  current_divider,           (* current divider estimate *)
  last_divider:              (* last divider estimate *)
    word;
  buffer:                    (* DMA buffer for timing *)
    pointer;

begin (* compute_divider *)
    (* compute initial estimate *)
  real_divider := (3.579865E+05 / inrate) - 1.276329;
  if is_SL then
    real_divider := real_divider / 1.15;
  divider := round( real_divider );

    (* if the user said not to adjust, return *)
  if not do_check then
    exit;

    (* allocate a sound DMA buffer (call can't fail - see $M directive
       above) *)
  getmem( buffer, 65535 );

    (* determine sampling rate for initial estimate *)
  current_rate := count2rate( time_div( buffer, divider ) );

    (* if current rate is less than desired, try lower dividers *)
  if current_rate < inrate then
    begin
    current_divider := divider;
    while current_rate < inrate do
      begin
      last_divider := current_divider;
      last_rate := current_rate;
      current_divider := current_divider - 1;
      current_rate := count2rate( time_div( buffer, current_divider ) );
      end; (* while *)
      (* find best of current_divider, last_divider *)
    if abs( current_rate/inrate - 1.0 ) < abs( last_rate/inrate - 1.0 ) then
      divider := current_divider
    else
      begin
      divider := last_divider;
      current_rate := last_rate;
      end; (* else if last_divider was better *)
    end (* if current_rate < inrate *)

    (* otherwise, if the current rate is more than desired, try higher
       dividers *)
  else if current_rate > inrate then
    begin
    current_divider := divider;
    while current_rate > inrate do
      begin
      last_divider := current_divider;
      last_rate := current_rate;
      current_divider := current_divider + 1;
      current_rate := count2rate( time_div( buffer, current_divider ) );
      end; (* while *)
      (* find best of current_divider, last_divider *)
    if abs( current_rate/inrate - 1.0 ) < abs( last_rate/inrate - 1.0 ) then
      divider := current_divider
    else
      begin
      divider := last_divider;
      current_rate := last_rate;
      end; (* else if last_divider was better *)
    end; (* if current_rate > inrate *)

    (* special case:  rate over 65535Hz not allowed *)
  while current_rate >= 65535.5 do
    begin
    divider := divider + 1;
    current_rate := count2rate( time_div( buffer, divider ) );
    end; (* while *)

    (* deallocate sound buffer *)
  freemem( buffer, 65535 );

    (* return rate *)
  inrate := round( current_rate );
end; (* compute_divider *)

(*********************************************************************)

{$L RECORDER}
procedure start_record(
  divider:                   (* DAC divider for recording *)
    word;
  buffer0,                   (* pointer to first sound DMA buffer *)
  buffer1:                   (* pointer to second sound DMA buffer *)
    pointer;
  var lastbuf,               (* "last buffer" flags *)
      fullbuf:               (* "full buffer" flags *)
    bool2;
  var lastbytes:             (* number of bytes in the last buffer *)
    word;
  var ovrflo:                (* true if input overflow occurred *)
    boolean ); external;
procedure stop_record; external;
  (*  These two subroutines handle the interface to the sound hardware.  
      start_record() divides each buffer into two parts so that neither 
      part crosses a 64k boundary, programs the PSSJ for sound input using 
      the given divider, hooks Int 0Fh and 15h, and programs the DMA 
      controller to start sound input on the first part of buffer0.  The 
      Int 0Fh handler takes over from then on, restarting DMA when it 
      reaches terminal count, marking buffers full, and halting sound input 
      if overflow (sound input faster than disk output) occurs.  The Int
      15h handler terminates sound input when a keystroke is detected.  
      start_record() returns immediately; sound input continues in the 
      background.  stop_record() stops sound DMA if it has not already been 
      and unhooks the interrupts.
          After start_record() has been called, stop_record() *must* be 
      called at some point to restore the interrupt vector table.  *)

(*********************************************************************)

function do_rec(
  divider,                   (* DAC divider value *)
  inrate:                    (* input sampling rate in Hz *)
    word;
  buffer0,                   (* pointers to 32k DMA buffers *)
  buffer1:
    pointer;
  strfile:                   (* output .wav file *)
    string;
  drive:                     (* drive where output file resides *)
    byte ): reccode;
  (*  This routine performs the actual sound recording.  It returns a code
      indicating the success or failure of its mission.  'recok' is returned
      if all went well.  'diskerr' is returned if a file I/O system call
      failed; the output file is deleted.  'fulldisk' is returned if a full
      disk was detected during recording; the output file is as large as
      possible.  'overflow' is returned if input overflow occurred, i.e.,
      if the system was unable to transfer sound data to disk as fast as it
      came in.  In case of overflow, the output file is deleted.  'openfail'
      is returned if the output file could not be opened.  *)

label 100, 200;              (* jump points for error conditions *)

var
  outfile:                   (* output file *)
    file;
  buffers:                   (* buffer pointers, as array *)
    array[0..1] of pointer;
  lastbuf,                   (* lastbuf[i] is true if last input buffer *)
  fullbuf:                   (* fullbuf[i] is true if buffer has sound to *)
                             (*   be written to disk                      *)
    bool2;
  nbytes,                    (* number of bytes to write to disk *)
  nwritten,                  (* number of bytes successfully written out *)
  lastbytes:                 (* number of bytes in the last buffer *)
    word;
  finished,                  (* true when done recording *)
  ovrflo:                    (* true if sound input overflowed *)
    boolean;
  current,                   (* current output buffer *)
  i:                         (* loop counter *)
    integer;
  filelength,                (* length of output file *)
  freespace:                 (* amount of free space left on the disk *)
    longint;
  returned:                  (* function return value *)
    reccode;
  regs:                      (* registers for Int 21h function 0Dh *)
    registers;

  (* RIFF WAVE header template.  The riffsize and datasize fields will be
     set to match the file length; other fields are constant. *)
  wavheader:
    record
      rifflabel:             (* 'RIFF' *)
        packed array[1..4] of char;
      riffsize:              (* length of file, less label above *)
        longint;
      wavefmtlabel:          (* 'WAVEfmt ' *)
        packed array[1..8] of char;
      fmtlen:                (* length of format chunk, always 16 *)
        longint;
      fmttag,                (* format tag, 1 = Microsoft PCM *)
      nchannels,             (* number of channels, 1 = mono *)
      outrate1,              (* sampling rate in Hz *)
      filler1,
      outrate2,              (* number of bytes per second of sound *)
      filler2,
      bytespersam,           (* bytes per (multichannel) sample, always 1 *)
      bitspersam:            (* bits per channel, always 8 *)
        word;
      datalabel:             (* 'data' *)
        packed array[1..4] of char;
      datasize:              (* number of bytes of actual sound data *)
        longint
    end; (* record *)

  (*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*)

  procedure keyflush;
    (*  This inline macro flushes the keyboard typeahead buffer.  *)

  inline(                  (* FLUSHLOOP: *)
    $B4/$01/               (*   MOV  AH,1 *)
    $CD/$16/               (*   INT  16h *)
    $74/$06/               (*   JZ   DONE *)
    $B4/$00/               (*   MOV  AH,0 *)
    $CD/$16/               (*   INT  16h *)
    $EB/$F4 );             (*   JMP  FLUSHLOOP *)
                           (* DONE: *)

  (*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*)

  procedure waitkey;
    (*  This inline macro waits for a (non-shift) key to be pressed.  *)

  inline(
    $B4/$00/               (*   MOV  AH,0 *)
    $CD/$16 );             (*   INT  16h *)

  (*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*)

begin (* do_rec *)
    (* open the output file *)
  assign( outfile, strfile );
  {$I-} rewrite( outfile, 1 ); {$I+}
  if ioresult <> 0 then
    begin
    do_rec := openfail;
    exit;
    end;

    (* initialize the .wav header template *)
  wavheader.rifflabel := 'RIFF';
  wavheader.riffsize := 0;
  wavheader.wavefmtlabel := 'WAVEfmt ';
  wavheader.fmtlen := 16;
  wavheader.fmttag := 1;
  wavheader.nchannels := 1;
  wavheader.outrate1 := inrate;
  wavheader.filler1 := 0;
  wavheader.outrate2 := inrate;
  wavheader.filler2 := 0;
  wavheader.bytespersam := 1;
  wavheader.bitspersam := 8;
  wavheader.datalabel := 'data';
  wavheader.datasize := 0;

    (* initialize buffer flags *)
  for i := 0 to 1 do
    begin
    lastbuf[i] := false;
    fullbuf[i] := false;
    end;

    (* initialize other local variables *)
  current := 0;
  finished := false;
  ovrflo := false;
  buffers[0] := buffer0;
  buffers[1] := buffer1;

    (* start out optimistic *)
  returned := recok;

    (* write the .wav header template to disk *)
  {$I-} blockwrite( outfile, wavheader, sizeof( wavheader ),
    nwritten ); {$I+}
  if (ioresult <> 0) or (nwritten <> sizeof( wavheader )) then
    begin
    returned := diskerr;
    goto 200;
    end;

    (* start recording sound *)
  writeln( 'Press a key to begin recording.' );
  keyflush;
  waitkey;
  start_record( divider, buffer0, buffer1, lastbuf, fullbuf, lastbytes,
    ovrflo );
  writeln( 'Press a key to stop recording.' );

    (* main loop:  wait for a buffer to be filled with sound, then write
       it to disk *)
  repeat

      (* wait for buffer to fill - or overflow *)
    while not ovrflo and not fullbuf[current] do;

      (* if overflow occurred, delete the file and exit *)
    if ovrflo then
      begin
      returned := overflow;
      goto 100;
      end; (* if ovrflo *)

      (* if this is the last buffer (user hit a key to stop recording) then
         we're finished:  unhook vectors and get the number of bytes in the
         last buffer *)
    if lastbuf[current] then
      begin
      stop_record;
      nbytes := lastbytes;
      finished := true;
      end

      (* otherwise, this is not the last buffer, so it must have 32768 bytes
         in it *)
    else
      nbytes := 32768;

      (* write the buffer to disk - halt on disk errors *)
    {$I-} blockwrite( outfile, buffers[current]^, nbytes, nwritten ); {$I+}
    if ioresult <> 0 then
      begin
      returned := diskerr;
      goto 100;
      end;

      (* if the requested number of bytes was not written (and there was no
         disk error), then the disk is full *)
    if nbytes <> nwritten then
      begin
        (* since the disk is full, we're finished *)
      stop_record;
      returned := fulldisk;
      finished := true;
        (* get the amount of free space left - check for errors *)
      {$I-} freespace := diskfree( drive ); {$I+}
      if (ioresult <> 0) or (freespace = -1) then
        begin
        returned := diskerr;
        goto 100;
        end;
      nbytes := freespace;
        (* write out whatever will fit on the disk - check for errors *)
      {$I-} blockwrite( outfile, buffers[current]^, nbytes, nwritten ); {$I+}
      if ioresult <> 0 then
        begin
        returned := diskerr;
        goto 100;
        end;
      end; (* if nbytes <> nwritten *)

      (* go do next output buffer *)
    fullbuf[current] := false;
    current := current XOR 1;
  until finished;

    (* set the length fields in the .wav header template *)
  {$I-} filelength := filesize( outfile ); {$I+}
  if ioresult <> 0 then
    begin
    returned := diskerr;
    goto 100;
    end;
  wavheader.riffsize := filelength - 8;
  wavheader.datasize := filelength - 44;

    (* seek to the beginning of the file *)
  {$I-} seek( outfile, 0 ); {$I+}
  if ioresult <> 0 then
    begin
    returned := diskerr;
    goto 100;
    end;

    (* write the .wav header template to disk *)
  {$I-} blockwrite( outfile, wavheader, sizeof( wavheader ), nwritten ); {$I+}
  if (ioresult <> 0) or (nwritten <> sizeof( wavheader )) then
    begin
    returned := diskerr;
    goto 100;
    end;

    (* close the output file *)
  {$I-} close( outfile ); {$I+}
  if ioresult <> 0 then
    begin
    returned := diskerr;
    goto 100;
    end;

    (* flush the disk buffers (prevents critical error at termination) *)
  regs.ah := $0D;
  {$I-} msdos( regs ); {$I+}

    (* return (code is recok or fulldisk) *)
  do_rec := returned;
  exit;

  (* jump to here in case of errors *)
100:
  stop_record;
  (* ... or here if you haven't started recording when the error happens *)
200:
  {$I-}
  close( outfile );
  erase( outfile );
  regs.ah := $0D;
  msdos( regs );
  {$I+}
  do_rec := returned;
end; (* do_rec *)

(*********************************************************************)
(************************ exit procedure *****************************)
(*********************************************************************)

{$F+} procedure mainexit; {$F-}
  (*  This exit procedure is executed automatically when the program
      terminates.  It unhooks interrupt vectors and finalizes the sound
      chip.  *)

begin (* mainexit *)
    (* unhook interrupt 1Bh; 23h and 24h will be done automatically by
       DOS *)
  setintvec( $1B, int1bvec );

    (* finalize the sound chip *)
  dac_init;

    (* execute next exit procedure in chain *)
  exitproc := nextexit;
end; (* mainexit *)

(*********************************************************************)
(************************** main program *****************************)
(*********************************************************************)

begin (* recwav *)
    (* check for presence of Tandy DAC *)
  if not dac_found then
    begin
    writeln( 'This program requires a Tandy DAC chip.' );
    halt;
    end; (* if not dac_found *)

    (* hook needed interrupts; <control>-C and <control>-<break> are
       disabled, critical error fails system call *)
  getintvec( $1B, int1bvec );
  setintvec( $1B, @int1b );
  setintvec( $23, @int23 );
  setintvec( $24, @int24 );

    (* link exit procedure to chain *)
  nextexit := exitproc;
  exitproc := @mainexit;

    (* get command line parameters and process *)
  process_command_line( inrate, do_check, is_SL, strfile );

    (* get drive number for output file *)
  drive := get_drive( strfile );

    (* set file extension to .wav if not specified *)
  set_extension( strfile );

    (* determine nearest available sampling rate and divider value *)
  if not do_check then
    writeln( 'Sampling rate not adjusted.' )
  else
    writeln( 'Adjusting sampling rate....' );
  compute_divider( inrate, is_SL, do_check, divider );

    (* display input sampling rate *)
  writeln( 'Recording RIFF WAVE file at ', inrate, 'Hz.' );

    (* allocate DMA buffer space from the heap - must do this *after*
       computing the divider *)
  allocate_buffers( buffer0, buffer1 );

    (* initialize DAC *)
  dac_init;

    (* do recording, report result *)
  case do_rec( divider, inrate, buffer0, buffer1, strfile, drive ) of
    recok:     writeln( 'Recording complete.' );
    diskerr:   writeln( 'Disk error - recording deleted.' );
    fulldisk:  writeln( 'Disk full - recording halted.' );
    overflow:  begin
               write( 'Input overflow - unable to maintain sampling ' );
               writeln( 'rate.  Record to a hard disk' );
               write( 'or RAM disk, or specify a lower rate.  ' );
               writeln( 'Recording deleted.' );
               end; (* overflow *)
    openfail:  writeln( 'Unable to create file "', strfile, '".' )
  end; (* case *)
end. (* recwav *)
