/*
 *  A standard set of graphics calls, defined on different machines
 *  to make the proper calls for that machine
 *
 *  The following functions are required:
 *     draw_init(void)  
 *       Inititialize system, called immediately after program starts
 *     setup_graphics(void)
 *       Enter the graphics mode
 *     cleanup_graphics(void)
 *       Exit the graphics mode, and return to text mode
 *     draw_fontcolor(int c)
 *       Set the font color to 'c'
 *     draw_text_left(int x,int y,char *string)
 *       Write the text string with its upper left corner = (x,y)
 *     draw_text_right(int x,int y,char *string)
 *       Write the text string with its upper right corner = (x,y)
 *     draw_text_centered(int x,int y,char *string)
 *       Write the text string with its upper edge centered on (x,y)
 *     draw_rectangle(int x1,int y1,int x2,int y2,int c)
 *       Draw a rectangle from (x1,y1) to (x2,y2), lines having color 'c'
 *     draw_line(int x1,int y1,int x2,int y2,int c)
 *       Draw a line from (x1,y1) to (x2,y2), having color 'c'
 *     draw_bar(int x1,int y1,int x2,int y2,int c)
 *       Draw a filled rectangle from (x1,y1) to (x2,y2) with color 'c'
 *     draw_setpalette(int n,int r,int g,int b)
 *       Set the paletted color number n to (r,g,b) where max(r)=63
 *     int draw_getch(void)
 *       Wait for a key press, and return the key value
 *     int draw_getkey(void)
 *       Check for key press, and return the key value if there, else return 0
 */

#ifdef SRGP
  /*
   *  Definitions for the SRGP library
   */
  #include "srgp.h"

  void draw_init(void);
  void draw_setup_graphics(void);
  void draw_cleanup_graphics(void);
  void draw_cleardisplay(void);
  void draw_fontcolor(int c);
  void draw_text_left(int x,int y,char *string);
  void draw_text_centered(int x,int y,char *string);
  void draw_text_right(int x,int y,char *string);
  void draw_text_vertical(int x,int y,char *string);
  void draw_rectangle(int x1,int y1,int x2,int y2,int c);
  void draw_line(int x1,int y1,int x2,int y2,int c);
  void draw_bar(int x1,int y1,int x2,int y2,int c);
  void draw_setpalette(int n,int r,int g,int b);
  void draw_setallpalette(unsigned char *p);
  void draw_putpixel(int x,int y,int c);
  int draw_getpixel(int x,int y);
  int draw_getch(void);
  int draw_getkey(void);

#endif /* SRGP */

#ifdef SVGALIB

  /*
   *  Definitions for the SVGALIB library for Linux
   */ 
  #include <vga.h>
  #include <vgagl.h>

  #define GMODE G800x600x256

  #define draw_init() vga_init()
  void draw_setup_graphics(void);
  #define draw_cleanup_graphics() vga_setmode(TEXT)
  #define draw_cleardisplay() vga_clear()
  void draw_fontcolor(int c);
  #define draw_text_left(x,y,string) gl_write(x,y,string)
  #define draw_text_centered(x,y,string) \
	  gl_write((x)-strlen(string)*_font_width/2,y,string)
  #define draw_text_right(x,y,string) \
	  gl_write((x)-strlen(string)*_font_width,y,string)
  void draw_text_vertical(int x,int y,char *string);
  void draw_rectangle(int x1,int y1,int x2,int y2,int c);
  #define draw_line(x1,y1,x2,y2,c) gl_line(x1,y1,x2,y2,c)
  #define draw_bar(x1,y1,x2,y2,c) gl_fillbox(x1,y1,(x2)-(x1)+1,(y2)-(y1)+1,c)
  #define draw_setpalette(n,r,g,b) gl_setpalettecolor(n,r,g,b)
  void draw_setallpalette(unsigned char *p);
  #define draw_putpixel(x,y,c) gl_setpixel(x,y,c)
  int draw_getpixel(int x,int y); 
  int draw_getch(void);
  int draw_getkey(void);

#endif /* SVGALIB */

#ifdef DOS

  /*
   *  Definitions for the VESA SVGA BIOS extensions for DOS
   */
  #include <conio.h>
  #include "vesa_vid.h"

  void draw_init(void);
  void draw_setup_graphics(void);
  #define draw_cleanup_graphics() RestoreState()
  #define draw_cleardisplay() clear800x600x8()
  #define draw_putpixel(x,y,c) putPixel(x,y,c)
  int draw_getpixel(int x,int y);
  void draw_fontcolor(int c);
  #define draw_text_left(x,y,string) writeText(x,y,_font_color,string)
  #define draw_text_centered(x,y,string) \
	  writeText((x)-strlen(string)*_font_width/2,y,_font_color,string)
  #define draw_text_right(x,y,string) \
	  writeText((x)-strlen(string)*_font_width,y,_font_color,string)
  void draw_text_vertical(int x,int y,char *string);
  void draw_rectangle(int x1,int y1,int x2,int y2,int c);
  #define draw_line(x1,y1,x2,y2,c) line(x1,y1,x2,y2,c)
  #define draw_bar(x1,y1,x2,y2,c) box(x1,y1,x2,y2,c)
  void draw_setallpalette(unsigned char *p);
  int draw_getch(void);
  int draw_getkey(void);

#endif /* DOS */
