#include <iostream.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>

#include "allegro.h"

#define MIN_MASS 12
#define MAX_MASS 200
#define MIN_SEGS 3
#define MAX_SEGS 60

struct Vector {
	float x;
	float y;
};

struct Segment {
	Vector pos;
	Vector vel;
	Vector acc;
	Vector force;
	float mass;
	int color, boxcolor;
	Segment *next;
	Segment *previous;
};

void append_seg(Segment *head, float x, float y, float xv, float yv, float xa, float ya, float mass);
void instructions_exit(void);

void set_forces(struct Segment *head);
void set_force(struct Segment *point);
void set_acceleration(struct Segment *point);
void set_velocity(struct Segment *point);
void set_position(struct Segment *point);


const float vel_mod = .99;
const float seg_length = 4;
float mass1 = 30;
float mass2 = 0;

int main (int argc, char *argv[]) {
	int i, x, y;
	RGB p;
	Segment *tip, *current;
	int hires = 0, segments = 20;
	unsigned char c;
	BITMAP *buffer;

	// startup routines
	allegro_init();
	install_keyboard();
	install_mouse();
	install_timer();

	i = 1;
	while (i < argc) {
		if (strcmp(argv[i], "-h") == 0) {
			hires = 1;
		} else if (strcmp(argv[i], "-s") == 0) {
			segments = atoi((argv[++i]));
			if (segments < MIN_SEGS) { segments = MIN_SEGS; }
			if (segments > MAX_SEGS) { segments = MAX_SEGS; }
		} else if (strcmp(argv[i], "-m") == 0) {
			mass1 = atof((argv[++i]));
			if (mass1 < MIN_MASS) { mass1 = MIN_MASS; }
			if (mass1 > MAX_MASS) { mass1 = MAX_MASS; }
		} else if (strncmp(argv[i], "-am", 3) == 0) {
			mass2 = atof((argv[++i]));
			if (mass2 < MIN_MASS) { mass2 = MIN_MASS; }
			if (mass2 > MAX_MASS) { mass2 = MAX_MASS; }
		} else {
			instructions_exit();
		}

		i++;
	}

	if (hires) {
		set_gfx_mode(GFX_AUTODETECT, 640, 480, 0, 0);
	} else {
		set_gfx_mode(GFX_AUTODETECT, 320, 200, 0, 0);
	}

	buffer = create_bitmap(SCREEN_W, SCREEN_H);

	for (i = 1; i < 64; i++) {
		p.r = i;
		p.g = i;
		p.b = 63;
		set_color(i, &p);
	}
	for (i = 0; i < 64; i++) {
		p.r = 63 - i;
		p.g = 63;
		p.b = 63 - i;
		set_color(i + 64, &p);
	}
	for (i = 0; i < 64; i++) {
		p.r = 63;
		p.g = i;
		p.b = 63 - i;
		set_color(i + 128, &p);
	}

	tip = (struct Segment *)malloc(sizeof(struct Segment));
	tip->previous = NULL;
	tip->next = NULL;
	tip->color = 4;

	for (i = 0; i < segments; i++) {
		c = ((c == 9) ? 15 : 9);
		append_seg(tip, SCREEN_W-1, SCREEN_H/2, 0, 0, 0, 0, ((i < segments / 2) || (!mass2)) ? mass1 : mass2);
	}

	textout(screen, font, "Press a Key.", 0, 0, 15);

	while (!keypressed()) {
	}

	while (!key[KEY_ESC]) {
		tip->pos.x = (float)mouse_x;
		tip->pos.y = (float)mouse_y;

		//set_forces(tip);

		current = tip;
		do {
			current = current->next;
			set_force(current);
		} while (current->next->next);

		current = tip;
		do {
			current = current->next;
			set_acceleration(current);
			set_velocity(current);
			set_position(current);
		} while (current->next->next);

		current = tip;

		clear(buffer);
		textout(buffer, font, "Cheez Wave Simulator", 0, 0, 63);

		while (current->next) {
			x = (int)(current->pos.x);
			y = (int)(current->pos.y);

			line(buffer, x, y,
				(int)(current->next->pos.x), (int)(current->next->pos.y), current->color);
			rectfill(buffer, x - 1, y - 1, x, y + 1, current->boxcolor);

			current = current->next;
		}

		//readkey();
		vsync();
		blit(buffer, screen, 0, 0, 0, 0, SCREEN_W, SCREEN_H);

	}

	return (0);
}

void set_force(struct Segment *point) {
	struct Vector dist, force;
	float ratio, length, fix;

	force.x = force.y = 0.0;

	if (point->previous != NULL) {
		dist.x = point->previous->pos.x - point->pos.x;
		dist.y = point->previous->pos.y - point->pos.y;

		length = sqrt((dist.x * dist.x) + (dist.y * dist.y));
		ratio = length / seg_length;

		if (ratio > 1) {
			fix = sqrt(ratio - 1);
			force.x += dist.x * fix;
			force.y += dist.y * fix;
		}
	}

	if (point->next != NULL) {
		dist.x = point->next->pos.x - point->pos.x;
		dist.y = point->next->pos.y - point->pos.y;

		length = sqrt((dist.x * dist.x) + (dist.y * dist.y));
		ratio = length / seg_length;

		if (ratio > 1) {
			fix = sqrt(ratio - 1);
			force.x += dist.x * fix;
			force.y += dist.y * fix;
		}
	}
	point->force.x = force.x;
	point->force.y = force.y;
}

void set_forces(struct Segment *head) {
	struct Vector dist, force;
	struct Segment *point;
	float ratio, length, fix;

	point = head;

	force.x = force.y = 0.0;

	while (point->next) {
		point = point->next;

		dist.x = point->previous->pos.x - point->pos.x;
		dist.y = point->previous->pos.y - point->pos.y;

		length = sqrt((dist.x * dist.x) + (dist.y * dist.y));
		ratio = length / seg_length;

		if (ratio > 1) {
			fix = sqrt(ratio - 1);
			force.x = dist.x * fix;
			force.y = dist.y * fix;
		} else {
			force.x = 0;
			force.y = 0;
		}
		point->force.x = force.x;
		point->force.y = force.y;
		//textprintf(screen, font, 0, (c++)*8, 15, "%f, %f", point->force.x, point->force.y);
	}

	force.x = force.y = 0.0;
	// now backwards

	while (point->previous) {
		point = point->previous;

		dist.x = point->next->pos.x - point->pos.x;
		dist.y = point->next->pos.y - point->pos.y;

		length = sqrt(dist.x * dist.x + dist.y * dist.y);
		ratio = length / seg_length;

		if (ratio > 1) {
			fix = sqrt(ratio - 1);
			force.x = dist.x * fix;
			force.y = dist.y * fix;
		} else {
			force.x = 0;
			force.y = 0;
		}

		point->force.x += force.x;
		point->force.y += force.y;

	}
}

void set_acceleration(struct Segment *point) {
		point->acc.x = point->force.x / point->mass;
		point->acc.y = point->force.y / point->mass;
}

void set_velocity(struct Segment *point) {
	point->vel.x = point->acc.x + point->vel.x * vel_mod;
	point->vel.y = point->acc.y + point->vel.y * vel_mod;
	point->color = (4 * ((int)fabs(point->vel.x) + (int)fabs(point->vel.y))) + 1;
	if (point->color > 127) { point->color = 127; }
}

void set_position(struct Segment *point) {
	point->pos.x += point->vel.x;
	point->pos.y += point->vel.y;
	if (point->pos.x < 0.0) { point->pos.x = 0.0; }
	if (point->pos.y < 0.0) { point->pos.y = 0.0; }
	if (point->pos.x >= SCREEN_W - 1) { point->pos.x = SCREEN_W - 1; }
	if (point->pos.y >= SCREEN_H - 1) { point->pos.y = SCREEN_H - 1; }
}


void append_seg(Segment *head, float x, float y, float xv, float yv, float xa, float ya, float mass) {
	Segment *current, *newseg;
	current = head;

	while (current->next) {
		current = current->next;
	}

	current->next = newseg = (struct Segment *)malloc(sizeof(struct Segment));

	newseg->previous = current;
	newseg->next = NULL;

	newseg->pos.x = x;
	newseg->pos.y = y;
	newseg->vel.x = xv;
	newseg->vel.y = yv;
	newseg->acc.x = xa;
	newseg->acc.y = ya;
	newseg->mass = mass;
	newseg->boxcolor = int(128 + 63 * (mass - MIN_MASS) / (MAX_MASS - MIN_MASS));
}

void instructions_exit(void) {
	cout << "\n -- Wave Simulator -- \n" << endl;
	cout << "Accepted arguments:" << endl;
	cout << "-s n     n is number of segments (default 20, valid range "
		<< MIN_SEGS << "-" << MAX_SEGS << ")" << endl;
	cout << "-m n     n is mass of each segment (default 30, valid range "
		<< MIN_MASS << "-" << MAX_MASS << ")" << endl;
	cout << "-am n    n is alternate mass for segments; if specified," << endl;
	cout << "         the first half of the segments will be the mass from -m," << endl;
	cout << "         and the second half will be as specified by -am. (same valid range)" << endl;
	cout << "-h       high resolution: 640x480 (default disbled - 320x200)." << endl;
	exit(0);
}
