#!/bin/bash

########################################################################
#
#          U P T I M E N O T I F I E R C L I E N T
#
#   A shell script client for use with UptimeNotifier
#
#   For options: UptimeNotifierClient
#
#   This UptimeNotifier client uses wget (http://www.gnu.org) to perform
#         http GET requests to the NotificationServer.  This means that
#         the event notification message is passed in the URL
#      SSL would not be useful when using wget
#
#   If the lynx browser (http://lynx.browser.org is specified, http POSTs
#   will be used to send event notifications
#
#   The awk and sed utilities need to be installed on your system.
#
#   Exit Codes
#   0: successful notification
#   1: unsuccessful notification
#   2: unable to locate httpPoster
#   3: incorrect parameters
#   6: unable to locate notification server
#
########################################################################
#
#   Copyright (C) 2000-2002  UptimeNotifier
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
########################################################################


########################################################################
#  Initialize variables
########################################################################

PN=`basename "$0"`
VER=`echo '$Revision: 1.7 $' | cut -d' ' -f2`
export DONE=1;
TMPFILE=/tmp/servers;
TMPRESPFILE=/tmp/resps;
TMPPROPS=/tmp/props;
CFGFILE=notifier.properties
VERBOSE=1
SILENT=1
LYNX=lynx
WGET=wget
WGETEXEC="$(which $WGET)"
WGETPARMS='-q -O'
LYNXGET='-get_data'
LYNXEXEC="$(which $LYNX)"
LYNXPOST='-post_data'
SRVSUCCESS='^[24]0[123]'


########################################################################
#
#                    C L E A N U P
#
#  Cleans up temporary files.  Called at normal termination and when
#  INT, TERM, HUP, and QUIT signals are encountered.
########################################################################

cleanup() {

  [[ -a $TMPFILE$$ ]] && rm -f $TMPFILE$$
  [[ -a $TMPRESPFILE$$ ]] && rm -f $TMPRESPFILE$$
  [[ -a $TMPPROPS$$ ]] && rm -f $TMPPROPS$$
  exit $DONE
}


########################################################################
#
#                      U S A G E
#
#  Prints a descriptive message on howto use this script
########################################################################

Usage () {

printf "$PN - sends UptimeNotifier event notifications, ver $VER\n\n"
printf "  -r <request type>   :  the event notification request type\n"
printf "  -a <accountID>      :  the UptimeNotifier accountID\n"
printf "  -p <password>       :  the UptimeNotifier account password\n\n"
printf "  -w <httpPoster>     :  the httpPoster utility (wget or lynx)\n"
printf "where options are:\n"
printf "  -f <properties file>:  location of properties file\n"
printf "  -u <user ID>        :  the UptimeNotifier userID\n"
printf "  -t <event ID>       :  the UptimeNotifier event id\n"
#printf "  -e <recipient ID>   :  the recipient id\n"
#printf "  -s <service ID>     :  the service id\n"
#printf "  -n <reference ID>   :  the reference id\n"
printf "  -m <message>        :  a notification message, e.g., \" Notify\"\n"
printf "  -v                  :  display more verbose output\n"
printf "  -q                  :  display less verbose output\n"
printf "  -D <directory>      :  the URI of the server directory\n"
printf "  -S <server>         :  the URI of the server to use\n"
printf "  -h                  :  prints this message\n"
}


########################################################################
#
#                 U R L E N C O D E
#
#  Formats the event notification message into url-encoded format.
#  Borrowed from Heiner's Shelldorado http://oase-shareware.org/shell/
########################################################################

urlencode () {

FORMATTEDMSG=`echo $MESSAGE | awk '
    BEGIN {
        EOL = "%0A"             # "end of line" string (encoded)
        split ("1 2 3 4 5 6 7 8 9 A B C D E F", hextab, " ")
        hextab [0] = 0
        for ( i=1; i<=255; ++i ) ord [ sprintf ("%c", i) "" ] = i + 0
        if ("'"$EncodeEOL"'" == "yes") EncodeEOL = 1; else EncodeEOL = 0
    }
    {
        encoded = ""
        for ( i=1; i<=length ($0); ++i ) {
            c = substr ($0, i, 1)
            if ( c ~ /[a-zA-Z0-9.-]/ ) {
                encoded = encoded c             # safe character
            } else if ( c == " " ) {
                encoded = encoded "+"   # special handling
            } else {
                # unsafe character, encode it as a two-digit hex-number
                lo = ord [c] % 16
                hi = int (ord [c] / 16);
                encoded = encoded "%" hextab [hi] hextab [lo]
            }
        }
        if ( EncodeEOL ) {
            printf ("%s", encoded EOL)
        } else {
            print encoded
        }
    }
    END {
        #if ( EncodeEOL ) print ""
    }
'`
}

#######################################################################
#
#               U P D A T E P R O P E R T I E S
#
#  Read the properties file and update properties as necessary.  If
#  a property is specified on the command-line, no action is taken.
#  A property listed in the properties file will only be used if it
#  was not coded on the command-line.
#######################################################################

updateproperties () {

##
#  If the configuration file is readable, use it
##
if [ -r $CFGFILE ]
then

  ##
  #  Filter out comments, blank lines, and empty lines and write remaining
  #  properties to a temp file. Modify permissions on the temp file to lessen
  #  chances of private data from being viewed
  ##
  sed -e '/^#/d' -e '/^ *$/d' -e '/^$/d' -e '/^ /d' $CFGFILE > $TMPPROPS$$
  chmod 600 $TMPPROPS$$

  ##
  #  Read the filtered properties, line by line, and apply as required
  ##
  while read CFGLINE
  do

    VALUE=$(echo $CFGLINE | cut -d'=' -f2)
    KEY=$(echo $CFGLINE | cut -d'=' -f1)

    ##
    #  Assign KEY property only if it is empty (it was not assigned on
    #  the command-line
    ##
    case $KEY in

      message)
              [[ -z $MESSAGE ]] && MESSAGE=$VALUE;;
      accountID)
              [[ -z $ACCOUNTID ]] && ACCOUNTID=$VALUE;;
      password)
              [[ -z $PASSWORD ]] && PASSWORD=$VALUE;;
      userID)
              [[ -z $USERID ]] && USERID=$VALUE;;
      eventID)
              [[ -z $EVENTID ]] && EVENTID=$VALUE;;
      request)
              [[ -z $REQUESTTYPE ]] && REQUESTTYPE=$VALUE;;
#      recipientID)
#              [[ -z $RECIPIENTID ]] && RECIPIENTID=$VALUE;;
#      serviceID)
#              [[ -z $SERVICEID ]] && SERVICEID=$VALUE;;
#      referenceID)
#              [[ -z $REFERENCEID ]]  && REFERENCEID=$VALUE;;
      defaultServerAddress)
              [[ -z $DEFSERVER ]]  && DEFSERVER=$VALUE;;
      verbose)
              [[ -z $VERBOSE ]]  && VERBOSE=0;;
      silent)
              [[ -z $SILENT ]]  && SILENT=0;;
      directoryAddress)
              [[ -z $DIRECTORY ]]  && DIRECTORY=$VALUE;;
      httpPoster)
              [[ -z $HTTPPOSTER ]] &&  HTTPPOSTER=$VALUE;;
    esac

  done < $TMPPROPS$$

else
  [[ $VERBOSE -eq 0 ]] && printf "Cannot open %s for reading...\n" $CFGFILE

fi
}


verifyposter () {
########################################################################
# Check for existence of the httpPoster utility.  Execution terminates
# if the http utility cannot be located.
########################################################################

##
# Test for existence of HTTPPOSTER property
# First check to see that it was specified, if not, get out now
##
if [ -z $HTTPPOSTER ]
then

  echo "httpPoster not specified (wget or lynx needed)" && exit 2
else

  ##
  # Check that the specified utility is available on this system
  ##
  type $HTTPPOSTER > /dev/null 2>&1

  if [ $? -ne 0 ]
  then

    printf "%s Cannot be located\n" "$HTTPPOSTER"
    Usage
    exit 2
  fi
fi
}


########################################################################
# Set up to call cleanup when INT, TERM, HUP, and QUIT signals received
########################################################################

trap cleanup INT TERM HUP QUIT


########################################################################
#  Process command line options
########################################################################

#while getopts hf:e:a:u:r:m:p:s:n:w:t:vqD:S: opt
while getopts hf:a:u:r:m:p:w:t:vqD:S: opt
do
  case $opt in
    f) CFGFILE=$OPTARG;;
    a) ACCOUNTID=$OPTARG;;
    u) USERID=$OPTARG;;
#    e) RECIPIENTID=$OPTARG;;
    r) REQUESTTYPE=$OPTARG;;
    m) MESSAGE=$OPTARG;;
    p) PASSWORD=$OPTARG;;
#    s) SERVICEID=$OPTARG;;
#    n) REFERENCEID=$OPTARG;;
    t) EVENTID=$OPTARG;;
    v) VERBOSE=0;;
    q) SILENT=0;;
    w) HTTPPOSTER=$OPTARG;;
    D) DIRECTORY=$OPTARG;;
    S) DEFSERVER=$OPTARG;;
    h) Usage;
       exit 0;;
  esac
done
shift $((OPTIND - 1))


########################################################################
# Synchronize properties, command-line takes precedence
########################################################################

updateproperties

verifyposter

########################################################################
# Get out now if verbose and silent are specified together
########################################################################

if [ $VERBOSE -eq 0 -a $SILENT -eq 0 ]
then
  echo "Verbose and Quiet modes cannot be specified together";
  exit 3;
fi


########################################################################
# Display properties -- sometimes useful as a debugging aid when event
#    notifications don't work as planned.
########################################################################

if [ $VERBOSE -eq 0 ]
then
  printf "CFGFILE    : %s\n" $CFGFILE
  printf "MESSAGE    : %s\n" "$MESSAGE"
  printf "ACCOUNTID  : %s\n" $ACCOUNTID
  printf "PASSWORD   : %s\n" $PASSWORD
  printf "USERID     : %s\n" $USERID
  printf "EVENTID    : %s\n" $EVENTID
  printf "REQUESTTYPE: %s\n" $REQUESTTYPE
#  printf "RECIPIENTID: %s\n" $RECIPIENTID
#  printf "SERVICEID  : %s\n" $SERVICEID
#  printf "REFERENCEID: %s\n" $REFERENCEID
  printf "VERBOSE    : %s\n" $VERBOSE
  printf "SILENT     : %s\n" $SILENT
  printf "DIRECTORY  : %s\n" $DIRECTORY
  printf "DEFSERVER  : %s\n" $DEFSERVER
  printf "HTTPPOSTER : %s\n" $HTTPPOSTER
fi

########################################################################
# URL-encode the event notification message
########################################################################

urlencode


########################################################################
# Format the event notification string for the NotificationServer
# Each request type has specific inclusion requirements.  The common
# parameters are handled by prepending request_type, accountID, and
# password.
########################################################################

case $REQUESTTYPE in
  P)
    NOTIFICATION=\&eventID=$EVENTID;;
  G)
    NOTIFICATION=\&userID=$USERID\&message=$FORMATTEDMSG;;
#  T)
#    NOTIFICATION=\&recipientID=$RECIPIENTID\&serviceID=$SERVICEID;
#    NOTIFICATION=$NOTIFICATION\&message=$MESSAGE;;
#  C)
#    NOTIFICATION=\&referenceID=$REFERENCEID;;
  *)
    printf "Request Type '$REQUESTTYPE' is invalid.\n"
    exit 3;;
esac


##
#  Prepend common parameters
##
NOTIFICATION=\&accountID=$ACCOUNTID\&password=$PASSWORD$NOTIFICATION
NOTIFICATION=request=$REQUESTTYPE$NOTIFICATION

[[ $VERBOSE -eq 0 ]] && echo $NOTIFICATION


########################################################################
# Get list of NotificationServers to try.  If a directory is coded,
# contact it to get a list of servers.  When no directory is coded,
# assign the default NotificationServer.
########################################################################

if [ -z $DIRECTORY ]
then
  if [ -z $DEFSERVER ]
  then
    echo "No server to contact" && exit 6
  else
    echo $DEFSERVER > $TMPFILE$$
  fi

else
  ##
  # Use the httpPoster utility specified to talk to the
  # NotificationDirectory.  Put the list of NotificationServers into
  # a temp file
  ##
  case "$HTTPPOSTER" in
    $LYNX)
      echo | $LYNXEXEC $LYNXGET $DIRECTORY | grep http > $TMPFILE$$;;
    $WGET)
      $WGETEXEC $DIRECTORY $WGETPARMS $TMPFILE$$;;
  esac
fi


########################################################################
# Send the event notification.  Servers will be tried one at a time until
# either a notification is successful or the list of NotificationServers
# is exhausted.
########################################################################

while read SERVER
do
  ##
  # if we are done, get out early and exit
  ##
  [[ $DONE -eq 0 ]] && break;

  ##
  #  Report the name of each server being tried, if requested
  ##
  [[ $VERBOSE -eq 0 ]] && echo $SERVER

  ##
  # Send the event notification.  wget uses http GET, lynx uses
  # http POST.
  ##
  case $HTTPPOSTER in
    $LYNX)
      echo $NOTIFICATION | $LYNXEXEC $LYNXPOST $SERVER > $TMPRESPFILE$$;;
    $WGET)
      $WGETEXEC $SERVER\?$NOTIFICATION $WGETPARMS $TMPRESPFILE$$
  esac

  ##
  # Check our notification server return code
  ##
  grep $SRVSUCCESS $TMPRESPFILE$$ > /dev/null 2>&1
  RETCODE=$?

  ##
  # Set a flag to allow us to break out of the loop when the event
  # notification was successful
  ##
  if [ $RETCODE -eq 0 ]
  then
    DONE=0
    cat $TMPRESPFILE$$
  else
    if [ $VERBOSE -eq 0 ]
    then
      cat $TMPRESPFILE$$
    fi
    echo "No reasonable return code found."
  fi

done  < $TMPFILE$$


########################################################################
# Clean up temp files that were created
########################################################################

cleanup
