/*****************************************************************************
*   "Irit" - the 3d polygonal solid modeller.				     *
*									     *
* Written by:  Gershon Elber				Ver 0.2, Mar. 1990   *
******************************************************************************
*   General, local to module, definitions for the Input Parser module.	     *
*   Note this module actually consists of InptPrsr/InptEval/OverLoad modules.*
*****************************************************************************/

#ifndef	INPT_PRSR_LH
#define	INPT_PRSR_LH

/* #define DEBUG        Print some intermediate results (InptPrsr/InptEval). */

#define MAX_PARSER_STACK	200	     /* Depth of expression nesting. */

typedef enum {
    POLY_EXPR =     0x0001,
    NUMERIC_EXPR =  0x0002,
    VECTOR_EXPR	=   0x0004,
    MATRIX_EXPR	=   0x0008,
    CURVE_EXPR =    0x0010,
    SURFACE_EXPR =  0x0020,
    STRING_EXPR	=   0x0040,
    OLST_EXPR =     0x0080,
    CTLPT_EXPR =    0x0100,

    NO_EXPR =       0x1000,
    POLY_CURVE_EXPR = POLY_EXPR | CURVE_EXPR,
    GEOM_EXPR =     POLY_EXPR | CURVE_EXPR | SURFACE_EXPR,
    OLST_GEOM_EXPR = OLST_EXPR | GEOM_EXPR,
    ANY_EXPR =      POLY_EXPR | NUMERIC_EXPR | VECTOR_EXPR |
    		    MATRIX_EXPR | CURVE_EXPR | SURFACE_EXPR |
		    STRING_EXPR | OLST_EXPR | CTLPT_EXPR,

    ERROR_EXPR =    0x4000
} IritExprType;


extern InptPrsrEvalErrType IPGlblEvalError;	 /* Global used by EvalTree. */

/*****************************************************************************
* The expression parse tree node definition:				     *
*****************************************************************************/
typedef	struct ParseTree {
    struct ParseTree *Right, *Left;
    int NodeKind;
    IritObjectType ObjType;
    union {
	RealType R;
	ObjectStruct *PObj;
    } U;
} ParseTree;

/* See Irit.h file for the different object possible: */
#define IS_POLY_NODE(Node)	((Node)->ObjType == POLY_OBJ)
#define IS_NUM_NODE(Node)	((Node)->ObjType == NUMERIC_OBJ)
#define IS_VEC_NODE(Node)	((Node)->ObjType == VECTOR_OBJ)
#define IS_CTLPT_NODE(Node)	((Node)->ObjType == CTLPT_OBJ)
#define IS_MAT_NODE(Node)	((Node)->ObjType == MATRIX_OBJ)
#define IS_STR_NODE(Node)	((Node)->ObjType == STRING_OBJ)
#define IS_OLST_NODE(Node)	((Node)->ObjType == OBJ_LIST_OBJ)
#define IS_CRV_NODE(Node)	((Node)->ObjType == CURVE_OBJ)
#define IS_SRF_NODE(Node)	((Node)->ObjType == SURFACE_OBJ)

/*****************************************************************************
* The include file stack - nesting is allowed up to FILE_STACK_SIZE.	     *
*****************************************************************************/
typedef struct FileStackStruct {
    char Name[FILE_NAME_LEN];
    FILE *f;
} FileStackStruct;

#define FILE_STACK_SIZE	10

/*****************************************************************************
* Aliases are simple strings substitution - each entry holds alias Name and  *
* alias Value, which replaces Name. Name id not NULL if active.		     *
*****************************************************************************/
typedef struct OneAliasStruct {
    char *Name, *Value;
} OneAliasStruct;

#define NUM_OF_ALIASES	10

typedef struct AliasesStruct {
    OneAliasStruct Aliases[NUM_OF_ALIASES];
} AliasesStruct;

/*****************************************************************************
* Function entry table looks like this (for table see InptPrsr.c module):    *
*****************************************************************************/
#define FUNC_NAME_LEN	11			    /* 10 + NULL terminator. */
#define FUNC_MAX_PARAM	4	    /* Max. of all the function types below. */
#define ANY_PARAM_NUM	127

typedef struct FuncTableType {
    char FuncName[FUNC_NAME_LEN];
    int FuncToken;
    void (*Func)();
    ByteType NumOfParam;
    IritExprType ParamObjType[FUNC_MAX_PARAM];
} FuncTableType;

typedef struct NumFuncTableType {
    char FuncName[FUNC_NAME_LEN];
    int FuncToken;
#ifdef __MSDOS__
    double cdecl (*Func)(); /* So we can use -pr option of Borland C++ 3.0. */
#else
    double (*Func)();
#endif /* __MSDOS__ */
    ByteType NumOfParam;
    IritExprType ParamObjType[FUNC_MAX_PARAM];
} NumFuncTableType;

typedef struct ObjFuncTableType {
    char FuncName[FUNC_NAME_LEN];
    int FuncToken;
    ObjectStruct *(*Func)();
    ByteType NumOfParam;
    IritExprType ParamObjType[FUNC_MAX_PARAM];
} ObjFuncTableType;

typedef struct GenFuncTableType {
    char FuncName[FUNC_NAME_LEN];
    int FuncToken;
    void (*Func)();
    ByteType NumOfParam;
    IritExprType ParamObjType[FUNC_MAX_PARAM];
} GenFuncTableType;

typedef struct ConstantTableType {
    char FuncName[FUNC_NAME_LEN];
    RealType Value;
} ConstantTableType;

/* The followings are defined in the InptEval.c module and are globals so   */
/* InptPrsr.c module will be able to access them...			    */
extern NumFuncTableType NumFuncTable[];
extern int NumFuncTableSize;
extern ObjFuncTableType ObjFuncTable[];
extern int ObjFuncTableSize;
extern GenFuncTableType GenFuncTable[];
extern int GenFuncTableSize;
extern ConstantTableType ConstantTable[];
extern int ConstantTableSize;

/*****************************************************************************
* Tokens used in the expression	to tree	conversion and tree definition.	     *
*****************************************************************************/

#define	TOKENERROR  0

/* Warning - changing the order of these constants, needs updating the order */
/* of them, in the tables in the begining of InptPrsr.c & OverLoad.c modules.*/

#define	ARCCOS	    100			   /* Real value returned functions. */
#define	ARCSIN	    101
#define	ARCTAN2	    102
#define	ARCTAN	    103
#define	COS	    104
#define	EXP	    105
#define	FABS	    106
#define	LN	    107
#define	LOG	    108
#define	SIN	    109
#define	SQRT	    110
#define	TAN	    111
#define CPOLY	    112
#define AREA	    113
#define VOLUME      114
#define TIME	    115

#define NUM_FUNC    100
#define NUM_FUNC_OFFSET	100
#define IS_NUM_FUNCTION(Token)		(Token >= 100 && Token < 200)

#define VECTOR	    200			       /* Object returned Functions. */
#define CTLPT	    201
#define ROTX	    202
#define ROTY	    203
#define ROTZ	    204
#define TRANS	    205
#define SCALE	    206
#define BOX	    207
#define GBOX	    208
#define CONE	    209
#define CONE2	    210
#define CYLIN	    211
#define SPHERE	    212
#define TORUS	    213
#define CIRCPOLY    214
#define POLY	    215
#define CROSSEC	    216
#define SURFREV     217
#define EXTRUDE     218
#define LIST	    219
#define LOAD	    220
#define CONVEX	    221
#define SBEZIER     222
#define CBEZIER     223
#define SBSPLINE    224
#define CBSPLINE    225
#define SEVAL       226
#define CEVAL       227
#define STANGENT    228
#define CTANGENT    229
#define SNORMAL     230
#define SDIVIDE	    231
#define CDIVIDE     232
#define SREGION	    233
#define CREGION     234
#define SREFINE     235
#define CREFINE     236
#define SRAISE      237
#define CRAISE      238
#define CSURFACE    239
#define CMESH       240
#define NTH         241
#define GPOLYGON    242
#define GPOLYLINE   243
#define CIRCLE      244
#define ARC         245
#define RULEDSRF    246
#define BOOLSUM     247
#define SFROMCRVS   248
#define SWEEPSRF    249
#define OFFSET      250
#define CEDITPT     251
#define SEDITPT     252
#define MERGEPOLY   253

#define OBJ_FUNC    200
#define OBJ_FUNC_OFFSET	200
#define IS_OBJ_FUNCTION(Token)		(Token >= 200 && Token < 300)

#define EXIT	    300	   /* General Functions/No value returned functions. */
#define VIEW	    301
#define DIR	    302
#define CHDIR	    303
#define NORMAL	    304
#define INCLUDE	    305
#define SAVE        306
#define FREEOBJ     307
#define INTERACT    308
#define PAUSE	    309
#define IFCOND	    310
#define FORLOOP	    311
#define PRHELP	    312
#define VARLIST	    313
#define ALIAS	    314
#define BEEP	    315
#define EDIT	    316
#define SYSTEM	    317
#define LOGFILE	    318
#define COLOR	    319
#define SNOC        320
#define ATTRIB	    321
#define CLOSED	    322

#define COMMENT	    399

#define GEN_FUNC    300
#define GEN_FUNC_OFFSET	300
#define IS_GEN_FUNCTION(Token)		(Token >= 300 && Token < 400)

#define IS_FUNCTION(Token)              (Token >= 100 && Token < 400)
#define IS_NO_PARAM_FUNC(Token)		(Token == EXIT || Token == SYSTEM || \
					 Token == VARLIST)

#define	PLUS	    400					       /* Operators. */
#define	MINUS	    401
#define	MULT	    402
#define	DIV	    403
#define	POWER	    404
#define	UNARMINUS   405
#define EQUAL	    406
#define COMMA	    407
#define COLON	    408
#define SEMICOLON   409

#define	OPENPARA    430					     /* Paranthesis. */
#define	CLOSPARA    431

#define	NUMBER	    450					    /* Numeric Data. */
#define	PARAMETER   451				 /* Point on new/old object. */
#define STRING	    452	     /* Sequence of characters within double quotes. */

#define TOKENSTART  490
#define TOKENEND    491

#define OPERATORS   400
#define OPERATORS_OFFSET	400

/*****************************************************************************
*   The local function (static) prototypes:				     *
*   Note that if DEBUG is defined for the preprocessor, few more function    *
* become available:							     *
*   Also note that some of the routines are defined globals as both the      *
* InptPrsr.c and InptEval.c modules needs them.				     *
*****************************************************************************/
/* Main parser routine (operator preceedence): Input stream to bin-tree. */
ParseTree *MyExprMalloc(void);
void MyExprFree(ParseTree *Ptr);
void UpdateCharError(char *StrMsg, int Token);
void AliasEdit(char *Name, char *Value);
void AliasExpand(char *Line);
void FileInclude(char *FileName);
IritExprType InptPrsrTypeCheck(ParseTree *Root, int Level);   /* Type check. */
ParseTree *InptPrsrEvalTree(ParseTree *Root, int Level);   /* Evaluate tree. */
void InptPrsrFreeTree(ParseTree *Root);			   /* Free all tree. */
void InptPrsrPrintTree(ParseTree *Root, char *str);	      /* print it... */
ParseTree *InptPrsrCopyTree(ParseTree *Root);			 /* Copy it. */

#endif	/* INPT_PRSR_LH */
