;****************************************************************************
;*
;* $ vfdb_se.asm 24/08/99 21:34 $
;* VFlat SE drivers database. Revision 1.0
;*
;* Copyright (C) 1999  Dmitry Uvarov <mit@pc2o205a.pnpi.spb.ru>
;*
;* This program is free software; you can redistribute it and/or modify
;* it under the terms of the GNU General Public License as published by
;* the Free Software Foundation; either version 2 of the License, or
;* (at your option) any later version.
;*
;* This program is distributed in the hope that it will be useful,
;* but WITHOUT ANY WARRANTY; without even the implied warranty of
;* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;* GNU General Public License for more details.
;*
;* You should have received a copy of the GNU General Public License
;* along with this program; if not, write to the Free Software
;* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;*
;**************************************************************************

MAGIC_ID = 11223344h

MACRO ProcStart
   @@_start:
ENDM

MACRO ProcEnd
   @@_end:
   ;; place procedure end flag
   dd MAGIC_ID
   ;; calculate code block size
   dd @@_end - @@_start
ENDM

MACRO RegisterDriver name
   PUBLIC name
   proc name
   ProcStart
ENDM

MACRO DriverEnd
   ProcEnd
   endp
ENDM

Code_JB = 820Fh
Code_JAE = 830Fh

CODESEG
;--------------------------------------------------------------------------;
; Start of main code for 64K handler                                       ;
;--------------------------------------------------------------------------;
proc Int14MainHndStart_64K
      ProcStart

      push    edx

      mov     edx,cr2                 ; load page fault address from CR2
      ; check if fault was in our vflat surface
      cmp     edx,VFLAT_START
_64_Put_JB_Offset:
      jb      @@dude1
      cmp     edx,VFLAT_END
_64_Put_JAE_Offset:
      jae     @@dude2

      ; now we know that this is our page fault, so handle it...
      pushad               ; save all dwords registers

      ; calculate bank number we need to swich to
      mov     eax,edx         ; edx=linear address of page fault
      shr     edx,16          ; edx=bank number
      shr     eax,10          ; calculate page number in eax
      and     edx,0FFh        ; mask bank number to 0-255
      and     eax,03FC0h      ; eax=page number aligned on bank boundary
                              ;    and multiplyed by four.
                              ;    i.e. eax=starting page offset in page table

      ; because we running in flat model and exeption was in our program,
      ; we can be sure that DS register is already loaded with program's
      ; data selector, so we can freely access our data

      mov     edi,[VFlatPageTablePtr]  ; load pointer to our page table
      mov     esi,[LastPage]
      add     edi,eax                  ; edi=absolute address of pages that
                                       ;     must be enabled
      ; disable old pages
offs = 0
REPT    16
         and  [DWORD esi+offs],00FFFFFFFEh     ; disable old pages
offs = offs+4
ENDM
      ; now enable new pages
offs = 0
REPT    16
          or  [DWORD edi+offs],0000000001h ; enable pages
offs = offs+4
ENDM
      mov     [LastPage],edi              ; store pointer to last mapped pages

      ; because we modify page table we must flush page cache
      mov     eax,cr3
      mov     cr3,eax    ; flush TLB cache

      ProcEnd
@@dude1:
@@dude2:
endp Int14MainHndStart_64K

;--------------------------------------------------------------------------;
; Start of main code for 4K handler                                       ;
;--------------------------------------------------------------------------;
proc Int14MainHndStart_4K
      ProcStart

      push    edx

      mov     edx,cr2                 ; load page fault address from CR2
      ; check if fault was in our vflat surface
      sub     edx,VFLAT_START
_4_Put_JL_Offset:
      jl      @@dude1
      cmp     edx,VFLAT_END - VFLAT_START
_4_Put_JAE_Offset:
      jae     @@dude2

      ; now we know that this is our page fault, so handle it...
      pushad               ; save all dwords registers

      ; calculate bank number we need to swich to
      shr     edx,12          ; edx = bank number

      ; because we running in flat model and exeption was in our program,
      ; we can be sure that DS register is already loaded with program's
      ; data selector, so we can freely access our data

      mov     edi,[VFlatPageTablePtr]  ; load pointer to our page table
      mov     esi,[LastPage]           ; load pointer to last mapped page
      lea     edi,[edi+edx*4]          ; edi=absolute address of page that
                                       ;     must be enabled

      and  [DWORD esi],00FFFFFFFEh     ; disable old page
      or   [DWORD edi],0000000001h     ; enable new page

      mov     [LastPage],edi              ; store pointer to last mapped page

      ; because we modify page table we must flush page cache
      mov     eax,cr3
      mov     cr3,eax    ; flush TLB cache

      ProcEnd
@@dude1:
@@dude2:
endp Int14MainHndStart_4K

; Calculate offsets needed for code modifying
Hnd_64_Put_JB_Offset  = offset _64_Put_JB_Offset - offset Int14MainHndStart_64K
Hnd_64_Put_JAE_Offset = offset _64_Put_JAE_Offset - offset Int14MainHndStart_64K

Hnd_4_Put_JL_Offset  = offset _4_Put_JL_Offset - offset Int14MainHndStart_4K
Hnd_4_Put_JAE_Offset = offset _4_Put_JAE_Offset - offset Int14MainHndStart_4K


;--------------------------------------------------------------------------;
; ending handler routine
;--------------------------------------------------------------------------;
proc Int14MainHndEnd
      ProcStart
      popad              ; restore all dword registers
      pop     edx        ; pop saved edx from stack
      add     esp,4      ; pop error code from stack, other registers stay
                         ;   unmodifyed
      iretd              ; interrupt return

_Put_Old_Hnd_Call:
      pop     edx        ; remove edx from stack
      jmp     [Int14CallGate] ; chain to old handler throught 386 callgate.
                              ; Leaving all parameters in stack
      ProcEnd
endp

Hnd_Put_Old_Hnd_Call = offset _Put_Old_Hnd_Call - offset Int14MainHndEnd

; Drivers for bank switching.
; Bank number is passed in edx register. All other registers can be modifyed

RegisterDriver HND_Int10
; swich bank using int10h. Under VBE 1.2 there is no other way
; of swiching banks.
; FIXME: Try to use direct 386 to 286 call to bank switching code.
; Int10h Input:   BX = window number ( always use window 0 )
;                 DX = bank number   ( already calculated )
;                 AX = function nuber ( function 04F05h - bank swich )
      xor ebx,ebx          ; window 0
      mov eax,04F05h       ; VESA switch bank function
      int 10h
DriverEnd


RegisterDriver HND_S3
; Driver for switching code on S3 chipsets.
      mov ah,dl
      mov dx,03D4h
      mov al,6Ah
      out dx,ax
DriverEnd


RegisterDriver HND_Trident
; Trident driver
      mov al,dl
      mov dx,03D8h
      out dx,al
DriverEnd


RegisterDriver HND_TSeng
; TSeng driver
      mov ax,dx
      shl dx,4
      and ax,0Fh
      or  ax,dx
      and dx,300h
      mov cx,dx
      shr dx,4
      or  cx,dx
      mov dx,03CDh
      shr cx,4
      out dx,al
      mov al,cl
      mov dx,03CBh
      out dx,al
DriverEnd


RegisterDriver HND_SiS
; SiS driver
      mov al,dl
      mov dx,03CBh
      out dx,al
      mov dx,03CDh
      out dx,al
DriverEnd


RegisterDriver HND_Cirrus
; Cirrus driver
      mov ah,dl
      shl ah,2
      mov al,09h
      mov dx,03CEh
      out dx,ax
DriverEnd


RegisterDriver HND_ATI
; ATI Mach 8 driver. I belive they have downwards compatibility, so it will
; work on Mach32 and Mach64 too.
      mov ah,dl
      mov dx,01CEh
      mov al,0B2h
      out dx,al
      inc dx
      in  al,dx
      shl ah,1
      and al,0E1h
      and ah,1Eh
      or  ah,al
      dec dx
      mov al,0B2h
      out dx,ax
DriverEnd


RegisterDriver HND_ARK
; ARK driver
      mov ah,dl
      mov al,15h
      mov dx,03C4h
      out dx,ax
      inc al
      out dx,ax
DriverEnd



; Function to return driver size.
; Input:  esi = driver pointer
; Output: eax = driver size
; All other registers, including esi, are preserved
proc CalcDriverSize
      push esi
      dec esi
      mov eax,-1
   @@checkit:
      inc esi
      inc eax
      cmp [dword ptr esi],MAGIC_ID
      jne @@checkit
      pop esi
      ret
endp


; dword GetHandlerSize(void *driver, dword banksize)
PUBLIC GetHandlerSize
proc GetHandlerSize
    ARG driver:dword, banksize:dword

    push edx
    mov esi,offset Int14MainHndStart_64K
    cmp [byte ptr banksize],4
    jne @@calcit
      mov esi,offset Int14MainHndStart_4K
@@calcit:
    call CalcDriverSize
    mov edx,eax
    mov esi,offset Int14MainHndEnd
    call CalcDriverSize
    add edx,eax
    mov esi,[driver]
    call CalcDriverSize
    add eax,edx
    pop edx
    ret
endp


; void BuildHandler(void *driver, dword banksize, void *dst)
PUBLIC BuildHandler
proc BuildHandler
    ARG driver:dword,banksize:dword,dst:dword

    pushad

    cld
    mov edi,[dst]
    ; first copy first part of handler
    mov esi,offset Int14MainHndStart_64K
    cmp [byte ptr banksize],4
    jne @@makeit
       mov esi,offset Int14MainHndStart_4K
@@makeit:
    call CalcDriverSize
    mov ecx,eax
    rep movsb        ; store first handler part

    mov ebx,eax       ; ebx = size of first handler part

    ; now modify offsets for jumps
    ; first calculate size of driver function
    mov esi,[driver]
    call CalcDriverSize
    mov ecx,eax
    add ebx,eax  ; add size of driver to the first part code size

    mov eax,[dst]

    cmp [byte ptr banksize],4
    je @@make4K
    ; Jump address calculation:
    ; ebx - Hnd_64_Put_JB_Offset = size of code from Put_JB label to the
    ;                              end of first handler part plus driver
    ; + Hnd_Put_Old_Hnd_Call     = plus offset for jump in the ending part
    ; - 6                        = substract instruction size
    lea edx, [ebx - Hnd_64_Put_JB_Offset + Hnd_Put_Old_Hnd_Call - 6] ; edx = new jump address
    ; +2 skips jump instruction code
    mov [eax+Hnd_64_Put_JB_Offset+2],edx
    lea edx, [ebx - Hnd_64_Put_JAE_Offset + Hnd_Put_Old_Hnd_Call - 6] ; edx = new jump address
    mov [eax+Hnd_64_Put_JAE_Offset+2],edx
    jmp @@done

@@make4K:
    ; modify offsets for 4K handler
    lea edx, [ebx - Hnd_4_Put_JL_Offset + Hnd_Put_Old_Hnd_Call - 6] ; edx = new jump address
    ; +2 skips jump instruction code
    mov [eax+Hnd_4_Put_JL_Offset+2],edx
    lea edx, [ebx - Hnd_64_Put_JAE_Offset + Hnd_Put_Old_Hnd_Call - 6] ; edx = new jump address
    mov [eax+Hnd_4_Put_JAE_Offset+2],edx


@@done:
    rep movsb        ; store driver inside buffer
    mov esi,offset Int14MainHndEnd
    call CalcDriverSize
    mov ecx,eax
    rep movsb           ; store ending part

    popad

    ret
endp
