#include <stdlib.h>
#include <stdio.h>

#include "dither.h"

typedef unsigned char byte;

/* 8 bit maps */
int graymap::alloc(int width,int height) {
  this->width=width,this->height=height;
  data=(byte *)malloc(width*height);
  return data==0;
}

void graymap::free() {
  if(data) ::free(data);
  width=height=0;
  data=0;
}

int graymap::export_pgm(FILE *f) {
  fprintf(f,"P5\n%d %d\n255\n",width,height);
  return 1==fwrite(data,width*height,1,f);
}

int graymap::import_pgm(FILE *f) {
  free();
  char line[256];
  int w,h;
  fgets(line,sizeof(line),f);    /* here should be P5 ? */
  if(line[0]!='P'||line[1]!='5') return 1;     /* bad format */
  while((fgets(line,sizeof(line),f))&&line[0]=='#');
  sscanf(line,"%d%d",&width,&height);
  fgets(line,sizeof(line),f);	/* here should be 255 */
  alloc(width,height);
  return 1==fread(data,width*height,1,f);
}

int graymap::export_tga(FILE *f) {
  char tgaheader[18]={0,0,3,0, 0,0,0,0, 0,0,0,0, 0,0,0,0, 0x8,0x20};
  *(unsigned short*)(tgaheader+12)=width;
  *(unsigned short*)(tgaheader+14)=height;
  fwrite(tgaheader,sizeof(tgaheader),1,f);
  return 1==fwrite(data,width*height,1,f);
}

int graymap::import_tga(FILE *f) {
  char tgaheader[18];
  fread(tgaheader,sizeof(tgaheader),1,f);
  if(tgaheader[2]!=3) return 1;
  width=*(unsigned short*)(tgaheader+12);
  height=*(unsigned short*)(tgaheader+14);
  alloc(width,height);
  fseek(f,*(unsigned short*)(tgaheader),SEEK_CUR);
  return 1==fread(data,width*height,1,f);
}

void graymap::dither(int levels) {
  graymap g2;
  g2.alloc(width,height);
  memcpy(g2.data,data,width*height);
  byte *p1,*p2;
  levels--;
  int x,y,r,e,e1,e2;
  int *el=(int*)malloc(sizeof(int)*(width+2)),*ep;
  e1=e2=0;
  for(x=0;x<width+2;x++) el[x]=0;
  for(p1=data,p2=g2.data,y=height;y;y--) {
    ep=el+1;
    e2=ep[0],e1=ep[1],ep[-1]=0,ep[0]=0;
    for(x=width;x;x--,p1++,p2++,ep++) {
      *p1=((*p2+e2)*levels+127)/255;
      e=(*p2+e2)-(*p1)*255/levels;
      ep[-1]+=e/9;
      ep[0]+=e*3/9;
      ep[+1]=e/9;
      e2=e1+e*3/9;
      e1=ep[2];
    }
  }
  ::free(el);
  g2.free();
}

// pseudomaps
int pseudomap::alloc(int width,int height) {
  this->width=width,this->height=height;
  data=(byte *)malloc(width*height);
  palette=(byte *)malloc(768);
  return data==0;
}

void pseudomap::free() {
  if(data) ::free(data);
  if(palette) ::free(palette);
  width=height=0;
  data=0;
}

void pseudomap::set_rgb_palette(int rbase,int rlevels,int gbase,int glevels,int bbase,int blevels) {
  int i;
  for(i=0;i<256;i++) {
    palette[3*i]=(i/rbase%rlevels*255)/(rlevels-1);
    palette[3*i+1]=(i/gbase%glevels*255)/(glevels-1);
    palette[3*i+2]=(i/bbase%blevels*255)/(blevels-1);
  }
}

// himaps

int himap::alloc(int width,int height) {
  this->width=width,this->height=height;
  data=(byte *)malloc(2*width*height);
  return data==0;
}

void himap::free() {
  if(data) ::free(data);
  width=height=0;
  data=0;
}

// rgbmaps

int rgbmap::alloc(int width,int height) {
  this->width=width,this->height=height;
  data=(byte *)malloc(3*width*height);
  return data==0;
}

void rgbmap::free() {
  if(data) ::free(data);
  width=height=0;
  data=0;
}

void rgbmap::swaprgb(int r,int g,int b) {
  byte rgb[3],*h;
  for(h=data+3*width*height;h>data;) {
    h-=3;
    rgb[0]=h[0],rgb[1]=h[1],rgb[2]=h[2];
    h[0]=rgb[r],h[1]=rgb[g],h[2]=rgb[b];
  }
}

int rgbmap::export_ppm(FILE *f) {
  fprintf(f,"P6\n%d %d\n255\n",width,height);
  return 1!=fwrite(data,3*width*height,1,f);
}

int rgbmap::import_ppm(FILE *f) {
  free();
  char line[256];
  int w,h;
  fgets(line,sizeof(line),f);    /* here should be P5 ? */
  if(line[0]!='P'||line[1]!='6') return 1;     /* bad format */
  while((fgets(line,sizeof(line),f))&&line[0]=='#');
  sscanf(line,"%d%d",&width,&height);
  fgets(line,sizeof(line),f);	/* here should be 255 */
  alloc(width,height);
  return 1!=fread(data,3*width*height,1,f);
}

int rgbmap::export_tga(FILE *f) {
  char tgaheader[18]={0,0,2,0, 0,0,0,0, 0,0,0,0, 0,0,0,0, 0x18,0x20};
  *(unsigned short*)(tgaheader+12)=width;
  *(unsigned short*)(tgaheader+14)=height;
  int w=fwrite(tgaheader,sizeof(tgaheader),1,f);
  if(w==1) {
    swaprgb(2,1,0);
    w=fwrite(data,3*width*height,1,f);
    swaprgb(2,1,0);
  }
  return w!=1;
}

int rgbmap::import_tga(FILE *f) {
  char tgaheader[18];
  fread(tgaheader,sizeof(tgaheader),1,f);
  if(tgaheader[2]!=2) return 1;
  width=*(unsigned short*)(tgaheader+12);
  height=*(unsigned short*)(tgaheader+14);
  alloc(width,height);
  fseek(f,*(unsigned short*)(tgaheader),SEEK_CUR);
  int r=fread(data,3*width*height,1,f);
  swaprgb(2,1,0);
  return r!=1;
}

void rgbmap::channel_from(graymap *gm,int offset) {
  byte *h,*g,*he;
  for(h=data+offset,he=h+3*width*height,g=gm->data;h<he;h+=3,g++)
    *h=*g;
}

void rgbmap::channel_to(graymap *gm,int offset) {
  byte *h,*g,*he;
  for(h=data+offset,he=h+3*width*height,g=gm->data;h<he;h+=3,g++)
    *g=*h;
}

void rgbmap::convert_from(pseudomap *pm) {
  free();
  alloc(pm->width,pm->height);
  byte *h,*he,*h2,*hp;
  for(h=pm->data,he=h+width*height,h2=data;h<he;h++,h2+=3) {
    hp=pm->palette+3*(*h);
    h2[0]=hp[0];
    h2[1]=hp[1];
    h2[2]=hp[2];
  }
}

static struct {
  int base,levels;
} rgb8[3]={{1,6},{6,7},{42,6}};

void rgbmap::convert_to(pseudomap *pm) {
  graymap g;
  int i;
  byte *h,*he,*p;
  g.alloc(width,height);
  for(i=0;i<3;i++) {
    channel_to(&g,i);
    g.dither(rgb8[i].levels);
    channel_from(&g,i);
  }
  pm->alloc(width,height);
  for(h=data,he=data+3*width*height,p=pm->data;h<he;h+=3,p++) {
    *p=rgb8[0].base*h[0]+rgb8[1].base*h[1]+rgb8[2].base*h[2];
  }
  pm->set_rgb_palette(rgb8[0].base,rgb8[0].levels,rgb8[1].base,rgb8[1].levels,
    rgb8[2].base,rgb8[2].levels);
  g.free();
}

void rgbmap::dither(int pixel_size,unsigned char *pixels,int rbase,int rlevels,int gbase,int glevels,int bbase,int blevels) {
  int x,y;
  unsigned int c,c2;
  int *m;
  unsigned char *p,*q;
  struct {
    int e,e1,e2,*el,*ep;
  } r,g,b;

  if(!pixels)
    pixels=(byte*)malloc(width*height*pixel_size);

  rlevels--,glevels--,blevels--;
  m=(int*)calloc(3*(width+2),sizeof(int));
  r.el=m;g.el=r.el+width+2;b.el=g.el+width+2;
  r.e1=r.e2=g.e1=g.e2=b.e1=b.e2=0;

  for(p=data,q=pixels,y=height;y;y--) {
    r.ep=r.el+1,r.e2=r.ep[0],r.e1=r.ep[1],r.ep[-1]=0,r.ep[0]=0;
    g.ep=g.el+1,g.e2=g.ep[0],g.e1=g.ep[1],g.ep[-1]=0,g.ep[0]=0;
    b.ep=b.el+1,b.e2=b.ep[0],b.e1=b.ep[1],b.ep[-1]=0,b.ep[0]=0;
    for(x=width;x;x--,r.ep++,g.ep++,b.ep++) {

      c2=((*p+r.e2)*rlevels+127)/255;
      c=rbase*c2;
      r.e=(*p+r.e2)-c2*255/rlevels;
      r.ep[-1]+=r.e/9,r.ep[0]+=r.e*3/9,r.ep[+1]=r.e/9,r.e2=r.e1+r.e*3/9;
      r.e1=r.ep[2];p++;

      c2=((*p+g.e2)*glevels+127)/255;
      c+=gbase*c2;
      g.e=(*p+g.e2)-c2*255/glevels;
      g.ep[-1]+=g.e/9,g.ep[0]+=g.e*3/9,g.ep[+1]=g.e/9,g.e2=g.e1+g.e*3/9;
      g.e1=g.ep[2];p++;

      c2=((*p+b.e2)*blevels+127)/255;
      c+=bbase*c2;
      b.e=(*p+b.e2)-c2*255/blevels;
      b.ep[-1]+=b.e/9,b.ep[0]+=b.e*3/9,b.ep[+1]=b.e/9,b.e2=b.e1+b.e*3/9;
      b.e1=b.ep[2];p++;

      switch(pixel_size) {
       case 4:q[3]=c>>24;
       case 3:q[2]=c>>16;
       case 2:q[1]=c>>8;
       case 1:q[0]=c;break;
      }

      q+=pixel_size;
    }
  }
  ::free(m);
}

void rgbmap::grayscale(unsigned char *pixels) {
  if(!pixels) return;
  unsigned char *q,*pe;
  for(q=data,pe=pixels+width*height;pixels<pe;q+=3)
    *pixels++=(19595*q[0]+38470*q[1]+7471*q[2])>>16;
}

/*
void main() {
  FILE *f;
  rgbmap m1;
  pseudomap m2;
  if(!(f=fopen("in.ppm","rb")))
    return 1;
  m1.import_ppm(f);
  fclose(f);
  m1.convert_to(&m2);
  m1.convert_from(&m2);
  if(!(f=fopen("out.ppm","wb")))
    return 1;
  m1.export_ppm(f);
  fclose(f);
  m2.free();
  m1.free();
  return 0;
} */
