/*
 * Disk-Image Transfer Utility v1.0
 * (C) Fauzan Mirza 1994
 */

#include <stdio.h>
#include <fcntl.h>
#include <bios.h>
#include <io.h>

#define THRESHOLD 18
#define SECTOR 512

#define DISKTOIMAGE isdigit(*source) && isalpha(*target)
#define IMAGETODISK isalpha(*source) && isdigit(*target)

char Program[]="Disk-Image Transfer Utility 1.0";
char Author[] ="Copyright (C) Fauzan Mirza 1994";

int buflen;
char *buffer;
int drive, ntracks, nheads, nsectors;

void init(char *disk, char *image)
{
    if(strchr(image,'.')==NULL)
        strcat(image,".IMG");
    strupr(image);
    sscanf(disk,"%u:%u:%u:%u", &drive, &ntracks, &nheads, &nsectors);

    if(ntracks==0 || nheads==0 || nsectors==0) {
        printf("Illegal disk geometry\n");
        usage();
    }

    buflen=nsectors * SECTOR;

    buffer=(char *)malloc(buflen);

    if(buffer==NULL) {
        printf("malloc(%u) failed\n",buflen);
        exit(1);
    }
}

void getenter(void)
{
    printf("<Enter>");
    fflush(stdout);
    if(getch()!=13) {
        printf("\rCopy aborted\n");
        exit(1);
    }
    putchar('\r');
}

int syncdrive(void)
{
    struct diskinfo_t disk;
    int result, i;

    for(i=0; i<2; i++) {
        disk.drive  = drive;
        disk.head   = 0;
        disk.track  = 0;
        disk.sector = 1;
        disk.nsectors=1;

        result=_bios_disk(_DISK_VERIFY, &disk) >> 8;
    }
    return result;
}

void usage(void)
{
    printf("Usage: DITU <Source> <Target>\n\n");
    printf("Source and Target must be either ...\n\n");
    printf("<Disk>:<Geometry>  - Where <Disk> is the BIOS disk number\n");
    printf("\t\t   - <Geometry> is the disk geometry in\n");
    printf("\t\t     <Tracks>:<Heads>:<Sector> format\n");
    printf("\t\t   - Examples  0:80:2:18, 1:40:2:9\n\n");
    printf("<Filename>         - File or path name of the disk image\n");
    printf("\t\t   - The filename should NOT start with a digit\n");
    printf("\t\t   - File extension of IMG will be assumed\n");
    printf("\t\t   - Examples  FLOPPY, FLOPPY.IMG\n\n");
    printf("You cannot have both Source and Target the same type of media.\n");
    exit(1);
}

int main(int argc, char *argv[])
{
    struct diskinfo_t disk;
    char source[80],target[80];
    int i,result,errors;
    int image;
    int track,head;

    errors=-1;

    if(argc != 3){
        printf("Specify Source and Target parameters\n");
        usage();
    }

    strcpy(source,argv[1]);
    strcpy(target,argv[2]);

    if(DISKTOIMAGE) {

        init(source,target);

        printf("Source: BIOS Drive: %u [%u:%u:%u]\n", drive, ntracks, nheads,
                nsectors);
        printf("Target: Image File: %s\n", target);

        getenter();

        image=creat(target, 0777);
        if(image==-1) {
            printf("creat() failed\n");
            exit(1);
        }

        setmode(image, O_BINARY);
        errors=0;

        syncdrive();

        for(track=0; track < ntracks; track++) {

            for(head=0; head < nheads; head++) {

                for(i=0; i<buflen; i++)
                    buffer[i]=0;

                disk.drive  = drive;
                disk.head   = head;
                disk.track  = track;
                disk.sector = 1;
                disk.nsectors=nsectors;
                disk.buffer=buffer;

                result=_bios_disk(_DISK_READ, &disk) >> 8;

                printf("Track %u, Head %u : 0x%2.2X\r",track,head,result);

                if(result != 0) {
                    errors++;
                    putchar('\n');
                    if(errors==THRESHOLD) {
                        printf("Copy aborted: Too many errors\n");
                        exit(-2);
                    }
                }

                if(write(image,buffer,buflen) == -1) {
                    printf("write() failed\n");
                    exit(2);
                }

            }
        }

    }

    if(IMAGETODISK) {

        init(target, source);

        printf("Source: Image File: %s\n", source);
        printf("Target: BIOS Drive: %u [%u:%u:%u]\n", drive, ntracks, nheads,
                nsectors);

        getenter();

        image=open(source, O_RDONLY | O_BINARY);
        if(image==-1) {
            printf("open() failed\n");
            exit(1);
        }

        errors=0;

        syncdrive();

        for(track=0; track < ntracks; track++) {

            for(head=0; head < nheads; head++) {

                for(i=0; i<buflen; i++)
                    buffer[i]=0;

                if((i=read(image,buffer,buflen)) == -1) {
                    printf("read() failed\n");
                    exit(2);
                }
                if(i==0) {
                    printf("\nPartial Copy Complete\n");
                    close(image);
                    free(buffer);
                    return 0;
                }

                disk.drive  = drive;
                disk.head   = head;
                disk.track  = track;
                disk.sector = 1;
                disk.nsectors=nsectors;
                disk.buffer=buffer;

                result=_bios_disk(_DISK_WRITE, &disk) >> 8;

                printf("Track %u, Head %u : 0x%2.2X\r",track,head,result);

                if(result != 0) {
                    errors++;
                    putchar('\n');
                    if(errors==THRESHOLD) {
                        printf("Copy aborted: Too many errors\n");
                        exit(-2);
                    }
                }

            }
        }

    }

    if(errors==-1) {
        printf("Transfer type must be disk-image or image-disk\n");
        usage();
    }

    if(errors)
        printf("\nCopy Completed with %u errors\n", errors);
    else
        printf("\nCopy Complete\n");

    close(image);

    free(buffer);

    return 0;
}


