#include "ntui.h"
#include "..\shared\syncdir.h"

#include <direct.h>
#include <io.h>
#include <sys\stat.h>
#include <fcntl.h>
#include <share.h>
#include <assert.h>
#include <time.h>
#include <errno.h>

static UINT32 scanCt;

static BOOL dirScan(FILETABLE *tbl, char path[MAX_PATH])
{
  WIN32_FIND_DATA fd;
  HANDLE          h;
  char           *ptr = path + strlen(path);
  char           *testPtr = path + strlen(tbl->name);
  BOOL            ret = TRUE;
  assert(ptr > path);
  assert(ptr[-1] == '\\');
  strcpy(ptr, "*");
  h = FindFirstFile(path, &fd);
  if (h != INVALID_HANDLE_VALUE)
  {
    do
    {
      if (strcmp(fd.cFileName, ".") && strcmp(fd.cFileName, ".."))
      {
        strcpy(ptr, fd.cFileName);
        if ((fd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) != 0)
          strcat(ptr, "\\");
        scanCt++;
        if (!optionTest(OPT_QUIET) && !showScanUpdate(tbl, scanCt))
          ret = FALSE;
        else if (isInList(includeList, testPtr) && !isInList(excludeList, testPtr))
        {
          FILEINFO *inf;
          inf = fileInfoAlloc(tbl, path + strlen(tbl->name));
          inf->osfi.mod = fd.ftLastWriteTime;
          inf->isDir = (fd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) != 0;
          inf->size = (fileInfoIsDirectory(inf)) ? 0 : fd.nFileSizeLow;
          fileTableAdd(tbl, inf);
          if (fileInfoIsDirectory(inf) && optionTest(OPT_RECURSE))
            ret = dirScan(tbl, path);
        }
      }
    } while (ret && FindNextFile(h, &fd));
    FindClose(h);
  }
  *ptr = 0;
  return ret;
}

BOOL directoryScan(FILETABLE *tbl)
{
  BOOL ret;
  char path[MAX_PATH];
  strcpy(path, tbl->name);
  strcat(path, "\\");
  scanCt = 0;
  if (!optionTest(OPT_QUIET))
    showScanBegin(tbl);
  ret = dirScan(tbl, path);
  if (!optionTest(OPT_QUIET))
    showScanComplete(tbl, scanCt);
  return ret;
}

static BOOL fileDeleteByPath(const char *path);
static BOOL dirDelete(char path[MAX_PATH])
{
  WIN32_FIND_DATA fd;
  HANDLE          h;
  char           *ptr = path + strlen(path);
  BOOL            ret = TRUE;
  strcpy(ptr, "\\*");
  h = FindFirstFile(path, &fd);
  if (h != INVALID_HANDLE_VALUE)
  {
    do
    {
      if (strcmp(fd.cFileName, ".") && strcmp(fd.cFileName, ".."))
      {
        strcpy(ptr+1, fd.cFileName);
        if (fd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)
          ret = dirDelete(path);
        else
          ret = fileDeleteByPath(path);
      }
    } while (FindNextFile(h, &fd));
    FindClose(h);
  }
  *ptr = 0;
  return (ret)
    ? (rmdir(path) == 0)
    : FALSE;
}

BOOL directoryDelete(const char *name)
{
  char path[MAX_PATH];
  strcpy(path, name);
  return dirDelete(path);
}

BOOL directoryCreate(const char *name)
{
  return (mkdir(name) == 0);
}

static BOOL fileDeleteByPath(const char *f)
{
  if (unlink(f) == 0)
    return TRUE;
  else
  {
    chmod(f, _S_IREAD | _S_IWRITE);
    return (unlink(f) == 0);
  }
}

BOOL fileDelete(const char *name)
{
  return fileDeleteByPath(name);
}

int fileInfoDateCompare(const FILEINFO *fA, const FILEINFO *fB)
{
  FILETIME a = fA->osfi.mod;
  FILETIME b = fB->osfi.mod;

  unsigned __int64 tA = (((unsigned __int64)a.dwHighDateTime) << 32) | a.dwLowDateTime;
  unsigned __int64 tB = (((unsigned __int64)b.dwHighDateTime) << 32) | b.dwLowDateTime;
  /*
   * diff is in 100 ns increments
   * diff / 10^6 = seconds
   * diff / 10^3 = ms
   */
  unsigned __int64 diff = (tA > tB)
    ? tA - tB
    : tB - tA;
  if ((diff / 10000) <= granularity)
    return 0;
  else if (tA > tB)
    return 1;
  else
    return -1;
}

int fileInfoNameCompare(const FILEINFO *a, const FILEINFO *b, int partial)
{
  return (partial)
    ? strnicmp(a->name, b->name, strlen(a->name))
    : stricmp(a->name, b->name);
}

BOOL fileAttrGet(const char *name, FILEATTR *sts)
{
  HANDLE h = CreateFile(
    name,
    GENERIC_READ,
    FILE_SHARE_READ | FILE_SHARE_WRITE,
    0,
    OPEN_EXISTING,
    FILE_ATTRIBUTE_NORMAL,
    0
  );
  BOOL ret = FALSE;
  if (h != INVALID_HANDLE_VALUE)
  {
    ret = GetFileTime(h, &sts->cr, &sts->ac, &sts->wr);
    CloseHandle(h);
    if (ret)
    {
      sts->attr = GetFileAttributes(name);
      ret = (sts->attr != (DWORD) -1);
    }
  }
  return ret;
}

BOOL fileAttrSet(const char *dst, const FILEATTR *attr)
{
  HANDLE h = CreateFile(
    dst,
    GENERIC_READ | GENERIC_WRITE,
    FILE_SHARE_READ | FILE_SHARE_WRITE,
    0,
    OPEN_EXISTING,
    0,//FILE_ATTRIBUTE_NORMAL,
    0
  );
  BOOL ret = FALSE;
  if (h != INVALID_HANDLE_VALUE)
  {
    ret = SetFileTime(h, &attr->cr, &attr->ac, &attr->wr);
    CloseHandle(h);
    if (ret)
      ret = SetFileAttributes(dst, attr->attr);
  }
  return ret;
}

FILETABLE *fileTableInit(const char *dir, BOOL create)
{
  char path[MAX_PATH];
  struct stat statBuf;

  if (dir)
    strcpy(path, dir);
  else
    path[0] = 0;

  if (path[0] == 0)
    strcpy(path, ".");
  else if (strcmp(path+1, ":") == 0)
    strcat(path, ".");
  else
  {
    char *ptr;
    for (ptr = path + strlen(path) - 1; (ptr >= path) && (*ptr == '\\'); ptr--)
      ;
    ptr[1] = 0;
  }
  if (strcmp(path+1, ":") == 0)
    strcat(path, "\\");

  if (stat(path, &statBuf) == -1) 
  {
    //
    // does not exist, if create is FALSE, ignore!
    //
    if (!create)
      return 0;
    else
    {
      //
      // attempt to create
      //
      char *ptr;
      for (ptr = strchr(path, '\\'); ptr; ptr = strchr(ptr+1, '\\'))
      {
        *ptr = 0;
        mkdir(path);
        *ptr = '\\';
      }
      return (mkdir(path) == -1)
        ? 0
        : fileTableAlloc(path);
    }
  } else if (!(statBuf.st_mode & _S_IFDIR))
    return 0; // exists but is not a directory
  else
  {
    if (strcmp(path+1, ":\\") == 0)
      path[2] = 0;
    return fileTableAlloc(path); // success!
  }
}


FILEHANDLE fileOpen(const char *name)
{
   return sopen(name, _O_BINARY | _O_RDONLY, _SH_DENYWR, 0);
}

FILEHANDLE fileCreate(const char *name)
{
  return  sopen(name, _O_BINARY | _O_WRONLY | _O_CREAT, _SH_DENYRW, _S_IREAD | _S_IWRITE);
}

UINT32     fileLength(FILEHANDLE f)
{
  return filelength(f);
}

UINT32     fileRead(FILEHANDLE f, void *dst, UINT32 len)
{
  return read(f, dst, len);
}

UINT32     fileWrite(FILEHANDLE f, const void *src, UINT32 len)
{
  return write(f, src, len);
}

BOOL       fileClose(FILEHANDLE f)
{
  return (close(f) == 0);
}

BOOL       fileHandleIsValid(FILEHANDLE f)
{
  return (f >= 0);
}

typedef struct
{
  FILE *file;
  char  name[MAX_PATH];
} EXT_FILE;

//
// shan't need more than two of these
//
static EXT_FILE tempFiles[2];

FILE *fileTempCreate(void)
{
  EXT_FILE *extFile;
  if (tempFiles[0].file)
  {
    if (tempFiles[1].file)
    {
      extFile = 0;
    } else
      extFile = tempFiles + 1;
  } else
    extFile = tempFiles;

  if (extFile)
  {
    //
    // find an appropriate file name & open it for ''w+b''
    //
    char *name = extFile->name;
    char *tmp = getenv("tmp");
    long  now = (long) time(0);
    int    ii;
    if (!tmp)
    {
      tmp = getenv("temp");
      if (!tmp)
        tmp = ".\\";
    }
    strcpy(name, tmp);
    if (name[0] && (name[strlen(name)-1] != '\\'))
      strcat(name, "\\");
    tmp = name + strlen(name);
    for (ii = 0, extFile->file = 0; !extFile->file && (ii < 100); ii++) // don't try more than 100 times
    {
      ultoa(now+ii, tmp, 16);
      if ((access(name, 0) == -1) && (errno == ENOENT))
        extFile->file = fopen(name, "w+b");
    }
  }
  return extFile->file;
}

BOOL fileTempDestroy(FILE *f)
{
  EXT_FILE *extFile;
  if (tempFiles[0].file == f)
  {
    extFile = tempFiles + 0;
  } else if (tempFiles[1].file == f)
  {
    extFile = tempFiles + 1;
  } else
    extFile = 0;
  fclose(extFile->file);
  fileDelete(extFile->name);
  return TRUE;
}
