{*************************************************************************}
{*                                                                       *}
{* DbDesc, Copyright 1994-1997 by Salvatore Besso, mc8505@mclink.it      *}
{*                                                                       *}
{* Prints Paradox tables structure and indexes informations              *}
{*                                                                       *}
{* This software is freeware.                                            *}
{*                                                                       *}
{* You are free to modify the source code for your personal use and      *}
{* redistribute this software provided that you leave the copyright      *}
{* notices unmodified everywhere in the code, also in the comments.      *}
{*                                                                       *}
{*                                                                       *}
{* Errorlevels returned:                                                 *}
{*                                                                       *}
{* 1  = File error (table(s) not found or other DOS error)               *}
{* 2  = Printer error                                                    *}
{* 99 = Command line parameter wrong or missing                          *}
{*                                                                       *}
{*                                                                       *}
{* Changes:                                                              *}
{* --------                                                              *}
{*                                                                       *}
{* In version 1.07 only an errorlevel bug regarding printer errors has   *}
{* been fixed. From this version also the LFN unit is included in the    *}
{* package.                                                              *}
{*                                                                       *}
{* Version 1.05 has never been released due to some bugs and it has been *}
{* immediatly replaced by version 1.06.                                  *}
{*                                                                       *}
{* From version 1.05 all Paradox formats, from 3.0 up to 7.0, are        *}
{* supported, and there are more informations about the table and        *}
{* its indexes (almost all). There is no need to still use the           *}
{* Paradox Engine, resulting in a substantial executable size            *}
{* decrease.                                                             *}
{*                                                                       *}
{* Also long file names support has been added if DbDesc is              *}
{* executed in a Windows 95 DOS box (thanks to Kim Kokkonen of           *}
{* Turbo Power Software for his LFN unit.                                *}
{*                                                                       *}
{* Fixed a major bug for which only tables created with pxASCII sort     *}
{* order were processed. Now all sort orders are processed.              *}
{*                                                                       *}
{* Error level 3 (Paradox Engine error) has been removed.                *}
{*                                                                       *}
{* Anyway, there is the possibility that some fields of the Paradox      *}
{* structure may be still not well interpreted. So any feedback will     *}
{* be appreciated. In this case send me an e-mail specifying the         *}
{* problem encountered.                                                  *}
{*                                                                       *}
{* N O T E:                                                              *}
{* ________                                                              *}
{*                                                                       *}
{* Don't use DbDesc from a path containing Windows 95 long directory     *}
{* names or Windows 95 directory names containing spaces. If you do it   *}
{* you will get a DbDesc printer error. This is due to a bug that I      *}
{* discovered in BP 7.0x for which any Write or WriteLn operation (it    *}
{* doesn't matter if on the screen or on the printer) returns with an    *}
{* IOResult = 3. For example:                                            *}
{*                                                                       *}
{* Executing DbDesc from:                                                *}
{*                                                                       *}
{*   C:\MYDIR\Very long name directory                                   *}
{*                                                                       *}
{* or:                                                                   *}
{*                                                                       *}
{*   C:\Very long name directory\MYDIR                                   *}
{*                                                                       *}
{* or (the real case):                                                   *}
{*                                                                       *}
{*   C:\Programmi\Borland\Delphi 2.0\Objrepos                            *}
{*                                                                       *}
{* returns with DbDesc printer error (IOResult = 3: Path not found ????) *}
{*                                                                       *}
{* I don't know why this happens and I'm not sure that it happens in     *}
{* every copy of BP 7.0x (my copy is BP 7.01). You could try.            *}
{*                                                                       *}
{* This bug is actually unresolved for me.                               *}
{*                                                                       *}
{*                                                                       *}
{* How to read abbreviations:                                            *}
{* --------------------------                                            *}
{*                                                                       *}
{* Tables:                                                               *}
{* -------                                                               *}
{*                                                                       *}
{* For BCD fields, a 'd' following the number in "Fld len" column means  *}
{* that length is referred to number of decimals instead of effective    *}
{* length.                                                               *}
{*                                                                       *}
{* Indexes:                                                              *}
{* --------                                                              *}
{*                                                                       *}
{*               Indexes of table "XYZ.DB"                               *}
{* --------------------------------------------....---------             *}
{*      | Num | Case |      | Desc |           .... |                    *}
{* Type | Fld | Sens | Incr | end  | Index name.... | Fields             *}
{* --------------------------------------------....---------             *}
{* |      |     |      |      |      |                |_ Fields by which *}
{* |      |     |      |      |      |                   index is formed *}
{* |      |     |      |      |      |                                   *}
{* |      |     |      |      |      |________________ Secondary index   *}
{* |      |     |      |      |                        name              *}
{* |      |     |      |      |                                          *}
{* |      |     |      |      |_____________________ Ascending order or  *}
{* |      |     |      |                             descending order    *}
{* |      |     |      |                             (7+ only)           *}
{* |      |     |      |                                                 *}
{* |      |     |      |__________________________ Incremental or not    *}
{* |      |     |                                  incremental index     *}
{* |      |     |                                                        *}
{* |      |     |_______________________________ Case sensitive or not   *}
{* |      |                                      case sensitive index    *}
{* |      |                                                              *}
{* |      |___________________________________ Number of fields in the   *}
{* |                                           index                     *}
{* |                                                                     *}
{* |________________________________________ Index type:                 *}
{*                                                                       *}
{*                                           Pri  = primary index        *}
{*                                           Comp = composite index      *}
{*                                                                       *}
{*                                           One field secondary indexes *}
{*                                           --------------------------- *}
{*                                           Alpha    : A                *}
{*                                           Short    : S                *}
{*                                           Number   : N                *}
{*                                           Money    : $                *}
{*                                           LongInt  : I                *}
{*                                           Date     : D                *}
{*                                           Time     : T                *}
{*                                           TimeStamp: @                *}
{*                                           BCD      : #                *}
{*                                           AutoInc  : +                *}
{*                                                                       *}
{*************************************************************************}

program DbDesc;

{$I-,V-}

uses
  Dos,Printer,
  Objects,LFN;

const

  Version = '1.07';

  OnPrinter    : Boolean     = True;
  TableName    : String      = '';
  Stream       : PBufStream  = NIL;
  IndexInfoColl: PCollection = NIL;

  { Paradox codes for field types in Header }

  pxAlpha     = $01;
  pxDate      = $02;
  pxShort     = $03;
  pxLong      = $04;
  pxMoney     = $05;
  pxNumber    = $06;
  pxLogical   = $09;
  pxMemo      = $0C;
  pxBinary    = $0D;
  pxFmtMemo   = $0E;
  pxOleObject = $0F;
  pxGraphic   = $10;
  pxTime      = $14;
  pxTimeStamp = $15;
  pxAutoInc   = $16;
  pxBcd       = $17;
  pxBytes     = $18;

type

  TWinMode =  (NoWin3,RealOrStd,Enhanced);

  PFieldDesc = ^TFieldDesc;
  TFieldDesc = record
    FldType: Byte;
    FldSize: Byte
  end;

  TTblNameInRam  = array[1..79] of Char;
  TTbl7NameInRam = array[1..261] of Char;

  PPdoxHeader = ^TPdoxHeader;
  TPdoxHeader = record
    RecordSize           : Word;
    HeaderSize           : Word;
    FileType             : Byte;
    MaxTableSize         : Byte;
    NumRecords           : LongInt;
    NextBlock            : Word;
    FileBlocks           : Word;
    FirstBlock           : Word;
    LastBlock            : Word;
    Reserved1            : Word;
    ModifiedFlags1       : Byte;
    IndexFieldNumber	 : Byte;
    PrimaryIndexWorkSpace: Pointer;
    Reserved2            : Pointer;
    Reserved3            : array[$001E..$0020] of Byte;
    NumFields            : Integer;
    PrimaryKeyFields     : Integer;
    Encryption1          : LongInt;
    SortOrder            : Byte;
    ModifiedFlags2       : Byte;
    Reserved4            : array[$002B..$002C] of Byte;
    ChangeCount1         : Byte;
    ChangeCount2         : Byte;
    Reserved5            : Byte;
    TableNamePtr         : ^PChar;
    FldInfoPtr           : PFieldDesc;
    WriteProtected       : Byte;
    FileVersionID        : Byte;
    MaxBlocks            : Word;
    Reserved6            : Byte;
    AuxPasswords         : Byte;
    Reserved7            : array[$003E..$003F] of Byte;
    CryptInfoStartPtr    : Pointer;
    CryptInfoEndPtr      : Pointer;
    Reserved8            : Byte;
    AutoIncVal           : LongInt;
    Reserved9            : array[$004D..$004E] of Byte;
    IndexUpdateRequired  : Byte;
    Reserved10           : array[$0050..$0054] of Byte;
    Descending           : Byte;  { 7.0 only }
    Reserved11           : array[$0056..$0057] of Byte;
    case Integer of
      3: (
        FieldInfo35 : array[1..255] of TFieldDesc
      );
      4: (
        FileVersionID2: Integer;
        FileVersionID3: Integer;
        Encryption2   : LongInt;
        FileUpdateTime: LongInt;  { 4.0 only }
        HiFieldID     : Word;
        HiFieldIDinfo : Word;
        NumFields2    : Integer;
        DosCodePage   : Word;
        Reserved12    : array[$006C..$006F] of Byte;
        ChangeCount4  : Integer;
        Reserved13    : array[$0072..$0077] of Byte;
        FieldInfo     : array[1..255] of TFieldDesc
      )
  end;

  PIndexCollection = ^TIndexCollection;
  TIndexCollection = object (TCollection)
    procedure FreeItem (Item: Pointer); virtual;
  end;

  PIndexInfo = ^TIndexInfo;
  TIndexInfo = object (TObject)
    IndexType    : Integer;
    NumFields    : Integer;                   { n. of fields in the index }
    CaseSensitive: Boolean;
    Incremental  : Boolean;                   { maintained or not }
    Descending   : Boolean;                   { 7.0 only }
    IndexName    : String[25];
    FldArray     : array[1..255] of Integer;  { Number of the fields }
  end;

const

  Header: PPdoxHeader = NIL;

  pxPrimary      = 100;
  pxComposite    = 101;
  pxIncComposite = 102;

  IndexColl: PIndexCollection = NIL;

var
  WinMode                      : TWinMode;
  WinVersion                   : Word;
  LFNAllowed                   : Boolean;
  Params,I                     : Word;
  TblName,ShortPath,ShortName,S: String;
  Line,Spaces                  : String[128];
  PrintHeader                  : array[1..8] of String;
  HeaderSize                   : Integer;

procedure TIndexCollection.FreeItem (Item: Pointer);

begin
  if Item <> NIL then DisposeStr (PString (Item))
end;

function GetWindowsMode (var Version: Word): TWinMode; assembler;

asm
      MOV     AX,01600H
      INT     02FH
      XCHG    AH,AL
      LES     DI,Version
      MOV     ES:[DI],AX
      OR      AH,AH
      JZ      @@2
      CMP     AH,1
      JE      @@2
      CMP     AH,080H
      JE      @@2
      CMP     AH,0FFH
      JE      @@2
      OR      AL,AL
      JNZ     @@1
      MOV     AX,04680H
      INT     02FH
      OR      AX,AX
      JNZ     @@1
      MOV     AL,RealOrStd
      JMP     @@3
@@1:  MOV     AL,Enhanced
      JMP     @@3
@@2:  MOV     AL,NoWin3
@@3:
end;

function GetParamCount: Word;

begin
  if LFNAllowed then
    GetParamCount := LFNParamCount
  else GetParamCount := ParamCount
end;

function GetParamStr (Index: Word): String;

begin
  if LFNAllowed then
    GetParamStr := LFNParamStr (Index)
  else GetParamStr := ParamStr (Index)
end;

function UpStr (S: String): String;

var
  I: Byte;

begin
  for I := 1 to Length (S) do S[I] := UpCase (S[I]);
  UpStr := S
end;

procedure FileSplit (const FName: String; var Dir,Name,Ext: String);

var
  I: Integer;
  S: String;

begin
  if LFNAllowed then
  begin
    I := Length (FName);
    while (I > 0) and (FName[I] <> '\') do Dec (I);
    Dir := Copy (FName,1,I);
    S := Copy (FName,I + 1,255);
    I := 1;
    while (I <= Length (S)) and (S[I] <> '.') do Inc (I);
    Name := Copy (S,1,I - 1);
    Ext := Copy (S,I,255)
  end
  else FSplit (FName,Dir,Name,Ext)
end;

function FileExpand (const Path: String): String;

var
  Dir,Name,Ext: String;

begin
  if LFNAllowed then
  begin
    FileSplit (Path,Dir,Name,Ext);
    if Dir = '' then Dir := '.\';
    if LFNGetLongPath (True,Dir,Dir) = 0 then
      FileExpand := Dir + Name + Ext
    else FileExpand := ''
  end
  else FileExpand := FExpand (Path)
end;

procedure FindFirstFile (const Path: String; ReqdAttr,Attr: Byte;
  var DosSR: SearchRec; var SR: TLFNSearchRec);

begin
  if LFNAllowed then
    DosError := LFNFindFirst (Path,ReqdAttr,Attr,SR)
  else FindFirst (Path,Attr,DosSR)
end;

procedure FindNextFile (var DosSR: SearchRec; var SR: TLFNSearchRec);

begin
  if LFNAllowed then
    DosError := LFNFindNext (SR)
  else FindNext (DosSR)
end;

procedure FindClose (var DosSR: SearchRec; var SR: TLFNSearchRec);

begin
  if LFNAllowed then LFNFindClose (SR)
end;

procedure GetShortNames (const FName: String; var SPath,SName: String);

var
  I         : Integer;
  S,Name,Ext: String;

begin
  SPath := '';
  SName := '';
  if LFNGetShortPath (True,FName,S) = 0 then
  begin
    FileSplit (S,SPath,Name,Ext);
    SName := Name + Ext
  end
end;

function HexString (Value: LongInt; Width: Byte): String; assembler;

asm
      PUSHF
      LES     DI,@Result
      XOR     CH,CH
      MOV     CL,Width
      MOV     ES:[DI],CL
      JCXZ    @@6
      ADD     DI,CX
      XOR     DX,DX
      CMP     CX,4
      JLE     @@1
      ADD     DL,CL
      SUB     DL,4
      MOV     CL,4
@@1:  MOV     BX,WORD [Value]
      STD
@@2:  MOV     AL,BL
      AND     AL,0FH
      OR      AL,30H
      CMP     AL,3AH
      JB      @@3
      ADD     AL,7
@@3:  STOSB
      SHR     BX,1
      SHR     BX,1
      SHR     BX,1
      SHR     BX,1
      LOOP    @@2
      MOV     CX,DX
      JCXZ    @@6
      MOV     BX,WORD [Value + 2]
      STD
@@4:  MOV     AL,BL
      AND     AL,0FH
      OR      AL,30H
      CMP     AL,3AH
      JB      @@5
      ADD     AL,7
@@5:  STOSB
      SHR     BX,1
      SHR     BX,1
      SHR     BX,1
      SHR     BX,1
      LOOP    @@4
@@6:  POPF
end;

function CenteredStr (const S: String; Width: Integer): String;

begin
  CenteredStr := Copy (Spaces,1,(Width - Length (S)) div 2) + S
end;

procedure Abort (const Message: String; HaltCode: Integer);

begin
  if IndexInfoColl <> NIL then Dispose (IndexInfoColl,Done);
  if IndexColl <> NIL then Dispose (IndexColl,Done);
  if Stream <> NIL then Dispose (Stream,Done);
  if Header <> NIL then FreeMem (Header,HeaderSize);
  WriteLn (#7#13#10,Message);
  Halt (HaltCode)
end;

function StrPas (Str: PChar): String; assembler;

asm
	PUSH	DS
	CLD
	LES	DI,Str
	MOV	CX,0FFFFH
	XOR	AL,AL
	REPNE	SCASB
	NOT	CX
	DEC	CX
	LDS	SI,Str
	LES	DI,@Result
	MOV	AL,CL
	STOSB
	REP	MOVSB
	POP	DS
end;

function StrLen (Str: PChar): Word; assembler;

asm
	CLD
	LES	DI,Str
	MOV	CX,0FFFFH
	XOR	AL,AL
	REPNE	SCASB
	MOV	AX,0FFFEH
	SUB	AX,CX
end;

procedure PrintStructure;

var
  Dir,Name,Ext: String;
  DosSR       : SearchRec;
  SR          : TLFNSearchRec;
  StrmErr,I   : Integer;
  Flds        : PFieldDesc;
  FldName     : PChar;
  S,T         : String;

procedure PrintFieldsInfo;

var
  I       : Integer;
  S,T     : String;
  Len,Size: LongInt;

begin
  for I := 1 to Header^.NumFields do
  begin
    Str (I:0,T);
    S := Copy (T + Spaces,1,4) + '| ';
    T := StrPas (FldName);
    S := S + Copy (T + Spaces,1,26) + '| ';
    case Flds^.FldType of
      pxAlpha          : T := 'Alpha';
      pxShort          : T := 'Short';
      pxDate           : T := 'Date';
      pxLong..pxNumber : T := 'Number';
      pxMemo..pxGraphic: T := 'Blob';
      pxLogical        : T := 'Logical';
      pxTime           : T := 'Time';
      pxTimeStamp      : T := 'TimeStamp';
      pxAutoInc        : T := 'AutoInc';
      pxBcd            : T := 'BCD';
      pxBytes          : T := 'Bytes'
    end;
    S := S + Copy (T + Spaces,1,10) + '| ';
    case Flds^.FldType of
      pxLong     : T := 'LongInt';
      pxMoney    : T := 'Money';
      pxMemo     : T := 'Memo';
      pxBinary   : T := 'Binary';
      pxFmtMemo  : T := 'Fmt Memo';
      pxOleObject: T := 'OLE';
      pxGraphic  : T := 'Graphic'
      else T := ''
    end;
    S := S + Copy (T + Spaces,1,9) + '| ';
    Size := Flds^.FldSize;
    Len := Size;
    if Flds^.FldType = pxBcd then Size := 17;
    if Flds^.FldType in [pxMemo..pxGraphic] then Dec (Len,10);
    Str (Size:0,T);
    S := S + Copy (T + Spaces,1,4) + '| ';
    Str (Len:0,T);
    if Flds^.FldType = pxBcd then T := T + 'd';
    S := S + Copy (T + Spaces,1,4) + '| ';
    if Flds^.FldType = pxAutoInc then
    begin
      Str (Header^.AutoIncVal:0,T);
      S := S + T
    end
    else Dec (S[0]);
    if OnPrinter then WriteLn (Lst,S) else WriteLn (S);
    if IOResult > 0 then Abort ('Printer error',2);
    Inc (PtrRec (Flds).Ofs,SizeOf (TFieldDesc));
    Len := StrLen (FldName) + 1;
    Inc (PtrRec (FldName).Ofs,Len)
  end
end;

procedure PrintIndexesInfo (Dir,Name: String);

var
  I,PriFlds: Integer;

function GetKeyFiles (Dir,Name: String): PIndexCollection;

var
  P    : PIndexCollection;
  DosSR: SearchRec;
  SR   : TLFNSearchRec;

begin
  P := New (PIndexCollection,Init (2,1));
  FindFirstFile (Dir + Name + '.PX',$00,AnyFile,DosSR,SR);
  if DosError = 0 then
  begin
    if LFNAllowed then GetShortNames (Dir + SR.Name,ShortPath,ShortName)
    else begin
      ShortPath := Dir;
      ShortName := DosSR.Name
    end;
    P^.Insert (NewStr (ShortPath + ShortName));
    FindClose (DosSR,SR);
    FindFirstFile (Dir + Name + '.X??',$00,AnyFile,DosSR,SR);
    while DosError = 0 do
    begin
      if LFNAllowed then GetShortNames (Dir + SR.Name,ShortPath,ShortName)
      else begin
        ShortPath := Dir;
        ShortName := DosSR.Name
      end;
      P^.Insert (NewStr (ShortPath + ShortName));
      FindNextFile (DosSR,SR)
    end;
    FindClose (DosSR,SR);
  end
  else FindClose (DosSR,SR);
  if P^.Count = 0 then
  begin
    Dispose (P,Done);
    P := NIL
  end;
  GetKeyFiles := P
end;

procedure DoBuild (P: PString); far;

var
  StrmErr: Integer;
  Info   : PIndexInfo;
  Dummy  : PChar;
  I,FldNo: Integer;

begin
  Stream := New (PBufStream,Init (P^,stOpenRead,4096));
  StrmErr := Stream^.Status;
  if StrmErr = stOk then
  begin
    Stream^.Seek (2);
    StrmErr := Stream^.Status;
    if StrmErr = stOk then
    begin
      Stream^.Read (HeaderSize,SizeOf (Integer));
      StrmErr := Stream^.Status;
      if StrmErr = stOk then
      begin
        Stream^.Seek (0);
        StrmErr := Stream^.Status;
        if StrmErr = stOk then
        begin
          GetMem (Header,HeaderSize);
          Stream^.Read (Header^,HeaderSize);
          StrmErr := Stream^.Status
        end
      end
    end
  end;
  Dispose (Stream,Done);
  Stream := NIL;
  if StrmErr <> stOk then Abort ('Index ' + P^ + ' read error',1);
  { assign the Header^'s FldInfoPtr field }
  if Header^.FileVersionID < $05 then
    Header^.FldInfoPtr := Addr (Header^.FieldInfo35)
  else Header^.FldInfoPtr := Addr (Header^.FieldInfo);
  Info := New (PIndexInfo,Init);
  case Header^.FileType of
    1: begin
      Info^.IndexType := pxPrimary;
      Info^.NumFields := Header^.NumFields;
      PriFlds := Header^.NumFields
    end;
    3: begin
      Info^.IndexType := Header^.FldInfoPtr^.FldType;
      Info^.NumFields := 1;
      Info^.CaseSensitive := True;
      Info^.Incremental := False;
      Info^.FldArray[1] := Header^.IndexFieldNumber
    end;
    5: begin
      Info^.IndexType := Header^.FldInfoPtr^.FldType;
      Info^.NumFields := 1;
      Info^.CaseSensitive := True;
      Info^.Incremental := True;
      Info^.FldArray[1] := Header^.IndexFieldNumber
    end;
    6: begin
      Info^.IndexType := pxComposite;
      Info^.NumFields := Header^.NumFields - PriFlds - 1;
      Info^.CaseSensitive := False;
      Info^.Incremental := False
    end;
    8: begin
      Info^.IndexType := pxIncComposite;
      Info^.NumFields := Header^.NumFields - PriFlds - 1;
      Info^.CaseSensitive := False;
      Info^.Incremental := True
    end
  end;
  Info^.Descending := Header^.Descending = $11;
  { assign Dummy variable }
  if Header^.FileVersionID < $05 then
    Dummy := Addr (Header^.FieldInfo35)
  else Dummy := Addr (Header^.FieldInfo);
  { skip fields info }
  for I := 1 to Header^.NumFields do
    Inc (PtrRec (Dummy).Ofs,SizeOf (TFieldDesc));
  { skip table name pointer and field name pointers }
  for I := 1 to Header^.NumFields + 1 do
    Inc (PtrRec (Dummy).Ofs,SizeOf (Pointer));
  { skip table name space }
  if Header^.FileVersionID < $0C then
    Inc (PtrRec (Dummy).Ofs,SizeOf (TTblNameInRam))
  else Inc (PtrRec (Dummy).Ofs,SizeOf (TTbl7NameInRam));
  { we are now at field name pointers }
  case Header^.FileType of
    1  : Info^.IndexName := '';
    3,5: Info^.IndexName := StrPas (Dummy);
    6,8: begin
      { skip field names }
      for I := 1 to Header^.NumFields do
        Inc (PtrRec (Dummy).Ofs,StrLen (Dummy) + 1);
      { retrieve field numbers }
      for I := 1 to Header^.NumFields do
      begin
        if I <= Header^.NumFields - PriFlds - 1 then
          Info^.FldArray[I] := Integer (Dummy^);
        Inc (PtrRec (Dummy).Ofs,SizeOf (Integer))
      end;
      { skip sort order name }
      Inc (PtrRec (Dummy).Ofs,StrLen (Dummy) + 1);
      { finally, get index name }
      Info^.IndexName := StrPas (Dummy)
    end
  end;
  IndexInfoColl^.Insert (Info);
  FreeMem (Header,HeaderSize);
  Header := NIL
end;

procedure DoPrint (Info: PIndexInfo); far;

var
  S,T: String;
  I  : Integer;
  N  : String[3];

begin
  case Info^.IndexType of
    pxAlpha                   : S := 'A';
    pxShort                   : S := 'S';
    pxNumber                  : S := 'N';
    pxMoney                   : S := '$';
    pxLong                    : S := 'I';
    pxDate                    : S := 'D';
    pxTime                    : S := 'T';
    pxTimeStamp               : S := '@';
    pxBcd                     : S := '#';
    pxAutoInc                 : S := '+';
    pxPrimary                 : S := 'Pri';
    pxComposite,pxIncComposite: S := 'Comp'
  end;
  S := Copy (S + Spaces,1,5) + '| ';
  Str (Info^.NumFields:0,T);
  T := Copy (T + Spaces,1,4) + '| ';
  S := S + T;
  if Info^.CaseSensitive then T := 'X' else T := ' ';
  T := Copy (T + Spaces,1,5) + '| ';
  S := S + T;
  if Info^.Incremental then T := 'X' else T := ' ';
  T := Copy (T + Spaces,1,5) + '| ';
  S := S + T;
  if Info^.Descending then T := 'X' else T := ' ';
  T := Copy (T + Spaces,1,5) + '| ';
  S := S + T;
  T := Copy (Info^.IndexName + Spaces,1,26) + '| ';
  S := S + T;
  T := '';
  for I := 1 to Info^.NumFields do
  begin
    if Info^.IndexType = pxPrimary then
      Str (I:0,N)
    else Str (Info^.FldArray[I]:0,N);
    if T <> '' then T := T + ', ';
    T := T + N
  end;
  if OnPrinter then Write (Lst,S) else Write (S);
  if IOResult > 0 then Abort ('Printer error',2);
  while T <> '' do
  begin
    if Length (T) > 16 then
    begin
      I := 16;
      while S[I] <> ',' do Dec (I);
      S := Copy (T,1,I) + #13#10 +
        '     |     |      |      |      |                            | ';
    end
    else begin
      I := Length (T);
      S := T
    end;
    Delete (T,1,I);
    if (T <> '') and (T[1] = ' ') then Delete (T,1,1);
    if OnPrinter then Write (Lst,S) else Write (S);
    if IOResult > 0 then Abort ('Printer error',2)
  end;
  if OnPrinter then WriteLn (Lst) else WriteLn;
  if IOResult > 0 then Abort ('Printer error',2)
end;

begin { PrintIndexesInfo }
  { first of all we collect all indexes' names, if any }
  IndexColl := GetKeyFiles (Dir,Name);
  { if there are indexes, we proceed }
  if IndexColl <> NIL then
  begin
    IndexInfoColl := New (PCollection,Init (IndexColl^.Count,0));
    { now we build the collection containing }
    { indexes' informations for this table   }
    IndexColl^.ForEach (@DoBuild);
    Dispose (IndexColl,Done);
    IndexColl := NIL;
    { and finally we print the result }
    PrintHeader[1] := '';
    PrintHeader[2] := CenteredStr ('Indexes of table ' + TblName,80);
    PrintHeader[3] := Copy (Line,1,79);
    PrintHeader[4] := '     | Num | Case |      | Desc |' +
      '                           | Fields in this';
    PrintHeader[5] := 'Type | Fld | Sens | Incr | end  | Index name' +
      '                | index';
    PrintHeader[6] := Copy (Line,1,79);
    for I := 1 to 6 do
    begin
      if OnPrinter then WriteLn (Lst,PrintHeader[I]) else WriteLn (PrintHeader[I]);
      if IOResult > 0 then Abort ('Printer error',2)
    end;
    IndexInfoColl^.ForEach (@DoPrint);
    Dispose (IndexInfoColl,Done);
    IndexInfoColl := NIL
  end
end;

begin { PrintStructure }
  FileSplit (FileExpand (TableName),Dir,Name,Ext);
  if Name = '' then Name := '*';
  Ext := '.DB';
  FindFirstFile (Dir + Name + Ext,$00,Archive,DosSR,SR);
  if DosError > 0 then
  begin
    FindClose (DosSR,SR);
    Abort ('Table(s) not found',1)
  end;
  while DosError = 0 do
  begin
    if LFNAllowed then
    begin
      GetShortNames (Dir + SR.Name,ShortPath,ShortName);
      Name := Copy (SR.Name,1,Pos ('.',SR.Name) - 1);
      TblName := SR.Name
    end
    else begin
      ShortPath := Dir;
      ShortName := DosSR.Name;
      Name := Copy (DosSR.Name,1,Pos ('.',DosSR.Name) - 1);
      TblName := DosSR.Name
    end;
    if Pos (' ',TblName) > 0 then TblName := '"' + TblName + '"';
    Stream := New (PBufStream,Init (ShortPath + ShortName,stOpenRead,4096));
    StrmErr := Stream^.Status;
    if StrmErr = stOk then
    begin
      Stream^.Seek (2);
      StrmErr := Stream^.Status;
      if StrmErr = stOk then
      begin
        Stream^.Read (HeaderSize,SizeOf (Integer));
        StrmErr := Stream^.Status;
        if StrmErr = stOk then
        begin
          Stream^.Seek (0);
          StrmErr := Stream^.Status;
          if StrmErr = stOk then
          begin
            GetMem (Header,HeaderSize);
            Stream^.Read (Header^,HeaderSize);
            StrmErr := Stream^.Status
          end
        end
      end
    end;
    Dispose (Stream,Done);
    Stream := NIL;
    if StrmErr <> stOk then Abort ('Table ' + TblName + ' read error',1);
    if (Header^.MaxTableSize < 1) or (Header^.MaxTableSize > 4) then
      Abort (TblName + ': Invalid max table size',1);
    { assign the Header^'s FldInfoPtr field }
    if Header^.FileVersionID < $05 then
    begin
      Header^.FldInfoPtr := Addr (Header^.FieldInfo35);
      FldName := Addr (Header^.FieldInfo35)
    end
    else begin
      Header^.FldInfoPtr := Addr (Header^.FieldInfo);
      FldName := Addr (Header^.FieldInfo)
    end;
    Flds := Header^.FldInfoPtr;
    { Skip fields info }
    for I := 1 to Header^.NumFields do
      Inc (PtrRec (FldName).Ofs,SizeOf (TFieldDesc));
    { skip table name pointer and field name pointers }
    for I := 1 to Header^.NumFields + 1 do
      Inc (PtrRec (FldName).Ofs,SizeOf (Pointer));
    { skip table name space }
    if Header^.FileVersionID < $0C then
      Inc (PtrRec (FldName).Ofs,SizeOf (TTblNameInRam))
    else Inc (PtrRec (FldName).Ofs,SizeOf (TTbl7NameInRam));
    PrintHeader[1] := '';
    PrintHeader[2] := CenteredStr ('Table: ' + TblName,80);
    S := 'Format: Paradox ';
    case Header^.FileVersionID of
      $03     : S := S + '3.0';
      $04     : S := S + '3.5';
      $05..$09: S := S + '4.0';
      $0A,$0B : S := S + '5.0';
      $0C     : S := S + '7.0'
      else S := S + '???: $' + HexString (Header^.FileVersionID,2);
    end;
    S := S + ' - Max table size: ';
    case Header^.MaxTableSize of
      1: S := S + '64';
      2: S := S + '128';
      3: S := S + '192';
      4: S := S + '256'
      else S := S + '???: $' + HexString (Header^.MaxTableSize,2);
    end;
    S := S + ' MB - Sort order: ';
    case Header^.SortOrder of
      $00: S := S + 'Ascii';
      $B7: S := S + 'Intl';
      $82: S := S + 'NorDan';
      $E6: S := S + 'NorDan 4.x';
      $F0: S := S + 'SwedFin'
      else S := S + '???: $' + HexString (Header^.SortOrder,2);
    end;
    PrintHeader[3] := CenteredStr (S,80);
    Str (Header^.DosCodePage:0,T);
    S := 'Code page: ' + T + ' - Write pro: ';
    if Header^.WriteProtected = 1 then S := S + 'YES' else S := S + 'NO';
    S := S + ' - Encrypted: ';
    if ((Header^.FileVersionID < 5) and (Header^.Encryption1 <> 0))
    or ((Header^.FileVersionID >= 5) and (Header^.Encryption2 <> 0)) then
      S := S + 'YES'
    else S := S + 'NO';
    PrintHeader[4] := CenteredStr (S,80);
    PrintHeader[5] := Copy (Line,1,79);
    PrintHeader[6] := 'Fld |                           |           | Fld ' +
      'sub  | Fld | Fld | Auto Inc';
    PrintHeader[7] := 'num | Field name                | Fld type  | ' +
      'type     | siz | len | cur value';
    PrintHeader[8] := Copy (Line,1,79);
    for I := 1 to 8 do
    begin
      if OnPrinter then
        WriteLn (Lst,PrintHeader[I])
      else WriteLn (PrintHeader[I]);
      if IOResult > 0 then Abort ('Printer error',2)
    end;
    PrintFieldsInfo;
    FreeMem (Header,HeaderSize);
    Header := NIL;
    PrintIndexesInfo (Dir,Name);
    if OnPrinter then Write (Lst,#12#13) else WriteLn;
    if IOResult > 0 then Abort ('Printer error',2);
    FindNextFile (DosSR,SR)
  end;
  FindClose (DosSR,SR)
end;

procedure Usage;

begin
  WriteLn (#13#10'Usage: DbDesc [/NOPRINTER] [d:\path\]tablename[.DB]' +
    #13#10'              [>] [>>] [output device or file]' +
    #13#10'              (wildcards are OK)' +
    #13#10'              (long file names are OK under Windows 95 only)');
  WriteLn ('              (long file names with spaces must be enclosed ' +
    'in "")'#13#10);
  WriteLn ('Use /NOPRINTER to view the structures without printing.');
  WriteLn ('Use /NOPRINTER together with "> file/device" or ">> file" to ' +
    'redirect output'#13#10'to a file or device without printing.');
  Halt (99)
end;

begin { Main }
  WinMode := GetWindowsMode (WinVersion);
  LFNAllowed := (WinMode = Enhanced) and (Hi (WinVersion) >= 4);
  WriteLn ('DbDesc ' + Version + ' - Prints Paradox tables structure'#13#10 +
    'by Salvatore Besso, mc8505@mclink.it');
  Params := GetParamCount;
  if Params = 0 then Usage;
  for I := 1 to Params do
  begin
    S := UpStr (GetParamStr (I));
    if S[1] = '/' then
    begin
      if (TableName <> '') or (S <> '/NOPRINTER') then
        Usage
      else OnPrinter := False
    end
    else if TableName = '' then
      TableName := GetParamStr (I)
    else if (S = '') or (S[1] = '>') then
      Break
    else Usage
  end;
  if TableName = '' then Usage;
  FillChar (Line,SizeOf (Line),'-');
  Line[0] := #128;
  FillChar (Spaces,SizeOf (Spaces),' ');
  Spaces[0] := #128;
  PrintStructure
end.
