/*
 *   Library searching off the keys is done as follows:
 *     Go Block_X, Bucket_X
 *		 This bucket correct name?  yes- found
 *     no, increment by bucket count.
 *		 This bucket correct name?  yes -found
 *		 This bucket = first bucket? no- loop
 *		 yes, is block totally full?  no- done but not found
 *		 yes, go to next block
 *		 This block first block?  yes -done but not found
 *		 no - loop
 */
#include <stdio.h>
#include <string.h>
#include "cmdline.h"
#include "umem.h"
#include "errors.h"
#include "data.h"
#include "section.h"
#include "public.h"
#include "extern.h"
#include "libs.h"
#include "module.h"

extern BOOL prm_case_sensitive;
extern LIST *unresolvedexterns;
extern char *modname;
extern HASHREC **publichash;
extern BOOL loadlib;

LIST *modulelist = 0; /* List of all modules loaded and the libraries they
											 * occur in.  This list is kept in sequential order
											 * the same order as the segment lists */

/*
 * For each library:
 * Search local directory and all directories in the search path
 *  until it is found or run out of directories
 */
void LocateLibraries(char *searchpath, LIST **list)
{
  while (*list) {
		char buffer[256];
		FILE *found;
		strcpy(buffer,(*list)->data);
		found = SearchPath(buffer,searchpath,"r");
		if (found) {
			fclose(found);
			DeallocateMemory( (*list)->data);
			(*list)->data = AllocateMemory(strlen(buffer)+1);
			strcpy((char *)(*list)->data, buffer);
			list = *list;
		}
		else {
  		LIST *p= *list;
  		fatal("Missing library %s, aborting", p->data);
			DeallocateMemory(p->data);
			*list = p->link;
			DeallocateMemory(p);
		}
	}
}
/*
 * Standard dictionary hash function as specified by OMF
 */
static void compute_hash(const char * name, int blocks, hash *h)
{
  int len = strlen(name);
  const char *pb = name, *pe = name + len;
  const ushort blank = ' ';

  uint block_x = len | blank, bucket_d = len | blank;
	uint block_d = 0, bucket_x = 0;

  while (1) {
    ushort cback = *(--pe) | blank;	/* This is a cute way to get lowercase */
					/* And uppercase names to hash the same way */
		ushort cfront = *(pb++) | blank;
		bucket_x = ROTR(bucket_x, 2) ^ cback;
		block_d = ROTL(block_d, 2) ^ cback;
		if (--len == 0)
			break;
		block_x = ROTL(block_x,2) ^ cfront;
    bucket_d = ROTR(bucket_d,2) ^ cfront;
  }
  h->block_x = block_x % blocks;
  h->block_d = block_d % blocks;
	h->block_d = MAX(h->block_d, 1);
  h->bucket_x = bucket_x % LIB_BUCKETS;
  h->bucket_d = bucket_d % LIB_BUCKETS;
  h->bucket_d = MAX(h->bucket_d, 1);
}
/* 
 * Position us somewhere in the library file
 */

static void PositionTo(FILE *in, long position)
{
  fpos_t thepos = position;
  fsetpos(in, &thepos);
}
/*
 * Read data from file
 */
static void ReadFileData(BYTE *buf, int len, FILE *f, char *name)
{
	if (len != (fread(buf,1,len,f)))
		fatal("Library has bad format");
}
/*
 * Read the library header
 */
static uint ReadLibraryHeader(FILE *in, char *name, long *dict_offset,
		uint *dict_size, uint *pagesize, uint *flags)
{
  char buffer[33];
	char ch;
	int pv,mc;
  PositionTo(in, 0);
  ReadFileData(&buffer, 32, in, name);
	buffer[32] = 0;
	if (strncmp(buffer,"LS",2) || (6 != sscanf(buffer+2,"%x,%lx,%x,%x,%x,%c.",
				&pv,dict_offset,dict_size,pagesize,&mc, &ch)))
    fatal("Library %s has bad format\n",name);
	if (ch == 'C')
		*flags = LIB_CASE_SENSITIVE;
	else
		*flags = 0;
  return(32);
}
/* 
 * Load a module from a library once we have the paragraph number 
 */
static void LoadModule(FILE *in, long paragraph, char *name,
			uint pagesize,uint pass)
{
  PositionTo(in, paragraph*pagesize);
  ReadModule(in, name, pass,TRUE);
}
/*
 * Search the dictionary for the specified public
 * It's still unclear exactly what the librarian does, but it only seems
 * to apply the bucket shift once per block
 */
static char *SearchForPublic(FILE *in, long dict_offset, uint dict_size, uint pagesize,
		char * public, char *name, long *paragraph)
{
	static DICTPAGE thispage;
  hash h,orig;
  BOOL exit = FALSE;
  uint current_block=-1;

  *paragraph = 0;
  compute_hash(public, dict_size, &h);
	orig = h;

  while (!exit) {
    uint q;
		if (h.block_x != current_block) {
		  PositionTo(in, h.block_x *LIB_PAGESIZE+dict_offset);
      ReadFileData(&thispage,LIB_PAGESIZE, in, name);
			current_block = h.block_x;
		}
		if ((q=thispage.f.htab[h.bucket_x]) == 0)
			if (thispage.f.fflag == 0xff) {
#ifdef DEBUG
					    printf("Diag: hash block shift searching public %s\n", public);
#endif /* DEBUG */
				h.block_x = (h.block_x+h.block_d) %dict_size ;
				if (h.block_x == orig.block_x)
		      exit = TRUE;
			}
			else
			  return(0);
		else {
			BYTE *pos = &thispage.bytes[q*2];
			  if (!strncmp((char *)(pos+1),public, *pos)) {
					int namehash;
				  BYTE *dataoffset;
#ifdef DEBUG
				  printf("Found public %s in library %s\n",public, name);
#endif /* DEBUG */
					dataoffset = pos + 1 + *pos;
					*paragraph = (*dataoffset << 24) + (*(dataoffset+1)<<16)
						+(*(dataoffset+2)<<8) + *(dataoffset+3);
					namehash = *((BYTE *)dataoffset + 4);
					namehash = thispage.f.htab[namehash] * 2;
					if (namehash == 0)
						return(name);
					return((char *)&thispage.bytes[namehash]);
        }
				else {
#ifdef DEBUG
					printf("Diag: hash bucket shift searching public %s\n",public);
#endif /* DEBUG */
					h.bucket_x = (h.bucket_x + h.bucket_d) %LIB_BUCKETS;
					if (h.bucket_x == orig.bucket_x) {
						if (thispage.f.fflag != 0xff)
							exit = TRUE;
						else {
#ifdef DEBUG
					    printf("Diag: hash block shift searching public %s\n", public);
#endif /* DEBUG */
					    h.block_x = (h.block_x+h.block_d) %dict_size ;
					    if (h.block_x == orig.block_x)
		            exit = TRUE;
						}
					}
				}
		}
	}
	return(0);
}
/*
 * Add a module to the linked list of modules to be loaded in pass 2
 */
static void AddToModuleList(char *modulename, char *libname, long paragraph, uint pagesize)
{                           
  MODLIST *temp = AllocateMemory(sizeof(MODLIST));
  temp->module = modulename;
  temp->lib = libname;
  temp->paragraph = paragraph;
	temp->pagesize = pagesize;
  AppendToList(&modulelist, temp);
}
/*
 * Read through a given library over and over until
 * no more publics can be resolved.  The library may have to be read
 * more than once because loading a module from it can result in
 * defining more publics.  However, each extern is flagged with libraries
 * it is not found in (up to 32 libraries allowed) to keep from repeated searches
 */
static void ResolveLibraryPublics(FILE *in, BOOL *ReadFromLibrary, 
		long offset, uint size, uint pagesize, char *name, long libmask)
{
  int i;
  BOOL FoundModules = TRUE;
	loadlib = TRUE;
  while (FoundModules) {
			LIST *q = unresolvedexterns;
#ifdef DEBUG
          printf("Searching library %s\n",name);
#endif /* DEBUG */
    	FoundModules = FALSE;
			while (q) {
      	EXTERN *p = q->data;
				long paragraph;
				if (!(p->nonlibs & libmask) && !LookupHash(publichash,p->name)) {
			    SearchForPublic(in, offset,size,pagesize,p->name, 
					  name,&paragraph);
				  if (paragraph) {
					  *ReadFromLibrary = TRUE;
					  FoundModules = TRUE;
					  LoadModule(in, paragraph, name,pagesize,SCAN);
#ifdef DEBUG
 					  printf("Loaded library %s module %s\n",name,modname);
#endif /* DEBUG */
 					  AddToModuleList(modname, name, paragraph,pagesize);
 					  break;
 					}
 					else
 						p->nonlibs |= libmask;
				}
#ifdef DEBUG
				else
					printf("Skipped search of %s in library %s\n",p->name, name);
#endif
				q = q->link;
			}
	}
	loadlib = FALSE;
}
/*
 * Error- missing library
 */
static void MissingLibrary(char *name)
{
  fatal("Missing library %s", name);
}
/*
 * Pass 1 Library search
 *   Reads dictionary and does external matching
 *   searches all libraries continuously until no more modules are loaded
 */
void SearchLibraries(LIST *liblist)
{
  if (liblist) {
		int SomethingFound = TRUE;
		BOOL case_sensitive = FALSE;
    while(SomethingFound) {
			LIST *p = liblist;
			long libmask = 1;
			SomethingFound = FALSE;
      while (p) {
  			FILE *in;
				long offset;
				uint size, flags, pagesize;
  			if ((in = fopen((char *)p->data,"rb")) == 0)
					MissingLibrary(p->data);
				ReadLibraryHeader(in, p->data, &offset, &size, &pagesize, &flags);
				if (flags & ~ LIB_CASE_SENSITIVE)
					Error("Unknown library flags %x", flags);
				case_sensitive |= (flags & LIB_CASE_SENSITIVE) && prm_case_sensitive;
 		    ResolveLibraryPublics(in,&SomethingFound, 
					offset, size,pagesize, p->data, libmask);
			  fclose(in);
				libmask <<= 1;
        p=p->link;
			}
    }
		if (case_sensitive&&!prm_case_sensitive)
			Error("One or more case-sensitive libraries included in non-sensitive link");
  }
}
/*   
 * Pass 2 library load
 * Pass through the linked list and load all required modules
 */

void LoadLibraries(void)
{
  LIST *p = modulelist;
  char *lib = 0;
  FILE *in = 0;
  while (p) {
		MODLIST *m = p->data;
    if (m->lib != lib) {
			lib = m->lib;
#ifdef DEBUG
			printf("Loading library %s\n",lib);
#endif
			if (in)
				fclose(in);
			in = fopen(lib,"rb");
			if (!in) MissingLibrary(lib);
		}
#ifdef DEBUG
		printf("  Module %s\n",m->module);
#endif
		LoadModule(in, m->paragraph, m->module,m->pagesize, RESOLVE);
    p=p->link;
	}
	if (in)
    fclose(in);
}  