{$U-,C-,I-,K-}
TYPE
    tCport = (com1, com2);
    tSaveInt = RECORD                  {Save interrupt vector address}
       IP :  INTEGER;
       CS :  INTEGER;
    END;
    tRegs    = RECORD                  {Dos registers}
       AX, BX, CX, DX, BP, SI, DI, DS, ES, Flags : INTEGER;
    END;
CONST
     RECV_BUF_SIZE = 2048;             {this may be changed to
                                        whatever size you need}
{ *** Port addresses *** }
     cRBR: ARRAY[com1..com2] OF INTEGER = ($3F8,$2F8);
                                       {Receive Buffer Register}
     cTHR: ARRAY[com1..com2] OF INTEGER = ($3F8,$2F8);
                                       {Transmitter Holding Register: the
                                        serial port address we use to send
                                        data}
     cIER: ARRAY[com1..com2] OF INTEGER = ($3F9,$2F9);
                                       {Interrupt Enable Register for the
                                        serial port}
     cLCR: ARRAY[com1..com2] OF INTEGER = ($3FB,$2FB);
                                       {Line Control Register for the serial
                                        port. Determines data bits, stop bits
                                        and parity, contributes to setting
                                        baud-rate}
     cMCR: ARRAY[com1..com2] OF INTEGER = ($3FC,$2FC);
                                       {Modem Control Register}
     cLSR: ARRAY[com1..com2] OF INTEGER = ($3FD,$2FD);
                                       {Line Status Register}
     cMSR: ARRAY[com1..com2] OF INTEGER = ($3FE,$2FE);
                                       {Modem Status Register}
     IMR = $021;                       {Interrupt Mask Register port address
                                        of Intel 8259A Programmable Interrupt
                                        controller}
{ *** Masks *** }
     ENABLE_OUT2 = 8;                  {Setting bit 3 of MCR enables OUT2}
     ENABLE_DAV = 1;                   {Setting bit 0 of IER enables Data
                                        AVailable interrupt from serial port}
     ENABLE_IRQ: ARRAY[com1..com2] OF INTEGER = ($00EF,$00F7);
                                       {Clearing bit of IMR enables serial
                                        interrupts to reach the CPU}
     DISABLE_OUT2 = 1;                 {Clearing MCR disables OUT2}
     DISABLE_DAV = 0;                  {Clearing IER disables Data
                                       AVailable interrupt from serial port}
     DISABLE_IRQ: ARRAY[com1..com2] OF INTEGER = ($0010,$0008);
                                       {Setting bit of IMR stops serial
                                        interrupts from reaching the CPU}
     cINVLIST: ARRAY[com1..com2] OF INTEGER = ($000C,$000B);
                                       {Interrupt vector number}
     SET_BAUD = $80;                   {Setting bit 7 of LCR allows us to set
                                        the baud rate of the serial port}
     SET_PARMS = $7F;                  {Clearing bit 7 of LCR allows us to set
                                        non-baud-rate parameters on the
                                        serial port}
TYPE
    parity_set        = (none,even);    {readability and expansion}
VAR
   SaveInt                           : tSaveInt;
   Regs                              : tRegs;
   RBR, THR, IER, LCR, MCR, LSR, MSR : INTEGER;
   INVLIST                           : INTEGER;
   SavIER, SavLCR, SavMCR, SavIMR    : INTEGER;
   buf_start, buf_end    : INTEGER;    {NOTE: these will change by them-
                                        selves in the background}
   recv_buffer           : ARRAY [1..RECV_BUF_SIZE] OF Byte;
                                       {also self-changing}
   speed                 : INTEGER;    {I don't know the top speed these
                                        routines will handle}
   dbits                 : 7..8;       {only ones most people use}
   stop_bits             : 1..2;       {does anyone use 2?}
   parity                : parity_set;  {even and none are the common ones}
   Cport                 : tCport;     {set at initialization}

FUNCTION cgetc(TimeLimit : INTEGER) : INTEGER;
{if a byte is recieved at COM1/COM2: in less than TimeLimit seconds,
 returns byte as an integer, else returns -1}
CONST
     TIMED_OUT = -1;
BEGIN
     TimeLimit := TimeLimit ShL 10;     {convert TimeLimit to millisecs}
     WHILE (buf_start = buf_end) AND (TimeLimit > 0) DO
     BEGIN
          Delay(1);
          TimeLimit := PRED(TimeLimit)
     END;
     IF (TimeLimit >= 0) AND (buf_start <> buf_end) THEN
     BEGIN
          InLine ($FA);            {suspend interrupts}
          cgetc := recv_buffer[buf_start];
          buf_start := SUCC(buf_start);
          IF buf_start > RECV_BUF_SIZE THEN
          buf_start := 1;
          InLine ($FB);            {resume interrupts}
     END
     ELSE
         cgetc := TIMED_OUT;
END;

PROCEDURE send(c : Byte);
VAR
   a : Byte;
BEGIN
{
  repeat
       a := port[LSR]
  until odd(a shr 5);
  port[THR] := c;
 }
  InLine(
     $8B/$16/LSR/   {in: mov dx,[LSR]}
     $EC/               {in  al,dx}
     $24/$20/           {and al,20}
     $74/$F7/           {jz  in}
     $8B/$16/THR/       {mov dx,[THR]}
     $36/               {ss:}
     $8B/$86/c/         {mov ax,[bp+c]}
     $EE                {out dx,al}
        );
END;


{Communications routines for TURBO Pascal written by Alan Bishop,
 modified slightly by Scott Murphy.  Modified by Peter Boswell to
 add COM2, saving the original interrupt vector, more inline code.
 Handles standart COM1/COM2: ports with interrupt handling.  Includes
 support for only one port, and with no overflow, parity, or other
 such checking.  However, even some of the best communication programs
 don't do this anyway, and I never use it.  If you make modifications,
 please send me a copy if you have a simple way of doing it (CIS EMAIL,
 Usenet, MCI Mail, etc)  Hope these are useful.

Alan Bishop - CIS      - 72405,647
              Usenet   - bishop@ecsvax
              MCI Mail - ABISHOP
}
PROCEDURE update_uart;
{uses dbits, stop_bits, and parity}
VAR
   newparm, oldLCR : Byte;
BEGIN
 newparm := dbits-5;
 IF stop_bits = 2 THEN newparm := newparm + 4;
 IF parity = even THEN newparm := newparm + 24;
 oldLCR := Port[LCR];
 Port[LCR] := oldLCR AND SET_PARMS;
 Port[LCR] := newparm;
END;

PROCEDURE term_ready(state : BOOLEAN);
{if state = TRUE then set RTS true else set false}
VAR
   OldMCR : Byte;
BEGIN
     OldMCR := Port[MCR];
     IF state THEN
        Port[MCR] := OldMCR OR 1
     ELSE
         Port[MCR] := OldMCR AND $FE
END;

FUNCTION carrier : BOOLEAN;
{true if carrier, false if not}
BEGIN
 carrier := ODD(Port[MSR] ShR 7);
END;

PROCEDURE set_up_recv_buffer;
BEGIN
 buf_start := 1;
 buf_end   := 1;
END;

PROCEDURE new_baud(rate : INTEGER);
{has no problems with non-standard bauds}
VAR
   OldLCR : Byte;
BEGIN
 IF rate <= 9600 THEN
 BEGIN
  speed := rate;
  rate := TRUNC(115200.0/rate);
  OldLCR := Port[LCR] OR SET_BAUD;
  Port[LCR] := OldLCR;
  Port[THR] := Lo(rate);
  Port[IER] := Hi(rate);
  Port[LCR] := OldLCR AND SET_PARMS;
 END;
END;

PROCEDURE init_port;
{installs interrupt sevice routine for serial port}
VAR a,b : INTEGER;
    buf_len : INTEGER;
BEGIN
 RBR := cRBR[Cport];
 THR := cTHR[Cport];
 IER := cIER[Cport];
 LCR := cLCR[Cport];
 MCR := cMCR[Cport];
 LSR := cLSR[Cport];
 MSR := cMSR[Cport];
 INVLIST := cINVLIST[Cport];
 update_uart;
 new_baud(speed);
 buf_len := RECV_BUF_SIZE;

 {save the original vector}

 WITH Regs DO BEGIN
   AX := $3500 + INVLIST;
   MSDos(Regs);
   SaveInt.CS := ES;
   SaveInt.IP := BX;
 END;

 {this is the background routine}

 InLine (
              $FA/                     {cli}
              $1E/                     {push ds}
              $0E/                     {push cs}
              $A1/INVLIST/             {mov  ax,[INVLIST]   interrupt vector}
              $1F/                     {pop  ds                  ;ds := cs}
              $BA/*+22/                {mov  dx, offset ISR}
              $B4/$25/                 {mov  ah,25H}
              $CD/$21/                 {int  21H}
              $8B/$BE/BUF_LEN/         {mov  di, buf_len}
              $89/$3E/*+88/            {mov  lcl_buf_len,di}
              $1F/                     {pop  ds}
              $2E/$8C/$1E/*+84/        {mov  lcl_ds, ds}
              $EB/$52/                 {jmp  exit}
{ISR:}        $1E/                     {push ds}
              $50/                     {push ax}
              $53/                     {push bx}
              $52/                     {push dx}
              $56/                     {push si}
              $FB/                     {sti}
              $2E/$8E/$1E/*+71/        {mov  ds,[lcl_ds]}
              $8B/$16/RBR/             {mov  dx,[RBR] 3F8H/2F8H ;address RBR}
              $EC/                     {in   al, dx             ;read rbr}
              $BE/RECV_BUFFER/         {mov  si, recv_buffer    ;address start of recv_buffer}
              $8B/$1E/BUF_END/         {mov  bx, [buf_end]      ;index of current char in recv_buffer}
              $88/$40/$FF/             {mov  [bx+si-1],al       ;copy char to recv_buffer}
              $43/                     {inc  bx                 ;update buf_end}
              $E8/$22/$00/             {call adj_idx}
              $89/$1E/BUF_END/         {mov  [buf_end],bx}
              $3B/$1E/BUF_START/       {cmp  bx, [buf_start]}
              $75/$0C/                 {jnz  ISR_DONE}
              $8B/$1E/BUF_START/       {mov  bx,buf_start}
              $43/                     {inc  bx}
              $E8/$10/$00/             {call adj_idx}
              $89/$1E/BUF_START/       {mov  [buf_start],bx}
              $BA/$20/$00/             {mov  dx,20H            ;EOI command for 8259A PIC}
              $B0/$20/                 {mov  al,20H            ;EOI port for 8259A PIC}
              $EE/                     {out  dx,al             ;End Of Interrupt}
              $5E/                     {pop  si}
              $5A/                     {pop  dx}
              $5B/                     {pop  bx}
              $58/                     {pop  ax}
              $1F/                     {pop  ds}
              $CF/                     {iret}
{adj_idx:}    $2E/$8B/$16/*+11/        {mov  dx,[lcl_buf_len]}
              $42/                     {inc  dx}
              $39/$DA/                 {cmp  dx,bx}
              $75/$03/                 {jnz  no_change}
              $BB/$01/$00/             {mov  bx,1}
{no_change:}  $C3/                     {ret}
{lcl_buf_len;}$00/$00/                 {dw  0}
              $00/$01/                 {dw  1}
{exit:}       $90/                     {nop}
              $FB                      {cli}
 );
 SavLCR    := Port[LCR];
 SavIER    := Port[IER];
 Port[IER] := ENABLE_DAV;              {interrupt enable}
 a         := Port[MCR];
 SavMCR    := a;
 Port[MCR] := a OR ENABLE_OUT2;        {preserve RTS and enable OUT2}
 a         := Port[IMR];
 SavIMR    := a;
 a         := a AND ENABLE_IRQ[Cport];
 Port[IMR] := a;
END;


PROCEDURE remove_port;
{Restores status to reflect the status upon entry}
VAR
   a : Byte;
BEGIN
     InLine($FA);                     {disable interrupts}
     Port[IMR] := SavIMR;
     Port[IER] := SavIER;
     Port[MCR] := SavMCR;
     Port[LCR] := SavLCR;

     {Restore the interrupt vector}
     WITH Regs DO BEGIN
        AX := $2500 + INVLIST;        {function 25H & INVLIST port}
        DS := SaveInt.CS;
        DX := SaveInt.IP;
        MSDos(Regs);
     END;
     InLine($FB);
END;


PROCEDURE break;
{send a break}
VAR a,b : Byte;
BEGIN
 a := Port[LCR];
 b := (a AND $7F) OR $40;
 Port[LCR] := b;
 Delay(750);
 Port[LCR] := a;
END;

