/*
  tad.c - Time and Date.

  Display the current time and date (using a variety of formats); set
  (optionally without display) the time and date from the real-time clock or
  from the command line; correct for real-time clock inaccuracies.

  Jason Hood, 14 & 15 January, 1997.

  30 November & 1 December, 1998:
    reformatted;
    able to set time and date from command line.

  3 April, 1999:
    display of "approximate" time.

  10 May, 2000:
    clock display.

  30 & 31 May, 3 June, 2000:
    rewrote option processing (and modified parse() accordingly);
    added correction to counter RTC drift;
    added option to set system time, but not RTC;
    return the current 10-minute interval (midnight is 0, noon is 72);
    used Time and Date structures.

  15 to 21 June, 2000, v1.00:
    corrected bug when displaying calibration end (was using same Date pointer);
    used a function to display calibration time (and removed seconds) and
     seconds-a-day calculation;
    improved all the display output;
    added get_ and set_{system,rtc}_tad() and {read,write}_calib();
    display of "brief" time;
    added copyright and version info for distribution;
    recognize '/' as a switch character;
    use brief option to shorten the duration display.

  15 July, 2000, v1.01 (Simtel.Net):
    more reliable monitoring;
    quiet clock - remove display when finished.

  26 July, 2000, v1.10:
    changed brief option to "H:mm:ss d/m/yy" (use two b's for concise);
    added "-t" option to use the real-time clock instead of system time;
    added "-e" option to select the return value;
    added "-y" option to display the current day of the year;
    imposed time restrictions on calibration (must be at least a second out).

  2 August, 2000, v1.11:
    added "-tt" option to display difference between RTC and system times;
    wait for RTC update with "-s" and "-k";
    added open_calib() function.

  20 February, 2001, v1.12:
    made "-t" work with all the calibration stuff (but still sets both and still
     displays "System");
    display the time (without date) when no adjustment was made.

  2 June, 2001, v1.13:
    corrected a bug with the approximate time when it was "to twelve" (midnight)
     (thanks to Waldemar Schultz for pointing it out).

  8 June, 2001:
    djgpp version.

  10 to 22 June, 2001, v2.00:
    use "midnight" and "midday" instead of "twelve" in approximate time;
    use a custom format string (added abbreviated month and day names, 12hr);
    added brief and concise approximate time displays;
    use "a quarter" instead of "quarter" in approximate time;
    add and subtract times and durations (introduced Duration type);
    added current week number option "-w" (January 1 to 7 is week 1, December
     31 (and 30 in a leap year) is week 53);
    German language support (courtesy of Waldemar);
    added "-rr <time>" to update the reference without calibration;
    added TAD structure (I really should be using C++), time_to_secs(),
     secs_to_time() and day_of_year() functions;
    rewrote str_time() and str_dur() with buffers;
    use local decimal symbol in secs_per_day();
    added German message to secs_per_day().

  20 to 23 October, 2004, v2.10:
    Win32 port (-t is ignored, -ks is done via file);
    setting only hour will no longer set minute and second to zero;
    allow setting any component of time/date without changing the others;
    fixed bug with 24-hour time ($C) when country was 12-hour.


  Feel free to use any portion of this code for your own use.
*/

//#define LANG_de

#define VERSION "2.10"
#define DATE	"23 October, 2004"


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <conio.h>
#ifndef __DJGPP__
#include <io.h>
#else
#include <unistd.h>
#include <sys/farptr.h>
#include <go32.h>
#include <dpmi.h>
#define _NAIVE_DOS_REGS
#endif
#ifndef _WIN32
#include <dos.h>
#endif


typedef struct
{
  int hour,
      minute,
      second;
} Time;

typedef struct
{
  int day,
      month,
      year;
} Date;

typedef struct
{
  Time t;
  Date d;
} TAD;

typedef struct
{
  unsigned long	seconds;
  long days;
  int  months,
       years;
} Duration;


#ifdef LANG_de
#include "lang-de.c"
#else
#include "lang.c"
#endif

char* duration_char = exit_char + 2;
#define YEARS	duration_char[0]
#define MONTHS	duration_char[1]
#define WEEKS	duration_char[2]
#define DAYS	duration_char[3]
#define HOURS	duration_char[4]
#define MINUTES duration_char[5]
#define SECONDS duration_char[6]


char* format  = NULL;
int   brief   = 0;
int   cont    = 0;
int   rtc2sys = 0;
int   leave   = 0;
int   calib   = 0;
int   ref     = 0;
int   monitor = 0;
int   drift   = 0;
int   view    = 0;
int   quiet   = 0;

char* tadfile;

int   dayofweek;
int   hundredths;


void  copyright( void );
void  help( void );
void  format_help( void );
void  format_list( void );
void  exit_help( void );
void  calc_help( void );

void  get_country_info( void );

void  get_system_time( Time* );
void  get_system_date( Date* );
void  get_system_tad( TAD* );

void  set_system_time( Time* );
void  set_system_date( Date* );
void  set_system_tad( TAD* );

#ifndef _WIN32
void  get_rtc_time( Time* );
void  get_rtc_date( Date* );
void  get_rtc_tad( TAD* );

void  set_rtc_time( Time* );
void  set_rtc_date( Date* );
void  set_rtc_tad( TAD* );

void  compare( void );
#endif

void  wait_for_rtc( Time* );

void  (*get_time)( Time* ) = get_system_time;
void  (*get_date)( Date* ) = get_system_date;
void  (*get_tad)( TAD* ) = get_system_tad;
//void	(*set_time)( Time* ) = set_system_time;
//void	(*set_date)( Date* ) = set_system_date;
void  (*set_tad)( TAD* ) = set_system_tad;

void  parse( char*, TAD*, Duration* );


FILE* open_calib( char* );
void  read_calib( FILE*, long*, long*, long* );
void  write_calib( FILE*, long, long, long );
void  calibrate( TAD*, TAD* );
void  update_ref( void );
void  monitor_time( void );
void  adjust( void );
void  view_calibration( void );
char* str_time( char*, TAD* );
char* str_dur( char*, long );
void  secs_per_day( long, long );

void  time_to_dur( TAD*, Duration* );
void  dur_to_time( Duration*, TAD* );
void  dur_add( Duration*, Duration* );
void  dur_sub( Duration*, Duration* );
void  date_duration( TAD*, TAD*, int, Duration* );
void  calc_date( TAD*, int, Duration* );
void  normalise( Duration* );
void  duration_add( Duration*, Duration* );
void  duration_sub( Duration*, Duration* );
int   no_duration( Duration* );
void  calc( TAD*, Duration*, int, TAD*, Duration*, int );
char* str_duration( char*, Duration*, int );
char* ultos( unsigned long );

int   isleap( unsigned );
unsigned months_to_days( unsigned );
long  years_to_days( unsigned );
long  date_to_days( Date* );
void  days_to_date( long, Date* );
long  time_to_secs( Time* );
void  secs_to_time( long, Time* );
long  tad_to_secs( TAD* );
void  secs_to_tad( long, TAD* );
int   day_of_year( Date* );

char* format_time( char*, char*, TAD* );
int   format_date( char*, Date*, int, int );
int   approx_time( char*, Time*, int );
char* place_str( int num );


int main( int argc, char *argv[] )
{
  TAD	    t  = { { -1, -1, -1 }, { -1, -1, -1 } },
	    t2 = t;
  Duration  d1 = { 0, 0, 0, 0 }, d2 = { 0, 0, 0, 0 };
  TAD*	    tp = &t;
  Duration* dp = &d1;
  int	    special = 0;
#ifdef _WIN32
  char*     rtcfile;
  FILE*     rtc;
  TAD	    t3;
#else
  int	    cmp = 0;
#endif
  char**    fmt = time_fmt;
  char	    buf[256];
  int	    op = 0, eq = -1;
  int	    j, k;
  int	    c;
  char*     prc;
  int	    rc = -1;

  for (j = 1; j < argc; ++j)
  {
    c = *argv[j];
    if ((c == '+' || c == '-') && argv[j][1] == '\0')
    {
      op = c;
      eq = 0;
      tp = &t2;
      dp = &d2;
      ++special;
    }
    else if (c == '=')
    {
      eq = argv[j][1];
      if (eq == '?' || strchr( duration_char, eq ) == NULL)
      {
	calc_help();
	return 255;
      }
      ++special;
    }
    else if ((c == '-' || c == '/')
	     && (!isdigit( argv[j][1] ) && argv[j][1] != c))
    {
      for (k = 1; argv[j][k] != '\0'; ++k)
      {
	switch (tolower( argv[j][k] ))
	{
	  case 'a': fmt = approx_fmt; break;
	  case 'y': fmt = year_fmt;   break;
	  case 'w': fmt = week_fmt;   break;

	  case 'f': c = argv[j][++k];
		    if (c == '?' || (c == '\0' && j == argc - 1))
		    {
		      format_help();
		      return 255;
		    }
		    if (c == 'l')
		    {
		      format_list();
		      return 255;
		    }
		    if (c == '\0') ++j, k = 0;
		    format = argv[j] + k;
		    while (argv[j][++k] != '\0') ;
		    --k;
		    break;

	  case 'b': ++brief; break;
	  case 'c': ++cont;  break;

	  case 't':
#ifndef _WIN32
		    if (++cmp > 1) ++special;
		    else
		    {
		      get_time = get_rtc_time;
		      get_date = get_rtc_date;
		      get_tad  = get_rtc_tad;
		      //set_time = set_rtc_time;
		      //set_date = set_rtc_date;
		      set_tad  = set_rtc_tad;
		    }
#endif
		    break;

	  case 's': ++rtc2sys; ++special; break;
	  case 'k': ++leave;   ++special; break;
	  case 'n': ++calib;   ++special; break;
	  case 'r': ++ref;     ++special; break;
	  case 'm': ++monitor; ++special; break;
	  case 'd': ++drift;   ++special; break;
	  case 'v': ++view;    ++special; break;
	  case 'q': ++quiet;              break;

	  case 'e': c = argv[j][++k];
		    if (c == '\0' || c == '?' ||
			(prc = strchr( exit_char, c )) == NULL)
		    {
		      exit_help();
		      return 255;
		    }
		    rc = prc - exit_char;
		    break;

	  case '/': break;

	  default:  help();
		    return 255;
	}
      }
    }
    else parse( argv[j], tp, dp );
  }
  if (brief > 2) brief = 2;
  if (!no_duration( &d1 ) && op == 0 && eq == -1)
  {
    op = '+';
    d2 = d1;
    d1.seconds = d1.days = 0;
    d1.months = d1.years = 0;
    ++special;
  }

  tadfile = malloc( (j = strlen( argv[0] )) + 8 );
  strcpy( tadfile, argv[0] );
  while (--j >= 0
	 && argv[0][j] != '\\' && argv[0][j] != '/' && argv[0][j] != ':')
    /* do nothing */;
  strcpy( tadfile+j+1, "tad.dat" );
#ifdef _WIN32
  rtcfile = strdup( tadfile );
  strcpy( rtcfile + strlen( rtcfile ) - 3, "rtc" );
#endif

  get_country_info();

  if (!quiet && isatty( 1 )) putchar( '\n' );

  if (op) calc( &t, &d1, op, &t2, &d2, eq );
  else if (eq != -1)
  {
    if (!no_duration( &d1 )) normalise( &d1 );
    else
    {
      if (t.t.hour == -1 && t.d.day == -1)
	get_tad( &t );
      if (t.t.hour != -1) d1.seconds = time_to_secs( &t.t );
      if (t.d.day  != -1) d1.days    = day_of_year( &t.d );
    }
    puts( str_duration( buf, &d1, eq ) );
  }

  if (!special) set_tad( &t );

  if (leave)
  {
    if (t.t.hour == -1 && t.d.day == -1)
    {
      fputs( "It would help to specify a time and/or date.\n", stderr );
      return 255;
    }
#ifdef _WIN32
    rtc = fopen( rtcfile, "wb" );
    if (rtc == NULL)
    {
      fputs( "Failed to create temporary file.\n", stderr );
      return 255;
    }
    wait_for_rtc( &t2.t );
    get_system_date( &t2.d );
    set_system_tad( &t );
    get_system_tad( &t );	// ensure both time and date are valid
    fwrite( &t2, sizeof(t2), 1, rtc );
    fwrite( &t, sizeof(t), 1, rtc );
    fclose( rtc );
#else
    wait_for_rtc( &t2.t );
    get_rtc_date( &t2.d );
    set_system_tad( &t );
    set_rtc_tad( &t2 );
#endif
  }

  if (rtc2sys)
  {
#ifdef _WIN32
    rtc = fopen( rtcfile, "rb" );
    if (rtc == NULL)
    {
      fputs( "Failed to open temporary file (use -k first).\n", stderr );
      return 255;
    }
    fread( &t, sizeof(t), 1, rtc );
    fread( &t2, sizeof(t2), 1, rtc );
    fclose( rtc );
    unlink( rtcfile );
    wait_for_rtc( &t3.t );
    get_system_date( &t3.d );
    date_duration( &t2, &t3, DAYS, &d1 );
    calc_date( &t, '+', &d1 );
#else
    wait_for_rtc( &t.t );
    get_rtc_date( &t.d );
#endif
    set_system_tad( &t );
  }

  if (ref)
  {
    if (access( tadfile, 0 ) != 0) ++calib;
    else if (t.t.hour == -1 || ref > 1) update_ref();
    else
    {
      get_tad( &t2 );
      set_system_time( &t.t );
      t.d = t2.d;
      calibrate( &t2, &t );
    }
  }

  if (calib && !monitor)
  {
    set_system_tad( &t );
    calibrate( NULL, &t );
  }

  if (monitor) monitor_time();

  if (drift) adjust();

  if (view) view_calibration();

#ifndef _WIN32
  if (cmp > 1) compare();
#endif

  if (cont)
  {
    if (quiet) putch( '\n' );
    buf[0] = '\r';
    while (!kbhit())
    {
      get_time( &t.t );
      format_time( buf+1, "$+2T", &t );
      cputs( buf );
#ifdef _WIN32
      Sleep( 100 );
#endif
    }
    while (kbhit()) getch();
    if (quiet)
    {
      putch( '\r' ); clreol();
      gotoxy( 1, wherey() - 1 );	// Counteract COMMAND.COM's newline
    }
    else putch( '\n' );
  }
  else if (!quiet && (!special || rtc2sys || leave))
  {
    if (format == NULL) format = fmt[brief];
    get_tad( &t );
    puts( format_time( buf, format, &t ) );
  }

  free( tadfile );

  get_tad( &t );
  switch (rc)
  {
    default: rc = (t.t.hour * 60 + t.t.minute) / 10; break;
    case 0:  rc = dayofweek;			     break;
    case 1:  rc = t.d.year / 100;		     break;
    case 2:  rc = t.d.year % 100;		     break;
    case 3:  rc = t.d.month;			     break;
    case 4:  rc = day_of_year( &t.d ) / 7 + 1;	     break;
    case 5:  rc = t.d.day;			     break;
    case 6:  rc = t.t.hour;			     break;
    case 7:  rc = t.t.minute;			     break;
    case 8:  rc = t.t.second;			     break;
  }
  return rc;
}


void parse( char* str, TAD* t, Duration* d )
{
  unsigned long	num;
  char* bp;
  int	hd = -1, mm = -1, sy = -1;
  int	seen_time;
  char* dur;
  char* fmt = str;

  num = strtoul( str, &bp, 10 );
  if (*bp == '\0' || (dur = strchr( duration_char, *bp )) == NULL)
  {
    if (bp != str)
      hd = (int)num;
    seen_time = (*bp == ':');
    if (*bp != '\0')
    {
      str = bp + 1;
      num = strtol( str, &bp, 10 );
      if (bp != str)
	mm = (int)num;
      if (*bp != '\0')
      {
	str = bp + 1;
	num = strtol( str, &bp, 10 );
	if (bp != str)
	  sy = (int)num;
      }
    }
    if (hd == -1 && mm == -1 && sy == -1)
      format = fmt;
    else if (seen_time)
    {
      get_time( &t->t );
      if (hd != -1) t->t.hour	= hd;
      if (mm != -1) t->t.minute = mm;
      if (sy != -1) t->t.second = sy;
    }
    else
    {
      get_date( &t->d );
      if (hd != -1) t->d.day   = hd;
      if (mm != -1) t->d.month = mm;
      if (sy != -1)
      {
	if (sy < 100) sy += (sy >= 80) ? 1900 : 2000;
	t->d.year = sy;
      }
    }
  }
  else
  {
    do
    {
      switch (dur - duration_char)
      {
	case 0: d->years   = (int)num;	  break;
	case 1: d->months  = (int)num;	  break;
	case 2: d->days    += num * 7;	  break;
	case 3: d->days    += num;	  break;
	case 4: d->seconds += num * 3600; break;
	case 5: d->seconds += num * 60;   break;
	case 6: d->seconds += num;	  break;
      }
      num = strtoul( bp+1, &bp, 10 );
      dur = strchr( duration_char, *bp );
    }
    while (*bp != '\0' && dur != NULL);
  }
}


// --------------------------------------------------------------------------

#ifndef _WIN32
void compare( void )
{
  TAD  rt, st;
  int  s, h, d;
  char rbuf[9], sbuf[9];

  wait_for_rtc( &rt.t );
  get_system_time( &st.t );
  h = (int)(time_to_secs( &rt.t ) - time_to_secs( &st.t )) * 100 - hundredths;
  d = (h < 0);
  if (d) h = -h;
  s = h / 100;
  h %= 100;
  printf( "RTC time:    %s\n"
	  "System time: %s%c%02d\n"
	  "\n"
	  "The RTC time is %s by %d%c%02d seconds.\n",
	  format_time( rbuf, "$+2C", &rt ),
	  format_time( sbuf, "$+2C", &st ), deci_sep, hundredths,
	  (d) ? "behind" : "ahead", s, deci_sep, h );
}
#endif


FILE* open_calib( char* mode )
{
  FILE* file = fopen( tadfile, mode );
  if (file == NULL)
  {
    fprintf( stderr, "Unable to %s calibration file %s!\n",
	     (*mode == 'r') ? "open" : "create", tadfile );
    exit( 255 );
  }
  return file;
}

void read_calib( FILE* dat, long* ref, long* diff, long* dur )
{
  fread( ref,  sizeof(long), 1, dat );
  fread( diff, sizeof(long), 1, dat );
  fread( dur,  sizeof(long), 1, dat );
}

void write_calib( FILE* dat, long ref, long diff, long dur )
{
  fwrite( &ref,  sizeof(long), 1, dat );
  fwrite( &diff, sizeof(long), 1, dat );
  fwrite( &dur,  sizeof(long), 1, dat );
}


void calibrate( TAD* told, TAD* tnew )
{
  FILE* dat;
  long	sold, snew, ref;
  long	diff, dur;
  char	tbuf[3][MAX_TIME];
  char	dbuf[2][MAX_DUR];

  if (told == NULL)
  {
    get_tad( tnew );
    snew = tad_to_secs( tnew );
    if (!quiet)
      printf( "Calibration started %s.\n", str_time( tbuf[0], tnew ) );
    diff = dur = 0;
    goto new_calibration;
  }

  sold = tad_to_secs( told );
  snew = tad_to_secs( tnew );

  if (snew == sold && quiet) return;

  dat = open_calib( "rb" );
  read_calib( dat, &ref, &diff, &dur );
  fclose( dat );
  if (!quiet)
  {
    TAD tref;
    secs_to_tad( ref, &tref );
    if (snew == sold)
    {
      puts( "The current time is correct - no reference made.\n" );
      if (dur == 0)
	printf( "Calibration has been in progress for %s\n"
		"(started at %s).\n",
		str_dur( dbuf[0], snew - ref ), str_time( tbuf[0], &tref ) );
      else
	printf( "Last reference was %s ago%c"
		"(at %s).\n",
		str_dur( dbuf[0], snew - ref ), (brief) ? ' ' : '\n',
		str_time( tbuf[0], &tref ) );
      return;
    }
    if (dur == 0)
    {
      printf( "Calibration started %s.\n"
	      "Calibration ended   %s.\n"
	      "System time was     %s.\n"
	      "\n"
	      "Calibration period was %s,\n"
	      "during which time the clock %s %s.\n",
	      str_time( tbuf[0], &tref ),
	      str_time( tbuf[1], tnew ),
	      str_time( tbuf[2], told ),
	      str_dur( dbuf[0], snew - ref ),
	      (snew <= sold) ? "gained" : "lost",
	      str_dur( dbuf[1], snew - sold ) );
    }
    else
    {
      printf( "Correct time: %s\n"
	      "System  time: %s\n"
	      "\n"
	      "It was %s since the last adjustment\n"
	      "or reference (which was at %s),\n"
	      "resulting in a %s of %s.\n",
	      str_time( tbuf[0], tnew ),
	      str_time( tbuf[1], told ),
	      str_dur( dbuf[0], snew - ref ),
	      str_time( tbuf[2], &tref ),
	      (snew <= sold) ? "gain" : "loss",
	      str_dur( dbuf[1], snew - sold ) );
    }
  }

  diff += snew - sold;
  dur  += sold - ref;

  if (!quiet) secs_per_day( diff, dur );

new_calibration:
  dat = open_calib( "wb" );
  write_calib( dat, snew, diff, dur );
  fclose( dat );
}


void update_ref( void )
{
  FILE* dat;
  TAD	t, tr;
  long	ref, old, diff, dur;
  char	tbuf[2][MAX_TIME];
  char	dbuf[MAX_DUR];

  get_tad( &t );
  ref = tad_to_secs( &t );
  dat = open_calib( "rb+" );
  read_calib( dat, &old, &diff, &dur );
  rewind( dat );
  write_calib( dat, ref, diff, dur );
  fclose( dat );
  if (!quiet)
  {
    secs_to_tad( old, &tr );
    printf( "%s %s.\n"
	    "\n"
	    "Previous reference or adjustment was at %s,\n"
	    "which was %s ago.\n",
	    (dur == 0) ? "Calibration restarted"
		       : "Reference set to",
	    str_time( tbuf[0], &t ),
	    str_time( tbuf[1], &tr ),
	    str_dur( dbuf, ref - old ) );
  }
}


void monitor_time( void )
{
  TAD  t, t1;
#ifdef _WIN32
  SYSTEMTIME st;
  int  diff;
#else
  long last, next, now;
# ifdef __DJGPP__
  #define ticks _farpeekl( _dos_ds, 0x46c )
# else
  volatile long far* const ticks_ptr = (long far*)0x46c;
  #define ticks *ticks_ptr
# endif
#endif
  static char* ind = "|/-\\";
  static char* str = "\b|";
  int p = 0;

  fputs( "Waiting for update...  ", stdout );
#ifdef __DJGPP__
  fflush( stdout );
#endif
  while (!kbhit())
  {
    get_time( &t.t );
#ifdef _WIN32
    Sleep( 100 );
    GetLocalTime( &st );
    cputs( str );
    diff = st.wMilliseconds / 10 - hundredths;
    if (diff < 0) diff += 100;
    if (diff > 12)
#else
    last = ticks;
    next = last + 4;
    while (ticks < next && ticks >= last)
      cputs( str );			// Need to be busy for Windows
    now = ticks;
    if (now < last || now > next + 4)   // next+3 was the slowest I saw
#endif
    {
      get_tad( &t1 ); t.d = t1.d;
      puts( "\bFound." ); if (!quiet) putchar( '\n' );
      if (calib || access( tadfile, 0 ) != 0)
	calibrate( NULL, &t );
      else
	calibrate( &t, &t1 );
      break;
    }
    if (++p == 4) p = 0;
    str[1] = ind[p];
  }
  if (kbhit())
  {
    puts( "\bAborted." );
    while (kbhit()) getch();
  }
}


void adjust( void )
{
  FILE* dat;
  TAD	t, tn, tr;
  long	s, n, r, diff, dur;
  char	buf[MAX_CLOCK];
  char	tbuf[2][MAX_TIME];
  char	dbuf[MAX_DUR];

  if (access( tadfile, 0 ) != 0)
    puts( "Calibration has not been started." );
  else
  {
    dat = open_calib( "rb+" );
    read_calib( dat, &r, &diff, &dur );
    get_tad( &t );
    s = tad_to_secs( &t );
    if (dur != 0)
    {
      n = s + (s - r) * diff / dur;
      if (n == s)
      {
	if (!quiet)
	printf( "Current time (%s) is correct - no adjustment made.\n",
		format_time( buf, "$+T", &t ) );
      }
      else
      {
	secs_to_tad( n, &tn );
	set_system_tad( &tn );
	if (!quiet)
	{
	  printf( "Expected time: %s\n"
		  "Current  time: %s (a %s of %s)\n",
		  str_time( tbuf[0], &tn ),
		  str_time( tbuf[1], &t ),
		  (n <= s) ? "gain" : "loss", str_dur( dbuf, n - s ) );
	}
	diff += n - s;
	dur  += s - r;
	rewind( dat );
	write_calib( dat, n, diff, dur );
      }
      if (!quiet)
      {
	secs_to_tad( r, &tr );
	printf( "\n"
		"It %s %s since the last adjustment\n"
		"(which was at %s).\n",
		(n == s) ? "has been" : "was",
		str_dur( dbuf, n - r ),
		str_time( tbuf[0], &tr ) );
      }
    }
    else if (!quiet)
    {
      secs_to_tad( r, &tr );
      printf( "Calibration started at %s\n"
	      "and has been in progress for %s.\n",
	      str_time( tbuf[0], &tr ), str_dur( dbuf, s - r ) );
    }
    fclose( dat );
  }
}


void view_calibration( void )
{
  FILE* dat;
  TAD	t, ts;
  long	ref, diff, dur, s;
  char	tbuf[MAX_TIME];
  char	dbuf[2][MAX_DUR];

  if (access( tadfile, 0 ) != 0)
    puts( "Calibration has not been started." );
  else
  {
    dat = open_calib( "rb" );
    read_calib( dat, &ref, &diff, &dur );
    fclose( dat );
    secs_to_tad( ref, &t );
    get_tad( &ts );
    s = tad_to_secs( &ts ) - ref;
    if (dur == 0)
    {
      printf( "Calibration started at %s\n"
	      "and has been in progress for %s.\n",
	      str_time( tbuf, &t ),
	      str_dur( dbuf[0], s ) );
    }
    else
    {
      printf( "Last reference or adjustment was at %s,\n"
	      "which was %s ago.\n",
	      str_time( tbuf, &t ),
	      str_dur( dbuf[0], s ) );
      secs_per_day( diff, dur );
      printf( "(An overall %s of %s%s over %s.)\n",
	      (diff < 0) ? "gain" : "loss", str_dur( dbuf[0], diff ),
	      (brief) ? "" : "\n", str_dur( dbuf[1], diff + dur ) );
    }
  }
}


char* str_time( char* buf, TAD* t )
{
#ifdef LANG_de
  dayofweek = (int)(date_to_days( &t->d ) % 7);
#endif
  return format_time( buf, time_fmt[1], t );
}


char* str_dur( char* buf, long secs )
{
  Duration d;

  if (secs == 0) return zero_dur[0];
  else
  {
    if (secs < 0) secs = -secs;
    d.days    = secs / 86400L;
    d.seconds = secs % 86400L;
    d.years   = 0;
    d.months  = 0;
    return str_duration( buf, &d, DAYS );
  }
}


void secs_per_day( long diff, long dur )
{
  char* d;
  char	buf[8];
  char* p = buf;
  static char* rel[] =
  {
#ifdef LANG_de
    "vor",   "nach"
#else
    "gains", "loses"
#endif
  };

  dur += diff;
  if (diff < 0) d = rel[0], diff = -diff;
  else		d = rel[1];
  sprintf( buf, "%.2f", 86400L * diff / (double)dur );
  while (isdigit( *++p )) ;
  *p = deci_sep;
#ifdef LANG_de
  printf( "\nDie Uhr geht um %s Sekunden pro Tag %s.\n", buf, d );
#else
  printf( "\nThe clock %s %s seconds a day.\n", d, buf );
#endif
}


// --------------------------------------------------------------------------

void time_to_dur( TAD* t, Duration* d )
{
  d->seconds = time_to_secs( &t->t );
  d->days    = date_to_days( &t->d );
  d->months  = 0;
  d->years   = 0;
}

void dur_to_time( Duration* d, TAD* t )
{
  secs_to_time( d->seconds, &t->t );
  days_to_date( d->days,    &t->d );
}


// d += a
void dur_add( Duration* d, Duration* a )
{
  d->seconds += a->seconds;
  if (d->seconds >= 86400L)
  {
    ++d->days;
    d->seconds -= 86400L;
  }
  d->days += a->days;
}

// d -= a (or d = a - d if a is later)
void dur_sub( Duration* d, Duration* a )
{
  Duration* sub = d;

  if (a->days > d->days || (a->days == d->days && a->seconds > d->seconds))
  {
    Duration* temp = a;
    a = d;
    d = temp;
  }
  sub->days    = d->days - a->days;
  sub->seconds = d->seconds - a->seconds;
  if ((long)sub->seconds < 0)
  {
    --sub->days;
    sub->seconds += 86400L;
  }
}


void date_duration( TAD* t1, TAD* t2, int eq, Duration* d )
{
  Duration d1, d2;
  TAD t;

  time_to_dur( t1, &d1 );
  time_to_dur( t2, &d2 );

  if (eq <= 0 || eq == YEARS || eq == MONTHS)
  {
    if (d2.days > d1.days || (d2.days == d1.days && d2.seconds > d1.seconds))
    {
      TAD* tt = t1;
      t1 = t2;
      t2 = tt;
      d1 = d2;
    }
    d->years = t1->d.year - t2->d.year;
    if (eq != YEARS)
    {
      d->months = t1->d.month - t2->d.month;
      if (d->months < 0)
      {
	--d->years;
	d->months += 12;
      }
      t.d.day	= t2->d.day;
      t.d.month = t1->d.month;
      t.d.year	= t1->d.year;
      if (t1->d.day < t2->d.day)
      {
	if (--t.d.month == 0)
	{
	  t.d.month = 12;
	  --t.d.year;
	}
	if (--d->months < 0) --d->years, d->months = 11;
      }
    }
    else
    {
      t.d.day	= t2->d.day;
      t.d.month = t2->d.month;
      t.d.year	= t1->d.year;
      if (t2->d.month > t1->d.month ||
	  (t2->d.month == t1->d.month && t2->d.day > t1->d.day))
      {
	--d->years;
	--t.d.year;
      }
    }
    t.t = t2->t;
    time_to_dur( &t, &d2 );
    dur_sub( &d1, &d2 );
    d->days    = d1.days;
    d->seconds = d1.seconds;
  }
  else
  {
    dur_sub( &d1, &d2 );
    *d = d1;
  }
}


// t op= d
void calc_date( TAD* t, int op, Duration* d )
{
  Duration d1;

  if (d->years || d->months)
  {
    if (op == '+')
    {
      t->d.year  += d->years;
      t->d.month += d->months;
      if (t->d.month > 12)
      {
	--t->d.month;
	t->d.year += t->d.month / 12;
	t->d.month = t->d.month % 12 + 1;
      }
    }
    else // op == '-'
    {
      t->d.year	 -= d->years;
      t->d.month -= d->months;
      if (t->d.month < 1)
      {
	t->d.month = -t->d.month;
	t->d.year -= (t->d.month + 12) / 12;
	t->d.month = 12 - t->d.month % 12;
      }
    }
  }
  if (d->seconds >= 86400L)
  {
    d->days += d->seconds / 86400L;
    d->seconds %= 86400L;
  }
  time_to_dur( t, &d1 );
  if (op == '+') dur_add( &d1, d );
  else		 dur_sub( &d1, d );
  dur_to_time( &d1, t );
}


void normalise( Duration* d )
{
  if (d->seconds >= 86400L)
  {
    d->days += d->seconds / 86400L;
    d->seconds %= 86400L;
  }
  if (d->months >= 12)
  {
    d->years += d->months / 12;
    d->months %= 12;
  }
  d->days += d->months * 30 + (d->months >> 1);
  if (d->days >= 365)
  {
    d->years += (int)(d->days / 365);
    d->days %= 365;
    d->days -= (d->years >> 2) - d->years / 100 + d->years / 400;
    if ((int)d->days < 0)
    {
      d->days += 365 + isleap( d->years );
      --d->years;
    }
  }
  if ((int)d->days >= 30)
  {
    d->months = 2 * ((int)d->days / 61);
    d->days %= 61;
    if ((int)d->days >= 30)
    {
      ++d->months;
      d->days -= 30;
    }
  }
  else d->months = 0;
}


// d1 += d2
void duration_add( Duration* d1, Duration* d2 )
{
  d1->seconds += d2->seconds;
  d1->days    += d2->days;
  d1->months  += d2->months;
  d1->years   += d2->years;
  normalise( d1 );
}

// d1 -= d2 (or d1 = d2 - d1 if d2 is longer)
void duration_sub( Duration* d1, Duration* d2 )
{
  Duration* sub = d1;
  int swap = 0;

  normalise( d1 );
  normalise( d2 );

  if (d1->years < d2->years) swap = 1;
  else if (d1->years == d2->years)
  {
    if (d1->months < d2->months) swap = 1;
    else if (d1->months == d2->months)
    {
      if (d1->days < d2->days) swap = 1;
      else if (d1->days == d2->days)
      {
	if (d1->seconds < d2->seconds) swap = 1;
      }
    }
  }
  if (swap)
  {
    Duration* temp = d1;
    d1 = d2;
    d2 = temp;
  }
  sub->years  = d1->years - d2->years;
  sub->months = d1->months - d2->months;
  if (sub->months < 0)
  {
    --sub->years;
    sub->months += 12;
  }
  sub->days = d1->days - d2->days;
  if (sub->days < 0)
  {
    if (--sub->months < 0)
    {
      sub->days += 29 + isleap( sub->years );
      sub->months = 11;
      --sub->years;
    }
    else sub->days += 30 + (sub->months & 1);
  }
  sub->seconds = d1->seconds - d2->seconds;
  if ((long)sub->seconds < 0)
  {
    if (--sub->days < 0)
    {
      if (--sub->months < 0)
      {
	sub->days = 29 + isleap( sub->years );
	sub->months = 11;
	--sub->years;
      }
      else sub->days = 29 + (sub->months & 1);
    }
    sub->seconds += 86400L;
  }
}


int no_duration( Duration* d )
{
  return (d->seconds == 0 && d->days == 0 && d->months == 0 && d->years == 0);
}


void calc( TAD* t1, Duration* d1, int op, TAD* t2, Duration* d2, int eq )
{
  char buf[256];

  if ((t1->t.hour != -1 || t1->d.day != -1) && !no_duration( d1 ))
  {
    fputs( "Operand one can only be time OR duration.\n", stderr );
    return;
  }
  if ((t2->t.hour != -1 || t2->d.day != -1) && !no_duration( d2 ))
  {
    fputs( "Operand two can only be time OR duration.\n", stderr );
    return;
  }

  if (!no_duration( d1 ) && !no_duration( d2 ))
  {
    if (op == '+') duration_add( d1, d2 );
    else	   duration_sub( d1, d2 );
    fputs( str_duration( buf, d1, eq ), stdout );
    if (!brief && !no_duration( d1 )) putchar( '.' );
    putchar( '\n' );
  }
  else
  {
    if (no_duration( d1 ))
    {
      if (t1->t.hour == -1) get_time( &t1->t );
      if (t1->d.day  == -1) get_date( &t1->d );
    }
    if (no_duration( d2 ))
    {
      if (t2->t.hour == -1) get_time( &t2->t );
      if (t2->d.day  == -1) get_date( &t2->d );
    }
    if (no_duration( d1 ) && no_duration( d2 ))
    {
      date_duration( t1, t2, eq, d1 );
      fputs( str_duration( buf, d1, eq ), stdout );
      if (!brief && !no_duration( d1 )) putchar( '.' );
      putchar( '\n' );
    }
    else
    {
      if (no_duration( d2 ))
      {
	t1 = t2;
	d2 = d1;
      }
      calc_date( t1, op, d2 );
      dayofweek = (int)(date_to_days( &t1->d ) % 7);
      if (format == NULL) format = time_fmt[brief];
      puts( format_time( buf, format, t1 ) );
    }
  }
}


char* ultos( unsigned long num )
{
  // 123456789012345
  // 4,294,967,295
  static char buf[14];
  char* ptr;
  int	pos;

  ptr = buf + 14;
  *--ptr = '\0';
  pos = -1;
  do
  {
    if (++pos == 3)
    {
      pos = 0;
      *--ptr = thou_sep;
    }
    *--ptr = num % 10 + '0';
    num /= 10;
  }
  while (num != 0);

  return ptr;
}


char* str_duration( char* buf, Duration* d, int fmt )
{
  int  years, months;
  long weeks, days;
  unsigned long hours, minutes, seconds;
  long total_days;
  int  pos;

  #define WORD(  idx, cnt ) durname[idx][cnt == 1]
  #define DWORD( idx, cnt ) cnt, WORD( idx, cnt )
  #define UWORD( idx, cnt ) ultos( cnt ), WORD( idx, cnt )

  years   = 0;
  months  = 0;
  weeks   = 0;
  days	  = 0;
  hours   = 0;
  minutes = 0;
  seconds = d->seconds;

  total_days = d->days + d->months * 30 + (d->months >> 1)
	       + years_to_days( d->years );

  if (fmt == SECONDS)
    seconds += total_days * 86400L;
  else
  {
    minutes  = seconds / 60;
    seconds %= 60;
    if (fmt == MINUTES)
      minutes += total_days * 1440;
    else
    {
      minutes %= 60;
      hours    = d->seconds / 3600;
      if (fmt == HOURS)
	hours += total_days * 24;
      else
      {
	if (fmt == DAYS)
	  days = total_days;
	else if (fmt == WEEKS)
	{
	  weeks = total_days / 7;
	  days	= total_days % 7;
	}
	else
	{
	  days	 = d->days;
	  months = d->months;
	  if (fmt == MONTHS)
	    months += d->years * 12;
	  else
	  {
	    years = d->years;
	    if (fmt == YEARS)
	    {
	      days  += months * 30 + (months >> 1);
	      months = 0;
	    }
	  }
	}
      }
    }
  }

  pos = 0;
  if (brief)
  {
    if (years)	 pos  = sprintf( buf,	  "%d%c ",        years,     YEARS );
    if (months)  pos += sprintf( buf+pos, "%d%c ",        months,    MONTHS );
    if (weeks)	 pos += sprintf( buf+pos, "%s%c ", ultos( weeks ),   WEEKS );
    if (days)	 pos += sprintf( buf+pos, "%s%c ", ultos( days ),    DAYS );
    if (hours)	 pos += sprintf( buf+pos, "%s%c ", ultos( hours ),   HOURS );
    if (minutes) pos += sprintf( buf+pos, "%s%c ", ultos( minutes ), MINUTES );
    if (seconds) pos += sprintf( buf+pos, "%s%c ", ultos( seconds ), SECONDS );
    if (pos == 0) *buf = '0', pos = 2;
    buf[pos-1] = '\0';
  }
  else
  {
    int and = 0;
    if (years)	 pos  = sprintf( buf,	  "%d %s, ", DWORD( 0, years ) );
    if (months)
      and = pos, pos += sprintf( buf+pos, "%d %s, ", DWORD( 1, months ) );
    if (weeks)
      and = pos, pos += sprintf( buf+pos, "%s %s, ", UWORD( 2, weeks ) );
    if (days)
      and = pos, pos += sprintf( buf+pos, "%s %s, ", UWORD( 3, days ) );
    if (hours)
      and = pos, pos += sprintf( buf+pos, "%s %s, ", UWORD( 4, hours ) );
    if (minutes)
      and = pos, pos += sprintf( buf+pos, "%s %s, ", UWORD( 5, minutes ) );
    if (seconds)
      and = pos, pos += sprintf( buf+pos, "%s %s, ", UWORD( 6, seconds ) );
    if (pos == 0) strcpy( buf, zero_dur[1] );
    else
    {
      buf[pos -= 2] = '\0';
      if (and)
      {
	buf[--and - 1] = ' ';
	memmove( buf+and+3, buf+and, pos - and + 1 );
	memcpy( buf+and, and_str, strlen( and_str ) );
      }
    }
  }
  return buf;
}


// --------------------------------------------------------------------------

/*
** scalar date routines    --    public domain by Ray Gardner
** Numerically, these will work over the range 1/01/01 thru 14699/12/31.
** Practically, these only work from the beginning of the Gregorian
** calendar thru 14699/12/31.  The Gregorian calendar took effect in
** much of Europe in about 1582, some parts of Germany in about 1700, in
** England and the colonies in about 1752ff, and in Russia in 1918.
*/
// From the C Snippets 9707, by Bob Stout. Slightly modified.


int isleap( unsigned year )
{
  return ((year & 3) == 0 && (year % 100 != 0 || year % 400 == 0));
}

unsigned months_to_days( unsigned month )
{
  return (month * 3057 - 3007) / 100;
}

long years_to_days( unsigned year )
{
  return (year * 365L + (year >> 2) - year / 100 + year / 400);
}


long date_to_days( Date* d )
{
  long days = d->day + months_to_days( d->month );
  if (d->month > 2) 			/* adjust if past February */
    days -= 2 - isleap( d->year );	// -= isleap( d->year ) ? 1 : 2;
  days += years_to_days( d->year - 1 );
  return days;
}

void days_to_date( long days, Date* d )
{
  unsigned n;

  for (n = (unsigned)((days * 400L) / 146097L); years_to_days(n) < days;)
    n++;				/* 146097 == years_to_days(400) */
  d->year = n;
  n = (unsigned)(days - years_to_days( n-1 ));
  if (n > 59)				/* adjust if past February */
  {
    n += 2;
    if (isleap( d->year ))
      n -= n > 62 ? 1 : 2;
  }
  d->month = (n * 100 + 3007) / 3057;	/* inverse of months_to_days() */
  d->day   = n - months_to_days( d->month );
}


long time_to_secs( Time* t )
{
  return (t->hour * 3600L + t->minute * 60 + t->second);
}

void secs_to_time( long secs, Time* t )
{
  t->hour   = (int)(secs / 3600); secs %= 3600;
  t->minute = (int)secs / 60;
  t->second = (int)secs % 60;
}


#define D19800101 722815L

long tad_to_secs( TAD* t )
{
  long secs = date_to_days( &t->d );
  secs -= D19800101;		      // normalize to 1 January, 1980
  secs *= 86400L;		      // convert days to seconds
  secs += time_to_secs( &t->t );
  return secs;
}

void secs_to_tad( long secs, TAD* t )
{
  secs_to_time( secs % 86400L, &t->t );
  secs /= 86400L;
  secs += D19800101;
  days_to_date( secs, &t->d );
}


// Note: this returns January 1 as zero.
int day_of_year( Date* d )
{
  int days = d->day + months_to_days( d->month );
  if (d->month > 2) 			/* adjust if past February */
    days -= 2 - isleap( d->year );	// -= isleap( d->year ) ? 1 : 2;
  return (days - 1);
}


// --------------------------------------------------------------------------

#define TS_APPROX	'a'
#define TS_APPROXC	'A'     // capitalise first letter
#define TS_DATE2Y 	'b'
#define TS_DATE 	'B'
#define TS_12CLOCK	'c'
#define TS_24CLOCK	'C'
#define TS_DAYMONTH	'd'
#define TS_DAYWEEK      'D'
#define TS_12HOUR       'h'
#define TS_24HOUR       'H'
#define TS_LINE 	'l'     // newline
#define TS_MONTHNUM     'm'
#define TS_MONTHWORD    'M'
#define TS_MINUTES      'n'
#define TS_MERIDIAN	'p'     // ie. am or pm
#define TS_SECONDS	's'
#define TS_TAB		't'
#define TS_TIME 	'T'
#define TS_WEEK 	'w'     // From January 1, not first Sunday or Monday
#define TS_YEARDAY	'y'
#define TS_YEAR 	'Y'

char* format_time( char* buffer, char* format, TAD* t )
{
  static char* fmt[] = { "%d", "%?d", "%0?d" };
  static char* align_fmt[] = { "%*s", "%-*s" };
  static char* clock_fmt[] = { "%c%02d","%c%02d%c%02d" };
  char*  b = buffer;
  int	 num;
  char** ptr;
  int	 yearday = -1;
  int	 f;
  int	 place;
  int	 align, width;
  int	 seconds;
  int	 j, len;
  int	 flag;

  *b = 0;
  len = strlen( format );
  for (j = 0; j < len; ++j)
  {
    if (format[j] != '$') *b++ = format[j];
    else
    {
      if (format[++j] == '$') *b++ = '$';
      else
      {
	num = -1;
	ptr = NULL;
	f = 0;
	place = 0;
	align = 0;
	width = 0;
	seconds = 0;
	flag = 0;
	while (!flag)
	{
	  switch (format[j])
	  {
	    case '0': if (f == 0) fmt[2][2] = '2';
		      else if (f == 1) fmt[2][2] = fmt[1][1];
		      f = 2;
		      break;
	    case '-': align = 1; // fall through
	    case '1': place = 1;
		      break;
	    case '2':
	    case '3': if (f == 2) fmt[2][2] = format[j];
		      else
		      {
			f = 1;
			fmt[1][1] = format[j];
		      }
		      break;
	    case '+': seconds = 1;
		      break;
	    default:  flag = 1;
		      continue;
	  }
	  ++j;
	}
	switch (format[j])
	{
	  case TS_APPROX:
	  case TS_APPROXC:   b += approx_time( b, &t->t, format[j] );
			     break;

	  case TS_DATE2Y:
	  case TS_DATE:      if (place) f = 1;
			     else if (f) ++f;
			     b += format_date( b, &t->d, f, format[j] );
			     break;

	  case TS_12CLOCK:
	  case TS_24CLOCK:
	  case TS_TIME:      num = t->t.hour;
			     if (format[j] == TS_12CLOCK ||
				 (format[j] == TS_TIME && time_hr == 0))
			     {
				    if (num > 12) num -= 12;
			       else if (num == 0) num  = 12;
			     }
			     b += sprintf( b, fmt[f], num );
			     b += sprintf( b, clock_fmt[seconds],
					   time_sep, t->t.minute,
					   time_sep, t->t.second );
			     if (format[j] == TS_24CLOCK ||
				 (format[j] == TS_TIME && time_hr == 1))
			     {
			       num = -1;
			       break;
			     }
			     *b++ = ' ';
			     // fall through

	  case TS_MERIDIAN:  ptr = ampm;
			     num = (t->t.hour >= 12);
			     if (place) width = longest_ampm;
			     break;

	  case TS_DAYMONTH:  num = t->d.day;
			     break;

	  case TS_DAYWEEK:   ptr = dayname[(f != 0)];
			     num = dayofweek;
			     if (place) width = longest_dayname;
			     break;

	  case TS_12HOUR:    num = t->t.hour;
				  if (num > 12) num -= 12;
			     else if (num == 0) num  = 12;
			     break;

	  case TS_24HOUR:    num = t->t.hour;
			     break;

	  case TS_LINE:      *b++ = '\n';
			     break;

	  case TS_MONTHNUM:  num = t->d.month;
			     break;

	  case TS_MONTHWORD: ptr = monthname[(f != 0)];
			     num = t->d.month - 1;
			     if (place) width = longest_monthname;
			     break;

	  case TS_MINUTES:   num = t->t.minute;
			     break;

	  case TS_SECONDS:   num = t->t.second;
			     break;

	  case TS_TAB:	     *b++ = '\t';
			     break;

	  case TS_WEEK:
	  case TS_YEARDAY:   if (yearday == -1)
			       yearday = day_of_year( &t->d ) + 1;
			     num = (format[j] == TS_WEEK)
				   ? (yearday - 1) / 7 + 1
				   : yearday;
			     break;

	  case TS_YEAR:      num = t->d.year;
			     if (f != 0)
			     {
			       num %= 100;
			       f = 2;
			       fmt[2][2] = '2';
			     }
			     break;

	  default:	     *b++ = '$';
			     *b++ = format[j];
	}
	if (ptr != NULL)
	{
	  if (place)
	    b += sprintf( b, align_fmt[align], width, ptr[num] );
	  else
	    b += sprintf( b, "%s", ptr[num] );
	}
	else if (num != -1)
	{
	  b += sprintf( b, fmt[f], num );
	  if (place) b += sprintf( b, "%s", place_str( num ) );
	}
      }
    }
  }
  *b = '\0';

  return buffer;
}


int format_date( char* buffer, Date* d, int format, int year )
{
  static char* fmt[2][4] = { { "%d%c%d%c%02d",
			       "%d%c%02d%c%02d",
			       "%2d%c%02d%c%02d",
			       "%02d%c%02d%c%02d" },
			     { "%02d%c%d%c%d",
			       "%02d%c%02d%c%d",
			       "%02d%c%02d%c%-2d",
			       "%02d%c%02d%c%02d" } };

  year = (year == TS_DATE2Y) ? d->year % 100 : d->year;

  switch (date_ord)
  {
    case 0: return sprintf( buffer, fmt[0][format],
			    d->month, date_sep, d->day, date_sep, year );
    default:
    case 1: return sprintf( buffer, fmt[0][format],
			    d->day, date_sep, d->month, date_sep, year );
    case 2: return sprintf( buffer, fmt[1][format],
			    year, date_sep, d->month, date_sep, d->day );
  }
}


int approx_time( char* buf, Time* t, int format )
{
  int h = t->hour, m = t->minute;
  int r = 0;
  int rc;

#ifdef LANG_de
  enum
  {
    Sharp,  Five,  Ten,  Quarter,
    Twenty, Threq, Half, Fv		// gnd
  };

  enum { Past, To, Rel };		// gnd

  int f, rr = Fv;

  f =
#endif
  m = ((m * 60) + t->second + 150) / 300;
  if (m > 6) m = 12 - m, ++h, ++r;
  if (h == 24) h = 0;
  else if (h > 12) h -= 12;

#ifdef LANG_de
  if (m == 0 && h == 1) h = 13; 	// Eins -> Ein Uhr
  else switch (f)
  {
#if 0
    case 4: m  = Ten;			// 10 vor halb
	    r  = To;
	    rr = Half;
	    ++h;
	    break;
#endif
    case 5: m  = Five;			// 5 vor halb
	    r  = To;
	    rr = Half;
	    ++h;
	    break;
    case 6: r = Rel;			// halb
	    ++h;			// half past 11 -> halb 12
	    break;
    case 7: m  = Five;			// 5 nach halb
	    r  = Past;
	    rr = Half;
	    break;
#if 0
    case 8: m  = Ten;			// 10 nach halb
	    r  = Past;
	    rr = Half;
	    break;
#endif
    case 9: m = Threq;			// 3/4
	    r = Rel;
	    break;
  }

  if (format == TS_APPROXC)
  {
    hourname[h][0] = toupper( hourname[h][0] );
    fives[m][0] = toupper( fives[m][0] );
  }
  else
  {
    hourname[h][0] = tolower( hourname[h][0] );
    fives[m][0] = tolower( fives[m][0] );
  }
  if (m == Sharp)
    rc = sprintf( buf, "%s %s", hourname[h], fives[Sharp] );
  else
    rc = sprintf( buf, "%s%s%s%s", fives[m], rel[r], fives[rr], hourname[h] );

#else

  if (m == 0) rc = sprintf( buf, "%s %s", hourname[h], fives[0] );
  else	      rc = sprintf( buf, "%s %s %s", fives[m], rel[r], hourname[h] );

  if (format == TS_APPROXC) *buf = toupper( *buf );

#endif

  return rc;
}


// Slightly modified from the algorithm by Bob Stout and Maynard Hogg,
// from the C Snippets 9707 (it was >9 && <20).
char* place_str( int num )
{
  static char* suffix[] = { "th", "st", "nd", "rd" };

  if (((num %= 100) > 3 && num < 21) || (num %= 10) > 3)
    num = 0;
  return suffix[num];
}


// --------------------------------------------------------------------------

void get_system_time( Time* t )
{
#ifdef _WIN32
  SYSTEMTIME st;

  GetLocalTime( &st );
  t->hour    = st.wHour;
  t->minute  = st.wMinute;
  t->second  = st.wSecond;
  hundredths = st.wMilliseconds / 10;

#else
  union REGS regs;

  regs.h.ah = 0x2c;
  intdos( &regs, &regs );
  t->hour    = regs.h.ch;
  t->minute  = regs.h.cl;
  t->second  = regs.h.dh;
  hundredths = regs.h.dl;
#endif
}

void get_system_date( Date* d )
{
#ifdef _WIN32
  SYSTEMTIME st;

  GetLocalTime( &st );
  d->year   = st.wYear;
  d->month  = st.wMonth;
  d->day    = st.wDay;
  dayofweek = st.wDayOfWeek;

#else
  union REGS regs;

  regs.h.ah = 0x2a;
  intdos( &regs, &regs );
  d->year   = regs.x.cx;
  d->month  = regs.h.dh;
  d->day    = regs.h.dl;
  dayofweek = regs.h.al;
#endif
}

void get_system_tad( TAD* t )
{
#ifdef _WIN32
  SYSTEMTIME st;

  GetLocalTime( &st );
  t->d.year   = st.wYear;
  t->d.month  = st.wMonth;
  t->d.day    = st.wDay;
  dayofweek   = st.wDayOfWeek;
  t->t.hour   = st.wHour;
  t->t.minute = st.wMinute;
  t->t.second = st.wSecond;
  hundredths  = st.wMilliseconds / 10;

#else
  get_system_time( &t->t );
  get_system_date( &t->d );
#endif
}


#ifdef _WIN32_privilege
// An NT system requires a privilege to change the time. It's not needed for
// my XP system, though. Has the default changed? Console process immune?
// API documentation incorrect?
void my_SetLocalTime( CONST SYSTEMTIME* st )
{
  HANDLE hToken;
  TOKEN_PRIVILEGES tkp;
  OSVERSIONINFO ovi;

  ovi.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
  GetVersionEx( &ovi );
  if (ovi.dwPlatformId == VER_PLATFORM_WIN32_NT)
  {
    if (!OpenProcessToken( GetCurrentProcess(), TOKEN_ADJUST_PRIVILEGES,
			   &hToken ))
      return;
    LookupPrivilegeValue( NULL, SE_SYSTEMTIME_NAME, &tkp.Privileges[0].Luid );
    tkp.PrivilegeCount = 1;
    tkp.Privileges[0].Attributes = SE_PRIVILEGE_ENABLED;
    AdjustTokenPrivileges( hToken, FALSE, &tkp, 0, NULL, 0 );
    SetLocalTime( st );
    AdjustTokenPrivileges( hToken, TRUE, NULL, 0, NULL, 0 );
  }
  else SetLocalTime( st );
}
#endif


void set_system_time( Time* t )
{
#ifdef _WIN32
  SYSTEMTIME st;

  GetLocalTime( &st );
  st.wHour   = t->hour;
  st.wMinute = t->minute;
  st.wSecond = t->second;
  st.wMilliseconds = 0;
  SetLocalTime( &st );

#else
  union REGS regs;

  regs.h.ch = t->hour;
  regs.h.cl = t->minute;
  regs.h.dh = t->second;
  regs.h.dl = 10;			// It goes back 1 to 6 hundredths,
  regs.h.ah = 0x2d;			//  depending on the time being set.
  intdos( &regs, &regs );
#endif
}

void set_system_date( Date* d )
{
#ifdef _WIN32
  SYSTEMTIME st;

  GetLocalTime( &st );
  st.wYear  = d->year;
  st.wMonth = d->month;
  st.wDay   = d->day;
  SetLocalTime( &st );

#else
  union REGS regs;

  regs.x.cx = d->year;			// 1980 - 2099
  regs.h.dh = d->month;
  regs.h.dl = d->day;
  regs.h.ah = 0x2b;
  intdos( &regs, &regs );
#endif
}

void set_system_tad( TAD* t )
{
#ifdef _WIN32
  SYSTEMTIME st;

  GetLocalTime( &st );
  if (t->d.day != -1)
  {
    st.wYear  = t->d.year;
    st.wMonth = t->d.month;
    st.wDay   = t->d.day;
  }
  if (t->t.hour != -1)
  {
    st.wHour   = t->t.hour;
    st.wMinute = t->t.minute;
    st.wSecond = t->t.second;
    st.wMilliseconds = 0;
  }
  SetLocalTime( &st );

#else
  set_system_time( &t->t );
  set_system_date( &t->d );
#endif
}


#ifndef _WIN32	// not really worth the effort

#define BCD2BIN( bcd ) ((((bcd) & 0xf0) >> 4) * 10 + ((bcd) & 0x0f))
#define BIN2BCD( bin ) ((((bin) / 10) << 4) | ((bin) % 10))

void get_rtc_time( Time* t )
{
  union REGS regs;

  regs.h.ah = 0x02;
  int86( 0x1a, &regs, &regs );
  t->hour   = BCD2BIN( regs.h.ch );
  t->minute = BCD2BIN( regs.h.cl );
  t->second = BCD2BIN( regs.h.dh );
}

void get_rtc_date( Date* d )
{
  union REGS regs;

  regs.h.ah = 0x04;
  int86( 0x1a, &regs, &regs );
  d->year  = BCD2BIN( regs.h.ch ) * 100 + BCD2BIN( regs.h.cl );
  d->month = BCD2BIN( regs.h.dh );
  d->day   = BCD2BIN( regs.h.dl );
  dayofweek = (int)(date_to_days( d ) % 7);
}

void get_rtc_tad( TAD* t )
{
  get_rtc_time( &t->t );
  get_rtc_date( &t->d );
}


void set_rtc_time( Time* t )
{
  union REGS regs;

  regs.h.ch = BIN2BCD( t->hour );
  regs.h.cl = BIN2BCD( t->minute );
  regs.h.dh = BIN2BCD( t->second );
  regs.h.ah = 0x03;
  int86( 0x1a, &regs, &regs );
}

void set_rtc_date( Date* d )
{
  union REGS regs;

  regs.h.ch = BIN2BCD( d->year / 100 );
  regs.h.cl = BIN2BCD( d->year % 100 );
  regs.h.dh = BIN2BCD( d->month );
  regs.h.dl = BIN2BCD( d->day );
  regs.h.ah = 0x05;
  int86( 0x1a, &regs, &regs );
}

void set_rtc_tad( TAD* t )
{
  if (t->t.hour != -1) set_rtc_time( &t->t );
  if (t->d.day	!= -1) set_rtc_date( &t->d );
}

#endif


void wait_for_rtc( Time* t )
{
  Time t1;
#ifdef _WIN32
  get_system_time( &t1 );
  do
  {
    Sleep( 10 );
    get_system_time( t );
  }
#else
  get_rtc_time( &t1 );
  do get_rtc_time( t );
#endif
  while (t->second == t1.second);
}



#ifdef _WIN32
void get_local_names( LCTYPE locale, char** name, int num, int* longest )
{
  int	j, len;
  char	buf[40];
  char* str;

  if (*longest < 0)
  {
    *longest = -*longest;
    for (j = 0; j < num; ++j)
    {
      len = GetLocaleInfo( LOCALE_USER_DEFAULT, locale + j, buf, sizeof(buf) );
      if (len && (str = strdup( buf )) != NULL)
      {
	name[j] = str;
	if (--len > *longest)
	  *longest = len;
      }
    }
  }
}
#endif

void get_country_info( void )
{
  char buf[40];

#ifdef _WIN32
  if (GetLocaleInfo( LOCALE_USER_DEFAULT, LOCALE_IDATE, buf, sizeof(buf) ))
    date_ord = *buf - '0';              // 0 = m/d/y, 1 = d/m/y, 2 = y/m/d
  if (GetLocaleInfo( LOCALE_USER_DEFAULT, LOCALE_STHOUSAND, buf, sizeof(buf) ))
    thou_sep = *buf;
  if (GetLocaleInfo( LOCALE_USER_DEFAULT, LOCALE_SDECIMAL, buf, sizeof(buf) ))
    deci_sep = *buf;
  if (GetLocaleInfo( LOCALE_USER_DEFAULT, LOCALE_SDATE, buf, sizeof(buf) ))
    date_sep = *buf;
  if (GetLocaleInfo( LOCALE_USER_DEFAULT, LOCALE_STIME, buf, sizeof(buf) ))
    time_sep = *buf;
  if (GetLocaleInfo( LOCALE_USER_DEFAULT, LOCALE_ITIME, buf, sizeof(buf) ))
    time_hr  = *buf - '0';              // 0 = 12-hour, 1 = 24-hour

  get_local_names( LOCALE_SMONTHNAME1, monthname[0], 12*2, &longest_monthname );
  get_local_names( LOCALE_S1159, ampm, 2, &longest_ampm );
  // LOCALE_DAYNAME1 is Monday, but I have Sunday first.
  if (longest_dayname < 0)
  {
    get_local_names( LOCALE_SDAYNAME7, dayname[0], 1, &longest_dayname );
    longest_dayname = -longest_dayname;
    get_local_names( LOCALE_SDAYNAME1, dayname[0] + 1, 6, &longest_dayname );
    longest_dayname = -longest_dayname;
    get_local_names( LOCALE_SABBREVDAYNAME7, dayname[1], 1, &longest_dayname );
    longest_dayname = -longest_dayname;
    get_local_names( LOCALE_SABBREVDAYNAME1, dayname[1]+1,6, &longest_dayname );
  }
#else
# if defined( __DJGPP__ )
  __dpmi_regs regs;

  regs.x.ax = 0x3800;
  regs.x.dx = __tb & 0x0f;
  regs.x.ds = __tb >> 4;
  __dpmi_int( 0x21, &regs );
  dosmemget( __tb, sizeof(buf), buf );

# else
  union REGS regs;

  regs.x.ax = 0x3800;
  regs.x.dx = (unsigned)buf;		// Assumes SS == DS
  intdos( &regs, &regs );
# endif

  if (regs.x.flags & 1) return; 	// Carry set - function failed

  date_ord = buf[0];			// 0 = m/d/y, 1 = d/m/y, 2 = y/m/d
  thou_sep = buf[7];
  deci_sep = buf[9];
  date_sep = buf[11];
  time_sep = buf[13];
  time_hr  = buf[17] & 1;		// 0 = 12-hour, 1 = 24-hour
#endif
}


// --------------------------------------------------------------------------

void copyright( void )
{
  if (isatty( 1 )) putchar( '\n' );
  puts( "Time and Date by Jason Hood <jadoxa@yahoo.com.au>.\n"
#ifdef LANG_de
	"German language support by Waldemar Schultz <schultz@ma.tum.de>.\n"
#endif
	"Version "VERSION" ("DATE"). Freeware.\n"
	"http://tad.adoxa.cjb.net/\n"
      );
};


void help( void )
{
  copyright();
  puts(
"tad [-aywbctksnrmdvq] [[-f] fmt] [-e?] [[hh]:[mm][:ss]] [[dd][/[mm][/yy[yy]]]]]\n"
"\n"
"\t<none>   display the time, day and date\n"
"\t-a       display the \"approximate\" time\n"
"\t-y       display the current day of the year\n"
"\t-w       display the current week of the year\n"
"\t-b[b]    display the date and time briefly [concisely]\n"
"\t-f       display the time using \"fmt\" (-f? for help)\n"
"\t-fl      list the default formats\n"
"\t-c       display as a clock\n"
#ifdef _WIN32
"\t-t       ignored\n"
"\t-k       remember the current time before setting new\n"
"\t-s       restore the updated time remembered above\n"
#else
"\t-t       use the real-time clock instead of the system\n"
"\t-tt      display the difference between RTC and system times\n"
"\t-k       set system time/date, keep the RTC as is\n"
"\t-s       set the system time/date from the real-time clock\n"
#endif
"\t-n       start a new calibration\n"
"\t-r[r]    take the given time/use the current time as reference\n"
"\t-m       monitor the time for external update\n"
"\t-d       correct for drift\n"
"\t-v       view calibration details\n"
"\t-q       quiet, no display\n"
"\t-e       select exit code (? for list)\n"
"\n"
"Notes: Hour is 24-hour format.\n"
"       Two-digit year is assumed 1980 to 2079.\n"
"       Missing time/date component defaults to current.\n"
"       Time separator must be ':', but date separator can be anything.\n"
"       Week of the year begins from 1 January, not first Sunday or Monday.\n"
"       Return value is in 10-minute intervals (midnight is 0, noon is 72).\n"
"       Calibration/reference/monitor should not be done across midnight.\n"
"\n"
"TAD can also calculate differences between times - use \"=?\" for help."
      );
}


void format_help( void )
{
  copyright();
  puts( "Format string:\n"
	"\n"
	"\t$a      approximate time"       "\t$m      month (number)\n"
	"\t$A      approximate time with"  "\t$M      month (word)\n"
	"\t         initial capital"       "\t$n      minutes\n"
	"\t$b      date (2-digit year)"    "\t$p      am or pm\n"
	"\t$B      date\t\t"               "\t$s      seconds\n"
	"\t$c      time (12-hour)\t"       "\t$t      tab\n"
	"\t$C      time (24-hour)\t"       "\t$T      time (12- or 24-hour)\n"
	"\t$d      day of month\t"         "\t$w      week of the year\n"
	"\t$D      day of week\t"          "\t$y      day of the year\n"
	"\t$h      hour (12-hour)\t"       "\t$Y      year\n"
	"\t$H      hour (24-hour)\t"       "\t$2Y     two-digit year\n"
	"\t$l      new line\t"             "\t$$      a dollar sign\n"
	"\n"
	"Words can be prefixed with:\n"
	"  0  - use the abbreviated form;\n"
	"  1  - right-aligned (-1 for left-aligned).\n"
	"\n"
	"Numbers can be prefixed with:\n"
	"  0  - use zero instead of space for padding (implies 2);\n"
	"  1  - add place suffix;\n"
	" 2,3 - pad to two or three characters.\n"
	"\n"
	"Time and date use the local conventions (as defined in COUNTRY).\n"
	"Prefix time with '+' to include seconds."
      );
}


void format_list( void )
{
  static char* type[] = { "Normal", "Brief", "Concise" };
  int j;

  if (isatty( 1 )) putchar( '\n' );

  puts( "Time\n"
	"----" );
  for (j = 0; j < 3; ++j)
    printf( "%s\t\t%s\n", type[j], time_fmt[j] );

  puts( "\nApproximate Time\n"
	  "----------------" );
  for (j = 0; j < 3; ++j)
    printf( "%s\t\t%s\n", type[j], approx_fmt[j] );

  puts( "\nDay of the Year\n"
	  "---------------" );
  for (j = 0; j < 3; ++j)
    printf( "%s\t\t%s\n", type[j], year_fmt[j] );

  puts( "\nWeek of the Year\n"
	  "----------------" );
  for (j = 0; j < 3; ++j)
    printf( "%s\t\t%s\n", type[j], week_fmt[j] );
}


void exit_help( void )
{
  copyright();
  puts( "Exit code character:\n"
	"\n"
	"\th - hour         (0..23)\n"
	"\tm - minute       (0..59)\n"
	"\ts - second       (0..59)\n"
	"\tD - day of week  (0..6, 0 = Sunday)\n"
	"\td - day of month (1..31)\n"
	"\tM - month        (1..12)\n"
	"\tw - week         (1..53)\n"
	"\ty - year         (00..99)\n"
	"\tY - century      (19 or 20)"
      );
}


void calc_help( void )
{
  copyright();
  puts(
"tad time1|duration1 [+|-] [time2|duration2] [=[?]]\n"
"\n"
"Add or subtract times and/or durations.\n"
"Convert times or durations to another format.\n"
"\n"
"? can be one of:\n"
"\n"
"\ts - seconds\n"
"\tm - minutes\n"
"\th - hours\n"
"\td - days\n"
"\tw - weeks\n"
"\tM - months\n"
"\ty - years\n"
"\n"
"which is also how duration is specified.\n"
"\n"
"Notes: Default output is years, months and days.\n"
"       Using \"=y\" gives years and days.\n"
"       Adding or subtracting two times gives the duration between them.\n"
"       For durations, months alternate between 30 and 31 days, with the\n"
"        twelfth month having 30 days, or 31 in a leap year."
      );
}
