/* simc.h

 Copyright (c) 1992-1996 TOH Choon Swee.
 All rights reserved.

 THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF 
 ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT 
 LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, 
 FITNESS FOR A PARTICULAR PURPOSE AND FREEDOM FROM 
 INFRINGEMENT.
 
 define data structures, prototypes and macros
 for Discrete and Combined Discrete-Continuous
     Simulation library
*/

#ifndef _SIMC_H
#define _SIMC_H "SimC/C++ Library Release 1.0 Feb 1996"

/*----------------------------------------------------------*/
void SimCNoMemory( char *);
void SimCUndefined( char *);

#ifdef __cplusplus
class   SimC  {};
typedef SimC  Sim_Data;
typedef SimC* Pointer;

#else
typedef void * Pointer;
typedef void * Set_ID;
#endif

#ifndef _SETS_H
#define _SETS_H
/*----------------------------------------------------------*/
                                                    /* Sets */
                                                    /*------*/
                                        /*-----*/
#ifdef __cplusplus                      /* C++ */
                                        /*-----*/
class Set_ID : public SimC
{
  private:
  class    Set_IDO *_Set;

  public:
           Set_ID( void);
          ~Set_ID( void);
  long     Clear( void);
  long     Find( Pointer);
  Pointer  Get( long);
  Pointer  GetFirst( void){ return Get( 1); }
  Pointer  GetLast( void){ return Get( Size()); }
  long     List( int (* )( Pointer));
  int      Rank( int (* )( Pointer, Pointer));
  Pointer  Read( long);
  Pointer  ReadFirst( void){ return Read( 1); }
  Pointer  ReadLast( void) { return Read( Size()); }
  Pointer  ReadNext( void);
  long     Remove( Pointer);
  long     Save( Pointer, long);
  long     SaveFirst( Pointer D){ return Save( D, 1); }
  long     SaveLast( Pointer);
  long     SaveRanked( Pointer, int (* )( Pointer, Pointer));
  long     Size( void);
  Set_ID&  operator =  ( char *);
  Set_ID&  operator =  ( Set_ID&);
  Set_ID   operator |  ( Set_ID&);
  Set_ID   operator -  ( Set_ID&);
  Set_ID   operator &  ( Set_ID&);
  int      operator == ( Set_ID&);
  int      operator <= ( Set_ID&);
  int      operator >= ( Set_ID&);
};

#define Create( E, S) \
        { E = new S; if( ! E) SimCNoMemory( #E " " #S ); }

#define SetClear( S)         S.Clear()
#define SetCreate()          "Init"
#define SetFind( S, D)       S.Find((Pointer) D)
#define SetGet( S, N)        S.Get( N)
#define SetGetFirst( S)      S.GetFirst()
#define SetGetLast( S)       S.GetLast()
#define SetKill( S)          S.Clear()
#define SetList( S, R)       S.List(( int(*)( Pointer)) R)
#define SetRank( S, R)       S.Rank(( int(*)( Pointer, Pointer)) R)
#define SetRead( S, N)       S.Read( N)
#define SetReadFirst( S)     S.ReadFirst()
#define SetReadLast( S)      S.ReadLast()
#define SetReadNext( S)      S.ReadNext()
#define SetRemove( S, D)     S.Remove((Pointer) D)
#define SetSave( S, D, N)    S.Save((Pointer) D, (long) N)
#define SetSaveFirst( S, D)  S.SaveFirst((Pointer) D)
#define SetSaveLast( S, D)   S.SaveLast((Pointer) D)
#define SetSaveRanked( S, D, R) \
        S.SaveRanked((Pointer) D, ( int(*)( Pointer, Pointer)) R)
#define SetSize( S)          S.Size()

                                        /*---*/
#else                                   /* C */
                                        /*---*/
/*--- Prototypes ---*/
long    SetClear( Set_ID);
Set_ID  SetCreate( void);
long    SetKill( Set_ID);
long    SetFind( Set_ID, Pointer);
Pointer SetGet( Set_ID, long);
long    SetListRoutine( Set_ID, int (* )( Pointer));
int     SetRankRoutine( Set_ID, int (* )( Pointer, Pointer));
Pointer SetRead( Set_ID, long);
Pointer SetReadNext( Set_ID);
long    SetRemove( Set_ID, Pointer);
long    SetSave( Set_ID, Pointer, long);
long    SetSaveLast( Set_ID, Pointer);
long    SetSaveRankedRoutine
        ( Set_ID, Pointer, int (* )( Pointer, Pointer));
long    SetSize( Set_ID);

/*--- Macros ---*/
#define SetGetFirst( S)     SetGet( S, 1)
#define SetGetLast( S)      SetGet( S, SetSize( S))
#define SetList( S, R)      SetListRoutine( S, ( int( *)( Pointer)) R)
#define SetRank( S, R)      SetRankRoutine( S, ( int( *)( Pointer, Pointer)) R)
#define SetReadFirst( S)    SetRead( S, 1)
#define SetReadLast( S)     SetRead( S, SetSize( S))
#define SetSaveFirst( S, D) SetSave( S, D, 1)
#define SetSaveRanked( S, D, R) \
        SetSaveRankedRoutine( S, D, ( int( *)( Pointer, Pointer)) R)

#define Create( E, S) \
        { E = ( S *) malloc( sizeof( S));       \
          if( ! E)   SimCNoMemory( #E " " #S ); }

#endif  /* __cplusplus */

#define ForEach( D, S)             \
        for( D= SetRead( S, 1); D; \
             D= SetReadNext( S)    )

#define ForEachC( D, C, S)               \
        for( D= ( C*) SetRead( S, 1); D; \
             D= ( C*) SetReadNext( S)    )

#endif  /* _SETS_H     */

#ifndef _EVENTS_H
#define _EVENTS_H
/*----------------------------------------------------------*/
                                                  /* events */
                                                  /*--------*/
typedef void * Ev_ID;
                                        /*-----*/
#ifdef __cplusplus                      /* C++ */
                                        /*-----*/
class Svr_ID : public SimC
{
  class    Svr_IDO *_S;

  public:
           Svr_ID( long = 1);
          ~Svr_ID( void);
  long     Available( void);
  long     Capacity( void);
  void     Init( long);
  long     QueueSize( void);
  long     S_Release( long);
  long     S_RequestP( long, long);
  long     Used( void);
  Svr_ID&  operator = ( char *);
  Svr_ID&  operator = ( long  );
  Svr_ID&  operator = ( Svr_ID&);
};

Ev_ID   EvScheduleRoutine
        ( int (* )( Pointer), SimC*, int (SimC::* )( Pointer),
          Pointer, double, char*);
long    EvSetPriorityRoutine( int (SimC::*)( Pointer), long);

#define EvSchedule( R, D, T)                            \
  EvScheduleRoutine(( int( *)( Pointer)) R, NULL, NULL, \
                    ( Pointer) D, ( double) T, #R)

#define EvScheduleC( O, R, D, T)                                \
  EvScheduleRoutine( NULL, O, ( int( SimC::*)( Pointer)) R, \
                    ( Pointer) D, ( double) T, #R)

#define EvSetPriorityC( R, P)                                  \
        EvSetPriorityRoutine(( int( SimC::*)( Pointer)) R, \
                             ( long) P)

#define SvrAvailable( S)      S.Available()
#define SvrCapacity( S)       S.Capacity()
#define SvrCreate( N)         N
#define SvrInit( S, N)        S.Init( N)
#define SvrQueueSize( S)      S.QueueSize()
#define SvrRequest( N, S, P)  S.S_RequestP( N, P)
#define SvrRelease( N, S)     S.S_Release( N)
#define SvrUsed( S)           S.Used()

                                        /*---*/
#else                                   /* C */
                                        /*---*/
typedef void * Svr_ID;

long    SvrAvailable( Svr_ID);
long    SvrCapacity( Svr_ID);
Svr_ID  SvrCreate( long);
void    SvrInit( Svr_ID, long);
long    SvrQueueSize( Svr_ID);
long    SvrRelease( long, Svr_ID);
long    SvrRequest( long, Svr_ID, long);
long    SvrUsed( Svr_ID);

Ev_ID   EvScheduleRoutine( int (* )( Pointer), Pointer, double, char*);

#define EvSchedule( R, D, T)                      \
  EvScheduleRoutine(( int( *)( Pointer)) R,       \
                    ( Pointer) D, ( double) T, #R)

#endif  /* __cplusplus */


/*--- Prototypes ---*/
long    EvBreak( Ev_ID);
long    EvClear( void);
long    EvClearPriority( void);
double  EvClock( void);
long    EvContinue( Ev_ID);
void    EvInit( void);
long    EvKill( Ev_ID);
long    EvList( void);
void    EvSimulate( void);
long    EvSetPriorityRoutine( int (* )( Pointer), long);
double  EvSetClock( double);
char   *EvVersion( void);

long    ProcessBeginRoutine( void);
void    ProcessBreakRoutine( void);
void    ProcessEndRoutine( void);
void    ProcessLine( long);
void    ProcessWaitRoutine( double);

/*--- Macros ---*/
#define EvSetPriority( R, P)                   \
  EvSetPriorityRoutine(( int( *)( Pointer)) R, \
                       ( long) P)

#define ProcessBegin    switch( ProcessBeginRoutine()) \
                        { case 0: ;

#define ProcessWait( T) { ProcessLine( __LINE__);         \
                          ProcessWaitRoutine((double) T); \
                          return( 0);                     \
                          case __LINE__: ; }

#define Wait( T)          ProcessWait( T)

#define ProcessBreak    { ProcessLine( __LINE__); \
                          ProcessBreakRoutine();  \
                          return( 0);             \
                          case __LINE__: ; }

#define ProcessEnd        default:             \
                          ProcessEndRoutine(); \
                          return( 0); }

#define RequestP( N, S, P)                   \
  { ProcessLine( __LINE__);                  \
    if( ! SvrRequest((long) N, S, (long) P)) \
    return( 0);                              \
    case __LINE__: ; }

#define Request( N, S)    RequestP( N, S, 0)

#define Release( N, S)    SvrRelease((long) N, S)

#endif  /* _EVENTS_H */

#ifndef _DATA_H
#define _DATA_H
/*----------------------------------------------------------*/
                                                    /* Data */
                                                    /*------*/
                                        /*-----*/
#ifdef __cplusplus                      /* C++ */
                                        /*-----*/
class Tally_ID : public SimC
{
  class     Tally_IDO *_T;
  public:
            Tally_ID( void);
           ~Tally_ID( void);
  long      Update( double);
  long      Count( void);
  int       Init( void);
  double    Max( void);
  double    Mean( void);
  double    Min( void);
  double    StdDev( void);
  Tally_ID& operator = ( double);
  Tally_ID& operator = ( char *);
  Tally_ID& operator = ( Tally_ID&);
};

class Accum_ID : public SimC
{
  class     Accum_IDO *_A;
  public:
            Accum_ID( void);
            Accum_ID( double);
           ~Accum_ID( void);
  long      Update( double V){ return UpdateT( V, EvClock()); }
  long      UpdateT( double, double);
  long      Count( void);
  int       Init( void){ return InitT( EvClock()); }
  int       InitT( double);
  double    Max( void);
  double    Mean( void){ return MeanT( EvClock()); }
  double    MeanT( double);
  double    Min( void);
  double    StdDev( void){ return StdDevT( EvClock()); }
  double    StdDevT( double);
  Accum_ID& operator = ( double);
  Accum_ID& operator = ( char *);
  Accum_ID& operator = ( Accum_ID&);
};

#define Accum( A, V)         A.Update( V)
#define AccumT( A, V, T)     A.UpdateT( V, T)
#define AccumCount( A)       A.Count()
#define AccumCreateT( T)     Accum_ID( T)
#define AccumInitT( A, T)    A.InitT( T)
#define AccumMax( A)         A.Max()
#define AccumMeanT( A, T)    A.MeanT( T)
#define AccumMin( A)         A.Min()
#define AccumStdDevT( A, T)  A.StdDevT( T)
#define AccumCreate()        "Init"
#define AccumInit( A)        A.Init()
#define AccumKill( A)        A.Kill()
#define AccumMean( A)        A.Mean()
#define AccumStdDev( A)      A.StdDev()

#define Tally( T, V)         T.Update( V)
#define TallyCreate()        "Init"
#define TallyCount( T)       T.Count()
#define TallyInit( T)        T.Init()
#define TallyKill( T)        (delete T)
#define TallyMax( T)         T.Max()
#define TallyMean( T)        T.Mean()
#define TallyMin( T)         T.Min()
#define TallyStdDev( T)      T.StdDev()

#else                                   /*---*/
                                        /* C */
/*--- Data Structures ---*/             /*---*/
typedef void * Accum_ID;
typedef void * Tally_ID;

/*--- Prototypes ---*/
long     AccumT( Accum_ID, double, double);
Accum_ID AccumCreateT( double);
long     AccumCount( Accum_ID);
int      AccumInitT( Accum_ID, double);
double   AccumMax( Accum_ID);
double   AccumMeanT( Accum_ID, double);
double   AccumMin( Accum_ID);
double   AccumStdDevT( Accum_ID, double);

long   Tally( Tally_ID, double);
void  *TallyCreate( void);
long   TallyCount( Tally_ID);
int    TallyInit( Tally_ID);
double TallyMax( Tally_ID);
double TallyMean( Tally_ID);
double TallyMin( Tally_ID);
double TallyStdDev( Tally_ID);

/*--- Macros ---*/
#define Accum( A, V)     AccumT( A, V, EvClock())
#define AccumCreate()    AccumCreateT( EvClock())
#define AccumInit( A)    AccumInitT( A, EvClock())
#define AccumMean( A)    AccumMeanT( A, EvClock())
#define AccumStdDev( A)  AccumStdDevT( A, EvClock())

#endif  /* __cplusplus */
#endif  /* _DATA_H     */

/*----------------------------------------------------------*/
#ifndef _CONT_H
#define _CONT_H

double ContinuousWaitRoutine
       ( int     NX,
         double *X,
         int  (* DX)( int, double *, double *, double));

double EvSetMax_dT( double);
double EvSetMin_dT( double);
double EvSetMax_dV( double);
double EvSetMax_Error( double);

#define EvReadMax_dT()    ( EvSetMax_dT( 0))
#define EvReadMin_dT()    ( EvSetMin_dT( 0))
#define EvReadMax_dV()    ( EvSetMax_dV( 0))
#define EvReadMax_Error() ( EvSetMax_Error( 0))


#define ContinuousBegin \
        ProcessBegin;   \
        while( 1) {

#define ContinuousEnd   \
        ; } ProcessEnd; \

#define ContinuousWait( N, V, DV) \
        { ProcessLine( __LINE__); \
          ProcessWaitRoutine      \
          ( ContinuousWaitRoutine \
            (( int)      N,       \
             ( double *) V,       \
             ( int( *)( int, double *, double *, double)) DV)); \
          return( 0);             \
          case __LINE__:;         \
        }

#endif  /* _CONT_H */

/*----------------------------------------------------------*/
#endif  /* _SIMC_H */

