/****************************************************************************
*
*				VFlat - DOS Virtual Flat Linear Framebuffer
*
*                   Copyright (C) 1995 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: vflat.c $
* Version:		$Revision: 1.1 $
*
* Language:		ANSI C
* Environment:	IBM PC (MS DOS)
*
* Description:	Main C module for the VFlat framebuffer routines. The page
*				fault handler is always installed to handle up to a 4Mb
*				framebuffer with a window size of 4Kb or 64Kb in size.
*
* $Id: vflat.c 1.1 1995/09/16 10:40:21 kjb release $
*
****************************************************************************/

#include <stdlib.h>
#include <dos.h>
#include "pmpro.h"
#include "debug.h"

#define	VFLAT_START_ADDR	0xF0000000U
#define	VFLAT_END_ADDR		0xF04FFFFFU
#define PAGE_PRESENT 		1
#define PAGE_NOTPRESENT 	0
#define PAGE_READ 			0
#define PAGE_WRITE			2

#if	defined(DOS4GW)

/*-------------------------------------------------------------------------*/
/* DOS4G/W, PMODE/W and FlashTek X32-VM support.                       	   */
/*-------------------------------------------------------------------------*/

PRIVATE	installed = false;
PUBLIC	VF_zeroSel = 0;

/* Low level assembler code */

int     _ASMAPI InitPaging(void);
void    _ASMAPI ClosePaging(void);
void    _ASMAPI MapPhysical2Linear(ulong pAddr, ulong lAddr, int pages, int flags);
void	_ASMAPI InstallFaultHandler(ulong baseAddr,int bankSize);
void	_ASMAPI RemoveFaultHandler(void);
void	_ASMAPI InstallBankFunc(int codeLen,void *bankFunc);

void * _cdecl VF_malloc(uint size)
{ return malloc(size); }

void _cdecl VF_free(void *p)
{ free(p); }

bool PMAPI VF_available(void)
/****************************************************************************
*
* Function:		VF_available
* Returns:		True if virtual buffer is available, false if not.
*
****************************************************************************/
{
	if (!VF_zeroSel)
		VF_zeroSel = PM_createSelector(0,0xFFFFFFFF);
	if (InitPaging() == -1)
		return false;
	ClosePaging();
	return true;
}

void * PMAPI VF_init(ulong baseAddr,int bankSize,int codeLen,void *bankFunc)
/****************************************************************************
*
* Function:		VF_init
* Parameters:	baseAddr	- Base address of framebuffer bank window
*				bankSize	- Physical size of banks in Kb (4 or 64)
*               codeLen		- Length of 32 bit bank switch function
* 				bankFunc	- Pointer to protected mode bank function
* Returns:		Near pointer to virtual framebuffer, or NULL on failure.
*
* Description:	Installs the virtual linear framebuffer handling.
*
****************************************************************************/
{
	int		i;

	if (installed)
		return (void*)VFLAT_START_ADDR;
	if (codeLen > 100)
		return NULL;				/* Bank function is too large!		*/
	if (!VF_zeroSel)
		VF_zeroSel = PM_createSelector(0,0xFFFFFFFF);
	if (InitPaging() == -1)
		return NULL;				/* Cannot do hardware paging!		*/

	/* Map 4MB of video memory into linear address space (read/write) */
	if (bankSize == 64) {
		for (i = 0; i < 64; i++) {
			MapPhysical2Linear(baseAddr,VFLAT_START_ADDR+(i<<16),16,
				PAGE_WRITE | PAGE_NOTPRESENT);
			}
		}
	else {
		for (i = 0; i < 1024; i++) {
			MapPhysical2Linear(baseAddr,VFLAT_START_ADDR+(i<<12),1,
				PAGE_WRITE | PAGE_NOTPRESENT);
			}
		}

	/* Install our page fault handler and banks switch function */
	InstallFaultHandler(baseAddr,bankSize);
	InstallBankFunc(codeLen,bankFunc);
	installed = true;
	return (void*)VFLAT_START_ADDR;
}

void PMAPI VF_exit(void)
/****************************************************************************
*
* Function:		VF_exit
*
* Description:	Closes down the virtual framebuffer services and
*				restores the previous page fault handler.
*
****************************************************************************/
{
	if (installed) {
		RemoveFaultHandler();
		ClosePaging();
		installed = false;
		}
}

#elif defined(X32VM)

/* All in assembler */

#elif	!defined(__WINDOWS__)

/*-------------------------------------------------------------------------*/
/* Support mapped out for other compilers. Windows support is provided	   */
/* by the WinDirect DLL's.										   		   */
/*-------------------------------------------------------------------------*/

bool PMAPI VF_available(void)
{
	return false;
}

void * PMAPI VF_init(ulong baseAddr,int bankSize,int codeLen,void *bankFunc)
{
	baseAddr = baseAddr;
	bankSize = bankSize;
	codeLen = codeLen;
	bankFunc = bankFunc;
	return NULL;
}

void PMAPI VF_exit(void)
{
}

#endif
