/*===================================================================
 sort.c

 Version 1.2

 Written by:
   Brent Worden
   WordenWare
   email:  brent@worden.org

 Copyright (c) 1998-2001 WordenWare

 Created:  August 28, 1998
 Revised:  January 7, 2000
===================================================================*/

#include "sort.h"
#include "vecutils.h"

#define MAX_QSORT 10

#define iter_swap(a, b) \
{                       \
    double tmp;         \
    tmp = *(a);         \
    *(a) = *(b);        \
    *(b) = tmp;         \
}

NUMERICS_EXPORT void isort1(double* first, double* last)
{
    double* iter;
    double* curr;
    double value;
    
    for(iter = first + 1; iter != last; ++iter){
		value = *iter;
		curr = iter - 1;
        while(curr >= first && *curr > value){
			*(curr + 1) = *curr;
            curr--;
        }
        *(curr + 1) = value;
    }
}

NUMERICS_EXPORT void isort2(double* first1, double* last1, double* first2)
{
    double* iter1;
    double* iter2;
    double* j1;
    double* j2;
    double m1;
    double m2;
    
    for(iter1 = first1 + 1, iter2 = first2 + 1; iter1 != last1; ++iter1, ++iter2){
        m1 = *iter1;
        m2 = *iter2;
        j1 = iter1 - 1;
        j2 = iter2 - 1;
        while(j1 >= first1 && *j1 > m1){
            *(j1 + 1) = *j1;
            *(j2 + 1) = *j2;
            --j1;
            --j2;
        }
        *(j1 + 1) = m1;
        *(j2 + 1) = m2;
    }
}

void _msort1(double* first, double* last, double* tmp)
{
	if(first < last){
		double *lp, *le, *rp, *re, *tp;
		int m = (last - first) / 2;

		lp = first;
		le = lp + m;
		rp = le;
		re = last;
		tp = tmp + m;

		_msort1(lp, le, tp);
		_msort1(rp, re, tp);
		
		while(lp < le && rp < re){
			if(*lp <= *rp){
				*tp++ = *lp++;
			} else {
				*tp++ = *rp++;
			}
		}
		while(lp < le){
			*tp++ = *lp++;
		}
		while(rp < re){
			*tp++ = *rp++;
		}

		while(first < last){
			*first++ = *tmp++;
		}
	}
}

NUMERICS_EXPORT void msort1(double* first, double* last)
{
	double *tmp;
	tmp = dvector(0, last - first - 1);
	_msort1(first, last, tmp);
	free_dvector(tmp, 0);
}

void _msort2(double* first1, double* last1, double* first2, double* tmp1, double* tmp2)
{
	if(first1 < last1){
		double *lp1, *le1, *rp1, *re1, *tp1;
		double *lp2, *rp2, *tp2;
		int m = (last1 - first1) / 2;

		lp1 = first1;
		lp2 = first2;
		le1 = lp1 + m;
		rp1 = le1;
		rp2 = first2 + m;
		re1 = last1;
		tp1 = tmp1 + m;
		tp2 = tmp2 + m;

		_msort2(lp1, le1, lp2, tp1, tp2);
		_msort2(rp1, re1, rp2, tp1, tp2);
		
		while(lp1 < le1 && rp1 < re1){
			if(*lp1 <= *rp1){
				*tp1++ = *lp1++;
				*tp2++ = *lp2++;
			} else {
				*tp1++ = *rp1++;
				*tp2++ = *rp2++;
			}
		}
		while(lp1 < le1){
			*tp1++ = *lp1++;
			*tp2++ = *lp2++;
		}
		while(rp1 < re1){
			*tp1++ = *rp1++;
			*tp2++ = *rp2++;
		}

		while(first1 < last1){
			*first1++ = *tmp1++;
			*first2++ = *tmp2++;
		}
	}
}

NUMERICS_EXPORT void msort2(double* first1, double* last1, double* first2)
{
	double *tmp1, *tmp2;
	int n = last1 - first1 - 1;
	tmp1 = dvector(0, n);
	tmp2 = dvector(0, n);
	_msort2(first1, last1, first2, tmp1, tmp2);
	free_dvector(tmp1, 0);
	free_dvector(tmp2, 0);
}

NUMERICS_EXPORT void qsort1(double* first, double* last)
{
    double* loSwap;
    double* hiSwap;
    
    if(last - first > MAX_QSORT){
        loSwap = first + 1;
        hiSwap = last - 1;
        
        do {
            while(loSwap <= hiSwap && *loSwap <= *first){
                ++loSwap;
            }
            while(*hiSwap > *first){
                --hiSwap;
            }
            if(loSwap < hiSwap){
                iter_swap(loSwap, hiSwap);
            }
        } while(loSwap < hiSwap);
        
        iter_swap(first, hiSwap);
        
        if(first < hiSwap - 1){
            qsort1(first, hiSwap);
        }
        if(hiSwap + 2 < last){
			qsort1(hiSwap + 1, last);
        }
    } else {
		isort1(first, last);
	}
}

NUMERICS_EXPORT void qsort2(double* first1, double* last1, double* first2)
{
    double* loSwap, *hiSwap;
    double* lo2, *hi2;
    
    if(last1 - first1 > MAX_QSORT){
        loSwap = first1 + 1;
        hiSwap = last1 - 1;
        lo2 = first2 + 1;
        hi2 = first2 + (last1 - first1) - 1;
        
        do {
            while(loSwap <= hiSwap && *loSwap <= *first1){
                ++loSwap;
                ++lo2;
            }
            while(*hiSwap > *first1){
                --hiSwap;
                --hi2;
            }
            if(loSwap < hiSwap){
                iter_swap(loSwap, hiSwap);
                iter_swap(lo2, hi2);
            }
        } while(loSwap < hiSwap);
        
        iter_swap(first1, hiSwap);
        iter_swap(first2, hi2);
        
        if(first1 < hiSwap - 1){
            qsort2(first1, hiSwap, first2);
        }
        if(hiSwap + 2 < last1){
			qsort2(hiSwap + 1, last1, hi2 + 1);
        }
    } else {
		isort2(first1, last1, first2);
	}
}

NUMERICS_EXPORT void qsort2key(double* first1, double* last1, double* first2)
{
	double* key;
	double* lo, *hi;
     
	qsort2(first1, last1, first2);

	hi = first2;
	while(first1 < last1){
         key = first1;
         lo = hi;
         while(first1 < last1 && *first1 == *key){
             ++first1;
             ++hi;
         }
         if(hi - lo > 0){
             qsort1(lo, hi);
         }
	}
}

/*===================================================================
 Revision History

 Version 1.0 - 08/28/1998 - New.
 Version 1.1 - 04/10/1999 - Added isort1 and isort2.
                            Applied isort1 and isort2 to small groups
                            in qsort1 and qsort2.
 Version 1.2 - 01/07/2000 - Added msort1 and msort2.
===================================================================*/
