/*===================================================================
 noncdist.c

 Version 1.2

 Written by:
   Brent Worden
   WordenWare
   email:  brent@worden.org

 Copyright (c) 1998-2001 WordenWare

 Created:  August 28, 1998
 Revised:  March 26, 2001
===================================================================*/

#include <math.h>

#include "algorthm.h"
#include "domain.h"
#include "mathx.h"
#include "noncdist.h"
#include "normdist.h"
#include "numerics.h"
#include "numerror.h"
#include "utility.h"

#define R2DIVPI ( sqrt(2.0 / NUMERICS_PI) )
#define LOGRPI  ( log(sqrt(NUMERICS_PI)) )

NUMERICS_EXPORT double ncbetap(double x, double a, double b, double lambda)
{
    double ret = 0.0, r1;
    double beta, temp, sumq, c, q, errbd, a0, x0, ax, gx;
    double xj;

 	if(!isPositive(lambda)){
		NUMERICS_ERROR("ncbetap", "Invalid lambda value");
	} else if(!isPositive(a)){
		NUMERICS_ERROR("ncbetap", "Invalid a value");
	} else if(!isPositive(b)){
		NUMERICS_ERROR("ncbetap", "Invalid b value");
	} else if(x <= 0.0){
		ret = 0.0;
	} else if(x >= 1.0){
		ret = 1.0;
	} else {
		c = lambda * 0.5;

		r1 = c - 5.0 * sqrt(c);
		x0 = (r1 > 0.0) ? r1 : 0.0;
		a0 = a + x0;
		r1 = a0 + b;
		beta = gammln(a0) + gammln(b) - gammln(r1);
		temp = betai(a0, b, x);
		gx = exp(a0 * log(x) + b * log(1.0 - x) - beta - log(a0));
		if (a0 > a) {
			r1 = x0 + 1.0;
			q = exp(-c + x0 * log(c)) - gammln(r1);
		} else {
			q = exp(-c);
		}
		xj = 0.0;
		ax = q * temp;
		sumq = 1.0 - q;
		ret = ax;

		do {
			xj += 1.0;
			temp -= gx;
			gx = x * (a + b + xj - 1.0) * gx / (a + xj);
			q = q * c / xj;
			sumq -= q;
			ax = temp * q;
			ret += ax;
			errbd = (temp - gx) * sumq;
		} while(xj < NUMERICS_ITMAX && errbd > NUMERICS_MAX_ERROR);

		if(errbd > NUMERICS_MAX_ERROR) {
			NUMERICS_ERROR("ncbetap", "Algorithm failed to converge");
		}
	}
    return ret;
}

typedef struct {
	double alpha;
	double beta;
	double lambda;
} ncbeta_t;

double ncbetap_evaluate(double x, void* s)
{
	ncbeta_t* state = (ncbeta_t*)s;
	return ncbetap(x, state->alpha, state->beta, state->lambda);
}

NUMERICS_EXPORT double ncbetav(double p, double a, double b, double lambda)
{
	double ret = 0.0;

 	if(!isPositive(lambda)){
		NUMERICS_ERROR("ncbetav", "Invalid lambda value");
	} else if(!isPositive(a)){
		NUMERICS_ERROR("ncbetav", "Invalid a value");
	} else if(!isPositive(b)){
		NUMERICS_ERROR("ncbetav", "Invalid b value");
	} else if(!isZeroOne(p)){
		NUMERICS_ERROR("ncbetav", "Invalid p value");
	} else {
		ncbeta_t state = {a, p, lambda};
		double x0, x1;
		x0 = x1 = .5;
		if(!bracketOutWithState(ncbetap_evaluate, (void*)&state, &x0, &x1)){
			x0 = x1 = 0.5;
			while(ncbetap(x0, a, b, lambda) > p) x0 -= 0.1;
			while(ncbetap(x1, a, b, lambda) < p) x1 += 0.1;
		}
		if(x0 < 0.0) x0 = 0.0;
		if(x1 > 1.0) x1 = 1.0;
		ret = bisectionWithState(ncbetap_evaluate, (void*)&state, x0, x1, NUMERICS_MAX_ERROR);
	}
	return ret;
}

NUMERICS_EXPORT double ncchisqp(double x, double df, double delta)
{
    double ret;
    double term, t, u, v, bound, f2, x2;
    double lam;
	int n;

	if(!isPositive(df)){
		NUMERICS_ERROR("ncchisqp", "Invalid df value");
	} else if(isNegative(delta)){
		NUMERICS_ERROR("ncchisqp", "Invalid delta value");
	} else if(!isPositive(x)){
		return 0.0;
	}

    lam = delta / 2.0;
    n = 1;
    u = exp(-lam);
    v = u;
    x2 = x / 2.0;
    f2 = df / 2.0;
    t = pow(x2, f2) * exp(-x2) / exp(gammln(f2 + 1.0));
    term = v * t;
    ret = term;

	while(df + 2.0 * n <= x){
		u = u * lam / n;
		v += u;
		t = t * x / (df + 2.0 * n);
		term = v * t;
		ret += term;
		++n;
	}
	
    bound = t * x / (df + 2.0 * n - x);
	while(bound > NUMERICS_MAX_ERROR && n <= NUMERICS_ITMAX){
		u = u * lam / n;
		v += u;
		t = t * x / (df + 2.0 * n);
		term = v * t;
		ret += term;
		++n;
	    bound = t * x / (df + 2.0 * n - x);
	}

    if (bound > NUMERICS_MAX_ERROR) {
		NUMERICS_ERROR("ncchisqp", "Algorithm failed to converge");
    }
    return ret;
}

typedef struct {
	double df;
	double delta;
} ncchisq_t;

double ncchisqp_evaluate(double x, void* s)
{
	ncchisq_t* state = (ncchisq_t*)s;
	return ncchisqp(x, state->df, state->delta);
}

NUMERICS_EXPORT double ncchisqv(double p, double df, double delta)
{
	double ret = 0.0;

	if(!isPositive(df)){
		NUMERICS_ERROR("ncchisqv", "Invalid df value");
	} else if(isNegative(delta)){
		NUMERICS_ERROR("ncchisqv", "Invalid delta value");
	} else if(!isZeroOne(p)){
		NUMERICS_ERROR("ncchisqv", "Invalid p value");
	} else {
		ncchisq_t state = {df, delta};
		double x0, x1;
		x0 = x1 = 5.0;
		if(!bracketOutWithState(ncchisqp_evaluate, (void*)&state, &x0, &x1)){
			x0 = x1 = 5.0;
			while(ncchisqp(x0, df, delta) > p) x0 -= 0.5;
			while(ncchisqp(x1, df, delta) < p) x1 += 0.5;
		}
		if(x0 < 0.0) x0 = 0.0;
		ret = bisectionWithState(ncchisqp_evaluate, (void*)&state, x0, x1, NUMERICS_MAX_ERROR);
	}
	return ret;
}

NUMERICS_EXPORT double ncstudtp(double x, double df, double delta)
{
    double tnc = 0.0;
    double del = delta;
    BOOL negdel = FALSE;
    int en;
    double xx;
    
	if(!isPositive(df)){
        NUMERICS_ERROR("ncstudtp", "Invalid df");
    } else {
	    if(x < 0.0){
	        negdel = TRUE;
		    del *= -1;
		}
    
		en = 1;
		xx = x * x / (x * x + df);
    
		if(xx > 0.0){
			double lambda = del*del;
			double p = .5*exp(-.5*lambda);
			double q = R2DIVPI * p * del;
			double s = .5 - p, a = .5, b = .5*df;
			double rxb = pow(1.0-xx, b);
			double albeta = LOGRPI + gammln(b) - gammln(a+b);
			double xodd = betai(a, b, xx);
			double godd = 2.0*rxb*exp(a*log(x)-albeta);
			double xeven = 1.0-rxb, geven = b*xx*rxb;
			double errbd;
			tnc = p * xodd + q * xeven;
			do {
				a += 1.0;
				xodd -= godd;
				xeven -= geven;
				godd *= (xx*(a+b-1.0)/a);
				geven *= (xx*(a+b-.5)/(a+.5));
				p *= (lambda/(2.0*en));
				q *= (lambda/(2.0*en+1.0));
				s -= p;
				++en;
				tnc += p*xodd + q*xeven;
				errbd = 2.0 * s * (xodd-godd);
			} while(errbd > NUMERICS_MAX_ERROR && en <= NUMERICS_ITMAX);
		}
    
		if(en > NUMERICS_ITMAX){
			NUMERICS_ERROR("ncstudtp", "Iteration failed to converge");
		} else {
			tnc += (1.0 - normalp(del, 0.0, 1.0));
			if(negdel){
				tnc = 1.0 - tnc;
			}
		}
	}    
    return tnc;
}

typedef struct {
	double df;
	double delta;
} ncstudt_t;

double ncstudp_evaluate(double x, void* s)
{
	ncstudt_t* state = (ncstudt_t*)s;
	return ncstudtp(x, state->df, state->delta);
}

NUMERICS_EXPORT double ncstudtv(double p, double df, double delta)
{
	double ret = 0.0;

	if(!isPositive(df)){
        NUMERICS_ERROR("ncstudtv", "Invalid df");
	} else if(!isZeroOne(p)){
		NUMERICS_ERROR("ncstudtv", "Invalid p value");
	} else {
		ncstudt_t state = {df, delta};
		double x0, x1;
		x0 = x1 = delta;
		if(!bracketOutWithState(ncstudp_evaluate, (void*)&state, &x0, &x1)){
			x0 = x1 = delta;
			while(ncstudtp(x0, df, delta) > p) x0 -= 0.5;
			while(ncstudtp(x1, df, delta) < p) x1 += 0.5;
		}
		return bisectionWithState(ncstudp_evaluate, (void*)&state, x0, x1, NUMERICS_MAX_ERROR);
	}
	return ret;
}

/*===================================================================
 Revision History

 Version 1.0 - 08/28/1998 - New.
 Version 1.1 - 04/10/1999 - Added use of domain calls.
 Version 1.2 - 03/26/2001 - Added ncbetap, ncbetav, ncstudtv,
                            ncchisqp, ncchisqv
===================================================================*/
