/************  Listing 1 *********************  OT.C  **********************
*
*  ot.c --   Object Record Traversal
*  Author:   Thomas E. Siering, 1991
* 
*  Usage:    This utility will list the Record Types present in a given
*            OBJ module.  Any command line parameters (other than the
*            OBJ file name) are considered Record Types.  In that case,
*            only the presence of those types will be reported.
*            Any Record Type 0 <= RT <= 255 will be searched for even if
*            it is not (universally) recognized as part of the OMF standard.
*            CAVEAT: Record Type on command line has to be DECIMAL!
* 
*  Compiler: Microsoft C 6.0, Turbo C++ 1.0
* 
*  Compile time switches: none
* 
*  Links with: This is a stand-alone module.
*
*  (c) Copyright 1991, C Gazette. May be used freely as long as authorship
*                      and publication are acknowledged.
***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <limits.h>


#pragma pack(1)         /* Byte-align the ObjHeader struct */

typedef enum { false, true } bool;


static char ProgName[] = { "OT" };


void main (int argc, char *argv[])
{
    FILE *ObjFH;
    int RecTypeCount[UCHAR_MAX + 1];
    bool DoShow[UCHAR_MAX + 1];

    struct {
        unsigned char RecordType;
        int RecordLength;
    } ObjHeader;

    int RecTotal;
    int i;

    /* Usage info for the uninitiated */
    if (argc < 2) 
    {
        fprintf(stderr, "\aUsage: %s fn.obj [options]\n", ProgName);
        fprintf(stderr, "Options: Record Types to list\n");
        exit(-1);
    }

    /* Check out the OBJ module file */
    if ( (ObjFH = fopen(argv[1], "rb")) == NULL ) 
    {
        fprintf(stderr, "\aFile %s could not be opened\n", argv[1]);
        exit(-1);
    }

    /* So far, no instances of any Record Type have been sighted */
    for (i = 0; i <= UCHAR_MAX; i++)
        RecTypeCount[i] = 0;

    /* Sort out which OBJ records we'll want to know about */
    if (argc == 2)
        /* Show them all! */
        for (i = 0; i <= UCHAR_MAX; i++)
            DoShow[i] = true;
    else 
    {
        int RecType;

        for (i = 0; i <= UCHAR_MAX; i++)
            DoShow[i] = false;

        for (i = argc; i > 2; i--) 
        {
            RecType = atoi(argv[i - 1]);
            if (RecType < 0 || RecType > UCHAR_MAX)
                fprintf(stderr, "\aIncorrect Record Type entered: %X\n",
                        RecType);
            else
                DoShow[RecType] = true;
        }
    }


    while ( fread(&ObjHeader, sizeof(ObjHeader), 1, ObjFH) == 1) 
    {
        /* Another OBJ record to account for */
        RecTypeCount[ObjHeader.RecordType]++;
        fseek(ObjFH, (long) ObjHeader.RecordLength, SEEK_CUR);

    #ifdef VERBOSE
        fprintf(stdout, "Record Type = %02X\nLength = %d\n",
                ObjHeader.RecordType & 0x00FF, ObjHeader.RecordLength);
    #endif
    }

    /* Make sure last read failed because of EOF. 
       Otherwise there's trouble! */
    if (feof (ObjFH) == 0) 
    {
        fprintf(stderr, "\aRead of OBJ Failed.  Aborting\n");
        exit(-1);
    }

    fclose(ObjFH);

    /* Finally, print the statistics */
    fprintf(stdout, "File %s:\n", argv[1]);
    for (i = 0, RecTotal = 0; i <= UCHAR_MAX; i++) 
    {
        if (RecTypeCount[i] > 0 && DoShow[i] == true) 
        {
            fprintf(stdout, "RecordType %X  --  %d\n", i, RecTypeCount[i]);
            RecTotal += RecTypeCount[i];
        }
    }
    fprintf(stdout, "Total Records: %d\n\n", RecTotal);
}