/*++

Module Name:

    ufsdapi.h

Abstract:

    This module is a common include file for
    linux vfs modules.

Author:

    Ahdrey Shedel

Revision History:

    27/12/2002 - Andrey Shedel - Created

--*/
#ifndef __UFSDAPI_INC__
#define __UFSDAPI_INC__

#ifdef __cplusplus
#define EXTERN_C extern "C"
#else
#define EXTERN_C
#endif


#ifdef __cplusplus
extern "C" {
#endif
#include "ufsdver.h"
#include "vfsdebug.h"
#include <linux/string.h>
#include <linux/kernel.h>
// kernel defined '_SIZE_T' but out 'types.h' will check for '_SIZE_T_DEFINED'
#ifdef _SIZE_T
#define _SIZE_T_DEFINED
#endif
#ifdef __cplusplus
} //extern "C"
#endif



//
// Missing but useful declarations.
//

#define IN
#define OUT
#define OPTIONAL

#define ARGUMENT_PRESENT(p) (NULL != (p))

typedef unsigned long long ULONGLONG;
typedef unsigned long ULONG;

typedef union _ULARGE_INTEGER {
	struct {
		ULONG LowPart;
		ULONG HighPart;
	};
	struct {
		ULONG LowPart;
		ULONG HighPart;
	} u;
	ULONGLONG QuadPart;
} ULARGE_INTEGER, *PULARGE_INTEGER;


typedef int BOOL;
#define FALSE ((BOOL)0)
#define TRUE  ((BOOL)1)

typedef void* PVOID;
typedef const void* PCVOID;

#ifndef SetFlag
#define SetFlag(flags, single_flag) (flags |= (single_flag))
#endif

#ifndef ClearFlag
#define ClearFlag(flags, single_flag) (flags &= ~(single_flag))
#endif

#ifndef FlagOn
#define FlagOn(flags, single_flag) (single_flag == ((flags) & (single_flag)))
#endif

#ifndef IN
#define IN
#endif

#ifndef OUT
#define OUT
#endif

#ifndef OPTIONAL
#define OPTIONAL
#endif

#ifndef NOTHING
#define NOTHING {;}
#endif




#ifndef min
#define min(x,y) ({ \
	const typeof(x) _x = (x);	\
	const typeof(y) _y = (y);	\
	(void) (&_x == &_y);		\
	_x < _y ? _x : _y; })

#define max(x,y) ({ \
	const typeof(x) _x = (x);	\
	const typeof(y) _y = (y);	\
	(void) (&_x == &_y);		\
	_x > _y ? _x : _y; })

#define min_t(type,x,y) \
	({ type __x = (x); type __y = (y); __x < __y ? __x: __y; })
#define max_t(type,x,y) \
	({ type __x = (x); type __y = (y); __x > __y ? __x: __y; })

#endif


#define _QUOTE1(name) #name
#define _QUOTE2(name) _QUOTE1(name)
#define QUOTED_UFSD_DEVICE  _QUOTE2(UFSD_DEVICE)


//
// Global shared data structure.
//

typedef struct _UFSD_DATA {

	//
	// The mask indicating which filesystems
	// compiled in UFSD are allowed to be used.
	//

	ULONG AllowedFilesystems;


	//
	// The mask indicating which filesystems
	// compiled in UFSD are allowed to be mounted in read/write mode.
	//

	ULONG ReadWriteFilesystems;


} UFSD_DATA, *PUFSD_DATA;




//
// Structure above can be referenced from any
// UFSD module using UFSD_Data variable.
// This variable will be defined in ufsddata.c file.
//

extern UFSD_DATA UFSD_Data;





//
// Memory allocation routines.
//

EXTERN_C
PVOID
UFSD_HeapAlloc (
    IN ULONG Size
   );

EXTERN_C
void
UFSD_HeapFree (
    IN PVOID Pointer
   );

EXTERN_C
PVOID
UFSD_HeapRealloc(
    IN PVOID Pointer,
    IN ULONG Size
   );


//
// CURRENT_TIME wrapper
//

EXTERN_C
time_t
UFSD_CurrentTime(void);

//
// Device IO functions.
//

struct super_block;

EXTERN_C
BOOL
UFSD_BdRead (
    IN struct super_block* sb,
    IN ULONGLONG StartSector,
    IN ULONG NrSectors,
    OUT PVOID Buffer
   );

EXTERN_C
BOOL
UFSD_BdWrite (
    IN struct super_block* sb,
    IN ULONGLONG StartSector,
    IN ULONG NrSectors,
    IN PCVOID Buffer
   );


EXTERN_C
BOOL
UFSD_BdSetBlockSize (
    IN struct super_block* sb,
    IN ULONG NewBlockSize
   );

EXTERN_C
ULONG
UFSD_BdGetBlockSize (
    IN struct super_block* sb
   );

EXTERN_C
BOOL
UFSD_BdIsReadonly (
    IN struct super_block* sb
   );

EXTERN_C
const char*
UFSD_BdGetName (
    IN struct super_block* sb
   );




typedef struct _UFSD_MUTEX* PUFSD_MUTEX;


EXTERN_C
PUFSD_MUTEX
UFSD_MutexCreate (void);

EXTERN_C
void
UFSD_MutexDelete (
    IN PUFSD_MUTEX Mutex
   );

EXTERN_C
int
UFSD_MutexEnter (
    IN PUFSD_MUTEX Mutex,
    IN BOOL Wait
   );

EXTERN_C
int
UFSD_MutexLeave (
    IN PUFSD_MUTEX Mutex
   );




EXTERN_C
unsigned long long
UFSD_TimePosix2Nt (
    IN time_t PosixTime
   );

EXTERN_C
time_t
UFSD_TimeNt2Posix (
    IN unsigned long long NtTime
   );



//
// Some forwards and
// helper API declarations.
//

typedef struct UFSD_VOLUME* PUFSD_VOLUME;
typedef struct UFSD_FILE* PUFSD_FILE;


EXTERN_C
int
UFSD_NamesEqual (
    IN PUFSD_VOLUME Volume,
    IN const char* Name1,
    IN unsigned Name1Len,
    IN const char* Name2,
    IN unsigned Name2Len
   );


EXTERN_C
int
UFSDAPI_VolumeMount (
    IN struct super_block * sb,
    IN long Codepage,
    IN int CaseInsensitive,
    OUT PUFSD_VOLUME* Volume
   );

EXTERN_C
void
UFSDAPI_VolumeFree (
    IN PUFSD_VOLUME Volume
   );

EXTERN_C
int
UFSDAPI_VolumeFlush (
    IN PUFSD_VOLUME Volume
   );

EXTERN_C
void
UFSDAPI_QueryVolumeInfo (
    IN PUFSD_VOLUME Volume,
    OUT ULONG* FsSignature OPTIONAL,
    OUT ULONG* BlocksTotal OPTIONAL,
    OUT ULONG* BlocksFree OPTIONAL,
    OUT ULONG* NameLength OPTIONAL
   );


#if UFSD_READONLY
#define UFSDAPI_VolumeIsReadonly(v) 1
#else
EXTERN_C
int
UFSDAPI_VolumeIsReadonly (
    IN PUFSD_VOLUME Volume
   );
#endif




EXTERN_C
int
UFSDAPI_FileOpen (
    IN PUFSD_VOLUME Volume,
    IN PUFSD_FILE ParentDir,
    IN const char* Name,
    IN unsigned NameLen,
    IN int Create,
    IN int CreateDir,
    IN int CreateReadonly,
    OUT PUFSD_FILE* File,
    OUT int* is_dir,
    OUT int* is_system,
    OUT int* is_readonly,
    OUT int* subdir_count,
    OUT unsigned long long* size,
    OUT time_t* atime,
    OUT time_t* ctime,
    OUT time_t* mtime
   );


EXTERN_C
int
UFSDAPI_FileClose (
    IN PUFSD_VOLUME Volume,
    IN OUT PUFSD_FILE* File,
    IN int Delete
   );


EXTERN_C
int
UFSDAPI_FileRead (
    IN PUFSD_VOLUME Volume,
    IN PUFSD_FILE File,
    IN unsigned long long Offset,
    IN size_t ByteCount,
    OUT void* Buffer,
    OUT unsigned long* BytesDone
   );


#if !UFSD_READONLY

EXTERN_C
int
UFSDAPI_FileWrite (
    IN PUFSD_VOLUME Volume,
    IN PUFSD_FILE FileHandle,
    IN unsigned long long Offset,
    IN size_t ByteCount,
    IN const void* Buffer,
    OUT unsigned long* BytesDone
   );


EXTERN_C
int
UFSDAPI_FileSetAttributes (
    IN PUFSD_VOLUME Volume,
    IN PUFSD_FILE FileHandle,
    IN int set_atime, IN time_t atime OPTIONAL,
    IN int set_mtime, IN time_t mtime OPTIONAL,
    IN int set_ctime, IN time_t ctime OPTIONAL,
    IN int set_size,  IN unsigned long long size OPTIONAL,
    IN int set_readonly, IN int readonly
   );


EXTERN_C
int
UFSDAPI_FileMove (
    IN PUFSD_VOLUME Volume,
    IN PUFSD_FILE FileHandle,
    IN PUFSD_FILE NewDirHandle,
    IN const char* NewName,
    IN size_t NewNameLen
   );


#endif //!UFSD_READONLY


//
// readdir() helpers.
//


typedef struct _UFSD_SEARCH* PUFSD_SEARCH;

EXTERN_C
int
UFSD_FindOpen (
    IN PUFSD_VOLUME Volume,
    IN PUFSD_FILE Dir,
    IN unsigned pos,
    OUT PUFSD_SEARCH* Scan
   );

EXTERN_C
int
UFSD_FindGet (
    IN PUFSD_SEARCH Scan,
    IN OUT unsigned* pos,
    OUT char** Name,
    OUT size_t* NameLen,
    OUT int* is_dir
   );

EXTERN_C
void
UFSD_FindClose (
    IN PUFSD_SEARCH Scan
   );



//
// cxx part.
//

#ifdef __cplusplus

#include "../include/types.h"
#include "../include/memmngr.h"
#include "../include/rwblock.h"
#include "../include/strings.h"
#include "../include/time.h"
#include "../include/fsbase.h"
// no UFSD wrapper!!!
//#include "../include/ufsdinl.h"
//#include "../include/ufsd.h"
#include "../include/errors.h"
#include "../include/uint64.h"


//
// Get memory manager implementation
// Implemented in ufsdmem.cpp
//

class UFSD::CMemoryManager* UFSD_MemoryManagerCreate(void);
void UFSD_MemoryManagerFree(IN class UFSD::CMemoryManager* mm);

class UFSD::CStrings* UFSD_StringsServiceCreate (IN long Codepage, IN int CaseInsensitive);
void UFSD_StringsServiceFree (class UFSD::CStrings* ss);

class UFSD::CRWBlock* UFSD_IOHandlerCreate (IN struct super_block* sb, IN BOOL Readonly);
void UFSD_IOHandlerFree (class UFSD::CRWBlock* iom);

class UFSD::CTime* UFSD_TimeServiceCreate(void);
void UFSD_TimeServiceFree(class UFSD::CTime* ts);

#endif

//
// Errors header will be included for all modules.
//

#include "../include/errors.h"

#endif
