; kmh's master boot sector loader

timeout equ 10 * 18   ;number of seconds to wait times 18.2
; maximum timeout is 65,525. see below for reasons why

start_of_prompt  equ  600h + 512
start_of_table_of_partition_tables  equ  start_of_prompt + 512
end_of_table_of_partition_tables  equ  start_of_table_of_partition_tables + (66*15)


; assume the bios has read these 512 bytes into memory starting at
; 0000:7C00, and then jumped to 0000:7C00

org 0600h  ;0600h instead of 7C00h because we will relocate to 0600h


; initialize stack, interrupts, segments
    xor  ax,ax
    mov  ss,ax
    mov  sp,7C00h
    sti
    mov  ds,ax
    mov  es,ax
; This is probably not needed because the bios should have done this
; already. We do this in case the bios is defective and failed to initialize
; the stack, interrupts, and segments. Instead of doing 'mov sp,7C00h', we
; could have done 'mov sp,ax', which would set sp to 0 instead of 7C00; that
; would save one byte and give us a little more stack space. Some source
; code from PC Techniques magazine for a custom floppy disk boot sector set
; sp to 0; so if it worked for them, it should work for us. But we already
; have more than enough stack space, so the size of the stack does not
; matter. Other partition table programs usually set sp to 7C00, so we will
; set sp to 7C00 too; because if we do things the same as everyone else
; whenever possible, then this is less likely to be incompatible with other
; operating systems and programs. We do not not initialize cs and ip here;
; the best way to initialize cs and ip is with a long jump; we are going to
; do a jump as soon as we relocate the code; so we will combine the long
; jump to initialize cs and ip with the jump to the new location, and save
; one line of code and a few bytes.




read_first_4_sectors:
; reset hard drive
    xor  ax,ax
    int  13h
; read the first four sectors of the hard drive into memory at 0000:0600
    mov  bx,600h  ;write data to es:bx
    mov  cx,1h
    mov  dx,80h  ;cx and dx are drive, cylinder, head, sector
    mov  ax,204h   ;al=04 is number of sectors to read
    int  13h
    jc  read_first_4_sectors
; this will create an endless loop if it fails to read the disk. But if it
; cannot read the disk, we cannot boot the computer, so what else is there
; to do? It should read the disk successfully on the first try, so it
; does not matter what we do if the read fails; we probably do not need
; to check for failure to read disk.
; we usually do not need to reset the hard drive. But sometimes we need
; to, and it is easier to reset the hard drive every time, and not
; worry about whether it is needed or not.
; The assembler will calculate the address of read_first_4_sectors
; incorrectly, because we told the assembler this code will be run from
; offset 600h, but we have not yet relocated the code to 600h. But the
; address of read_first_4_sectors is irrelevant, because the only jump to
; read_first_4_sectors is jc, and jc is a relative jump, it jumps relative
; to the current ip. The assembler will correctly calculate the number of
; bytes between label read_first_4_sectors and instruction 'jc
; read_first_4_sectors', and when this code runs, the cpu will subtract that
; much from ip, and the jump will land at the correct instruction.
; When we read four sectors into memory, the first sector is this program we
; are running, which is already in memory. Instead of reading four sectors,
; we could copy 512 bytes from 7C00h to 0600h, then read three sectors
; starting with the second sector to 0800h; that would probably be faster,
; but this is simpler.


; jump to the new location
; this also initializes cs and ip
    jmp  0:jump_to_here_after_relocation
jump_to_here_after_relocation:





; display the prompt
    mov  si,start_of_prompt
    call  display_message_pointed_to_by_si





; get the current time in dx
    mov  ah,0
    int  1Ah

; add the timeout to the current time to get the time when done, and store
; the time when done in di; set bl to 0 if the add did NOT result in a
; carry, set bl to 1 if the add did result in a carry.
    mov  bl,0
    add dx,timeout
    jnc do_not_set_bl_to_1
    mov bl,1
do_not_set_bl_to_1:
    mov  di,dx
; Usually we want to wait until the current time is greater than the time
; when done. However, if we start at time FFF0h, and the timeout is 0010h,
; then the time when done will be 0000h; then if we wait until the current
; time is greater than 0000h, the current time is already greater than
; 0000h, so we will not wait at all. In other words, there are two
; possibilities: usually we wait until the current time is greater than the
; time when done. But sometimes we wait until the current time is 0, then
; wait until the current time is greater than the time when done. We set bl
; to 0 for the first possibility, or to 1 for the second possibility.





check_timeout:
; get the current time in dx
    mov  ah,0
    int  1Ah

; if bl is 0, check if we are out of time
    cmp  bl,0
    je   are_we_out_of_time
; else check to see if the current time has gone to 0

; if dx (current time) is greater than 10, then the current time has not
; yet gone to 0; check for a keystroke
    cmp  dx,10
    ja   check_for_key
; else the current time has gone to 0; set bl to 0 because we are done
; waiting for the current time to go to 0, and then check for a keystroke
    mov  bl,0
    jmp short  check_for_key
; we are waiting for dx to go to 0000h, so you might think that we should
; compare dx and 0. However, we might miss the time of 0000h; we might check
; the time and it would be FFFFh, and then we might check the time again and
; it might be 0001h. This error is very unlikely to occur; and if it did
; occur, it would not crash the computer, it would wait for the time to go
; to 0000h again, so it would make the timeout longer. But don't you hate
; intermittant random bugs? However, by dealing with one intermittant random
; bug, we have created another: if the timeout is FFF7h, and the starting
; time is 0007h, then the time when done will be 0000h; first we wait for
; the current time to be less than 10, but it already is, then we wait for
; the current time to be greater than the time when done, but it already is,
; so we get no timeout. To prevent this bug, do not use a timeout of more
; than FFFFh - 10, or FFF5h, or 65,525. Feel free to change 10 to 2 or 1 or
; 0 if you want; or use 32 bit or 64 bit times.

are_we_out_of_time:
; if dx (current time) is above (unsigned integer greater than) di (time
; when done), then it is time to stop waiting for a keystroke and boot from
; partition 1.
    cmp dx,di
    ja  boot_default_partition_table
; else check for a keystroke

check_for_key:
; set the zero flag if there is no keystroke waiting in the keyboard buffer
    mov  ah,01h
    int  16h

; if there is no keystroke waiting (if the zero flag is set), check the time
; again
    jz   check_timeout

; else read one character from the keyboard into al
    mov     ah, 0
    int     16h

; check if the key in al matches any of the hotkeys
    mov si,start_of_table_of_partition_tables
compare_key_in_al_to_next_partition_table:
    cmp  al,[si]
    je boot_partition_table_pointed_to_by_si
    add  si,66
    cmp  si,end_of_table_of_partition_tables
    jae  check_timeout
    jmp short  compare_key_in_al_to_next_partition_table






display_message_pointed_to_by_si:
; set bh to the page, bl to the color and ah to 0Eh for int 10.
; I am not sure what to put for the page and the color, but bx=0 works.
; The example from How It Works did bx=7
    xor  bx,bx
    mov  ah,0Eh
display_next_character:
; put the next character of the message in al
    mov  al,[si]
; if al=0, we are done displaying the prompt
    cmp  al,0
    je  done_displaying_message
; display the character
    int  10h   ;ah=0Eh,al=char,bh=page,bl=color
; increment si to point to the next character of the message
    inc  si
    jmp short  display_next_character
done_displaying_message:
    ret








boot_default_partition_table:
    mov  si,start_of_table_of_partition_tables
boot_partition_table_pointed_to_by_si:

; if the partition table pointed to by si is the same as the current
; partition table, go ahead and boot using the current partition table
    cmp  si,[current_partition_table_identifier]
    je  find_active_partition

; otherwise overwrite the current partition table with the partition table
; pointed to by si
    mov  [current_partition_table_identifier],si
    cld
    push  si
    inc  si
    inc  si
    mov  di,start_of_current_partition_table
    mov  cx,32  ;32 words means 64 bytes
    repnz
    movsw
    pop  si

; if the byte at [si+1] is 1, then the current partition table needs to be
; written to the hard drive. if the current partition table does not need to
; be written to the hard drive, go ahead and boot using the current
; partition table
    cmp byte  [si + 1],1
    jne  find_active_partition

; otherwise write the current partition table to the hard drive
write_partition_table:
    mov  bx,600h
    mov  cx,1h
    mov  dx,80h  ;cx and dx are drive, cylinder, head, sector
    mov  ax,301h   ;ah=3 means write  al=1 means 1 sectors
    int  13h
    jc write_partition_table
; this will go into an infinite loop if it cannot write the partition table
; to the hard drive. If it cannot write the partition hard table to the hard
; drive, there is probably something wrong with the hard drive, and we will
; probably not be able to boot the computer anyway.




find_active_partition:
;point si to the first entry in the partition table
    mov  si,start_of_current_partition_table

is_partition_table_entry_active:
; if the first byte of the entry is 80h, then the entry is active
    cmp byte  [si],80h
    jz  load_boot_sector_from_partition_pointed_to_by_si

; point si to the next partition table entry
    add  si,16

; if si is before the end of the partition table, check if the entry
; is active.
    cmp  si,end_of_current_partition_table
    jb  is_partition_table_entry_active

; otherwise, if we are already at the end of the partition table, and we
; have not found an active entry, then there are no active entries. The
; traditional thing to do here is 'int 18h' to enter rom basic, but that is
; kind of dumb because computers do not have rom basic anymore, and usually
; if you try to enter rom basic, the computer displays a message saying 'no
; rom basic', which fails to tell you the real problem, which is that there
; are no active entries in the partition table. So instead of int 18h, we
; will display a message saying no active partitions.
    mov  si,message_no_active_partitions
    call  display_message_pointed_to_by_si
    mov  si,message_reboot
    call  display_message_pointed_to_by_si
    jmp short  reboot
     






load_boot_sector_from_partition_pointed_to_by_si:
; reset hard drive
    xor  ax,ax
    int  13h
; read boot sector
    mov  dx,[si]
    mov  cx,[si+2]  ;cx and dx are device, cylinder, head, sector
    mov  bx,7C00h  ;write sector to es:bx
    mov  ax,201h  ;al=1 means read 1 sector
    int  13h
    jc  load_boot_sector_from_partition_pointed_to_by_si
; this will create an endless loop if it fails to read the disk. But if it
; cannot read the disk, we cannot boot the computer, so what else is there
; to do? It should read the disk successfully on the first try, so it
; does not matter what we do if the read fails; we probably do not need
; to check for failure to read disk.
; we usually do not need to reset the hard drive. But sometimes we need
; to, and it is easier to reset the hard drive every time, and not
; worry about whether it is needed or not.





; check boot sector signature
    cmp word  [0x7DFE],0xAA55
    jne  wrong_boot_sector_signature
; if the boot sector signature is ok, jump to the boot sector code
    jmp  7C00h

wrong_boot_sector_signature:
    mov  si,message_wrong_boot_sector_signature
    call  display_message_pointed_to_by_si
    mov  si,message_reboot
    call  display_message_pointed_to_by_si
    jmp short  reboot


reboot:
; wait for any key (read one character from the keyboard)
    mov     ah, 0
    int     16h
; warm boot
; my book says to write 1234h to 0040h:0072h. That is the same as 0:472h;
; and since we already have the segment registers set to 0, we will write
; to 472h
    mov word  [472h],1234h
    jmp  0FFFFh:0







message_wrong_boot_sector_signature:
 db 'Wrong boot sector signature. Try running SYS.',0Dh,0Ah,0
message_no_active_partitions:
 db 'No active partitions. Try running FDISK.',0Dh,0Ah,0
message_reboot:
 db 'Press any key to reboot.',0Dh,0Ah,0


; add enough nuls to make this exactly 512 bytes
times 512 - (2 + 64 + 2 + ($ - $$)) db 0

current_partition_table_identifier:
times 2 db 0
start_of_current_partition_table:
times 64 db 0
end_of_current_partition_table:
dw 0AA55h  ;partition table signature



; Parts of this were copied from the MBR section of How It Works by Hale
; Landis (landis@sugs.tware.com), which includes a disassembled partition
; table, disassembled from a partition table created by Microsoft DOS FDISK,
; I am not sure which version of DOS, maybe 5? I found How It Works on the
; internet somewhere.

; Parts of this were copied from the source code for PART.EXE, Ranish
; Partition Manager by Mikhail Ranish (ranish@intercom.com). The source code
; was included with PART.EXE, which I downloaded from the internet
; somewhere, probably from simtel.
