/****************************************************************

	gr_def.c        default line, rectangle, and circle
			routines for Bywater
			Graphics Interface standard

			Copyright (c) 1991, Ted A. Campbell

			Bywater Software
			P. O. Box 4023
			Duke Station
			Durham, NC  27706

			email: tcamp@hercules.acpub.duke.edu

	Copyright and Permissions Information:

	All U.S. and international copyrights are claimed by the
	author. The author grants permission to use this code
	and software based on it under the following conditions:
	(a) in general, the code and software based upon it may be
	used by individuals and by non-profit organizations; (b) it
	may also be utilized by governmental agencies in any country,
	with the exception of military agencies; (c) the code and/or
	software based upon it may not be sold for a profit without
	an explicit and specific permission from the author, except
	that a minimal fee may be charged for media on which it is
	copied, and for copying and handling; (d) the code must be
	distributed in the form in which it has been released by the
	author; and (e) the code and software based upon it may not
	be used for illegal activities.

****************************************************************/


#include "stdio.h"
#include "math.h"

#ifdef __STDC__
#include "malloc.h"
#else
extern char * malloc();
#define size_t  int
#endif

#include "bw.h"
#include "gr.h"
#include "kb.h"

#define CHECK_PARAMS    FALSE
#ifndef DEG_RAD
#define RAD_DEG         5.729577951e1   /* Convert radians to degrees   */
#define DEG_RAD         1.745329252e-2  /* Convert degrees to radians   */
#endif
#ifndef ACCURACY
#define ACCURACY        10
#endif

#ifdef  STANDALONE
struct gr_window main_window;
char tbuf[ 128 ];
char bw_ebuf[ BW_EBUFSIZE ];
int x_pos, y_pos, b_stat;

main()
   {
   register int c;
   int x, y;

   gr_init( &main_window, NULL );
   kb_init();

   x = main_window.xmax / 8;
   y = ( main_window.ymax / 4 ) * 3;
   c = 0;
   while ( ( c < gr_colors ) && ( y > main_window.fysize ) )
      {
      def_line( GR_PRIMARY, x, y, x + main_window.xmax / 20, y,
	 c, SOLID );
      x += main_window.xmax / 10;
      def_line( GR_PRIMARY, x, y, x + main_window.xmax / 20, y,
	 c, HATCH );
      sprintf( tbuf, " <- lines, color %d at x = %d, y = %d",
	 c, x, y );
      gr_text( GR_PRIMARY, x + ( 3 * ( main_window.xmax / 20 )),
	 y - ( main_window.fysize / 2 ), tbuf, WHITE, BLACK );
      ++c;
      x -= main_window.xmax / 10;
      y -= ( main_window.fysize / 2 ) * 3;
      }
   kb_rx();

   gr_cls( GR_PRIMARY );
   def_line( GR_PRIMARY, main_window.xmax / 2, main_window.ymax,
      main_window.xmax, main_window.ymax / 2, WHITE, GRID );
   def_line( GR_PRIMARY, main_window.xmax, main_window.ymax / 2,
      main_window.xmax / 2, 0, WHITE, GRID );
   def_line( GR_PRIMARY, main_window.xmax / 2, 0,
      0, main_window.ymax / 2, WHITE, GRID );
   def_line( GR_PRIMARY, 0, main_window.ymax / 2,
      main_window.xmax / 2, main_window.ymax, WHITE, GRID );
   def_line( GR_PRIMARY, 0, main_window.ymax / 2,
      main_window.xmax, main_window.ymax / 2, WHITE, SOLID );
   def_line( GR_PRIMARY, main_window.xmax / 2, 0,
      main_window.xmax / 2, main_window.ymax, WHITE, SOLID );
   kb_rx();

   gr_cls( GR_PRIMARY );
   x = main_window.xmax / 40;
   y = main_window.ymax / 20;
   def_rectangle( GR_PRIMARY, x * 4, y * 16, x * 6, y * 18, WHITE, HOLLOW );
   gr_text( GR_PRIMARY, x * 10, y * 17, "Hollow rectangle", WHITE, BLACK );
   def_rectangle( GR_PRIMARY, x * 4, y * 12, x * 6, y * 14, WHITE, SOLID );
   gr_text( GR_PRIMARY, x * 10, y * 13, "Solid rectangle", WHITE, BLACK );
    def_rectangle( GR_PRIMARY, x * 4, y * 8, x * 6, y * 10, WHITE, GRID );
   gr_text( GR_PRIMARY, x * 10, y * 9, "Grid rectangle", WHITE, BLACK );
    def_rectangle( GR_PRIMARY, x * 4, y * 4, x * 6, y * 6, WHITE, HATCH );
   gr_text( GR_PRIMARY, x * 10, y * 5, "Hatch rectangle", WHITE, BLACK );
   kb_rx();

   gr_cls( GR_PRIMARY );
   x = main_window.xmax / 40;
   y = main_window.ymax / 20;
   def_circle( GR_PRIMARY, x * 5, y * 17, (y/2)*3, WHITE, HOLLOW );
   gr_text( GR_PRIMARY, x * 10, y * 17, "Hollow circle", WHITE, BLACK );
   def_circle( GR_PRIMARY, x * 5, y * 13, (y/2)*3, WHITE, SOLID );
   gr_text( GR_PRIMARY, x * 10, y * 13, "Solid circle", WHITE, BLACK );
    def_circle( GR_PRIMARY, x * 5, y * 9, (y/2)*3, WHITE, GRID );
   gr_text( GR_PRIMARY, x * 10, y * 9, "Grid circle", WHITE, BLACK );
    def_circle( GR_PRIMARY, x * 5, y * 5, (y/2)*3, WHITE, HATCH );
   gr_text( GR_PRIMARY, x * 10, y * 5, "Hatch circle", WHITE, BLACK );

   kb_rx();
   kb_deinit();
   gr_deinit();
   }

#endif

/****************************************************************

	def_line()

****************************************************************/

def_line( screen, x1, y1, x2, y2, color, style )
   int screen;
   int x1, y1, x2, y2;
   int color, style;
   {
   register int t, dist;
   int xerr, yerr, dx, dy, incx, incy;
   int x_val, y_val;

#if CHECK_PARAMS
   if ( ( x1 < 0 ) || ( x1 > main_window.xmax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_line(): x1 value is %d", x1 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( x2 < 0 ) || ( x2 > main_window.xmax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_line(): x2 value is %d", x2 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y1 < 0 ) || ( y1 > main_window.ymax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_line(): y1 value is %d", y1 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y2 < 0 ) || ( y2 > main_window.ymax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_line(): y2 value is %d", y2 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
#endif

   xerr = yerr = 0;
   dx = x2 - x1;
   dy = y2 - y1;
   x_val = x1;
   y_val = y1;

   if ( dx > 0 )
      {
      incx = 1;
      }
   else if ( dx == 0 )
      {
      incx = 0;
      }
   else
      {
      incx = -1;
      }

   if ( dy > 0 )
      {
      incy = 1;
      }
   else if ( dy == 0 )
      {
      incy = 0;
      }
   else
      {
      incy = -1;
      }

   dx = abs(dx);
   dy = abs(dy);

   dist = (dx > dy) ? dx : dy;

   for ( t = 0; t <= ( dist + 1 ); t++ )
      {
      switch( style )
	 {
	 case HOLLOW:
	    gr_pixel( screen, x_val, y_val, BLACK );
	    break;
	 case SOLID:
	    gr_pixel( screen, x_val, y_val, color );
	    break;
	 case GRID:
	 case HATCH:
	    if ( ( ( x_val + y_val ) % 2 ) == 1 )
	       {
	       gr_pixel( screen, x_val, y_val, color );
	       }
	    break;
	 }
      xerr += dx;
      yerr += dy;
      if ( xerr > dist )
	 {
	 xerr -= dist;
	 x_val += incx;
	 }
      if ( yerr > dist )
	 {
	 yerr -= dist;
	 y_val += incy;
	 }
      }
   }

/****************************************************************

	def_rectangle()

****************************************************************/

def_rectangle( screen, x1, y1, x2, y2, color, style )
   int screen;
   int x1, y1, x2, y2;
   int color, style;
   {
   register int x_val, y_val;

#if CHECK_PARAMS
   if ( ( x1 < 0 ) || ( x1 > main_window.xmax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_rectangle(): x1 value is %d", x1 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( x2 < 0 ) || ( x2 > main_window.xmax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_rectangle(): x2 value is %d", x2 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y1 < 0 ) || ( y1 > main_window.ymax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_rectangle(): y1 value is %d", y1 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y2 < 0 ) || ( y2 > main_window.ymax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_rectangle(): y2 value is %d", y2 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
#endif

   /*  First draw border around the rectangle */

   gr_line( screen, x1, y1, x2, y1, color, SOLID );
   gr_line( screen, x2, y1, x2, y2, color, SOLID );
   gr_line( screen, x1, y1, x1, y2, color, SOLID );
   gr_line( screen, x1, y2, x2, y2, color, SOLID );

   switch( style )
      {
      case HOLLOW:
	 break;
      case SOLID:
	 for ( y_val = y1 + 1; y_val < y2; ++y_val )
	    {
	    gr_line( screen, x1 + 1, y_val, x2 - 1, y_val, color, SOLID );
	    }
	 break;
      case GRID:
	 for ( y_val = y1 + 1; y_val < y2; ++y_val )
	    {
	    gr_line( screen, x1 + 1, y_val, x2 - 1, y_val, color, GRID );
	    }
	 break;
      case HATCH:
	 for ( y_val = y1 + 1; y_val < y2; ++y_val )
	    {
	    if ( ( y_val % 3 ) == 1 )
	       {
	       gr_line( screen, x1 + 1, y_val, x2 - 1, y_val, color, SOLID );
	       }
	    }
	 for ( x_val = x1 + 1; x_val < x2; ++x_val )
	    {
	    if ( ( x_val % 3 ) == 1 )
	       {
	       gr_line( screen, x_val, y1 + 1, x_val, y2 - 1, color, SOLID );
	       }
	    }
	 break;
      }

   }

/****************************************************************

   def_circle()    Draw circle using gr line routines

****************************************************************/

def_circle( screen, x, y, radius, color, style )
   int screen;
   int x, y, radius;
   int color, style;
   {
   register int y_val;
   int top_y, bot_y, prev_ty, prev_by;
   int x1, x2, prev_x1, prev_x2;
   int width, x_val;
   double angle, y_cor;

#ifdef  OLD_DEBUG
   sprintf( bw_ebuf, "circle: r %d, y %d, x %d   ",
      radius, y, x );
   bw_error( bw_ebuf );
#endif
#if CHECK_PARAMS
   if ( ( x < 0 ) || ( x > main_window.xmax ))
      {
      sprintf( bw_ebuf, "[pr:] def_circle(): x value is %d", x );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y < 0 ) || ( y > main_window.ymax ))
      {
      sprintf( bw_ebuf, "[pr:] def_circle(): y value is %d", y );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
#endif

   width = ( radius * gr_pysize * ACCURACY ) / ( gr_pxsize * ACCURACY );
   prev_by = prev_ty = bot_y = top_y = y;
   prev_x1 = x1 = x - width;
   prev_x2 = x2 = x + width;

   switch( style )
      {
      case HOLLOW:
	 break;
      case SOLID:
	 gr_line( screen, x1 + 1, top_y, x2 - 1, top_y, color, SOLID );
	 gr_line( screen, x1 + 1, bot_y, x2 - 1, bot_y, color, SOLID );
	 break;
      case GRID:
	 gr_line( screen, x1 + 1, top_y, x2 - 1, top_y, color, GRID );
	 gr_line( screen, x1 + 1, bot_y, x2 - 1, bot_y, color, GRID );
	 break;
      case HATCH:
	 if ( ( top_y % 2 ) == 1 )
	    {
	    gr_line( screen, x1 + 1, top_y, x2 - 1, top_y, color, SOLID );
	    gr_line( screen, x1 + 1, bot_y, x2 - 1, bot_y, color, SOLID );
	    }
	 break;
      }
   for ( y_val = 1; y_val <= ( radius - 1 ); ++y_val )
      {
      top_y = y + y_val;
      bot_y = y - y_val;

      y_cor = ( y_val * gr_pysize * ACCURACY ) / ( gr_pxsize * ACCURACY );
      angle = asin( ( y_cor ) / ((double) width ) );

#ifdef  OLD_DEBUG
      sprintf( bw_ebuf, "angle %.2lf deg   ",
	 angle * RAD_DEG );
      bw_error( bw_ebuf );
#endif

#ifdef  OLD_DEBUG
      sprintf( bw_ebuf, "tan( angle ) %.2lf deg   ",
	 tan( angle ) );
      bw_error( bw_ebuf );
#endif
      x_val = (int) ( y_cor / tan( angle ));
#ifdef  OLD_DEBUG
      sprintf( bw_ebuf, "angle %.2lf deg, y %d, x %d   ",
	 angle * RAD_DEG, y_val, x_val );
      bw_error( bw_ebuf );
#endif

      x1 = x - x_val;
      x2 = x + x_val;

      gr_line( screen, x2, top_y, prev_x2, prev_ty, color, SOLID );
      gr_line( screen, x1, top_y, prev_x1, prev_ty, color, SOLID );
      gr_line( screen, x2, bot_y, prev_x2, prev_by, color, SOLID );
      gr_line( screen, x1, bot_y, prev_x1, prev_by, color, SOLID );

      switch( style )
	 {
	 case HOLLOW:
	    break;
	 case SOLID:
	    gr_line( screen, x1 + 1, top_y, x2 - 1, top_y, color, SOLID );
	    gr_line( screen, x1 + 1, bot_y, x2 - 1, bot_y, color, SOLID );
	    break;
	 case GRID:
	    gr_line( screen, x1 + 1, top_y, x2 - 1, top_y, color, GRID );
	    gr_line( screen, x1 + 1, bot_y, x2 - 1, bot_y, color, GRID );
	    break;
	 case HATCH:
	    if ( ( top_y % 2 ) == 1 )
	       {
	       gr_line( screen, x1 + 1, top_y, x2 - 1, top_y, color, SOLID );
	       gr_line( screen, x1 + 1, bot_y, x2 - 1, bot_y, color, SOLID );
	       }
	    break;
	 }

      prev_x1 = x1;
      prev_x2 = x2;
      prev_ty = top_y;
      prev_by = bot_y;

      }

   x2 = x1 = x;
   top_y = y + radius;
   bot_y = y - radius;

   gr_line( screen, x2, top_y, prev_x2, prev_ty, color, SOLID );
   gr_line( screen, x1, top_y, prev_x1, prev_ty, color, SOLID );
   gr_line( screen, x2, bot_y, prev_x2, prev_by, color, SOLID );
   gr_line( screen, x1, bot_y, prev_x1, prev_by, color, SOLID );
   }

#ifdef  STANDALONE

bw_error( s )
   char *s;
   {
   gr_text( GR_PRIMARY, 10, 10, s, WHITE, BLACK );
   kb_rx();
   }

#endif

