/***************************************************************

	sfsm_om.c	Orbital Modeling routines for
			Space Flight Simulator,
			Modeling Module

			Copyright (c) 1991, Ted A. Campbell

			Bywater Software
			P. O. Box 4023 
			Duke Station 
			Durham, NC  27706

			email: tcamp@hercules.acpub.duke.edu

	Copyright and Permissions Information:

	All U.S. and international copyrights are claimed by the
	author. The author grants permission to use this code
	and software based on it under the following conditions:
	(a) in general, the code and software based upon it may be 
	used by individuals and by non-profit organizations; (b) it
	may also be utilized by governmental agencies in any country,
	with the exception of military agencies; (c) the code and/or
	software based upon it may not be sold for a profit without
	an explicit and specific permission from the author, except
	that a minimal fee may be charged for media on which it is
	copied, and for copying and handling; (d) the code must be 
	distributed in the form in which it has been released by the
	author; and (e) the code and software based upon it may not 
	be used for illegal activities. 

***************************************************************/

#include "stdio.h"
#include "sfsm.h"

#ifdef  __STDC__
#include "malloc.h"
#else
extern   char * malloc();
#endif

#ifndef   __STDC__
#define   time_t    long
#define   size_t   int
#endif

#define OM_TITLES       7
#define SP_TITLES       9
#define AP_TITLES       6
#define SYS_TITLES      4

extern   struct as_focus * pr_getfd();

struct uiwindow *set_window;
int      sp_x1, sp_y1, sp_x2, sp_y2;
int      sp_ready;

static int spmem;

static char *om_titles[ OM_TITLES ];
static char *sp_titles[ SP_TITLES ];
static char *ap_titles[ AP_TITLES ];
static char *sys_titles[ SYS_TITLES ];
static char t_buffer[ BW_EBUFSIZE ];
static char set_buffer[ 64 ];
static struct sfs_orbit *ap_oldorbit = NULL;

extern int null_poll();

sfs_om( sorbit_array, n_orbits )
   struct sfs_orbit **sorbit_array;
   int n_orbits;
   {
   int orbit, keep_on, item, c;
   long l;
   static int x_elements = VI_ORBOUTLINE | VI_GRIDFRONT | VI_SURFRONT;

   ui_fbox( main_window->u_x1, main_window->u_y1,
      main_window->u_x2, main_window->u_y2, BLACK, SOLID );

   if ( sp_ready == 0 )
      {
      sp_init();
      sp_ready = 1;
      }

   tl_window( OM_TITLE );
   tr_window( SOP_TITLE );
   bl_window( SFSM_PEBL );
   br_window( AP_TITLE );
   w1.is_drawn = FALSE;
   w2.is_drawn = FALSE;
   ap_oldorbit = NULL;
   sfsm_sp( FALSE, sorbit_array, sfs_curorbit, n_orbits );

   keep_on = TRUE;
   while ( keep_on == TRUE )
      {

#ifdef  BLOCKEDOUT
      pe_calc( sorbit_array, n_orbits );
      pe_draw( w3.window, sorbit_array[ sfs_curorbit ]->aorbit->focus,
	 sorbit_array, n_orbits, NULL, NULL, (long) 0, x_elements );
#endif

      bw_message( MES_OM );

      pr_stripcr( sfs_prname );
      sprintf( om_titles[ 0 ], OM_SPT, sfs_prname );
      sprintf( om_titles[ 1 ], OM_SON, sfs_curorbit + 1 );
      strcpy( om_titles[ 2 ],  OM_SOP );
      sprintf( om_titles[ 3 ], OM_SIP, sfs_insertion );
      strcpy( om_titles[ 4 ],  OM_SSP );
      strcpy( om_titles[ 5 ],  OM_VCO );
      strcpy( om_titles[ 6 ],  OM_EXIT );

      item = ui_list( MENU_TITLED, w1.window->x1, w1.window->y1,
	 w1.window->x2, w1.window->y2,
	 OM_TTL,
	 OM_TITLES, om_titles, cl_mfore, cl_mback,
	 cl_marker, &w1 );

      switch( item )
         {
	 case 0:
	    get_string( SPT_TITLE, SPT_LEGEND, SPT_PROMPT,
	       sfs_prname );
	    sprintf( om_titles[ 0 ], OM_SPT, sfs_prname );
	    w1.is_drawn = FALSE;
            break;
	 case 1:
	    sprintf( sfs_tbuf, SON_LEGEND, ORBITS );
	    sfs_curorbit = get_int( SON_TITLE,
	       sfs_tbuf, SON_PROMPT,
	       sfs_curorbit, 1, ORBITS ) - 1;
	    if ( sorbit_array[ sfs_curorbit ] == NULL )
               {
	       pr_default( sorbit_array, sfs_curorbit, n_orbits, DEFAULTFD );
               }
	    sprintf( om_titles[ 1 ],
	       OM_SON, sfs_curorbit + 1 );
            sfsm_sp( FALSE, sorbit_array, sfs_curorbit, n_orbits );
	    w1.is_drawn = FALSE;
	    w2.is_drawn = FALSE;
	    break;
	 case 2:
	    w2.is_drawn = FALSE;
	    sfsm_sp( TRUE, sorbit_array, sfs_curorbit, n_orbits );
            break;
	 case 3:
	    sfs_insertion = get_long( SIP_TITLE,
	       SIP_LEGEND, SIP_PROMPT,
               (long) sfs_insertion, 
	       (long) 0,
	       (long) sorbit_array[ sfs_curorbit ]->aorbit->period );
	    sprintf( om_titles[ 3 ],
	       OM_SIP, sfs_insertion );
	    w1.is_drawn = FALSE;
	    break;
	 case 4:
	    w2.is_drawn = FALSE;
            sfs_sysp();
            break;
	 case 5:
	    pe_calc( sorbit_array, n_orbits, null_poll );
	    pe_draw( w3.window, sorbit_array[ sfs_curorbit ]->aorbit->focus,
	       sorbit_array, n_orbits, NULL, NULL, (long) 0, x_elements );
            break;
	 case TK_EXIT:
	 case 6:
	    keep_on = FALSE;
	    c = get_yn( OM_SF );
	    if ( c == TRUE )
               {
	       get_string( SF_TITLE,
		  SF_LEGEND,
		  SF_PROMPT,
                  sfs_program );
	       pr_save( sfs_program, sorbit_array, n_orbits, sfs_prname );
               }
	    w1.is_drawn = FALSE;
	    break;
         }
      }
   }

sp_init()
   {
   int height, length, midx, midy, c;

   midx = main_window->u_x1 + ( ( main_window->u_x2 - main_window->u_x1 ) / 2 );
   midy = main_window->u_y1 + ( ( main_window->u_y2 - main_window->u_y1 ) / 2 );

   length = ( main_window->u_x2 * 3 ) / 5;
   sp_x1  = midx - length / 2;
   sp_x2  = midx + length / 2;
   height = main_window->u_y2 / 2;
   sp_y1  = midy - height / 2;
   sp_y2  = midy + height / 2;

   for ( c = 0; c < OM_TITLES; ++c )
      {
      if ( ( om_titles[ c ] = malloc( 64 ) ) == NULL )
         {
	 bw_error( MM_MEMERR );
         }
      }
   for ( c = 0; c < SP_TITLES; ++c )
      {
      if ( ( sp_titles[ c ] = malloc( 64 ) ) == NULL)
         {
	 bw_error( OPM_MEMERR );
         }
      }
   for ( c = 0; c < AP_TITLES; ++c )
      {
      if ( ( ap_titles[ c ] = malloc( 64 ) ) == NULL)
         {
	 bw_error( APM_MEMERR );
         }
      }
   for ( c = 0; c < SYS_TITLES; ++c )
      {
      if ( ( sys_titles[ c ] = malloc( 64 ) ) == NULL)
         {
	 bw_error( SPM_MEMERR );
         }
      }

   }

sfs_sysp()
   {
   int c, y, item;

   /* set up titles for menu */

   sprintf( sys_titles[ 0 ], SSP_TF,
      (long) sfs_tfactor );
   sprintf( sys_titles[ 1 ], SSP_UI,
      (long) sfs_inc );
   sprintf( sys_titles[ 2 ], SSP_TPL,
      vpt_level );
   sprintf( sys_titles[ 3 ], SSP_EXIT );
   w2.is_drawn = FALSE;

   bw_message( MES_SSP );

   y = TRUE;
   while( y == TRUE )
      {

      item = ui_list( MENU_TITLED, w2.window->x1, w2.window->y1,
	 w2.window->x2, w2.window->y2,
	 SSP_TITLE,
	 SYS_TITLES, sys_titles,
	 cl_mfore, cl_mback,
	 cl_marker, &w2 );

      /* Process the argument */

      switch( item )
         {
         case 0:
	    sfs_tfactor = get_long( TF_TITLE,
	       TF_LEGEND,
	       TF_PROMPT,
               (long) sfs_tfactor,
               (long) 1, (long) 0 );
	    sprintf( sys_titles[ 0 ], SSP_TF,
	       (long) sfs_tfactor );
	    w2.is_drawn = FALSE;
            break;
         case 1:
	    sfs_inc = get_long( UI_TITLE,
	       UI_LEGEND,
	       UI_PROMPT,
	       (long) sfs_inc,
	       (long) 1, (long) 0 );
	    sprintf( sys_titles[ 1 ], SSP_UI,
	       (long) sfs_inc );
	    w2.is_drawn = FALSE;
            break;
         case 2:
	    vpt_level = get_int( TPL_TITLE,
	       TPL_LEGEND,
	       TPL_PROMPT,
               vpt_level, 1, 2 );
	    sprintf( sys_titles[ 2 ], SSP_TPL,
	       vpt_level );
	    w2.is_drawn = FALSE;
            break;
	 case TK_EXIT:
         case 3:
	    y = FALSE;
            break;
         }

      }
   }

/***************************************************************

   sfsm_sp()    set orbital parameters

***************************************************************/

sfsm_sp( execute, sorbit_array, orbit_n, n_orbits )
   int execute;
   struct sfs_orbit **sorbit_array;
   int orbit_n, n_orbits;
   {
   register int c;
   int item, y;
   static char x_buffer[ BW_EBUFSIZE ];
   static struct menu_box fs_box;

   if ( execute == TRUE )
      {
      bw_message( MES_SP );
      }

   y = TRUE;
   while( y == TRUE )
      {

      if ( sorbit_array[ orbit_n ] != ap_oldorbit )
	 {
	 or_init( sorbit_array[ sfs_curorbit ]->aorbit,
	    sorbit_array[ sfs_curorbit ]->aorbit->periapsis,
	    sorbit_array[ sfs_curorbit ]->aorbit->apoapsis );

	 sfs_ap( sorbit_array, sfs_curorbit );
	 ap_oldorbit = sorbit_array[ orbit_n ];

	 /* set up titles for menu */

	 sprintf( sp_titles[ 0 ], SOP_SOF,
	    sorbit_array[ sfs_curorbit ]->aorbit->focus->fdfile );
	 sprintf( sp_titles[ 1 ], SOP_SN,
	    sorbit_array[ sfs_curorbit ]->name );
	 sprintf( sp_titles[ 2 ], SOP_SDF,
	    sorbit_array[ sfs_curorbit ]->surfile );
	 sprintf( sp_titles[ 3 ], SOP_PRI,
	    sorbit_array[ sfs_curorbit ]->aorbit->periapsis );
	 sprintf( sp_titles[ 4 ], SOP_APO,
	    sorbit_array[ sfs_curorbit ]->aorbit->apoapsis );
	 sprintf( sp_titles[ 5 ], SOP_INC,
	    sorbit_array[ sfs_curorbit ]->aorbit->inclination );
	 sprintf( sp_titles[ 6 ], SOP_ARP,
	    sorbit_array[ sfs_curorbit ]->aorbit->arg_per );
	 sprintf( sp_titles[ 7 ], SOP_LAN,
	    sorbit_array[ sfs_curorbit ]->aorbit->lon_an );
	 strcpy( sp_titles[ 8 ], SOP_EXIT );

	 tr_window( x_buffer );
	 w2.is_drawn = FALSE;
	 }

      sprintf( x_buffer, SOP_TITLE2, sfs_curorbit + 1 );

      if ( execute == FALSE )
	 {
	 uil_draw( MENU_TITLED, w2.window->x1, w2.window->y1,
	    w2.window->x2, w2.window->y2,
	    x_buffer, SP_TITLES, sp_titles,
	    cl_mfore, cl_mback, cl_marker, &w2 );

	 return TRUE;
	 }

      item = ui_list( MENU_TITLED, w2.window->x1, w2.window->y1,
	 w2.window->x2, w2.window->y2,
	 x_buffer, SP_TITLES, sp_titles,
	 cl_mfore, cl_mback, cl_marker, &w2 );

      /* Process the argument */

      switch( item )
	 {
	 case 0:
	    sfsm_getspw();
	    fs_box.is_drawn = FALSE;
	    item = ui_ftext( sp_x1, sp_y1, sp_x2, sp_y2, "*.fd",
	       SOF_TITLE,
	       &fs_box, d_titles, d_entries, MAX_ENTRIES,
	       cl_mback, cl_mfore, cl_marker );
	    fntolc( d_titles[ item ] );

	    sorbit_array[ sfs_curorbit ]->aorbit->focus
	       = pr_getfd( sorbit_array, sfs_curorbit, n_orbits,
	       d_titles[ item ]  );
	    sprintf( sp_titles[ 0 ], SOP_SOF,
	       sorbit_array[ sfs_curorbit ]->aorbit->focus->fdfile );
	    sfsm_putspw();
	    w2.is_drawn = FALSE;
	    break;

	 case 1:
	    sprintf( t_buffer, SN_TITLE, sfs_curorbit + 1);
	    get_string( set_buffer, SN_LEGEND,
	       SN_PROMPT, sfs_tbuf );
	    strcpy( sorbit_array[ sfs_curorbit ]->name, sfs_tbuf );
	    sprintf( sp_titles[ 1 ], SOP_SN,
	       sorbit_array[ sfs_curorbit ]->name );
	    w2.is_drawn = FALSE;
            break;

	 case 2:
	    sfsm_getspw();
	    fs_box.is_drawn = FALSE;
	    item = ui_ftext( sp_x1, sp_y1, sp_x2, sp_y2, "*.spd",
	       SDF_TITLE,
	       &fs_box, d_titles, d_entries, MAX_ENTRIES,
	       cl_mback, cl_mfore, cl_marker );
	    fntolc( d_titles[ item ] );
	    strncpy( sorbit_array[ sfs_curorbit ]->surfile,
	       d_titles[ item ], SFS_FILESIZE - 2 );
	    sprintf( sp_titles[ 2 ], SOP_SDF,
	       sorbit_array[ sfs_curorbit ]->surfile );
	    sfsm_putspw();
	    w2.is_drawn = FALSE;
	    break;

	 case 3:
	    sprintf( t_buffer, PRI_TITLE, sfs_curorbit + 1 );
	    sprintf( sfs_tbuf, PRI_LEGEND,
	       (double) sorbit_array[ sfs_curorbit ]->aorbit->apoapsis );
	    sorbit_array[ sfs_curorbit ]->aorbit->periapsis = get_double( t_buffer,
	       sfs_tbuf,
	       PRI_PROMPT,
	       (double) sorbit_array[ sfs_curorbit ]->aorbit->periapsis,
	       (double) 0.0,
	       (double) sorbit_array[ sfs_curorbit ]->aorbit->apoapsis );
	    sfsm_clpe();
	    sprintf( sp_titles[ 3 ], SOP_PRI,
	       sorbit_array[ sfs_curorbit ]->aorbit->periapsis );
	    w2.is_drawn = FALSE;
            break;

	 case 4:
	    sprintf( t_buffer, APO_TITLE, sfs_curorbit + 1 );
	    sprintf( sfs_tbuf, APO_LEGEND,
	       (double) sorbit_array[ sfs_curorbit ]->aorbit->periapsis );
	    sorbit_array[ sfs_curorbit ]->aorbit->apoapsis = get_double( t_buffer,
	       sfs_tbuf,
	       APO_PROMPT,
	       (double) sorbit_array[ sfs_curorbit ]->aorbit->apoapsis,
	       (double) sorbit_array[ sfs_curorbit ]->aorbit->periapsis,
	       (double) 0.0 );
	    sfsm_clpe();
	    sprintf( sp_titles[ 4 ], SOP_APO,
	       sorbit_array[ sfs_curorbit ]->aorbit->apoapsis );
	    w2.is_drawn = FALSE;
            break;
	 case 5:
	    sprintf( t_buffer, INC_TITLE, sfs_curorbit + 1);
	    sprintf( sfs_tbuf, INC_LEGEND,
	       (double) OR_INC_MIN, (double) OR_INC_MAX );
	    sorbit_array[ sfs_curorbit ]->aorbit->inclination = get_double( t_buffer,
	       sfs_tbuf,
	       INC_PROMPT,
	       sorbit_array[ sfs_curorbit ]->aorbit->inclination,
	       (double) OR_INC_MIN, (double) OR_INC_MAX );
	    sfsm_clpe();

            /*** inclination cannot be 90 or -90 ***/

            if ( sorbit_array[ sfs_curorbit ]->aorbit->inclination == 90.0 )
               {
               sorbit_array[ sfs_curorbit ]->aorbit->inclination = 89.9999;
               }
            if ( sorbit_array[ sfs_curorbit ]->aorbit->inclination == -90.0 )
               {
               sorbit_array[ sfs_curorbit ]->aorbit->inclination = -89.9999;
               }

	    sprintf( sp_titles[ 5 ], SOP_INC,
	       sorbit_array[ sfs_curorbit ]->aorbit->inclination );
	    w2.is_drawn = FALSE;
            break;
	 case 6:
	    sprintf( t_buffer, ARP_TITLE, sfs_curorbit + 1 );
	    sprintf( sfs_tbuf, ARP_LEGEND,
	       (double) OR_ARP_MIN, (double) OR_ARP_MAX );
	    sorbit_array[ sfs_curorbit ]->aorbit->arg_per = get_double( t_buffer,
	       sfs_tbuf,
	       ARP_PROMPT,
	       sorbit_array[ sfs_curorbit ]->aorbit->arg_per,
	       (double) OR_ARP_MIN, (double) OR_ARP_MAX );
	    sfsm_clpe();
	    sprintf( sp_titles[ 6 ], SOP_ARP,
	       sorbit_array[ sfs_curorbit ]->aorbit->arg_per );
	    w2.is_drawn = FALSE;
            break;
	 case 7:
	    sprintf( t_buffer, LAN_TITLE, sfs_curorbit +1 );
	    sprintf( sfs_tbuf, LAN_LEGEND,
	       OR_LAN_MIN, OR_LAN_MAX );
	    sorbit_array[ sfs_curorbit ]->aorbit->lon_an = get_double( t_buffer,
	       sfs_tbuf,
	       LAN_PROMPT,
	       sorbit_array[ sfs_curorbit ]->aorbit->lon_an,
	       (double) OR_LAN_MIN, (double) OR_LAN_MAX );
	    sfsm_clpe();
	    sprintf( sp_titles[ 7 ], SOP_LAN,
	       sorbit_array[ sfs_curorbit ]->aorbit->lon_an );
	    w2.is_drawn = FALSE;
            break;
	 case TK_EXIT:
	 case 8:
            y = 0;
            break;
         }

      if ( w2.is_drawn == FALSE )
	 {
	 or_init( sorbit_array[ sfs_curorbit ]->aorbit,
	    sorbit_array[ sfs_curorbit ]->aorbit->periapsis,
	    sorbit_array[ sfs_curorbit ]->aorbit->apoapsis );
	 sfs_ap( sorbit_array, sfs_curorbit );
	 }
      }
   }

sfs_ap( sorbit_array, orbit_n )
   struct sfs_orbit **sorbit_array;
   int orbit_n;
   {
   static char x_buffer[ BW_EBUFSIZE ];

   /* set up titles for menu */

   sprintf( ap_titles[ 0 ], AP_MAS,
      sorbit_array[ orbit_n ]->aorbit->focus->mass );
   sprintf( ap_titles[ 1 ], AP_RAD,
      sorbit_array[ orbit_n ]->aorbit->focus->radius );
   sprintf( ap_titles[ 2 ], AP_SID,
      sorbit_array[ orbit_n ]->aorbit->focus->sid_day  );
   sprintf( ap_titles[ 3 ], AP_SMI,
      sorbit_array[ sfs_curorbit ]->aorbit->semiminor );
   sprintf( ap_titles[ 4 ], AP_ECC,
      sorbit_array[ sfs_curorbit ]->aorbit->eccentricity );
   sprintf( ap_titles[ 5 ], AP_PER,
      sorbit_array[ sfs_curorbit ]->aorbit->period,
      sorbit_array[ sfs_curorbit ]->aorbit->period / (double) 86400 );

   sprintf( x_buffer, AP_TITLE2, orbit_n + 1 );

   br_window( t_buffer );
   w4.is_drawn = FALSE;
   uil_draw( MENU_TITLED, w4.window->x1, w4.window->y1,
      w4.window->x2, w4.window->y2, 
      x_buffer, AP_TITLES, ap_titles,
      cl_mfore, cl_mback, cl_marker, &w4 );


   }

get_yn( text )
   char *text;
   {
   register int c;
   static struct uiwindow *yn_window;

   sfsm_getspw();

   c = ui_yn( sp_x1, sp_y1, sp_x2, sp_y2,
      cl_mback, cl_mfore, cl_marker, TRUE,
      text, GETYN_NO, GETYN_YES, &yn_window );

   sfsm_putspw();
   return c;
   }

get_string( title, text, prompt, buffer )
   char *title, *text, *prompt, *buffer;
   {
   sfsm_getspw();

   get_window();
   ui_dial( sp_x1, sp_y1, sp_x2, sp_y2, cl_mback, cl_mfore, cl_marker,
      1, title,
      text, prompt, set_buffer, &set_window );
   if ( set_buffer[ 0 ] != 0 )
      {
      strcpy( buffer, set_buffer );
      }

   sfsm_putspw();
   }

get_int( title, text, prompt, oldvalue, minimum, maximum )
   char *title, *text, *prompt;
   int oldvalue, minimum, maximum;
   {
   char *_text;
   int y;
   static int n;

   sfsm_getspw();

   _text = text;
   y = 1;
   while( y == 1 )
      {
      get_window();
      ui_dial( sp_x1, sp_y1, sp_x2, sp_y2, cl_mback, cl_mfore,
         cl_marker, 1, title,
         _text, prompt, set_buffer, &set_window );
      if ( set_buffer[ 0 ] == 0 )
	 {
	 sfsm_putspw();
         return oldvalue;
         }

      sscanf( set_buffer, "%d",  &n );
      y = 0;
      if ( n < minimum )
         {
	 sprintf( sfs_tbuf, "%s %ld; %s.", MINERR, minimum - 1, PTA );
         _text = sfs_tbuf;
         y = 1;
         }
      if ( ( maximum != 0 ) && ( n > maximum ))
         {
         y = 1;
	 sprintf( sfs_tbuf, "%s %ld; %s.", MAXERR, maximum + 1, PTA );
         _text = sfs_tbuf;
         }
      }

   sfsm_putspw();
   return n;
   }

long
get_long( title, text, prompt, oldvalue, minimum, maximum )
   char *title, *text, *prompt;
   long oldvalue, minimum, maximum;
   {
   char *_text;
   int y;
   static long n;

   sfsm_getspw();                       /* save the set param window */

   _text = text;
   y = TRUE;
   while( y == TRUE )
      {
      get_window();
      ui_dial( sp_x1, sp_y1, sp_x2, sp_y2, cl_mback, cl_mfore,
         cl_marker, 1, title,
         _text, prompt, set_buffer, &set_window );

      if ( set_buffer[ 0 ] == 0 )
	 {
	 sfsm_putspw();
         return oldvalue;
         }

      sscanf( set_buffer, "%ld",  &n );
      y = 0;
      if ( n < minimum )
         {
	 sprintf( sfs_tbuf, "%s %ld; %s.", MINERR, minimum - 1, PTA );
         _text = sfs_tbuf;
         y = 1;
         }
      if ( ( maximum != 0 ) && ( n > maximum ))
         {
         y = 1;
	 sprintf( sfs_tbuf, "%s %ld; %s.", MAXERR, maximum + 1, PTA );
         _text = sfs_tbuf;
         }
      }

   sfsm_putspw();
   return n;
   }

double
get_double( title, text, prompt, oldvalue, minimum, maximum )
   char *title, *text, *prompt;
   double oldvalue, minimum, maximum;
   {
   char *_text;
   int y;
   static double n;

   sfsm_getspw();

   _text = text;
   y = 1;
   while( y == 1 )
      {
      get_window();
      ui_dial( sp_x1, sp_y1, sp_x2, sp_y2, cl_mback, cl_mfore,
         cl_marker, 1, title,
         _text, prompt, set_buffer, &set_window );

      if ( set_buffer[ 0 ] == 0 )
	 {
	 sfsm_putspw();
         return oldvalue;
         }

      sscanf( set_buffer, "%lf",  &n );
      y = 0;
      if ( n < minimum )
         {
	 sprintf( sfs_tbuf, "%s %.2lf; %s.", MINERR, minimum - 1, PTA );
         _text = sfs_tbuf;
         y = 1;
         }
      if ( ( maximum != 0.0 ) && ( n > maximum ))
         {
         y = 1;
	 sprintf( sfs_tbuf, "%s %.2lf; %s.", MAXERR, maximum + 1, PTA );
         _text = sfs_tbuf;
         }
      }

   sfsm_putspw();
   return n;
   }

/***************************************************************

   get_window()         This function is called prior to ui_dial
			in get_xxx() functions in order to set
			the title bar background color to cl_marker.
			Since ui_dial does not allow specification
			of a title bar color, this will set it, and
			once set it will remian the same.

***************************************************************/

get_window()
   {
   if ( set_window == NULL )
      {
      set_window = ui_window( sp_x1, sp_y1, sp_x2, sp_y2, TRUE, cl_marker,
	 BLACK, "", TRUE, cl_mfore, TRUE, BLACK, cl_mback, SOLID, FALSE );
      }
   }

sfsm_clpe()
   {
   if ( w3.window == NULL )
      {
      return BW_ERROR;
      }
   ui_fbox( w3.window->u_x1, w3.window->u_y1,
      w3.window->u_x2, w3.window->u_y2, BLACK, SOLID );
   ui_fbox( w3.window->u_x1, w3.window->u_y1,
      w3.window->u_x2, w3.window->u_y2, /* cl_mback */ WHITE, GRID );
   ui_rewindow( w3.window, SFSM_PEBL );

   }

sfsm_getspw()
   {
   size_t x_size;

   gr_imsave( GR_PRIMARY, TRUE, sp_x1, sp_y1, sp_x2, sp_y2, &spmem );
   }

sfsm_putspw()
   {
   gr_imsave( GR_PRIMARY, FALSE, sp_x1, sp_y1, sp_x2, sp_y2, &spmem );
   gr_imfree( spmem );
   }

null_poll()
   {
   }
