/***************************************************************

	map.c           map utility for Space Flight Simulator

			Copyright (c) 1991, Ted A. Campbell

			Bywater Software
			P. O. Box 4023 
			Duke Station 
			Durham, NC  27706

			email: tcamp@hercules.acpub.duke.edu

	Copyright and Permissions Information:

	All U.S. and international copyrights are claimed by the
	author. The author grants permission to use this code
	and software based on it under the following conditions:
	(a) in general, the code and software based upon it may be 
	used by individuals and by non-profit organizations; (b) it
	may also be utilized by governmental agencies in any country,
	with the exception of military agencies; (c) the code and/or
	software based upon it may not be sold for a profit without
	an explicit and specific permission from the author, except
	that a minimal fee may be charged for media on which it is
	copied, and for copying and handling; (d) the code must be 
	distributed in the form in which it has been released by the
	author; and (e) the code and software based upon it may not 
	be used for illegal activities. 

***************************************************************/

#include "stdio.h"
#include "time.h"
#include "signal.h"
#include "ctype.h"
#include "math.h"

#include "bw.h"
#include "gr.h"
#include "kb.h"
#include "dr.h"
#include "ui.h"
#include "as.h"

#include "map.h"

#ifdef __STDC__
#include "malloc.h"
#else
extern char * malloc();
#endif

#ifndef   __STDC__
#define   time_t    long
#define   size_t   int
#endif

extern  int map_sigint();

/***    Colors */

int   cl_mback;      /* Main background color   */
int   cl_mfore;      /* Main foreground color   */
int   cl_marker;      /* Color for prompt markers   */
int   cl_grid;      /* Color for planetary grid   */
int   cl_surface;      /* Color for planetary surface   */

struct spj_pt spd_start, spd_end;       /* surface data pointers */
struct spj_pt new_start, new_end;       /* new data pointers */
struct spj_pt *lastspj;       		/* pointer to last new spj point */

struct uiwindow *main_window;           /* structure for main window    */
struct uiwindow *map_window;            /* structure for map window    */
char bw_ebuf[ BW_EBUFSIZE ];
char map_tbuf[ BW_EBUFSIZE ];
char map_datafile[ 128 ];
int dr_xsize;            /* Size of ground track map, x */
int dr_ysize;            /* Size of ground track map, y */
int dr_mapbase;          /* Bottom of dr map */
int dr_mapedge;          /* Left edge of groundtrack map */
int mesg_y;
int map_curpos = 0;	 /* current position (default: full) */
double map_defalt = 0.0; /* default altitude */
FILE *outfile;		 /* file for output */

int map_pos[ 10 ][ 4 ] = {
	{ -180,  -90,  180,   90 },         /* 0 - full          */
	{ -180,  -90,    0,    0 },         /* 1 - lower left    */
	{  -90,  -90,   90,    0 },         /* 2 - lower middle  */
	{    0,  -90,  180,    0 },         /* 3 - lower right   */
	{ -180,  -45,    0,   45 },         /* 4 - center left   */
	{  -90,  -45,   90,   45 },         /* 5 - center middle */
	{    0,  -45,  180,   45 },         /* 6 - center right  */
	{ -180,    0,    0,   90 },         /* 7 - upper left    */
	{  -90,    0,   90,   90 },         /* 8 - upper middle  */
	{    0,    0,  180,   90 } };       /* 9 - upper right   */

static struct uiwindow *dial_window;
static int dial_x1 = 0, dial_x2, dial_y1, dial_y2;
static int midx, midy, length, height;
static char buffer[ BW_EBUFSIZE ];

char *mm_titles[ MM_TITLES ] = {
	 MM_POSITION,
	 MM_ENTITLE,
         MM_SEQUENCE,
         MM_ALTITUDE,
	 MM_EXIT };

extern int map_sigint();

/****************************************************************

	main()

****************************************************************/

main( argc, argv )
   int argc;
   char **argv;
   {
   int offs_y, offs_x, m_x1, m_y1, m_x2, m_y2;

   /***    set a signal handler so that a SIGINT           */
   /***    (control-C under DOS, DEL under Unix) will      */
   /***    exit gracefully                                 */

   signal( SIGINT, map_sigint );

   /***    process argument 1 for data file */

   if ( argc > 1 )
      {
      strcpy( map_datafile, argv[ 1 ] );
      }
   else
      {
      strcpy( map_datafile, MAP_DEFAULTDATA );
      }

   /* Initialize the user interface */

   ui_init();

   if ( ui_ready != TRUE )
      {
      gr_deinit();
      fprintf( stderr, "Graphics system not initialized.\n" );
      exit( -1 );
      }

   if ( gr_ismouse != TRUE )
      {
      gr_deinit();
      fprintf( stderr,
	 "The map utility requires the use of a pointer device (\"mouse\").\n" );
      exit( -1 );
      }

   map_colors();         /* Set colors       */

   /* reset icon colors */

   if ( gr_colors > 2 )
      {
      ui_seticons( cl_mback, cl_marker );
      }

   /* Initialize variable-precision trig */

   vpt_init();

   map_font( 0 );

   /* Clear the screen */

   gr_cls( GR_PRIMARY );

   /* Draw main screen */

   sprintf( map_tbuf, MAP_TITLE, MAP_VERSION );
   main_window = ui_window( 0, 0, ui_grwind->xmax, ui_grwind->ymax,
      TRUE, cl_marker, cl_mfore,
      map_tbuf, TRUE, cl_mback,
      FALSE, BLACK, BLACK, SOLID );

   /* delimit message area */

   mesg_y = 5 + ui_grwind->fysize + 3;
   gr_line( GR_PRIMARY, main_window->u_x1, mesg_y,
      main_window->u_x2, mesg_y, WHITE, SOLID );

   /* read spd data */

   spj_readspd( map_datafile, &spd_start, &spd_end );

   /* initialize new data pointers */

   new_start.next = &new_end;
   lastspj = &new_start;

   /* open datafile for append */

   if ( ( outfile = fopen( map_datafile, "a" ) ) == NULL )
      {
      sprintf( bw_ebuf, "Cannot open file %s for append.",
         map_datafile );
      }

   /* set up display */

   offs_x = gr_strlen( "  000: " );
   offs_y = ui_grwind->fysize + 4;
   m_x1 = main_window->u_x1 + offs_x;
   m_y1 = mesg_y + offs_y;
   m_x2 = main_window->u_x2 - offs_x;
   m_y2 = main_window->u_y2 - offs_y;
   map_window = ui_window( m_x1, m_y1, m_x2, m_y2,
      FALSE, cl_marker, cl_mfore,
      map_tbuf, FALSE, cl_mback,
      FALSE, BLACK, BLACK, SOLID );

/*    dr_draw( map_window, TRUE, &spd_start, &spd_end, 0 ); */
   map_draw( map_curpos );

   bw_message( MAP_HELP );
   while( TRUE )
      {
      map_poll();
      }
   }

/****************************************************************

	map_poll()

****************************************************************/

map_poll()
   {
   register int c;
   static int x, y, b;

   /* Check for keyboard input */

   if ( kb_rxstat() == TRUE )
      {
      c = kb_rx();
      if ( c == 0x1b )
         {
         map_esc();
         }
      }

   /* Check for mouse input */

   if ( gr_ismouse == TRUE )
      {
      if ( gr_mouse( SAMPLE, &x, &y, &b ) == TRUE )
	 {

	 gr_mouse( WAIT, &x, &y, &b );           /* click down */
	 gr_mouse( WAIT, &x, &y, &b );           /* click up */

#ifdef  OLD_DEBUG
	 sprintf( sfsx_tbuf, "Mouse hit: %d %d", x, y );
	 bw_debug( sfsx_tbuf );
#endif

	 /* Is the release within bounds ? */

	 if ( uil_bounds( x, y, main_window->tbar_x1, main_window->tbar_y1,
	    main_window->tbar_x2, main_window->tbar_y2) == TRUE )
	    {
	    map_esc();
	    }

	 }
      }
   }

/****************************************************************

	map_esc()

****************************************************************/

map_esc()
   {
   static struct menu_box esc_box;
   static int x_x1 = 0, x_y1 = 0, x_x2 = 0, x_y2 = 0, item;
   static int x_ebuf;
   int xcenter, ycenter;
   
   ui_push();                   /* save current ui screen */
   ui_setscreen( GR_PRIMARY );  /* set primary screen */
   if ( x_x1 == 0 )
      {
      xcenter = main_window->u_x1 +
         (( main_window->u_x2 - main_window->u_x1 ) / 2 );
      ycenter = main_window->u_y1 +
         (( main_window->u_y2 - main_window->u_y1 ) / 2 );
      x_x1 = (int) xcenter - ( ui_grwind->xmax / 4 );
      x_y1 = (int) ycenter - ( ui_grwind->fysize * 7 );
      x_x2 = (int) xcenter + ( ui_grwind->xmax / 4 );
      x_y2 = (int) ycenter + ( ui_grwind->fysize * 5 );
      }

   gr_imsave( GR_PRIMARY, TRUE, x_x1, x_y1, x_x2, x_y2, &x_ebuf );

   bw_message( MAP_MM );

   esc_box.is_drawn = FALSE;
   item = ui_list( MENU_SLIDERS, x_x1, x_y1, x_x2, x_y2,
      MM_TITLE, MM_TITLES,
      mm_titles, cl_mfore, cl_mback, cl_marker,
      &esc_box );

   gr_imsave( GR_PRIMARY, FALSE, x_x1, x_y1, x_x2, x_y2, &x_ebuf );
   gr_imfree( x_ebuf );
   bw_message( MAP_HELP );

   switch( item )
      {
      case 0:
         map_select();
         bw_message( MAP_HELP );
	 break;
      case 1:
	 map_entitle();
	 bw_message( MAP_HELP );
	 break;
      case 2:
         map_sequence();
         bw_message( MAP_HELP );
         break;
      case 3:
         map_altitude();
         bw_message( MAP_HELP );
         break;
      case 4:
	 map_exit();
	 exit(0);
      default:
         break;
      }

   ui_pop();                            /* restore ui screen */

   }

/****************************************************************

	map_select()

****************************************************************/

map_select()
   {
   register int selpos, c;

   selpos = map_curpos;
   if ( selpos != 0 )
      {
      map_draw( 0 );
      }
   c = TRUE;
   bw_message( MAP_SELECT );
   while( c )
      {
      map_bound( TRUE, selpos );
      c = kb_rx();
      map_bound( FALSE, selpos );
      switch( c )
         {
         case KB_UP:
            if ( selpos < 7 )
               {
               selpos += 3;
               }
            break;
         case KB_DOWN:
            if ( selpos > 3 )
               {
               selpos -= 3;
               }
            break;
         case KB_RIGHT:
            ++selpos;
            selpos %= 10;
            break;
         case KB_LEFT:
            --selpos;
            selpos += 10;
            selpos %= 10;
            break;
         case CR:
         case LF:
            c = FALSE;
            map_curpos = selpos;
            break;
         }
      }
   map_draw( map_curpos );
   }

/****************************************************************

	map_draw()

****************************************************************/

map_draw( pos )
   int pos;
   {
   register int i, c;
   int x;

   /* Blank the whole area */

   ui_fbox( main_window->u_x1, mesg_y + 1, main_window->u_x2,
      main_window->u_y2, BLACK, SOLID );

   /* draw original lines */

   dr_draw( map_window, TRUE, &spd_start, &spd_end, pos );

   /* plot new (added) lines */

   dr_plot( &new_start, &new_end, cl_surface, SOLID, pos );

   /* write bounds */

   if ( pos == 0 )
      {
      i = 30;
      }
   else
      {
      i = 15;
      }

   /* write latitudes */

   x = dr_mapbase - ( ui_grwind->fysize / 2 );
   sprintf( map_tbuf, " %d ", map_pos[ pos ][ 1 ] );
   gr_text( GR_PRIMARY, dr_mapedge - ( gr_strlen( map_tbuf ) + 1 ),
      dr_mapbase, map_tbuf, BLACK, WHITE );
   gr_text( GR_PRIMARY, dr_mapedge + dr_xsize + 1,
      dr_mapbase, map_tbuf, BLACK, WHITE );
   x += dr_ysize
      / ( ( map_pos[ pos ][ 3 ] - map_pos[ pos ][ 1 ] ) / i );
   for ( c = map_pos[ pos ][ 1 ] + i; c < map_pos[ pos ][ 3 ]; c += i )
      {
      sprintf( map_tbuf, " %d ", c );
      gr_text( GR_PRIMARY, dr_mapedge - ( gr_strlen( map_tbuf ) + 2 ),
	 x, map_tbuf, WHITE, BLACK );
      gr_text( GR_PRIMARY, dr_mapedge + dr_xsize + 1,
	 x, map_tbuf, WHITE, BLACK );
      x += dr_ysize
	 / ( ( map_pos[ pos ][ 3 ] - map_pos[ pos ][ 1 ] ) / i );
      }
   sprintf( map_tbuf, " %d ", map_pos[ pos ][ 3 ] );
   gr_text( GR_PRIMARY, dr_mapedge - gr_strlen( map_tbuf ) - 1,
      dr_mapbase + dr_ysize - ui_grwind->fysize, map_tbuf, BLACK, WHITE );
   gr_text( GR_PRIMARY, dr_mapedge + dr_xsize + 1,
      dr_mapbase + dr_ysize - ui_grwind->fysize, map_tbuf, BLACK, WHITE );

   /* write longitudes */

   x = dr_mapedge;

   sprintf( map_tbuf, " %d ", map_pos[ pos ][ 0 ] );
   gr_text( GR_PRIMARY, dr_mapedge,
      dr_mapbase - ( ui_grwind->fysize + 1 ), map_tbuf, BLACK, WHITE );
   gr_text( GR_PRIMARY, dr_mapedge,
      dr_mapbase + dr_ysize + 1, map_tbuf, BLACK, WHITE );
   x += dr_xsize
      / ( ( map_pos[ pos ][ 2 ] - map_pos[ pos ][ 0 ] ) / i );

   for ( c = map_pos[ pos ][ 0 ] + i; c < map_pos[ pos ][ 2 ]; c += i )
      {
      sprintf( map_tbuf, " %d ", c );
      gr_text( GR_PRIMARY, x - ( gr_strlen( map_tbuf ) / 2 ),
	 dr_mapbase - ( ui_grwind->fysize + 1 ), map_tbuf, WHITE, BLACK );
      gr_text( GR_PRIMARY, x - ( gr_strlen( map_tbuf ) / 2 ),
	 dr_mapbase + dr_ysize + 1, map_tbuf, WHITE, BLACK );
      x += dr_xsize
	 / ( ( map_pos[ pos ][ 2 ] - map_pos[ pos ][ 0 ] ) / i );
      }
   sprintf( map_tbuf, " %d ", map_pos[ pos ][ 2 ] );
   gr_text( GR_PRIMARY, dr_mapedge + dr_xsize - gr_strlen( map_tbuf ),
      dr_mapbase - ( ui_grwind->fysize + 1 ), map_tbuf, BLACK, WHITE );
   gr_text( GR_PRIMARY, dr_mapedge + dr_xsize - gr_strlen( map_tbuf ),
      dr_mapbase + dr_ysize + 1, map_tbuf, BLACK, WHITE );
   }

/****************************************************************

	map_bound()

****************************************************************/

map_bound( highlight, pos )
   int highlight, pos;
   {
   int style;
   double x1, y1, x2, y2;

   if ( highlight == TRUE )
      {
      style = SOLID;
      }
   else
      {
      style = GRID;
      }

#ifdef OLD_DEBUG
   sprintf( map_tbuf, "highlight %d, position %d, style %d",
      highlight, pos, style );
   bw_debug( map_tbuf );
#endif

   x1 = map_pos[ pos ][ 0 ];
   y1 = map_pos[ pos ][ 1 ];
   x2 = map_pos[ pos ][ 0 ];
   y2 = map_pos[ pos ][ 3 ];
   dr_line( y1, x1, y2, x2, BLACK, SOLID, 0 );
   dr_line( y1, x1, y2, x2, WHITE, style, 0 );

   x1 = map_pos[ pos ][ 2 ];
   y1 = map_pos[ pos ][ 1 ];
   x2 = map_pos[ pos ][ 2 ];
   y2 = map_pos[ pos ][ 3 ];
   dr_line( y1, x1, y2, x2, BLACK, SOLID, 0 );
   dr_line( y1, x1, y2, x2, WHITE, style, 0 );

   x1 = map_pos[ pos ][ 0 ];
   y1 = map_pos[ pos ][ 1 ];
   x2 = map_pos[ pos ][ 2 ];
   y2 = map_pos[ pos ][ 1 ];
   dr_line( y1, x1, y2, x2, BLACK, SOLID, 0 );
   dr_line( y1, x1, y2, x2, WHITE, style, 0 );

   x1 = map_pos[ pos ][ 0 ];
   y1 = map_pos[ pos ][ 3 ];
   x2 = map_pos[ pos ][ 2 ];
   y2 = map_pos[ pos ][ 3 ];
   dr_line( y1, x1, y2, x2, BLACK, SOLID, 0 );
   dr_line( y1, x1, y2, x2, WHITE, style, 0 );

   gr_line( GR_PRIMARY, dr_mapedge, dr_mapbase, dr_mapedge,
      dr_mapbase + dr_ysize, cl_grid, SOLID );

   gr_line( GR_PRIMARY, dr_mapedge + dr_xsize, dr_mapbase,
      dr_mapedge + dr_xsize, dr_mapbase + dr_ysize,
      cl_grid, SOLID );

   gr_line( GR_PRIMARY, dr_mapedge, dr_mapbase,
      dr_mapedge + dr_xsize,
      dr_mapbase, cl_grid, SOLID );

   gr_line( GR_PRIMARY, dr_mapedge, dr_mapbase + dr_ysize,
      dr_mapedge + dr_xsize, dr_mapbase + dr_ysize,
      cl_grid, SOLID );

   }

/****************************************************************

	map_sequence()

****************************************************************/

map_sequence()
   {
   int qu_x1, qu_y1, qu_x2, qu_y2;
   int bu_x1, bu_y1, bu_x2, bu_y2;
   register int c;
   static int x, y, b;
   double xpos, ypos;
   double prevx, prevy, lastx, lasty;
   struct spj_pt *curspj;
   int n, allow_backup;

   bw_message( MAP_SEQM );

   if ( gr_ismouse != TRUE )
      {
      bw_error( ERR_MOUSE );
      return BW_ERROR;
      }

   /* calculate sizes for quit and backup spaces */

   qu_x1 = ( main_window->u_x2 - main_window->u_x1 ) / 2;
   qu_x2 = qu_x1 + ( ( main_window->u_x2 - main_window->u_x1 ) / 4 );
   qu_y1 = main_window->tbar_y1;
   qu_y2 = main_window->tbar_y2;
   bu_x1 = qu_x2 + 1;
   bu_x2 = main_window->u_x2;
   bu_y1 = main_window->tbar_y1;
   bu_y2 = main_window->tbar_y2;

   /* draw delimiters for quit and backup spaces */

   gr_line( GR_PRIMARY, qu_x1, qu_y1, qu_x1, qu_y2, BLACK, SOLID );
   gr_line( GR_PRIMARY, bu_x1, bu_y1, bu_x1, bu_y2, BLACK, SOLID );

   /* denote quit and backup spaces */

   gr_text( GR_PRIMARY, qu_x1 + 3, qu_y1, SEQ_QUIT, BLACK, WHITE );
   gr_text( GR_PRIMARY, bu_x1 + 3, bu_y1, SEQ_BACKUP, BLACK, WHITE );

   lastx = lasty = prevx = prevy = xpos = ypos = 0.0;
   n = 0;
   allow_backup = FALSE;
   c = TRUE;
   while ( c == TRUE )
      {

      gr_mouse( WAIT, &x, &y, &b );           /* click down */
      gr_mouse( WAIT, &x, &y, &b );           /* click up */

      /* Is the release within bounds of the map? */

      if ( uil_bounds( x, y, dr_mapedge, dr_mapbase,
         dr_mapedge + dr_xsize, dr_mapbase + dr_ysize ) == TRUE )
         {
         xpos = map_pos[ map_curpos ][ 0 ] +
	    (( x - (double) dr_mapedge )
	    / ( dr_xsize / (double) ( map_pos[ map_curpos ][ 2 ] - (double) map_pos[ map_curpos ][ 0 ] ) ));
         ypos = map_pos[ map_curpos ][ 1 ] +
	    (( y - (double) dr_mapbase )
	    / ( dr_ysize / ( (double) map_pos[ map_curpos ][ 3 ] - (double) map_pos[ map_curpos ][ 1 ] ) ));
         sprintf( map_tbuf, "Position: Latitude %lf  Longitude %lf",
            ypos, xpos );
         bw_message( map_tbuf );

         if ( ( curspj = ( struct spj_pt  *) malloc( sizeof( struct spj_pt ) )) == NULL )
            {
            sprintf( bw_ebuf, MEMERR_SEQ );
            bw_error( bw_ebuf );
            return BW_ERROR;
            }
         lastspj->next = curspj;
         curspj->next = &new_end;
         curspj->latitude = ypos;
         curspj->longitude = xpos;
         curspj->radius = map_defalt;

         if ( n == 0 )
            {
            dr_line( ypos, xpos, ypos, xpos,
               WHITE, SOLID, map_curpos );
            curspj->code = 1001;
            }
         else if ( n == 1 )
            {
            dr_line( ypos, xpos, prevy, prevx,
               WHITE, SOLID, map_curpos );
            curspj->code = 5;
            }
         else
            {
            dr_line( ypos, xpos, prevy, prevx,
               WHITE, SOLID, map_curpos );
            curspj->code = 5;
            if ( n == 2 )
               {
               fprintf( outfile, "1001\t%lf\t%lf\t%lf\n",
                  lasty, lastx, map_defalt );
               }
            else
               {
               fprintf( outfile, "5\t%lf\t%lf\t%lf\n",
                  lasty, lastx, map_defalt );
               }
            }

         /* Get storage for the point */

         lastspj = curspj;
         lastx = prevx;
         lasty = prevy;
         prevx = xpos;
	 prevy = ypos;
	 allow_backup == TRUE;
         ++n;
         }

      /* Is the release within bounds of quit area? */

      else if ( uil_bounds( x, y, qu_x1, qu_y1,
	 qu_x2, qu_y2 ) == TRUE )
         {
         c = FALSE;
         }

      /* Is the release within bounds of backup area? */

      else if ( ( uil_bounds( x, y, bu_x1, bu_y1,
	 bu_x2, bu_y2 ) == TRUE ) && ( allow_backup == TRUE ) )
	 {
	 xpos = prevx;
	 ypos = prevy;
	 prevx = lastx;
	 prevy = lasty;
	 allow_backup = FALSE;
	 --n;
	 if ( n == -1 )
	    {
	    n = 0;
	    }
	 else if ( n == 0 )
	    {
	    curspj = lastspj;
	    curspj->next = &new_end;
	    dr_line( ypos, xpos, ypos, xpos,
	       BLACK, SOLID, map_curpos );
	    }
	 else
	    {
	    curspj = lastspj;
	    curspj->next = &new_end;
	    dr_line( ypos, xpos, prevy, prevx,
	       BLACK, SOLID, map_curpos );
	    }
	 }

      else
         {
         bw_message( " " );
	 }

      }

   switch ( n )
      {
      case 0:
         return -1;
         break;
      case 1:
         fprintf( outfile, "1001\t%lf\t%lf\t%lf\n",
            ypos, xpos, map_defalt );
         break;
      case 2:
         fprintf( outfile, "1001\t%lf\t%lf\t%lf\n",
            prevy, prevx, map_defalt );
         fprintf( outfile, "5\t%lf\t%lf\t%lf\n",
            ypos, xpos, map_defalt );
         break;
      default:
         fprintf( outfile, "5\t%lf\t%lf\t%lf\n",
            prevy, prevx, map_defalt );
         fprintf( outfile, "5\t%lf\t%lf\t%lf\n",
            ypos, xpos, map_defalt );
         break;
      }

   /* clear the quit and backup areas */

   ui_fbox( qu_x1 - 1, qu_y1, bu_x2, bu_y2,
      WHITE, SOLID );

   }

/****************************************************************

	map_entitle()

****************************************************************/

map_entitle()
   {
   static int image;

   if ( dial_x1 == 0 )
      {
      midx = main_window->u_x1
	 + ((main_window->u_x2 - main_window->u_x1 ) / 2 );
      midy = main_window->u_y1
	 + ((main_window->u_y2 - main_window->u_y1 ) / 2 );
      length = ( main_window->u_x2 - main_window->u_x1 ) / 2;
      dial_x1 = midx - length / 2;
      dial_x2 = midx + length / 2;
      height = ( main_window->u_y2 - main_window->u_y1 ) / 2;
      dial_y1 = midy - height / 2;
      dial_y2 = midy + height / 2;

      }

   gr_imsave( GR_PRIMARY, TRUE, dial_x1, dial_y1,
      dial_x2, dial_y2, &image );
   ui_dial( dial_x1, dial_y1, dial_x2, dial_y2,
      cl_mback, cl_mfore, cl_marker,
      TRUE, ENT_TITLE,
      ENT_TEXT, ENT_PROMPT, buffer, &dial_window );
   gr_imsave( GR_PRIMARY, FALSE, dial_x1, dial_y1,
      dial_x2, dial_y2, &image );
   gr_imfree( image );

   fprintf( outfile, ";----------------------------------------------------------------\n" );
   fprintf( outfile, ";\n" );
   fprintf( outfile, ";\t%s\n", buffer );
   fprintf( outfile, ";\n" );

   }

/****************************************************************

	map_altitude()

****************************************************************/

map_altitude()
   {
   static double n;
   static int image;
   
   if ( dial_x1 == 0 )
      {
      midx = main_window->u_x1
         + ((main_window->u_x2 - main_window->u_x1 ) / 2 );
      midy = main_window->u_y1
         + ((main_window->u_y2 - main_window->u_y1 ) / 2 );
      length = ( main_window->u_x2 - main_window->u_x1 ) / 2;
      dial_x1 = midx - length / 2;
      dial_x2 = midx + length / 2;
      height = ( main_window->u_y2 - main_window->u_y1 ) / 2;
      dial_y1 = midy - height / 2;
      dial_y2 = midy + height / 2;

      }

   gr_imsave( GR_PRIMARY, TRUE, dial_x1, dial_y1,
      dial_x2, dial_y2, &image );
   ui_dial( dial_x1, dial_y1, dial_x2, dial_y2,
      cl_mback, cl_mfore, cl_marker,
      TRUE, ALT_TITLE,
      ALT_TEXT, ALT_PROMPT, buffer, &dial_window );
   gr_imsave( GR_PRIMARY, FALSE, dial_x1, dial_y1,
      dial_x2, dial_y2, &image );
   gr_imfree( image );

   sscanf( buffer, "%lf",  &n );
   }

/****************************************************************

	map_exit()

****************************************************************/

map_exit()
   {
   kb_deinit();
   gr_deinit();
   }

/****************************************************************

	map_font()

****************************************************************/

map_font( size )
   int size;
   {
   static oldsize = 159;
   int rq_height;

   if ( size == oldsize )
      {
      return;
      }

   switch( size )
      {
      case 1:
	 rq_height = ui_grwind->ymax / 25;
	 break;
      case 2:
	 rq_height = ui_grwind->ymax / 15;
	 break;
      default:
	 rq_height = ui_grwind->ymax / 30;
	 break;
      }

   gr_font( GR_PRIMARY, F_DEFAULT, rq_height );
   }

/****************************************************************

	map_colors()

****************************************************************/

map_colors()
   {

   if ( gr_colors > 8 )
      {
      cl_mback = WHITE;
      cl_mfore = BLACK;
      cl_marker = DARK_CYAN;
      cl_grid = DARK_BLUE;
      cl_surface = DARK_GREEN;
      }
   else if ( gr_colors > 2 )
      {
      cl_mback = WHITE;
      cl_mfore = BLACK;
      cl_marker = BLACK;
      cl_grid = WHITE;
      cl_surface = WHITE;
      }
   else
      {
      cl_mback = WHITE;
      cl_mfore = BLACK;
      cl_marker = WHITE;
      cl_grid = WHITE;
      cl_surface = WHITE;
      }

   }

/****************************************************************

	bw_error()

****************************************************************/

bw_error( mes )
   char *mes;
   {
   static char buffer[ BW_EBUFSIZE ];
   static int image;
   static int x1 = 0, x2, y1, y2, height, length, midx, midy, c;
   static struct uiwindow *err_window;

   if ( x1 == 0 )
      {

      midx = main_window->u_x1 + ( ( main_window->u_x2 - main_window->u_x1 ) / 2 );
      midy = main_window->u_y1 + ( ( main_window->u_y2 - main_window->u_y1 ) / 2 );

      length = main_window->u_x2 / 2;
      x1 = midx - length / 2;
      x2 = midx + length / 2;
      height = main_window->u_y2 / 2;
      y1 = midy - height / 2;
      y2 = midy + height / 2;

      }

   gr_imsave( GR_PRIMARY, TRUE, x1, y1, x2, y2, &image );
   buffer[ 0 ] = 0;
   sprintf( buffer, ERR_PROMPT, mes );
   c = ui_yn( x1, y1, x2, y2, cl_mback, cl_mfore, cl_marker, TRUE,
      buffer, ERRP_NO, ERRP_YES, &err_window );
   gr_imsave( GR_PRIMARY, FALSE, x1, y1, x2, y2, &image );
   gr_imfree( image );

   if ( c == FALSE )
      {
      map_exit();
      exit( 1 );
      }
   }

/****************************************************************

	bw_debug()

****************************************************************/

#ifdef  DEBUG
bw_debug( mes )
   char *mes;
   {
   static char buffer[ 256 ];

   sprintf( buffer, "DEBUG: %s     ", mes );
   bw_message( buffer );
   ui_wait();
   }
#endif

/****************************************************************

	map_sigint()

****************************************************************/

map_sigint()
   {
   kb_deinit();
   gr_deinit();
   exit( 1 );
   }

/****************************************************************

	bw_message()

****************************************************************/

bw_message( m )
   char *m;
   {
   ui_fbox( main_window->u_x1, main_window->u_y1,
      main_window->u_x2, mesg_y - 1, BLACK, SOLID );
   ui_str( 5, 5, main_window->x2, BLACK, WHITE, m );
   }

ui_poll()
   {
   }
