/***************************************************
   N-Body Globular Cluster INTERACTION ROUTINES
   Modifications (C) Copyright 1998 Graeme Fenwick
   Based on original BBC-BASIC routines by ?
***************************************************/

#include "gl.h"                                /* SETTINGS structure typedef */
#include <math.h>
#include <stdlib.h>
#include <time.h>
#include "gl_inter.h"
#include "gl_gfx.h"

#define   RAN1   (rand()/(double)RAND_MAX)      /* random double from 0 to 1 */
#ifndef   PI                                         /* DJGPP pre-defines PI */
   #define   PI   3.1415923536
#endif

/****** INITIALIZE bodies with random values */

void initialize(int n, unsigned seed, double **pos, double **vel,
                    double *vbar, double *ke)
{
   int      star, cmp;
   double   u[2][3],
            cdist, theta, phi, s, c, s1, c1;

   *ke = 0;                                          /* not auto-initialized */
   srand(seed);
   for (star = 0; star < n; ++star) {
      cdist = pow(RAN1, 1.0 / 3.0);
      c = -1 + 2 * RAN1;
      theta = acos(c);
      s = sin(theta);
      phi = 2 * PI * RAN1;
      c1 = cos(phi);
      s1 = sin(phi);
      pos[star][0] = cdist * s * c1;                         /* assign x pos */
      pos[star][1] = cdist * s * s1;                         /* assign y pos */
      pos[star][2] = cdist * c;                              /* assign z pos */
      u[0][0] = -s1;        /* vectors at right angles to radius vector (??) */
      u[0][1] = c1;
      u[0][2] = 0;
      u[1][0] = -c * c1;
      u[1][1] = -c * s1;
      u[1][2] = s;
      phi = 2 * PI * RAN1;
      c = cos(phi);
      s = sin(phi);
      for (cmp = 0; cmp < 3; ++cmp) {
         vel[star][cmp] = cdist * (c * u[0][cmp] + s * u[1][cmp]);
         vbar[cmp] += vel[0][cmp];
      }
      *ke += (cdist * cdist);                      /* cdist is speed as well */
   }
}

/****** REMOVE MOTION of CENTRE of gravity from bodies */

void remove_centre_motion(int n, double **vel, double *vbar, double *ke)
{
   int   cmp, star;

   *ke = 0;
   for (cmp = 0; cmp < 3; ++cmp) {
      vbar[cmp] /= n;
      for (star = 0; star < n; ++star) {
         vel[star][cmp] -= vbar[cmp];     /* remove centr-mass-vel from body */
         *ke += (vel[star][cmp] * vel[star][cmp]); /* add vel cmp to kin nrg */
      }
   }
}

/****** CALCulate Potential Energy in system */

double calc_pe(int n, double **pos, SETTINGS cfg)
{
   double   pe, mass, dsqu, posdif;
   int      a_body, b_body, cmp;                           /* loop variables */

   if (n == 1)
      return 0.0;                                    /* pe is zero if n == 1 */
   else {
      pe = 0.0;
      mass = (double) cfg.totmass / n;         
      for (a_body  = 0; a_body < n - 1; ++a_body)
	 for (b_body = a_body + 1; b_body < n; ++b_body) {
            dsqu = 0.0;                                  /* distance squared */
            for (cmp = 0; cmp < 3; ++cmp) {
               posdif = pos[a_body][cmp] - pos[b_body][cmp];
               dsqu += (posdif * posdif);
            }
            pe -= (mass / sqrt(dsqu));
         }
   return pe;
   }
}

/****** INTERACTion sequence */

void interact(int n, double pe, double ke, double **frc, double **pos,
                 double **vel, SETTINGS cfg)
{
   int      star, cmp, a_body, b_body;
   time_t   then;
   double   difpos[3],
            mass, initial_e, e, passed, dsqu, r, r3, part, vold;

   put_bodies(pos, n, cfg.cndist, cfg.size, cfg.p_source, 0);
   mass = (double) cfg.totmass / n;
   initial_e = pe + 0.5 * ke;
   passed = 0;
   if (cfg.e_disp == 1)
      status(initial_e, 0, 0.0);            /* initial energy status display */
   time(&then);                                         /* pause 3-4 seconds */
   while (time(NULL) < then + 4)
      ;
   time(&then);                                      /* realtime clock start */
   while (time(NULL) - then < (time_t) cfg.runtime) {    /* interaction loop */
      pe = 0;
      for (star = 0; star < n; ++star)
         for (cmp = 0; cmp < 3; ++cmp)
            frc[star][cmp] = 0;                   /* reset force every cycle */
      if (n > 1)
         for (a_body = 0; a_body < n - 1; ++a_body)
            for (b_body = a_body + 1; b_body < n; ++b_body)
	       if (a_body != b_body) {               /* two different bodies */
                  dsqu = 0;
                  for (cmp = 0; cmp < 3; ++cmp) {
                     difpos[cmp] = pos[a_body][cmp] - pos[b_body][cmp];
                     dsqu += (difpos[cmp] * difpos[cmp]);
                  }
                  r = sqrt(dsqu);
                  r3 = dsqu * r;
                  pe -= 1 / r;
                  for (cmp = 0; cmp < 3; ++cmp) {
                     part = mass * difpos[cmp] / r3;
                     frc[a_body][cmp] -= part;
                     frc[b_body][cmp] += part;
                  }
	       } 
      pe *= mass;
      ke = 0;
      if (cfg.trails == 0)
         put_bodies(pos, n, cfg.cndist, cfg.size, cfg.p_source, 1);/* unplot */
      for (star = 0; star < n; ++star) {
         for (cmp = 0; cmp < 3; ++cmp) {
            vold = vel[star][cmp];
            ke += vold * vold;
            vel[star][cmp] += cfg.timestep * frc[star][cmp];
            pos[star][cmp] += cfg.timestep * vel[star][cmp];
         }
      }
      put_bodies(pos, n, cfg.cndist, cfg.size, cfg.p_source, 0);
      passed += cfg.timestep;
      e = 0.5 * ke + pe;
      if (cfg.e_disp == 1)
         status(e, labs((long) e - initial_e) / labs((long) initial_e) > 1,
            passed);
   }
}
































