--  -----------------------------------------------------------------------
--  Title:       primitive_functions
--  Last Mod:    Thu Apr 19 11:24:34 1990
--  Author:      Vincent Broman <broman@nosc.mil>
--   Copyright 1990 Vincent Broman
--     Permission granted to copy, modify, or compile this software for
--   one's own use, provided that this copyright notice is preserved intact.
--     Permission granted to distribute compiled binary copies of this
--   software which are linked in with some other application.
--     Permission granted to distribute other copies of this software,
--   provided that (1) any copy which is not source code, i.e. not in the
--   form in which the software is usually maintained, must be accompanied
--   by a copy of the source code from which it was compiled, and (2) the
--   one distributing it must refrain from imposing on the recipient
--   further restrictions on the distribution of this software.
--   
--  Visibility:  
--  Description: 
--               functions on floating point types whose implementation
--               does not depend on access to the mantissa/exponent 
--               representation of the floating point number.  this includes
--               integer/fraction operations.
--               
--  Exceptions:  numeric_error upon overflow,
--               constraint_error only if type Float is constrained.
--  -----------------------------------------------------------------------
package primitive_functions is
-- 
   function exponent( x: Float) return integer;
--
-- return the exponent k such that 1/2 <= x/(2**k) < 1,
-- or zero for x = 0.0 .
-- 

   function mantissa (x: Float) return Float;
--
-- return scale( x, - exponent( x)) if x is nonzero,  0.0 otherwise.
--

   function scale (x: Float;
		   k: integer) return Float;
--
-- return x * 2**k quickly, or quietly underflow to zero,
-- or raise an exception on overflow.
-- 

   function shorten (x: Float;
		     k: integer) return Float;
-- 
-- set all but the k most significant bits in the mantissa of x to zero,
-- i.e. reduce the precision to k bits, truncating, not rounding.
-- shorten( x, k) = 0.0 if k < 1 and
-- shorten( x, k) = x   if k >= Float'machine_mantissa.
-- 

   function odd (x: Float) return boolean;
--
-- predicate indicates whether or not truncate( x) is an odd integer.
--

   function truncate (x: Float) return Float;
--
-- truncate x to the nearest integer value with absolute value
-- not exceeding abs( x).  No conversion to an integer type
-- is expected, so truncate cannot overflow for large arguments.
--

   function floor (x: Float) return Float;
--
-- return as a Float the greatest integer value <= x.
--

   function ceiling (x: Float) return Float;
--
-- return as a Float the least integer value >= x.
--

   function round (x: Float) return Float;
--
-- return as a Float the integer value nearest x.
-- in case of a tie, prefer the even value.
--

end primitive_functions;

-- $Header: g_primitive_functions_s.ada,v 3.13 90/04/19 12:34:11 broman Rel $
