/*
    Gn: A Server for the Internet Gopher Protocol(*).
    File: gn/chkcache.c
    Version 2.14
    
    Copyright (C) 1993  <by John Franks>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    (*) Gopher is a registered trademark of the Univ. of Minn.
*/
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include "gn.h"
#include "chkcache.h"

char		myhost[MIDLEN],
		myport[SMALLLEN];

static int	match_select();

extern void	www_unreplace();
static void	get_stat();


/*
 * chkcache( ip)  checks to see that the selector string sent by the client
 * is a valid one, i.e. that the selector exists in a cache file in the 
 * designated directory. It returns the (local not gopher) type of the item 
 * which ip->selector refers to.  It also enters some additional fields
 * in the Item struct pointed to by ip, namely "name", "content_type",
 * "encoding" and "suffix."  Taking the information from the .cache file this
 * function also fills in the ip->name, ip->content_type, ip->suffix,
 * myhost, and myport values.
 */

int
chkcache( ip)
Item	*ip;
{
	struct stat stat_buf;

	FILE	*fp,
		*fopen();

	char	altgtype[MIDLEN],
		altselector[PATHLEN];

	Item		citem;
	Cache_entry	*cep,
			entry;

	altgtype[0] = altselector[0] = '\0';

	if ( *(ip->cachepath) == '\0') {
		/*
		 * root directory is the unique item not contained IN
                 * a cache file, handle it here.
                 */
		strcpy( ip->name, root_title);
		strcpy( ip->content_type, "text/html");
		strcpy( ip->suffix, "html");
		if ( *(ip->gtype) == '7' )
			return ( http & HTTP1_0HEAD ? HEADER : SEARCH);
		get_stat( ip);
		return ( http & HTTP1_0HEAD ? HEADER : ROOT);
	}

	if ( chk_cache_id == CACHE_USE_UID ) {
		if ( stat( ip->cachepath, &stat_buf) != 0 ) {
			senderr2( "Can't stat cache file [chkcache]",
					ip->cachepath);
			exit( 2);
		}
		if ( stat_buf.st_uid != (uid_t) cache_id) {
			senderr2( "Bad cache uid", ip->cachepath);
			exit( 2);
		}
	}
	else if ( chk_cache_id == CACHE_USE_GID ) {
		if ( stat( ip->cachepath, &stat_buf) != 0 ) {
			senderr2( "Can't stat cache file [chkcache]",
					ip->cachepath);
			exit( 2);
		}
		if ( stat_buf.st_gid != (gid_t) cache_id) {
			senderr2( "Bad cache gid", ip->cachepath);
			exit( 2);
		}
	}
        if ( (fp = fopen( ip->cachepath, "r")) == (FILE *) NULL ) {
		if ( !http) {
			senderr2( "Can't open cache file [chkcache]",
				ip->cachepath);
			exit( 2);
		}
		if ( !streq( ip->gtype, "XRETRY") ) {
                        sprintf( altselector, "%s/%s%s", "XRETRY",
                                        ip->gtype, ip->relpath);
                        process_url( altselector, ip);
                        exit( 0);
                }
		/*
		 * We failed on an XRETRY path.  The URL probably
                 * points to something that does not exist.
		 */
		return DENIED;
	}

	cep = &entry;
	while ( read_cache( cep, fp)) {
		switch ( cep->entrytype) {
		case REMGOPHER:
		case REMGN:
		case ILINE:
		case REMHTTP:
		case FTPLINK:
		case TELNETLINK:
		case HTTPTEXT:
				continue;
		}

		parse_selector( &citem, cep->path, PARTIAL);

		if ( match_select( &citem, ip, altgtype)) {
			strcpy( ip->name, entry.name); 
			www_unreplace( ip->name);
			strcpy( myhost, entry.host);
			strcpy( myport, entry.port);

			if ( streq( ip->gtype, "0h") && http )
				strcpy( ip->content_type, "text/html");
			else
				strcpy( ip->content_type, entry.content);
			
			strcpy( ip->encoding, entry.encoding);
			strcpy( ip->suffix, entry.suffix);
			
			get_stat( ip);
			if ( http & HTTP1_0HEAD )
				return HEADER;
			switch (*(ip->gtype)) {
				case '0':
				case '4':
				case '6':
					return TEXT;
				case '1':
					if ( http && (ip->gtype)[1] == 'h' )
						return TEXT;
					else
						return DIR;
				case '5':
				case '9':
				case 's':
				case 'I':
					return BIN;
				case '7':
					return SEARCH;
#ifndef FORBID_EXEC
				case 'e':
					return EXEC;
				case 'C':
					return CGI;
#endif
				case 'r':
				case 'R':
					return RANGE;
				case 'i':
					return COMMENT;
				default:
					return UNKNOWN;
			}
		}
	}
	if ( (!http) &&  (*(ip->gtype) == 'i'))
		return COMMENT;

	if ( http && *altgtype ) {
		strcpy( altselector, altgtype);
		strcat( altselector, ip->relpath);
		process_url( altselector, ip);
		exit( 0);
	}
	if ( http && !streq( ip->gtype, "XRETRY") ) {
		sprintf( altselector, "%s/%s%s", "XRETRY", 
				ip->gtype, ip->relpath);
		process_url( altselector, ip);
		exit( 0);
	}

	writelog( "", "No match for selector", ip->selector);
	return DENIED;
}

static int
match_select( ipc, ipr, alt)
Item	*ipc,
	*ipr;
char	*alt;
{

	/*
	 * Check that ipc (from .cache) and ipr have the same filepath
	 * and the same gtype.  One exception is the gytpe for
	 * ipc can be "1s" (searchable directory) and for ipr
	 * it can be "7g" (search).
	 */

	if ( ( !streq( ipc->relpath, ipr->relpath))
			&&  (!streq( ipc->filepath, ipr->filepath)))
		return FALSE;
	if ( streq( ipc->gtype, ipr->gtype))
		return TRUE;
	if ( !*alt )
		strcpy( alt, ipc->gtype);

	if ( streq(ipr->gtype, "7g") && *ipc->gtype == '1') {
		switch (ipc->gtype[1]) {
		case 's':
			return TRUE;
		case 'h':
			return ((ipc->gtype[2] == 's') ? TRUE : FALSE);
		default:
			writelog("", "Directory not searchable",
					ipc->filepath);
			return ( FALSE );
		}
	}
	if ( streq(ipr->gtype, "1") && streq( ipc->gtype, "1s")) {
		strcpy( ipr->gtype, ipc->gtype);
		return TRUE;
	}
	return ( FALSE );
}


static void
get_stat( ip)
Item	*ip;
{
	struct stat stat_buf;
	struct tm *gmt;


	if ( stat( ip->filepath, &stat_buf) != 0 ) {
		senderr2( DENYMSG, ip->filepath);
		exit( 2);
	}

	switch (*(ip->gtype)) {
	case '0':
	case '4':
	case '5':
	case '9':
	case 's':
	case 'I':
		sprintf( ip->length, "%lu", (unsigned long) stat_buf.st_size);
		break;
	case 'R':
	case 'r':
		sprintf( ip->length, "%ld", ip->range_end - ip->range_start);
		break;
	}
	gmt = gmtime(&(stat_buf.st_mtime));
	strftime( ip->mod_date, SMALLLEN, "%A, %d-%h-%y %T GMT", gmt);
}

