Program TestFFT;
{ E.Sorokin, 1996 }
{
  Demo program for LArrayS unit.

  Create an array, and transform it two times using
	Fast Fourier Transform (FFT) algorithm.

  The program illustrates different methods of addressing.
}

uses LArrays
{$IFDEF WINDOWS}, WinCrt {$ENDIF}
;
const TestSize = $1000*4;   {Must be a power of 2}

type TComplex = record Re, Im : double; s : string[6] end;
     PComplex =^TComplex;

type  PCArr =^TCArr;
      TCArr = object (TAbstArray)
                constructor make( pts : longint);
              end;

constructor TCArr.make;
begin
  if not inherited Init( pts, sizeof(TComplex), 1) then Fail;
end;


{Algorithm from "Numerical recipes", adapted for 0-based arrays}

procedure FFT( var Data : TCArr; isign : integer);
var
	 istep, mmax, i,j,m,n, ii,jj  : longint;
	 wtemp,wr,wpr,wpi,wi,theta : extended;
	 tempr,tempi               : extended;
   ith,jth : PComplex;
BEGIN
   If Data.size=0 then exit;
   n:=1;
   while n < Data.Size do n:=n*2;
   If n <> Data.Size then exit; {Only powers of 2 are allowed}

   {Bit-invert commutations using Swap function}
	 j := 0;
	 FOR i := 0 to n-2 DO BEGIN
			IF (j > i) THEN
			   Data.Swap(i,j);
			m := n DIV 2;
			WHILE (j >= m) DO BEGIN
				 j := j-m;
				 m := m DIV 2
			END;
			j := j+m
	 END;

   mmax := 1;
	 WHILE (mmax < n) DO BEGIN
      istep:=mmax*2;
			theta := 6.28318530717959/(isign*istep);
			wpr := -2.0*sqr(sin(0.5*theta));
			wpi := sin(theta);
			wr := 1.0;
			wi := 0.0;
			FOR ii := 0 to mmax-1 DO BEGIN
         {Perform butterflies, using indirect typecasting}
				 FOR jj := 0 to ((n DIV istep)-1) DO BEGIN
						i := ii + jj*istep;
						j := i+mmax;
            ith:=Data.Nth(i);
            jth:=Data.Nth(j);
						tempr := wr*jth^.Re - wi*jth^.Im;
						tempi := wr*jth^.Im + wi*jth^.Re;
						jth^.Re := ith^.Re-tempr;
            jth^.Im := ith^.Im-tempi;
						ith^.Re := ith^.Re+tempr;
						ith^.Im := ith^.Im+tempi
         END;
         wtemp := wr;
         wr := wr*wpr-wi*wpi+wr;
         wi := wi*wpr+wtemp*wpi+wi;
      END;
      mmax := istep;
   END;
   if isign=-1 then begin  {Normalize the result}
     tempr:=Data.size;
	   for i:=0 to Data.size-1 do begin
       ith:=NthOf(Data,i);
       ith^.re:=ith^.re/tempr;
       ith^.im:=ith^.im/tempr;
     end;
   end;
END;

var a,b : TCArr;
    c,c1 : TComplex;
    i : longint;
    Rerr, Ierr : extended;

begin
    {Simple  constructor}
  if not a.make( TestSize ) then exit;
  Writeln('Creating an array of ',TestSize, ' random complex points.');
  Write(a.Pages,' pages, ', a.PageSize, ' complex points per page');
  If a.LastPageSize <> 0 then
    Writeln(', except the last.')
  else
    Writeln ('.');
  Randomize;
  for i:=0 to a.size-1 do begin
    c.re:=Random {sin(2*pi*i/a.size)};
    c.Im:=0;
    {Using Put function}
    a.Put(i,c);
  end;

    {Using copy constructor}
  if not b.Copy(@a) then begin writeln('Not enough memory for copy array.'); exit; end;

  writeln('Transforming . .');
  fft(b,1);
  { Check FFT algorithm:
  for i:=0 to b.size-1 do begin
    b.get(i,c);
    writeln(c.re:8:4,#9,c.im:8:4);
  end;}
  writeln;
  writeln('Transforming back . .');
  fft(b,-1);
  writeln('Checking errors . .');
  rerr:=0; Ierr:=0;
  for i:=0 to a.size-1 do begin
        {Using Get function}
    b.get(i,c);
        {Using direct typecasting with Nth function}
    rerr:=rerr + sqr(c.re-PComplex(a.Nth(i))^.Re);
    Ierr:=Ierr + sqr(c.Im-PComplex(a.Nth(i))^.Im);
  end;

  writeln('Standard deviation (', Sqrt(rerr)/(a.size-1):7, ', ' , Sqrt(Ierr)/(a.size-1):8,')');

end.