{$A+}
{$B-}
{$D+}
{$E-}
{$F+}
{$I-}
{$L+}
{$N-}
{$O-}
{$R-}
{$S-}
{$V-}

UNIT MetaDirE;
{+H
---------------------------------------------------------------------------
  Version     - 0.11

  File        - METADIRE.PAS

  Copyright   - None. Public Domain.

  Author      - Keith S. Brown (except where otherwise noted)
                Surface mail:              Email:(brown@smd4.jsc.nasa.gov)
                  K.Brown
                  Code:NASA/JSC/ES64
                  Houston, TX 77058 (USA)  Voice:713-483-8952

  Purpose     - Directory search engine using Meta File Names (UNIX style
                wild cards).

  Language    - Borland International's Turbo Pascal V:4.x+ for MS-DOS

  Remarks     - Uses regular expression parsing to handle UNIX style file
                name wild cards such as '*ss.[^ef]*'
                  --> file names ending with 'SS' and whose extension starts
                      with neither 'E' nor 'F'.  Has enough smarts to
                      decide if DOS can handle the wild card (faster) or
                      if it requires special Unix-style processing (slower).

                The SearchEngine procedure searches through a named directory
                applying the procedure type PROC to each matching file.

                The SearchEngineAll procedure calls SearchEngine to process
                all subdirectories of the named directory and apply the
                procedure type PROC to each matching file.

  Requires    - METADIRE requires access to Borland's DOS unit.  Access to
                TurboPower Software's TPSTRING unit is optional.

  Reference   - Turbo Technix; September/October 1988 p27-36.
  Revised     - 1989.1128 (KSB) Entered code. Variant of DIRENGNE.PAS with Meta File Names.
              - 1990.0808 (KSB) bypass Meta search if DOS can handle it.
              - 1991.0628 (KSB) Merged with METANAME & removed redundant code & procedures. Fixed MakeAutomaton.
              - 1993.0901 (KSB) Emulated TP functs. if unit not available.  Updated documentation.
---------------------------------------------------------------------------}
INTERFACE
USES
  DOS;

{+B}
CONST
  NoZeroAttr  : BOOLEAN = FALSE;
  {If NoZeroAttr is TRUE then SearchEngine & SearchEngineAll
    will not process files whose file attribute is zero}

TYPE
  FullNameStr = STRING[12];
  ProcType    = PROCEDURE(VAR s:SearchRec; p:PathStr);
{+E}

 {searchRec   =
    RECORD   -- defined in DOS unit --
      fill : ARRAY[1..21]OF BYTE; -- used by DOS.
      attr : BYTE;                -- file attribute: ReadOnly, Hidden, etc.
      time : LongINT;             -- packed date & time
      size : LongINT;             -- file size in bytes
      name : STRING[12];          -- file name
    END;}

  {Directory Search routines}

{}PROCEDURE SearchEngine(
     mask    : PathStr;     {is the directory mask}
     attr    : BYTE;        {special files to include}
     proc    : ProcType;    {procedure to handle search records}
 VAR errCode : BYTE);       {Dos error value}

{}PROCEDURE SearchEngineAll(
     path    : PathStr;     {directory mask}
     mask    : FullNameStr; {file name mask}
     attr    : BYTE;        {file attribute: ReadOnly, Hidden, etc.}
     proc    : ProcType;    {procedure to handle search records}
 VAR errCode : BYTE);       {Dos error value}

{}FUNCTION  GoodDirectory(s: SearchRec):BOOLEAN;

{}PROCEDURE MatchFirst(Path:STRING; Attr:WORD; VAR s:SearchRec);
{}PROCEDURE MatchNext(VAR s:SearchRec);


  {Other routines}

{}FUNCTION  AddFilePath(Mask : STRING; FName : FullNameStr) : STRING;
{}FUNCTION  AddWildCard(Mask : STRING) : STRING;
{}FUNCTION  CompleteFileName(NAME : STRING) : STRING;
{}FUNCTION  CompletePath(Path : STRING) : STRING;
{}FUNCTION  HasWildCards(Mask : STRING) : BOOLEAN;
{}FUNCTION  MultiDirs(VAR Mask : STRING) : BOOLEAN;
{}FUNCTION  MetaErrorMsg(errCode: BYTE):STRING;
{}FUNCTION  SafeChDir(Path : STRING) : WORD;

     {====================================================================}

IMPLEMENTATION
{$DEFINE UseTPcode} {If defined, uses Turbo Power Software's UNITs}
{$IFDEF UseTPcode}
USES
  TPString;
{$ENDIF}

CONST
  DosDelimSet : SET OF CHAR  = ['\',':',#0];

VAR
  engineMask  : FullNameStr;
  engineAttr  : BYTE;
  engineProc  : ProcType;
  engineCode  : BYTE;


{$IFDEF UseTPcode}
{}FUNCTION AddBackSlash(DirName : STRING) : STRING;
{---------------------------------------------------------------------------
  Purpose     - Add a default backslash to a directory name
  Remarks     - Added to emulate function with same name in Turbo-Power
                TPSTRING unit.
---------------------------------------------------------------------------}
  BEGIN
    IF DirName[Length(DirName)] IN DosDelimSet THEN
      AddBackSlash := DirName
    ELSE
      AddBackSlash := DirName+'\';
{}END {AddBackSlash};




{}FUNCTION  JustFileName(path : PathStr):STRING;
{---------------------------------------------------------------------------
  Purpose     - Return just the file name & extension.
  Remarks     - Added to emulate function with same name in Turbo-Power
                TPSTRING unit.
---------------------------------------------------------------------------}
  VAR
    p    : BYTE;
    dir  : DirStr;
    nm   : NameStr;
    ext  : ExtStr;
  BEGIN
    FSplit(path,dir,nm,ext);
    JustFileName := nm+ext;
{}END {JustFileName};




{}FUNCTION  JustPathName(path : PathStr):STRING;
{---------------------------------------------------------------------------
  Purpose     - Return just the file name & extension.
  Remarks     - Added to emulate function with same name in Turbo-Power
                TPSTRING unit.
---------------------------------------------------------------------------}
  VAR
    p    : BYTE;
    dir  : DirStr;
    nm   : NameStr;
    ext  : ExtStr;
  BEGIN
    FSplit(path,dir,nm,ext);
    JustPathName := dir;
{}END {JustPathName};




{$ENDIF}


{}FUNCTION AddFilePath(Mask : STRING; FName : FullNameStr) : STRING;
{+H
---------------------------------------------------------------------------
  Purpose     - Concatenate a pathmask and filename

  Declaration - function AddFilePath(Mask:STRING; FName:FullNameStr):STRING;

  Remarks     - Concatenates a pathMask with a file name adding a backslash
                between the two strings if required.  Mask is a string-type
                expression of the DOS path.  Fname is a string-type
                expression containing the file name and extension.
---------------------------------------------------------------------------}
  BEGIN
    AddFilePath := AddBackSlash(Mask)+FName;
{}END {AddFilePath};




{}FUNCTION AddWildCard(Mask : STRING) : STRING;
{+H
---------------------------------------------------------------------------
  Purpose     - Add a default wild card to Mask if it needs it

  Declaration - function AddWildCard(Mask : STRING) : STRING;

  Remarks     - Append a general wild card file specification '*.*' to the
                path mask, if the mask does not already contain a wild card
                specification.
---------------------------------------------------------------------------}
  BEGIN
    IF HasWildCards(Mask) THEN
      AddWildCard := Mask
    ELSE
      AddWildCard := AddFilePath(Mask, '*.*');
{}END {AddWildCard};




{}FUNCTION HasWildCards(Mask : STRING) : BOOLEAN;
{+H
---------------------------------------------------------------------------
  Purpose     - Return true if Mask has Meta wildcards

  Declaration - function HasWildCards(Mask : STRING) : BOOLEAN;

  Remarks     - Returns true if Mask contains either a DOS or Unix style
                wild card specification.
---------------------------------------------------------------------------}
  BEGIN
    Mask := JustFilename(Mask);
    HasWildCards := (Pos('*', Mask) <> 0) OR
                    (Pos('?', Mask) <> 0) OR
                    (Pos('[', Mask) <> 0);
{}END {HasWildCards};




{}FUNCTION CompletePath(Path : STRING) : STRING;
{+H
---------------------------------------------------------------------------
  Purpose     - Convert a potentially relative path into a complete one

  Declaration - function CompletePath(Path : STRING) : STRING;

  Remarks     - Path is a string type expression that contains a (potentially)
                relative DOS path name.  The result is a fully qualified path
                name.
---------------------------------------------------------------------------}
  VAR
    ColPos    : BYTE;
    DrNum: BYTE;
    i    : WORD;
    SaveDir   : PathStr;
    CurDir    : PathStr;
  BEGIN
    GetDir(0, CurDir);            {get current directory, default drive}
    ColPos := Pos(':', Path);     {find colon if drive specified}

              {Get current directory on specified drive}

    IF ColPos > 1 THEN BEGIN
      DrNum := Pos(UpCase(Path[ColPos-1]), 'ABCDEFGHIJKLMNOPQRSTUVWXYZ');
      IF DrNum = 0 THEN
        ColPos := 0
      ELSE
        GetDir(DrNum, SaveDir);   {get current directory on specified drive}
    END {IF};
    {$I-}
    ChDir(Path);                  {change to specified path}
    IF IOresult = 0 THEN BEGIN    {if no error then}
      GetDir(0, Path);            {get the directory that is now current}

              {Restore current directory on other drive}

      IF ColPos > 1 THEN BEGIN
        ChDir(SaveDir);           {change to the previously current directory on specified drive}
        i := IOresult;            {Watch out! ChDir may set IoResult}
      END {IF};
    END {IF};
    {$I+}
    ChDir(CurDir);                {change back to original directory}
    CompletePath := Path;
{}END {CompletePath};




{}FUNCTION SafeChDir(Path : STRING) : WORD;
{---------------------------------------------------------------------------
  Purpose     - Change to a directory, restoring current directory if error
---------------------------------------------------------------------------}
  VAR
    Status    : WORD;
    CurDir    : PathStr;
  BEGIN
    GetDir(0, CurDir);            {get current directory}
    ChDir(Path);                  {change to desired directory}
    Status := IOresult;           {did an error occur?}

    IF Status <> 0 THEN
           {if so then}
      ChDir(CurDir);              {  change back to original directory}
    SafeChDir := Status;          {return error status}
{}END {SafeChDir};




{}FUNCTION CompleteFileName(NAME : STRING) : STRING;
{+H
---------------------------------------------------------------------------
  Purpose     - Convert a potentially relative file name into a complete one

  Declaration - function CompleteFileName(NAME : STRING) : STRING;
---------------------------------------------------------------------------}
  VAR
    JustName  : FullNameStr;
  BEGIN
    JustName := JustFilename(NAME);
    NAME     := CompletePath(JustPathname(NAME));
    CompleteFileName := AddFilePath(NAME, JustName);
{}END {CompleteFileName};




{}FUNCTION MultiDirs(VAR Mask : STRING) : BOOLEAN;
{+H
---------------------------------------------------------------------------
  Purpose     - Return true if Mask could specify more than 1 directory

  Declaration - function MultiDirs(VAR Mask : STRING) : BOOLEAN;
---------------------------------------------------------------------------}
  BEGIN
    MultiDirs := (Length(Mask) = 0) OR HasWildCards(Mask);
{}END {MultiDirs};




              {-------------------------------------------------
              {------------- UNIX style wild cards -------------
              {-------------------------------------------------}

CONST
  MaxState    = 11;
  {$IFDEF DOSCOMPAT}
  AllChars    : SET OF CHAR  = ['A'..'Z','0'..'9','#'..')',#96,'~','{','}','_'];
  {$ELSE}
  AllChars    : SET OF CHAR  = ['A'..'Z','0'..'9','#'..')',#96,'~','{','}','_','.'];
  {$ENDIF}
  EmptySet    : SET OF CHAR  = [];
CONST
  Mrange = '-';  {indicates range in set, ie. [a-z]}
  Mliteral    = '\';  {indicates literal follows, ie. \?}
  MnotSet= '^';  {indicates every thing but, ie. [^a-z]}

TYPE
  Str12  = STRING[12];
  CharSet= SET OF CHAR;
  states = 0..MaxState;
  metaCharType= (null,tClosure,tCCl,tLitChar);
  metaCharRec = RECORD
    _set   : CharSet;
    _next  : States;
    _type  : MetaCharType;
  END {RECORD};

VAR
  arcs   : ARRAY[0..MaxState] OF MetaCharRec;
  lastState   : states;



{}PROCEDURE MakeAutomaton(metaCharExpr:STRING);
{---------------------------------------------------------------------------
  Purpose     - Convert a meta character expression to a finite state machine.
  Revised     - 1991.0628 (KSB) Fixed DoRange.
---------------------------------------------------------------------------}
  VAR
    i    : WORD;
    L    : BYTE ABSOLUTE metaCharExpr;

{}{}PROCEDURE GetNextMetaChar(VAR m:MetaCharType; VAR ms:CharSet);
    VAR
      c  : CHAR;
      IsIn    : BOOLEAN;

{}{}{}PROCEDURE Include(c:CHAR);
      BEGIN
        CASE IsIn OF
          TRUE : ms := ms + [c];
          ELSE
            ms := ms - [c];
        END {CASE};
{}{}{}END {Include};


{}{}{}PROCEDURE GetChar(VAR c:CHAR);
      BEGIN
        IF i <= L THEN BEGIN
          c := UpCase(metaCharExpr[i]);
          Inc(i);
          {Write(c);}
        END ELSE
          c := #0;
{}{}{}END {GetChar};


{}{}{}PROCEDURE DoRange;
{---------------------------------------------------------------------------
  Revised     - 1991.0628 (KSB) Changed exit loop if i>L rather than i=L.
---------------------------------------------------------------------------}
      VAR
        c1,c2 : CHAR;
      BEGIN
        m := tCCl;
        c1:= #0;

        GetChar(c);
        IF c = MnotSet THEN BEGIN
          GetChar(c);
          IsIn := FALSE;
          ms   := AllChars;
        END ELSE
          IsIn := TRUE;

        REPEAT
          CASE c OF
            Mrange :
            IF c1<>#0 THEN BEGIN
              IF metaCharExpr[Pred(Pred(i))] < metaCharExpr[i] THEN BEGIN
                FOR c2 := Succ(c1) TO UpCase(metaCharExpr[i]) DO
                  Include(c2);
                GetChar(c);
              END ELSE BEGIN
                GetChar(c);
                Include(c);
              END {FOR};
            END ELSE
              Include(c);

            Mliteral :
            BEGIN
              GetChar(c);
              Include(c);
              c := ' ';
            END {BEGIN};

            ']' :;

            ELSE BEGIN
              Include(c);
              c1 := c;
            END {BEGIN};

          END {CASE};

          GetChar(c);
        UNTIL (c=']') OR (i>L);
{}{}{}END {DoRange};


    BEGIN {-- GetNextMetaChar --}
      ms := EmptySet;
      m  := Null;
      GetChar(c);

      CASE c OF
        '*' :
        BEGIN
          m := tClosure;
          ms:= AllChars;
        END {BEGIN};

        '[' : DoRange;

        '.' :
        BEGIN
          m  := tLitChar;
          ms := [c];
        END {BEGIN};

        ELSE
        IF c IN AllChars THEN BEGIN
          m  := tLitChar;
          ms := [c];
        END ELSE
        IF c = '?' THEN BEGIN
          m  := tLitChar;
          ms := AllChars;
        END ELSE
          Exit;
      END {CASE};
{}{}END {GetNextMetaChar};


  VAR
    n    : WORD;
    m    : MetaCharType;
    mset : CharSet;
  BEGIN
    FillChar(arcs,SizeOf(arcs),0);

    n := 0;
    i := 1;
    m := tLitChar;

    WHILE (i <= L) AND (m<>Null) DO BEGIN
      GetNextMetaChar(m,mset);
      CASE m OF
        tClosure :
        BEGIN
          WITH arcs[n] DO BEGIN
            _set  := AllChars;
            Inc(n);
            _next := n;
            _type := m;
          END {WITH};

          GetNextMetaChar(m,mset);
          IF (m<>Null) THEN
            WITH arcs[n] DO BEGIN
              _set  := mset;
              _next := Succ(n);
              _type := m;
                {WriteLn(n,' tClosure set2=EmptySet ',arcs[n][2]._set=EmptySet);}
            END {WITH};
        END {BEGIN};

        Null : ;

        ELSE
          WITH arcs[n] DO BEGIN
            _set  := mset;
            _next := Succ(n);
            _type := m;
            {
            CASE m OF
             tCCl    : WriteLn(n,' tCCl     set1=EmptySet ',arcs[n][1]._set=EmptySet);
             tLitChar: WriteLn(n,' tLitChar set1=EmptySet ',arcs[n][1]._set=EmptySet);
            END;
            }
          END {WITH};
      END {CASE};

      Inc(n);
    END {WHILE};

    IF m = Null THEN
      Dec(n);
    lastState := n;
{}END {MakeAutomaton};




{}FUNCTION  IsAcceptable(s:Str12):BOOLEAN;
{---------------------------------------------------------------------------
  Purpose     - Determines if S fits the pattern established in the ARCS.
---------------------------------------------------------------------------}
  VAR
    L    : BYTE ABSOLUTE s;

{}{}FUNCTION Amatch(offset:INTEGER; patNdx:WORD):INTEGER;
{---------------------------------------------------------------------------
  Purpose     - Look for a match of pattern arcs starting at PATNDX with
                S[OFFSET].  Return the last position that matched.
---------------------------------------------------------------------------}

{}{}{}FUNCTION Omatch(VAR i:INTEGER; patNdx:WORD):BOOLEAN;
{---------------------------------------------------------------------------
  Purpose     - Match one pattern element at pattern pointed to by PATNDX
                and S[I].
---------------------------------------------------------------------------}
      VAR
        advance    : INTEGER;
        ttok  : MetaCharType;
        c: CHAR;
      BEGIN
        advance := -1;
        ttok    := arcs[patNdx]._type;
        IF i <= L THEN BEGIN
          c := UpCase(s[i]);
          CASE ttok OF
            tLitChar :
            IF c IN arcs[patNdx]._set THEN
              advance := 1;
            tCCL     :
            IF c IN arcs[patNdx]._set THEN
              advance := 1;
            tClosure :
            IF c IN arcs[patNdx]._set THEN
              advance := 1;
          END {CASE};
        END {IF};

        IF advance >= 0 THEN BEGIN
          Omatch := TRUE;
          i := i + advance;
        END ELSE
          Omatch := FALSE;
{}{}{}END {Omatch};


    VAR
      done    : BOOLEAN;
      j,k: WORD;
      i  : INTEGER;
      ttok    : MetaCharType;
    BEGIN {--- Amatch ---}
      done := FALSE;
      j    := patNdx;

      WHILE NOT(done) AND (j<=lastState) DO BEGIN
        ttok := arcs[j]._type;

        IF (ttok = Null) AND (offset > L) THEN
          done := TRUE
        ELSE
        IF ttok = tClosure THEN BEGIN
          i := offset;
          WHILE NOT(done) AND (i<=L) DO BEGIN    {match as many as possible}
            IF NOT(Omatch(i,j)) THEN
              done := TRUE;
          END {WHILE};

          done := FALSE;
          WHILE NOT(done) AND (i >= offset) DO BEGIN
            k := Amatch(i,Succ(j));              {i is location of non-match}
            IF k > 0 THEN
              done := TRUE
            ELSE
              Dec(i);                            {shrink closure by one after each failure}
          END {WHILE};

          offset := k;
          done   := TRUE;
        END ELSE
        IF NOT (Omatch(offset,j)) THEN BEGIN
          offset := 0;
          done   := TRUE;
        END ELSE BEGIN
          Inc(j);
        END {BEGIN};
      END {WHILE};
      Amatch := offset;
{}{}END {Amatch};


  BEGIN {--- IsAcceptable ---}
    IsAcceptable := (Amatch(1,0) > 0);
{}END {IsAcceptable};




{}PROCEDURE AddDot(VAR s:SearchRec);
{---------------------------------------------------------------------------
  Purpose     - Terminate file names with no extension with a "."
---------------------------------------------------------------------------}
  BEGIN
    IF (Pos('.',s.NAME) = 0) AND        {no "."}
       (((s.attr SHR 4)AND 1)<>1) THEN
  {not a Directory}
      s.NAME := s.NAME+'.';
{}END {AddDot};




{}FUNCTION  DosOnlyPattern(s:STRING):BOOLEAN;
{---------------------------------------------------------------------------
  Purpose     - Determine if S can be handled by DOS's pattern matching.
  Remarks     - DOS can't handle character classes, ie., [a-x] or [^1-3].
                nor can it handle correctly a non-final closure, ie., AB*C
---------------------------------------------------------------------------}
{}{}FUNCTION DosOk(s:STRING):BOOLEAN;
    VAR
      L  : BYTE ABSOLUTE s;
    BEGIN
      IF Pos('*',s) > 0 THEN
        DosOk := Succ(Pos('*',s)) > L
      ELSE
        DosOk := TRUE;
{}{}END {DosOk};


  VAR
    fp   : STRING;
    fn   : STRING;
    fe   : STRING;
    Ln   : BYTE ABSOLUTE fn;
    Le   : BYTE ABSOLUTE fe;
    i    : WORD;
    ok   : BOOLEAN;
  BEGIN
    IF (Pos('[',s) > 0) THEN
            {character class found}
      ok := FALSE
    ELSE
    IF (Pos('*',s) > 0) THEN BEGIN      {closure}
      FSplit(s,fp,fn,fe);
      ok := DosOk(fn) AND DosOk(fe);
    END ELSE
      ok := TRUE;

    DosOnlyPattern := ok;
{}END {DosOnlyPattern};




{}PROCEDURE MatchFirst(Path:STRING; Attr:WORD; VAR s:SearchRec);
{+H
---------------------------------------------------------------------------
  Purpose     - Searches the specified (or current) directory for the first
                entry matching the specified name and set of attributes.
                Similar to DOS.FindFirst, but with the ability to handle
                Unix-style wild cards.

  Declaration - procedure MatchFirst(Path:STRING; Attr:WORD; VAR s:SearchRec);

  Remarks     - Path is the directory mask (for example '*.*'). The Attr
                parameter specifies the special files to include (in addition
                to all normal files).

                The result of the directory search is returned in the search
                record.

                Errors are reported in DosError; possible error codes are 3
                (path not found) and 18 (no more files).

  See also    - MatchNext.
---------------------------------------------------------------------------}
  BEGIN
    MakeAutomaton(JustFileName(Path));
    FindFirst(AddBackSlash(JustPathName(Path))+'*.*',Attr,s);
    AddDot(s);

    IF dosError = 0 THEN
      IF IsAcceptable(s.NAME) THEN
        Exit
      ELSE
        WHILE dosError=0 DO BEGIN
          FindNext(s);
          AddDot(s);
          IF IsAcceptable(s.NAME) THEN
            Exit;
        END {WHILE};
{}END {MatchFirst};




{}PROCEDURE MatchNext(VAR s:SearchRec);
{+H
---------------------------------------------------------------------------
  Purpose     - Returns the next entry that matches the name and attributes
                specified in a previous call to MatchFirst.  Similar to
                DOS.FindNext, but can handle Unix-style wild cards.

  Declaration - procedure MatchNext(VAR s:SearchRec);

  Remarks     - The search record must be the same search record passed to
                MatchFirst.  Errors are reported in DosError; the only
                possible error is 18 (no more files).

  See also    - MatchFirst.
---------------------------------------------------------------------------}
  BEGIN
    WHILE dosError = 0 DO BEGIN
      FindNext(s);
      AddDot(s);
      IF IsAcceptable(s.NAME) THEN
        Exit;
    END {WHILE};
{}END {MatchNext};




              {--------------------- End of --------------------
              {------------- UNIX style wild cards -------------
              {-------------------------------------------------}


{}PROCEDURE SearchEngine(
       mask    : PathStr;
       attr    : BYTE;
       proc    : ProcType;
   VAR errCode : BYTE);
{+H
---------------------------------------------------------------------------
  Purpose     - Search for files within a single sub-directory.

  Declaration - procedure SearchEngine(mask:PathStr; attr:BYTE; proc:ProcType;
                  VAR errCode:BYTE);

  Remarks     - Mask is the directory mask (for example, 'C:\*.*') specifing
                the directory and the file (with optional wildcard) to be
                searched.

                The Attr parameter specifies the special files to include.
                If NoZeroAttr is True then normal files (attribute zero) are
                excluded.

                The Proc parameter is a procedure-type that is to be applied
                to any valid files found.

                The errCode parameter returns the status of the operation
                where 0 indicates success.

  See also    - SearchEngineAll.
---------------------------------------------------------------------------}
  VAR
    s    : SearchRec;
    p    : PathStr;
    useDos    : BOOLEAN;

{}{}FUNCTION AttribMatch:BOOLEAN;
    VAR
      Lattr   : WORD;
      Fattr   : WORD;
      i  : WORD;
    BEGIN
      AttribMatch := TRUE;
      IF (attr > 0) AND (attr = s.attr) THEN
        Exit;

      Lattr := attr;
      Fattr := s.attr;
      FOR i := 1 TO 6 DO BEGIN
        IF ((Lattr AND $01) = 1) AND ((Fattr AND $01) = 1) THEN
          Exit;
        Lattr := Lattr SHR 1;
        Fattr := Fattr SHR 1;
      END {FOR};
      AttribMatch := FALSE;
{}{}END {AttribMatch};


  BEGIN
    p := JustPathname(mask);

    useDos := DosOnlyPattern(mask);

    IF useDOS THEN
      FindFirst(mask,attr,s)
    ELSE
      MatchFirst(mask,attr,s);

    IF dosError <> 0 THEN BEGIN
      errCode := dosError;
      Exit;
    END {IF};

    WHILE dosError = 0 DO BEGIN
      IF NoZeroAttr THEN BEGIN
        IF AttribMatch THEN
          Proc(s,p);
      END ELSE
        Proc(s,p);                      {process all files}

      IF useDOS THEN
        FindNext(s)
      ELSE
        MatchNext(s);
    END {WHILE};
    IF dosError = 18 THEN
      errCode := 0
    ELSE
      errCode := dosError;
{}END {SearchEngine};




{}FUNCTION GoodDirectory(s : SearchRec):BOOLEAN;
{+H
---------------------------------------------------------------------------
  Purpose     - Returns TRUE if the parameter S is a full subdirectory
                specification and not "." or "..".

  Declaration - function GoodDirectory(s : SearchRec):BOOLEAN;
---------------------------------------------------------------------------}
  BEGIN
    GoodDirectory := (s.NAME <> '.') AND (s.NAME <> '..') AND
                     (s.Attr AND Directory = Directory);
{}END {GoodDirectory};




{}PROCEDURE SearchOneDir(VAR s:SearchRec; p:PathStr);
{---------------------------------------------------------------------------
  Purpose     - Recursively calls SearchEngine to examine subdirectories of
                the current directory.
---------------------------------------------------------------------------}
  BEGIN
    IF GoodDirectory(s) THEN BEGIN
      p := AddBackSlash(p) + s.NAME;
{$IFDEF debug} WriteLn('1.SearchOneDir -> ',p,' -> ',p+'\'+engineMask); {$ENDIF}
      SearchEngine(AddBackSlash(p)+engineMask,engineAttr,engineProc,engineCode);
      SearchEngine(AddBackSlash(p)+'*.*',Directory OR Archive,SearchOneDir,engineCode);
    END {IF};
{}END {SearchOneDir};




{}PROCEDURE SearchEngineAll(
       path    : PathStr;
       mask    : FullNameStr;
       attr    : BYTE;
       proc:ProcType;VAR errCode : BYTE);
{+H
---------------------------------------------------------------------------
  Purpose     - Uses SearchEngine to search all subdirectories of the current
                directory.

  Declaration - procedure SearchEngineAll(path:PathStr; mask:FullNameStr;
                  attr:BYTE; proc:ProcType; VAR errCode:BYTE);

  Remarks     - Path is the directory mask (for example, 'C:\') specifing
                the directory at which to start the search.  Mask specifies
                a file name or (DOS or Unix) wild-card file specification
                to look for.

                The Attr parameter specifies the special files to include.
                If NoZeroAttr is True then normal files (attribute zero) are
                excluded.

                The Proc parameter is a procedure-type that is to be applied
                to any valid files found.

                The errCode parameter returns the status of the operation
                where 0 indicates success.

  See also    - SearchEngine.
---------------------------------------------------------------------------}
  BEGIN
{$IFDEF debug}
    WriteLn('SearchEngineAll(path:',path,'; mask:',mask,'; attr:',attr,
      ';
      proc:',Seg(proc),':',Ofs(proc),'; errCode:',errCode,');');
{$ENDIF}
    engineMask := mask;
    engineProc := proc;
    engineAttr := attr;
    SearchEngine(path+mask,attr,proc,errCode);

    SearchEngine(path+'*.*',Directory OR Attr,SearchOneDir,errCode);

    errCode := engineCode;
{}END {SearchEngineAll};




{}FUNCTION  MetaErrorMsg(errCode:BYTE):STRING;
{+H
---------------------------------------------------------------------------
  Purpose     - Provides a description for a error code value.

  Declaration - function  MetaErrorMsg(errCode:BYTE):STRING;
---------------------------------------------------------------------------}
  VAR
    s    : STRING[5];
  BEGIN
    CASE errCode OF
      0 : MetaErrorMsg := '';   { -- no error }
      2 : MetaErrorMsg := 'File not found';
      3 : MetaErrorMsg := 'Path not found';
      5 : MetaErrorMsg := 'Access denied';
      6 : MetaErrorMsg := 'Invalid handle';
      8 : MetaErrorMsg := 'Not enough memory';
      10 : MetaErrorMsg := 'Invalid environment';
      11 : MetaErrorMsg := 'Invalid format';
      18 : MetaErrorMsg := '';   { -- no more files }
      ELSE
        Str(errCode,s);
      MetaErrorMsg := 'ERROR #'+s;
    END {CASE};
{}END {MetaErrorMsg};




BEGIN
END {BEGIN}.
