{$A+,B-,D+,E+,F+,G+,I+,L+,N-,O-,P-,Q-,R+,S+,T-,V-,X-,Y-}
{$M 16384,0,655360}

PROGRAM Pilot;
{
  Reference   - (SLAM) 10.13  Example 10-1 Pilot Ejection; p.374

  Remarks     - (Paraphrased from SLAM)

     Pilot Ejection model [CONTROL DATA MIMIC -- A Digital Simulation
     Language Reference Manual, Publication No. 44610400, Revision D,
     Control Data Corporation, Minneapolis, Minn., 1970]

     For this simulation assume an aircraft flying straight and level
     at constant velocity with no atmospheric turbulance.  When
     activated, the pilot ejection system causes the pilot and seat to
     travel along the ejection rails, which are canted backwards from
     the vertical, at a specified velocity.  Once a known vertical
     distance is traveled, the pilot and seat are considered separated
     from the aircraft.  At this point the force of gravity and
     atmospheric drag become important factors in determining whether
     or not the pilot will clear the top of the tail, located 60 feet
     behind and 12 feet above the cockpit.


     For the simulation use a aircraft-body-centered coordinate system
     +Y is vertical up, +X is forward through the nose, +Z out the
     right wing.

  Revised     - 1993.0804 (KSB) Reformatted with KPPF.
}
USES
 Ksim_Com,
 Ksim_A,
 Ksim_B;

TYPE
  pilotModelType = RECORD
    cD   : REAL; {coefficient of Drag}
    G    : REAL; {acceleration of gravity}
    rho  : REAL; {air density}
    xd   : REAL; {pilot velocity, x direction}
    theta: REAL; {launch angle}

    vAirplane    : REAL; {velocity of airplane}
    vEjection    : REAL; {pilot ejection velocity}
    pilotMass    : REAL; {mass of pilot}
    pilotSurface : REAL; {surface area of pilot}
    Yrelease     : REAL; {vertical distance above ejection point}
  END;

VAR
  pem : PilotModelType;



{}PROCEDURE Event(n:INTEGER);
  BEGIN

    {handle events}
    WITH Ksim0,Ksim2 DO
    CASE n OF
      1 : Ksim1.stop  := TRUE;  {stop the simulation}
      2 : Ksim5.xx[1] :=  1.;   {trigger change of state equations}
    END {CASE};
{}END {Event};



{}PROCEDURE Intlc;
  VAR
    atb  : StoreReal;
  BEGIN
    FillChar(atb,SizeOf(atb),0);

    WITH pem DO BEGIN   {pilot ejection model}
      cD    :=  1.0;         {ND}
      G     := 32.2;         {ft/sec/sec}
      rho   :=  2.3769e-3;   {slug/ft3}
      xd    :=  0.0;         {ft/sec}
      theta := 15.0/57.3;    {degrees --> radians}

      vAirplane    :=900.0;  {ft/sec}
      vEjection    := 40.0;  {ft/sec}
      pilotMass    :=  7.0;  {slugs}
      pilotSurface := 10.0;  {ft2}
      Yrelease     :=  4.0;  {ft}

      atb[1] := vAirplane - vEjection*Sin(theta);
      atb[2] := vEjection*Cos(theta);

    {set initial values}
      Ksim5.xx[1] := 0.;
      Ksim5.ss[3] := Sqrt(Sqr(atb[1]) + Sqr(atb[2]));
      Ksim5.ss[4] := Arctan(atb[2]/atb[1]);
    END;

    {set start and ending times}
    WITH Ksim2 DO BEGIN
      tStart := 0.0;
      tFin   := 4.0;
    END {WITH};

    {set integration controls}
    WITH Ksim4 DO BEGIN
      dtSave := 0.01;        {frequency at which data is recorded}
      dtMax  := 0.01;        {max allowable step size}
      dtMin  := 0.0001;      {min allowable step size}
      neqdt  := 4;           {number of derivative equations}
      neqst  := 0;           {number of state equations}
      aerr   := 0.0;         {absolute error limit}
      rerr   := 0.000005;    {relative error limit}
      depf   := TRUE;        {print msg when a descrete event is released}
    END;

    {set up threshold checking}
    WITH Ksim6 DO BEGIN
      nthres := 3;

      {Release an event#1 when SS[1] exceeds |60|ft
      (pilot has either cleared the tail or is a red spot)}
      WITH svent[1] DO BEGIN
        eventNo  := 1;
        crossVar := _ss;
        varIndex := 1;
        directn  := _neg;    {SS[1] < -60 }
        ThresVar := _const;  {(-60 is a constant), not a XX,SS,or DD var}
        value    := -60.;
       {tolerance:= 0.1;}
      END;

      {Release an event#1 when SS[2] exceeds 30 ft (clears tail with mucho margin)}
      WITH svent[2] DO BEGIN
        eventNo  := 1;
        crossVar := _ss;
        varIndex := 2;
        directn  := _pos;    {SS[2] > 30}
        ThresVar := _const;  {(30 is a constant), not a XX,SS,or DD var}
        value    :=  30.;
       {tolerance:= 0.1;}
      END;

      {Release an event#2 when SS[2] exceeds 4 ft (clears cockpit)}
      WITH svent[3] DO BEGIN
        eventNo  := 2;
        crossVar := _ss;
        varIndex := 2;
        directn  := _pos;    {SS[2] > 4}
        ThresVar := _const;  {(4 is a constant), not a XX,SS,or DD var}
        value    := pem.Yrelease;
       {tolerance:= 0.1;}
      END;
    END;

    {set up variables to record}
    WITH Ksim7 DO BEGIN
      nvar := 5;
      {vvar[1] is automatically <TNOW>}
      WITH vvar[2] DO BEGIN variable := _ss;  index := 1;  description := 'X-Pos';       END;
      WITH vvar[3] DO BEGIN variable := _ss;  index := 2;  description := 'Y-Pos';       END;
      WITH vvar[4] DO BEGIN variable := _ss;  index := 3;  description := 'Speed';       END;
      WITH vvar[5] DO BEGIN variable := _ss;  index := 4;  description := 'Theta';       END;
    END {WITH};
{}END {Intlc};


{}PROCEDURE State;
{
  Purpose     - Compute the current value of each state variable or its
                derivative.
}
  BEGIN
    WITH Ksim5 DO BEGIN
      dd[1] := ss[3]*Cos(ss[4]) - pem.vAirplane;
      dd[2] := ss[3]*Sin(ss[4]);

      IF xx[1] < 1.0 THEN BEGIN
        dd[3] := 0.0;
        dd[4] := 0.0;
      END ELSE WITH pem DO BEGIN
        xd     := 0.5*rho*cd*pilotSurface*Sqr(ss[3]);  {pCdSv}

        dd[3]  := -xd/pilotMass - G*Sin(ss[4]);
        dd[4]  := -pem.g*Cos(ss[4])/ss[3];
      END;
    END;
{}END {State};


BEGIN
  EventPtr := @Event;        {link in Event handler}
  IntlcPtr := @Intlc;        {  and initialization}
  StatePtr := @State;        {  and state, differential equations}
  DfltPtr  := @Ksim_B.Dflt;  {  and menu system}

  WriteLn(MemAvail);

  cfgName      := 'PILOT.CFG';
  RprtFileName := 'PILOT.RPT';
  DataFileName := 'PILOT.DAT';
  Ksim;                      {execute the simulation}

  WriteLn(MemAvail);
END {Test}.