{$A+}
{$B-}
{$D+}
{$E+}
{$F+}  {must be +}
{$I+}
{$L+}
{$N-}
{$O-}
{$R+}
{$S+}
{$V-}

UNIT Ksim_A;
{+H
---------------------------------------------------------------------------
  Version     - 5.01

  File        - KSIM_A.PAS

  Copyright   - None. Public Domain.

  Author      - Keith S. Brown (except where otherwise noted)
                713-483-8952
                Surface mail:              Email:(brown@smd4.jsc.nasa.gov)
                  K.Brown                     or (brownk@samnet.jsc.nasa.gov)
                  Code:NASA/JSC/ES64
                  Houston, TX 77058 (USA)  Voice:(713)483-8952

  Purpose     - Performs actual continuous & discrete event simulation.

  Remarks     - KSIM is based on SLAM (see references) and follows SLAM
                conventions and syntax rather closely.

  o  For a minimal model that integrates differental equations, the
     following modules will be required:
     1)  a main program  (sets output files array sizes, calls KSIM)
     2)  a STATE subroutine which has all differental equations
         written in canonical form as state equations, ie. given the
         second order differental equation in X:

            Xdd(t) = A Xd(t) + B X(t) + C
                                              then
            ss(1) = X(t)
            dd(1) = Xd(t) = ss(2)
            dd(2) = Xdd(t)
                                              which would be coded in
                                              subroutine STATE as:
            dd(1) = ss(2)
            dd(2) = A * ss(2) + B * ss(1) + C
                                              where dd(.) and ss(.)
                                              are variables provided
                                              by KSIM.

     3) an initial conditions routine which can either be UINPUT or
        INTLC.  Normally values are initialized in INTLC and UINPUT
        is written as a menu to change the defaults established in
        INTLC.

  Requires    - TPSTRING.PAS  - Turbo Power Software v:5.x or later
                TPDATE.PAS    - Turbo Power Software v:5.x or later
                KSIM_COM.PAS  - (part of KSIM package)
                KSIM_STR.PAS  - (part of KSIM package)
                KSIMDATE.PAS  - (part of KSIM package)

  Language    - Borland International's Turbo Pascal 5.5 or later.

  References  - Introduction to Simulation and SLAM.  A. Alan B. Pritsker
                and Claude Dennis Pegden (c) 1979 by A. Alan B. Pritsker.
                Halsted Press, a Division of John Wiley & Sons, Inc. NY
                QA76.9.C65P74  ISBN 0-470-26588-4.
                See chapters 7:Discrete Event Simulation Using SLAM;
                            10:Continuous Models

  Revised     - 1991.1030 (KSB) Converted version 4.30 from Harris Fortran to
                Borland International's Turbo Pascal 5.5.
              - 1992.0122 (KSB) Added ScheduleOneAttribute procedure.
              - 1992.0129 (KSB) Mod'd to do threshold checking on XX.
              - 1993.0804 (KSB) Reformatted with KPPF.
---------------------------------------------------------------------------
  COMPONENT PROCEDURES PURPOSE:
     A. EVENT          User provided Event processing procedure.
     B. INTLC          User provided Initialization procedure.
     C. UINPUT         User provided menuing system for changing values of user defined variables.
     D. UPRNTIN        User provided procedure for printing initial values of user defined variables.
     E. UREPORT        User provided procedure for printing final values of user defined variables.
     F. STATE          User provided procedure defining differential or difference equations.

     1. CONTROL        CONTROL DISCRETE & CONTINUOUS SIMULATION.
     2. DFLT           INTERACTIVE CONTROL OF DEFAULT VALUES.
     3. ERCODE         SUPPLY ERROR MESSAGES.
     4. FNDUMP         DUMP CALENDAR FILE ARRAY.
     5. IINZ           INITIALIZE KSIM VARIABLES.
     6. PRNTIN         PRINT KSIM VARIABLES.
     7. REMOVE         REMOVE EVENT FROM CALENDAR FILE.
     8. REPORT         REPORT KSIM VARIABLE FINAL STATUS.
     9. RKF            RUNGE-KUTTA-FEHLBERG 4-5 INTEGRATOR.
    10. SCHEDUL        SCHEDULE AN EVENT ON THE CALENDAR FILE.
    11. SEC            CHECK FOR STATE EVENT.
    12. SORT           SORT SIMULTANEOUS EVENTS BY PRIORITY.
    13. SSAVE          SAVE STATE VARIABLES FOR PLOTTING.
    14. TUNE           TIME UNTIL NEXT DISCRETE EVENT.
    15. ScheduleOneAttribute
                       Schedule an event with one attribute on the calendar file.
    16. ExamineNext_N_Event
                       Returns the attributes for the next "N" event.
---------------------------------------------------------------------------}
INTERFACE
USES
  KSIM_com;


{}PROCEDURE Ksim;
{}PROCEDURE SCHEDUL( dt:REAL; nvent:INTEGER; atb:StoreReal );
{}PROCEDURE ScheduleOneAttribute(dt:REAL; nvent:INTEGER; a:REAL);
{}PROCEDURE ExamineNext_N_Event(N:INTEGER; VAR t:REAL; VAR atb:StoreReal);

     {====================================================================}

IMPLEMENTATION

USES
  TPstring, TPdate,
  KsimDate, Ksim_str;


TYPE
  oBase   = OBJECT                             {oBase object type}

    DESTRUCTOR  Done; VIRTUAL;
  END {OBJECT};




  listNodePtr = ^oNode;
  oNode    = OBJECT(oBase)                  {abstract linked oList node type}
    Next   : listNodePtr;

    FUNCTION    Prev:listNodePtr;
  END {OBJECT};




  listAction  = PROCEDURE(N:listNodePtr);


  listPtr= ^oList;
  oList  = OBJECT(oBase)                       {circularly linked oList.  last points to last item in oList}
    last      : listNodePtr;
    ilSize    : WORD;

    CONSTRUCTOR Init;
    DESTRUCTOR  Done; VIRTUAL;
    FUNCTION    Size: WORD;
    PROCEDURE   Append(N:listNodePtr);
    PROCEDURE   Delete;
    FUNCTION    Empty : BOOLEAN;
    FUNCTION    First : listNodePtr;
    PROCEDURE   ForEach(action:listAction);
    PROCEDURE   Insert(N:listNodePtr);
    FUNCTION    Next(N:listNodePtr):listNodePtr;
    FUNCTION    Prev(N:listNodePtr):listNodePtr;
    PROCEDURE   Remove(N:listNodePtr);
  END {OBJECT};




              {oBase methods}

{}DESTRUCTOR oBase.Done;
  BEGIN
{}END {Done};




              {oNode methods}

{}FUNCTION oNode.Prev:ListNodePtr;
  VAR
    p    : ListNodePtr;
  BEGIN
    p := @Self;
    WHILE p^.Next <> @Self DO
      p := p^.Next;
    Prev := p;
{}END {Prev};




{}PROCEDURE DelListNode(N:ListNodePtr);
  BEGIN
    Dispose(N,Done);
{}END {DelListNode};




              {oList methods}

{}CONSTRUCTOR oList.Init;
  BEGIN
    Last   := NIL;
    ilSize := 0;
{}END {Init};




{}DESTRUCTOR oList.Done;
  BEGIN
    Delete;
{}END {Done};




{}FUNCTION  oList.Size:WORD;
  BEGIN
    Size := ilSize;
{}END {Size};




{}PROCEDURE oList.Append(N:ListNodePtr);
  BEGIN
    Insert(N);
    Last := N;
{}END {Append};




{}PROCEDURE oList.Delete;
  BEGIN
    ForEach(DelListNode);
    Last := NIL;
{}END {Delete};




{}FUNCTION oList.Empty:BOOLEAN;
  BEGIN
    Empty := Last = NIL;
{}END {Empty};




{}PROCEDURE oList.ForEach(action:ListAction);
  VAR
    p,q  : ListNodePtr;
  BEGIN
    p := First;
    WHILE p <> NIL DO BEGIN
      q := p;
      p := Next(p);
      Action(q);
    END {WHILE};
{}END {ForEach};




{}FUNCTION oList.First:ListNodePtr;
  BEGIN
    IF Last = NIL THEN
      First := NIL
    ELSE
      First := Last^.Next;
{}END {First};




{}PROCEDURE oList.Insert(N:ListNodePtr);
  BEGIN
    IF Last = NIL THEN
      Last := N
    ELSE
      N^.Next := Last^.Next;

    Last^.Next := N;
    Inc(ilSize);
{}END {Insert};




{}FUNCTION oList.Next(N:ListNodePtr):ListNodePtr;
  BEGIN
    IF N = Last THEN
      Next := NIL
    ELSE
      Next := N^.Next;
{}END {Next};




{}FUNCTION oList.Prev(N:ListNodePtr):ListNodePtr;
  BEGIN
    IF N = First THEN
      Prev := NIL
    ELSE
      Prev := N^.Prev;
{}END {Prev};




{}PROCEDURE oList.Remove(N:ListNodePtr);
  VAR
    p    : ListNodePtr;
  BEGIN
    IF Last <> NIL THEN BEGIN
      p := Last;
      WHILE (p^.Next <> N) AND (p^.Next <> Last) DO
        p := p^.Next;

      IF p^.Next = N THEN BEGIN
        p^.Next := N^.Next;
        IF Last = N THEN
          IF p = N THEN
            Last := NIL
          ELSE
            Last := p;
        Dec(ilSize);
      END {IF};
    END {IF};
{}END {Remove};




              {Inherited oList node types}
TYPE
  AtbObjPtr   = ^AtbObj;
  AtbObj = OBJECT(oNode)
    Value     : REAL;

    CONSTRUCTOR Init(r:REAL);
  END {OBJECT};




{}CONSTRUCTOR AtbObj.Init(r:REAL);
  BEGIN
    value := r;
{}END {Init};




TYPE
  CalObjPtr   = ^CalendarObj;
  CalendarObj = OBJECT(oNode)
    eventNo   : INTEGER;
    exeTime   : REAL;
    schTime   : REAL;
    atb       : ListPtr;

    CONSTRUCTOR Init(i:INTEGER; dt:REAL);
    DESTRUCTOR  Done; VIRTUAL;
  END {OBJECT};




{}CONSTRUCTOR CalendarObj.Init(i:INTEGER; dt:REAL);
  BEGIN
    eventNo := i;
    schTime := Ksim2.tnow;
    exeTime := schTime + dt;
    atb     := NIL;
{}END {Init};




{}DESTRUCTOR CalendarObj.Done;
  BEGIN
    eventNo := 0;
    exeTime := 0.;
    schTime := 0.;
    IF atb <> NIL THEN
      atb^.Done;
{}END {Done};




CONST
  big    = 1.0E20;





{}PROCEDURE FarCallUinput;
INLINE($FF/$1E/UinputPtr);

{}PROCEDURE FarCallUprntIn;
INLINE($FF/$1E/UprntInPtr);

{}PROCEDURE FarCallUreport;
INLINE($FF/$1E/UreportPtr);

{}PROCEDURE FarCallIntlc;
INLINE($FF/$1E/IntlcPtr);

{}PROCEDURE FarCallState;
INLINE($FF/$1E/StatePtr);

{}PROCEDURE FarCallEvent(n:INTEGER);
INLINE($FF/$1E/EventPtr);

{}PROCEDURE FarCallDflt;
INLINE($FF/$1E/DfltPtr);



{}PROCEDURE CallUINPUT;
  BEGIN
    IF UinputPtr <> NIL THEN
      FarCallUinput;
{}END {CallUINPUT};




{}PROCEDURE CallUPRNTIN;
  BEGIN
    IF UprntInPtr <> NIL THEN
      FarCallUprntIn;
{}END {CallUPRNTIN};




{}PROCEDURE CallUREPORT;
  BEGIN
    IF UreportPtr <> NIL THEN
      FarCallUreport;
{}END {CallUREPORT};




{}PROCEDURE Intlc;
  BEGIN
    IF IntlcPtr <> NIL THEN
      FarCallIntlc;
{}END {Intlc};




{}PROCEDURE State;
  BEGIN
    IF StatePtr <> NIL THEN
      FarCallState;
{}END {State};




{}PROCEDURE Event(n:INTEGER);
  BEGIN
    IF EventPtr <> NIL THEN
      FarCallEvent(n);
{}END {Event};




{}PROCEDURE Dflt;
  BEGIN
    IF DfltPtr <> NIL THEN
      FarCallDflt;
{}END {Dflt};




{}PROCEDURE RecoverAttributes(n:ListPtr; VAR atb:StoreReal);
{---------------------------------------------------------------------------
  Revised     - 1992.0122 (KSB) Changed from "if count>0" to "if p<>nil" for error.
---------------------------------------------------------------------------}
  VAR
    p,q  : ListNodePtr;
    count: WORD;
  BEGIN
    count := Ksim3.matrib;
    p     := n^.First;
    WHILE (p <> NIL) AND (count>0) DO BEGIN
      q := p;
      p := n^.Next(p);
      atb[count] := AtbObjPtr(q)^.value;
      Dec(count);
    END {WHILE};

    IF p <> NIL THEN BEGIN
      WriteLn(' *** KSIM PROCESSOR ERROR 0007 ***',CRLF,
              '     SUB: RECOVERATTRIBUTES -- TOO MANY ATTRIBUTES');
    END {IF};
{}END {RecoverAttributes};




{}PROCEDURE AtbEntryDump(n:ListNodePtr);
{---------------------------------------------------------------------------
---------------------------------------------------------------------------}
  VAR
    q    : CalObjPtr ABSOLUTE n;
    atb  : StoreReal;
    i,j  : WORD;
  BEGIN
    RecoverAttributes(q^.atb,atb);
    j := 0;
    FOR i := 1 TO Ksim3.matrib DO BEGIN
      Inc(j);
      IF j = 7 THEN BEGIN
        j := 0;
        WriteLn(nprnt);
        Write(nprnt,CharStr(' ',59));
      END {IF};

      Write(nprnt,'  ',atb[i]:8:2);
    END {FOR};
{}END {AtbEntryDump};




{}PROCEDURE CalEntryDump(n:ListNodePtr);
{---------------------------------------------------------------------------
---------------------------------------------------------------------------}
  VAR
    p    : CalObjPtr ABSOLUTE n;
  BEGIN
    Inc(Ksim0.count);
    Write(nprnt, CharStr(' ',10),
          '$',HexPtr(p),' ',
    Ksim0.count:5,' ',p^.exeTime:7:2,' ',p^.schTime:7:2,' ',p^.eventNo:5,
          ' $',HexPtr(p^.next));

    IF p^.atb <> NIL THEN BEGIN
      AtbEntryDump(n);
    END {IF};

    WriteLn(nprnt);
{}END {CalEntryDump};




{}PROCEDURE FnDump;
{--------------------------------------------------------------------------
  Purpose     - Dump the calendar stored in circular linked list.
                                CALENDAR FILE DUMP AT: 99999.9999999
                            X-TIME  S-TIME EVENT            ATTRIBUTES
          ....0 $....:.... xxxx.xx xxxx.xx ..... $....:....  mmmmm.mm  mmmmm.mm ...

  Written     - 1981.1008 (KSB) Original in VAX-11 FORTRAN 77.
  Revised     - 1982.1020 (KSB) Converted to Harris FORTRAN-77.
              - 1991.1028 (KSB) Converted to Turbo pascal
---------------------------------------------------------------------------}
  VAR
    i    : WORD;
  BEGIN
    WITH Ksim0,Ksim2 DO BEGIN
      count := 0;
      WriteLn(nprnt,CRLF,CRLF,
      CharStr(' ',32),'CALENDAR FILE DUMP AT: ',tnow:13:7,CRLF,
      CharStr(' ',11),'THIS ADDR        X-TIME  S-TIME EVENT  NEXT ADDR  ATTRIBUTES',CRLF);

      ListPtr(Ksim3.Calendar)^.ForEach(CalEntryDump);
    END {WITH};
{}END {FnDump};




{}PROCEDURE REPORT;
{---------------------------------------------------------------------------
  Purpose     - Write KSIM Summary Report.

  Written     - 1980.1105 (KSB) Original in VAX-11 Fortran 77
  Revised     - 1981.1208
              - 1982.1020 (KSB) Converted to Harris FORTRAN-77.
              - 1985.0423 (KSB) Updated format 1000 for variable version numbers.
              - 1991.1030 (KSB) Converted to Turbo Pascal.
---------------------------------------------------------------------------}
  VAR
    i    : WORD;
  BEGIN
    WITH Ksim0,Ksim1,Ksim2,Ksim3,Ksim4,Ksim5,Ksim6 DO BEGIN
      WriteLn(nprnt,FormFeed,CRLF,CRLF,
      CharStr(' ',24),'    K S I M   S U M M A R Y   R E P O R T',CRLF,
      CharStr(' ',24),'KSIM VERSION ',version,' / OCT 1991 (MS-DOS 3.x+)',CRLF,CRLF);   {v:2.01}

      WriteLn(nprnt,
      CharStr(' ',10),'SIMULATION PROJECT: ',Pad(nprjct,15),'        BY: ',nname,CRLF,
      CharStr(' ',10),'DATE:               ',DateStr(cDate),CharStr(' ',15),'RUN NUMBER:',nnrns:5,CRLF,
      CharStr(' ',10),'SIM-TIME AT COMPLETION: ',tnow:12:6);

      WriteLn(nprnt,CRLF,CRLF,
      CharStr(' ',10),'NUMBER OF EVENTS SCHEDULED  ',nsched:5,CRLF,
      CharStr(' ',10),'NUMBER OF EVENTS REMAINING  ',ListPtr(calendar)^.Size:5,CRLF,
      CharStr(' ',10),'NUMBER OF EVENTS EXECUTED   ',nxqt:5,CRLF,
      CharStr(' ',10),'NUMBER OF S.EVENTS EXECUTED ',nsvent:5);
{
      ;---For continuous simulations...
}
      IF ( neqtt > 0 ) THEN BEGIN
        WriteLn(nprnt,CRLF,CRLF,
        CharStr(' ',10),'**STATE AND DERIVATIVE VARIABLES***',CRLF,CRLF,
            '              (I)        SS(I)            DD(I)');
        FOR i := 1 TO neqtt DO
          WriteLn(nprnt,CharStr(' ',12),i:4,'     ',ss[i]:12:4,'    ',dd[i]:12:4);
      END {IF};
    END {WITH};
{
      ;---Report on user variables...
}
    CallUREPORT;
{}END {REPORT};




{}PROCEDURE ErCode(nerror:INTEGER);
{---------------------------------------------------------------------------
  Purpose     - Write error messages and close out an aborted simulation.

  Written     - 1981.1008 (KSB) Original in VAX-11 Fortran 77.
  Revised     - 1982.1020 (KSB) Converted to Harris FORTRAN-77
              - 1991.1028 (KSB) Converted to Turbo Pascal.
---------------------------------------------------------------------------}
  BEGIN
    WITH Ksim0,Ksim2,Ksim3 DO BEGIN
      WriteLn(nprnt, CRLF,CRLF,CharStr(' ',44),'K S I M   E R R O R   R E P O R T',CRLF,CRLF);

      WriteLn(nprnt, CharStr(' ',32),'SIMULATION TERMINATED AT ',tnow:12:4,CRLF,CRLF,
      CharStr(' ',32),nsched:5,'  EVENTS SCHEDULED',CRLF,
      CharStr(' ',32),nxqt:5,   '  EVENTS EXECUTED',CRLF,
      CharStr(' ',32),ListPtr(calendar)^.Size:5,  '  EVENTS REMAINING ON CALENDAR',CRLF,CRLF,
      CharStr(' ',32),'MEMORY AVAILABLE FOR CALENDAR       ',MemAvail:5);
      IF calendar <> NIL THEN
        WriteLn(nprnt,
        CharStr(' ',32),'CURRENT NUMBER OF CALENDAR ENTRIES  ',ListPtr(calendar)^.ilsize:5);

      WriteLn(nprnt);

      CASE nerror OF
        0 : WriteLn(nprnt,CharStr(' ',32),'UNDEFINED KSIM PROCESSOR ERROR 1000');

        1 : WriteLn(nprnt,CharStr(' ',32),'KSIM PROCESSOR ERROR  1001 SUB:CONTROL, VAR:nsim NEGATIVE OR ZERO VALUE.');
        2 : WriteLn(nprnt,CharStr(' ',32),'KSIM PROCESSOR ERROR  1002 SUB:SORT, VAR:nsim VALUE GT 100.');
        3 : WriteLn(nprnt,CharStr(' ',32),'KSIM PROCESSOR ERROR  1003 SUB:REMOVE, VAR:INDEX IS NIL.');
        4 : WriteLn(nprnt,CharStr(' ',32),'KSIM PROCESSOR ERROR  1004 SUB:SCHEDUL, VAR:DT NEGATIVE VALUE');
        5 :
        BEGIN
          WriteLn(nprnt,CharStr(' ',32),'KSIM PROCESSOR ERROR  1005 SUB:SCHEDUL, VAR:maxcal');
          WriteLn(nprnt,CharStr(' ',32),'MEMORY EXHAUSTED. NO ROOM FOR EVENT.');
        END {BEGIN};
        6 :
        BEGIN
          WriteLn(nprnt,CharStr(' ',32),'KSIM PROCESSOR ERROR  1006 SUB:SCHEDUL, VAR:maxcal');
          WriteLn(nprnt,CharStr(' ',32),'MEMORY EXHAUSTED. NO ROOM FOR ATTRIBUTES.');
        END {BEGIN};
        7 :
        BEGIN
          WriteLn(nprnt,CharStr(' ',32),'KSIM PROCESSOR ERROR  1007 SUB:REMOVE, VAR:atb');
          WriteLn(nprnt,CharStr(' ',32),'TOO MANY ATTRIBUTES FOR EVENT IN CALENDAR.');
        END {BEGIN};
        8 :
        BEGIN
          WriteLn(nprnt,CharStr(' ',32),'KSIM PROCESSOR ERROR  1008 RprtFileName INVALID');
          WriteLn(nprnt,CharStr(' ',32),'UNABLE TO OPEN FILE: "',RprtFileName,'"');
        END {BEGIN};
        9 :
        BEGIN
          WriteLn(nprnt,CharStr(' ',32),'KSIM PROCESSOR ERROR  1009 DataFileName INVALID');
          WriteLn(nprnt,CharStr(' ',32),'UNABLE TO OPEN FILE: "',DataFileName,'"');
        END {BEGIN};

        10 :
        BEGIN
          WriteLn(nprnt,CharStr(' ',32),'KSIM PROCESSOR ERROR  1010 ILLEGAL STATE EVENT');
          WriteLn(nprnt,CharStr(' ',32),'ONLY DD() OR SS() VARIABLES CAN BE CHECKED FOR THRESHOLDS');
        END {BEGIN};
      END {CASE};
    END {WITH};

    FnDump;
    Report;

    Close(nprnt);
    Close(iplot);
    Halt;
{}END {ErCode};




{}PROCEDURE PRNTIN;
{---------------------------------------------------------------------------
  Purpose     - Write the KSIM echo report.  List KSIM options and initial
                conditions.

  Written     - 1980.1105
  Revised     - 1981.1208 (KSB) Original in VAX-11 Fortran 77
              - 1982.1020 (KSB) Converted to Harris FORTRAN-77.
              - 1983.0303 (KSB) Corrected format 1110
              - 1985.0423 (KSB) Changed Format 1000 to include variable version number.
              - 1991.1029 (KSB) Converted to Turbo Pascal.
              - 1991.1108 (KSB) Corrected indexing error in Record Card section.
---------------------------------------------------------------------------}

  VAR
    i    : WORD;
    vas  : STRING[3];
    dir  : STRING[3];
    Val  : STRING[3];
    vx   : REAL;
    nwc  : LongINT;
  BEGIN
    WriteLn(nprnt,CRLF,CRLF,
    CharStr(' ',24),'    K S I M   E C H O   R E P O R T',CRLF,
    CharStr(' ',24),'KSIM VERSION ',version,' / OCT 1991 (MS-DOS 3.x+)',CRLF,CRLF);

    WITH Ksim0,Ksim1,Ksim2,Ksim3,Ksim4,Ksim5,Ksim6,Ksim7 DO BEGIN
      WriteLn(nprnt,CharStr(' ',10),'SIMULATION PROJECT: ',Pad(nprjct,23) ,'BY: ',nname);
      WriteLn(nprnt,CharStr(' ',10),'DATE ',DateStr(cDate),CharStr(' ',30),'RUN NUMBER:',nnrns,CRLF);
{
      ;---KSIM options...
}
      WriteLn(nprnt,CharStr(' ',10),'GENERAL OPTIONS',CRLF,
      CharStr(' ',16), 'LIST INPUT STATEMENTS:',CharStr(' ',15),YesNo[ilist],CRLF,
      CharStr(' ',16), 'EXECUTE SIMULATION:   ',CharStr(' ',15),YesNo[ixqt],CRLF,
      CharStr(' ',16), 'PRINT SUMMARY REPORT: ',CharStr(' ',15),YesNo[isimrep],CRLF,
      CharStr(' ',16), 'SECONDARY CALENDAR PRIORITY:',CharStr(' ',9),RankImage[lpsym],CRLF);
{
      ;---CONTINOUS CARD options...
}
      IF ( neqtt > 0 ) THEN BEGIN
        WriteLn(nprnt,CharStr(' ',10),'CONTINUOUS VARIABLES');
        WriteLn(nprnt,CharStr(' ',16), 'NUMBER OF DD EQUATIONS:',  CharStr(' ',15),neqdt:3);
        WriteLn(nprnt,CharStr(' ',16), 'NUMBER OF SS EQUATIONS:',  CharStr(' ',15),neqst:3);
        WriteLn(nprnt,CharStr(' ',16), 'TIME BETWEEN SAVE POINTS:',CharStr(' ', 7),dtsave:16:6);
        WriteLn(nprnt,CharStr(' ',16), 'MAXIMUM STEP SIZE:'       ,CharStr(' ',14),dtmax:16:6);
        WriteLn(nprnt,CharStr(' ',16), 'MINIMUM STEP SIZE:'       ,CharStr(' ',14),dtmin:16:6);
        WriteLn(nprnt,CharStr(' ',16), 'ABSOLUTE ERROR LIMIT:'    ,CharStr(' ',11),aerr:16:6);
        WriteLn(nprnt,CharStr(' ',16), 'RELATIVE ERROR LIMIT:'    ,CharStr(' ',11),rerr:16:6,CRLF);
{
      ;---SVENT CARD options...
}
        IF ( nthres > 0 ) THEN BEGIN
          WriteLn(nprnt,CharStr(' ',10),'STATE EVENTS');
          WriteLn(nprnt,CharStr(' ',16),'         EVENT   CROSSING   DIRECTION     THRESHOLD VALUE   TOLERANCE');
          WriteLn(nprnt,CharStr(' ',16),'NUMBER   CODE    VARIABLE   OF CROSSING   OR VARIABLE       OF CROSSING');
          WriteLn(nprnt,CharStr(' ',16),'------   -----   --------   -----------   ---------------   -----------');

          FOR i := 1 TO nthres DO BEGIN
            IF svent[i].thresVar = _const THEN
              Val := VarImage[svent[i].thresVar] + ' '
            ELSE
              Val := VarImage[svent[i].thresVar] + '(';

            Write(nprnt,CharStr(' ',17), i:5,
                        '   ',           svent[i].eventNo:5,
                        '   ',           VarImage[svent[i].crossVar],'(',svent[i].varIndex:3,')',
            CharStr(' ', 8), DirImage[svent[i].directn]);

            IF ( svent[i].thresVar <> _const ) THEN
              WriteLn(nprnt,CharStr(' ',07), Val,svent[i].thresIndex:3,')',
              CharStr(' ',09), svent[i].tolerance:12:4)
            ELSE
              WriteLn(nprnt,CharStr(' ',07), Val,svent[i].value:12:4,'  ',svent[i].tolerance:12:4);

          END {FOR};
        END {IF};
      END {IF};
{
      ;---RECORD CARD options...
}
      IF ( nvar > 0 ) THEN BEGIN
        WriteLn(nprnt,CRLF,CRLF,CharStr(' ',10),'RECORDED VARIABLES',CRLF);
        WriteLn(nprnt,CharStr(' ',16),'RECORDED VARIABLES SAVED TO FILE: [',dataFileName,']');
        WriteLn(nprnt,CharStr(' ',16),'NUMBER    RECORDED VARIABLE    IDENTIFIER         INITIAL VALUE');
        WriteLn(nprnt,CharStr(' ',16),'------    -----------------    ---------------    -------------');
        WriteLn(nprnt,CharStr(' ',16),'    1           TNEXT          TIME              ',tnow:14:5);

        FOR i := 2 TO nvar DO BEGIN
          CASE vvar[i].variable OF
            _ss : vx := ss[vvar[i].INDEX];                {1991.1108}
            _dd : vx := dd[vvar[i].INDEX];                {1991.1108}
            _xx : vx := xx[vvar[i].INDEX];                {1991.1108}
          END {CASE};
          WriteLn(nprnt,CharStr(' ',16),i:5,
          CharStr(' ',10),VarImage[vvar[i].variable],'(',vvar[i].INDEX:3,')         ',
          Pad(vvar[i].description,15),'   ',vx:14:5);
        END {FOR};
      END {IF};
{
      ;---Initialization options....
}
      WriteLn(nprnt,CRLF,CRLF,CharStr(' ',10),'INITIALIZATION OPTIONS',CRLF,
      CharStr(' ',16), 'BEGINNING TIME OF SIMULATION:   ',tstart:16:6,CRLF,
      CharStr(' ',16), 'ENDING TIME OF SIMULATION:      ',  tfin:16:6,CRLF,CRLF);
{
      ;---Calendar initialization...
}
      nwc := Round((MemAvail DIV (SizeOf(CalendarObj) + Matrib*SizeOf(AtbObj)))*0.95);
      WriteLn(nprnt, CharStr(' ',10),'CALENDAR STORAGE ALLOCATION',CRLF,
      CharStr(' ',16),'MEMORY AVAILABLE TO CALENDAR:     ',MemAvail:7,CRLF,
      CharStr(' ',16),'ESTIMATED MAX CALENDAR ENTRIES:   ',nwc:7);

      IF ( ixqt ) THEN
        WriteLn(nprnt,CRLF,CRLF,CharStr(' ',10),'EXECUTION WILL BE ATTEMPTED')
      ELSE
        WriteLn(nprnt,CRLF,CRLF,CharStr(' ',10),'EXECUTION WILL NOT BE ATTEMPTED');

      CallUPRNTIN;

      WriteLn(nprnt,FormFeed,CRLF,CRLF,CharStr(' ',24),'K S I M   P R O C E S S   R E P O R T');
      WriteLn(nprnt,CharStr(' ',24),'KSIM VERSION ',version,' / OCT 1991 (MS-DOS 3.x+)',CRLF,CRLF);
    END {WITH};
{}END {PRNTIN};




{}PROCEDURE IINZ;
{---------------------------------------------------------------------------
  Purpose     - Initialize KSIM variables and linked lists.

  Written     - 1980.1105 (KSB) Wrote in VAX-11 Fortran 77
  Revised     - 1982.0407
              - 1982.1020 (KSB) Converted to Harris FORTRAN-77.
              - 1991.1028 (KSB) Converted to Turbo Pascal
---------------------------------------------------------------------------}
  CONST
    Inited    : BOOLEAN = FALSE;
  VAR
    previus   : INTEGER;
    i,ii : WORD;
    p    : ListNodePtr;
  BEGIN
    WITH Ksim1 DO BEGIN
      stop   := FALSE;
    END {WITH};

    WITH Ksim2 DO BEGIN
      tstart := 0.0;
      tfin   := big;
      tnext  := big;
      tlast  := 0.0;
      tstep  := 0.05;
      tnow   := tstart;
      ttsave := 0.;
      dtnow  := 0.;
    END {WITH};

    WITH Ksim3 DO BEGIN
      FillChar(atrib,SizeOf(atrib),0);
      FillChar(krank,SizeOf(krank),0);
      nxqt   := 0;
      nsim   := 0;
      nsched := 0;
{
      ;---Initialize the calendar queue...
}
      IF NOT Inited THEN BEGIN
        New(ListPtr(Calendar),Init);
        Inited := TRUE;
      END ELSE BEGIN
        p := ListPtr(Calendar)^.First;
        WHILE p <> NIL DO BEGIN
          ListPtr(calendar)^.Remove(p);
          ListPtr(p)^.Done;
          p := ListPtr(Calendar)^.First;
        END {WHILE};

        ListPtr(Calendar)^.Done;
        ListPtr(Calendar)^.Init;
      END {BEGIN};
    END {WITH};

    WITH Ksim4 DO BEGIN
      dtsave := big;
      dtfull := big;
      dtmax  := dtsave;
      dtmin  := 0.01 * dtmax;
      aerr   := 0.00001;
      rerr   := 0.00001;
    END {WITH};

    WITH Ksim5 DO BEGIN
{
      ;---Initialize arrays...
}
      FillChar(dd,   SizeOf(dd),   0);
      FillChar(ddl,  SizeOf(ddl),  0);
      FillChar(ss,   SizeOf(ss),   0);
      FillChar(ssl,  SizeOf(ssl),  0);
      FillChar(xx,   SizeOf(xx),   0);
      FillChar(xxl,  SizeOf(xxl),  0);  {1992.0129}
    END {WITH};

    WITH Ksim6 DO BEGIN
      nsvent := 0;
    END {WITH};
{}END {IINZ};




{}PROCEDURE SORT( iopt:Priorities );
{---------------------------------------------------------------------------
  Purpose     - Arrange the calendar indexes stored in krank (events scheduled
                to be executed at the same time) according to the priority
                specified by "iopt".

   Parameters :
         iopt - The ranking or priority:
                1) FIRST-IN FIRST-OUT  2) LAST-IN FIRST-OUT
                3) LOW EVENT FIRST     4) HIGH EVENT FIRST

  Written     - 1980.1105 (KSB) Original in VAX-11 Fortran 77
  Revised     - 1982.0402
              - 1982.1020 (KSB) Converted to Harris FORTRAN-77.
              - 1985.0420 (KSB) Rewrote using block "ifs" and "for" loops.
              - 1985.0819 (KSB) Corrected typos with store array (referencing
                store(*,3)) that cause compiler errors.
              - 1991.1030 (KSB) Converted to Turbo Pascal.
              - 1991.1101 (KSB) Fixed the swap code to properly exchange elements.
---------------------------------------------------------------------------}

  TYPE
    simultaneousRec= RECORD
      p : POINTER;
      CASE INTEGER OF
        0:(eNo : INTEGER);
        1:(time: REAL);
    END {RECORD};
  VAR
    store: ARRAY[1..100]OF SimultaneousRec;
    save : SimultaneousRec;
    ii,jj,kk  : WORD;
    ll,nnsim  : WORD;
{
      ;---Check for too many simultaneous events...
}
  BEGIN
    WITH Ksim3 DO BEGIN
      IF ( nsim > 100 ) THEN BEGIN
        WriteLn(' *** KSIM PROCESSOR ERROR 0002 ***',CRLF,
                '     SUB: SORT  NUMBER OF SIMULTANEOUS EVENTS > 100');
        ERCODE(2);
      END {IF};
{
      ;---Set sorting parameters...
      ;---Extract the pointers and values to be sorted...
}
      FOR ii := 1 TO nsim DO BEGIN
        store[ii].p := krank[ii];
        CASE iopt OF
          _FIFO,
          _LIFO : store[ii].time := CalObjPtr(krank[ii])^.schTime;
          _LEF,
          _HEF  : store[ii].time := CalObjPtr(krank[ii])^.eventNo;
        END {CASE};
      END {FOR};
{
      ;---Now use a bubble sort to reorder the values...
}
      nnsim := nsim - 1;
      FOR jj := 1 TO nnsim DO BEGIN                                        {v:2.01}
        kk := jj + 1;
        FOR ll := kk TO nsim DO BEGIN                                      {v:2.01}
          IF (( iopt = _FIFO )  AND ( store[jj].time <= store[ll].time )) THEN
             {v:2.02}
          ELSE
          IF (( iopt = _LIFO )  AND ( store[jj].time >= store[ll].time )) THEN
             {v:2.02}
          ELSE
          IF (( iopt = _LEF  )  AND ( store[jj].eNo  <= store[ll].eNo  )) THEN
             {v:2.02}
          ELSE
          IF (( iopt = _HEF  )  AND ( store[jj].eNo  >= store[ll].eNo  )) THEN
             {v:2.02}
          ELSE BEGIN
{
                swap values
}
            save      := store[ll];
            store[ll] := store[jj];
            store[jj] := save;
          END {IF};
        END {FOR};
      END {FOR};
{
      ;---Return the sorted values and indexes to krank...
}
      FOR jj := 1 TO nsim DO
        krank[jj] := store[jj].p;

    END {WITH};
{}END {SORT};




{}PROCEDURE REMOVE( INDEX:POINTER; VAR nvent:INTEGER; VAR atb:StoreReal );
{---------------------------------------------------------------------------
  Purpose     - Remove from the event calendar, the event and the associated
                attributes specified by index.  Return the event code and the
                to the calling subprogram module.

   Parameters :
        index - (PTR.INPUT) The calendar node ptr of the event to be removed and returned.
        nvent - (INT.OUTPUT) The event code of the event to be removed.
          atb - (F.P.(),OUTPUT) Attributes associated with the removed event.

  Written     - 1980.1105 (KSB) Original in VAX-11 Fortran 77
  Revised     - 1981.1007
              - 1982.1020 (KSB) Converted to Harris FORTRAN-77.
              - 1991.1028 (KSB) Converted to Turbo Pascal
---------------------------------------------------------------------------}
  VAR
    p    : CalObjPtr ABSOLUTE INDEX;
    q    : CalObjPtr;
  BEGIN
    WITH Ksim3 DO BEGIN
      IF ( INDEX = NIL ) THEN BEGIN
        WriteLn(' *** KSIM PROCESSOR ERROR 0003 ***',CRLF,
                '     SUB: REMOVE  VAR: INDEX IS NIL');
        ERCODE(3);
      END {IF};

      FillChar(atb,SizeOf(atb),0);               {zero out atb}
      ListPtr(calendar)^.Remove(INDEX);          {remove index from calendar}
      nvent := p^.eventNo;                       {set the event number}

      IF p^.atb <> NIL THEN
        RecoverAttributes(p^.atb,atb);           {get attributes}

      p^.Done;
      Dispose(p);                                {release memory held by index}
    END {WITH};
{}END {REMOVE};




{}PROCEDURE ExamineNext_N_Event(N:INTEGER; VAR t:REAL; VAR atb:StoreReal);
{+H
---------------------------------------------------------------------------
  Purpose     - Looks for the next event (with code N) passed the current
                time.

  Declaration - procedure ExamineNext_N_Event(N:INTEGER; VAR t:REAL;
                   VAR atb:StoreReal);

  Remarks     - If an event N cannot be found, then T is set to -1.0,
                otherwise T is set to the scheduled time of release of event
                N.  The associated attributes are returned in ATB. Cannot be
                used for state events as they are not placed on the calendar.
---------------------------------------------------------------------------}
  VAR
    p,q,r: ListNodePtr;
    s,u  : REAL;
  BEGIN
    WITH Ksim3 DO BEGIN
      t := -1;
      s := big;
      u := big;
      p := ListPtr(calendar)^.First;
      WHILE p <> NIL DO BEGIN
        q := p;
        p := ListPtr(calendar)^.Next(p);
{
          Examine the scheduled execution time for each N event on the
          calendar, taking the earliest such event after TNOW.  If more
          than one N event is scheduled for execution at time T, then
          take the one that was first scheduled...
}
        WITH CalObjPtr(q)^ DO
          IF (eventNo = N) AND (schTime <= s) AND
           (exeTime > Ksim2.tnow) AND (exeTime < u) THEN BEGIN
            t := CalObjPtr(q)^.exeTime;
            s := CalObjPtr(q)^.schTime;
            u := t;
            r := q;
          END {IF};
      END {WHILE};

      IF t >= 0.0 THEN
        RecoverAttributes(CalObjPtr(r)^.atb,atb)
      ELSE
        FillChar(atb,SizeOf(atb),0);

    END {WITH};
{}END {ExamineNext_N_Event};




{}FUNCTION HeapFunc(size:WORD):INTEGER;
  BEGIN
    HeapFunc := 1;
{}END {HeapFunc};




{}PROCEDURE MemProtection(onOff:BOOLEAN);
{---------------------------------------------------------------------------
---------------------------------------------------------------------------}
  CONST
    save : POINTER = NIL;
  BEGIN
    IF onOff THEN BEGIN
      save      := heapError;
      heapError := @HeapFunc;
    END ELSE
    IF save <> NIL THEN BEGIN
      heapError := save;
    END {IF};
{}END {MemProtection};




{}PROCEDURE SCHEDUL( dt:REAL; nvent:INTEGER; atb:StoreReal );
{+H
---------------------------------------------------------------------------
  Purpose     - Schedul(e) is called by the user to schedule an event with
                the event code of 'nvent' to occur 'dt' time units in the
                future.

  Declaration - procedure SCHEDUL( dt:REAL; nvent:INTEGER; atb:StoreReal );

  Remarks     - DT is the number of time units in the future at which the
                event is to occur.
                NVENT is the event number of the event.
                ATB   is the array of event attributes and are stored with
                      the event

  Written     - 1980.1105 (KSB) Written in VAX-11 Fotran 77
  Revised     - 1981.1007
              - 1982.1020 (KSB) Converted to Harris FORTRAN-77.
              - 1991.1028 (KSB) Converted to Turbo Pascal
---------------------------------------------------------------------------}
  VAR
    p    : CalObjPtr;
    q    : AtbObjPtr;
    i    : WORD;
  BEGIN
    IF ( dt < 0.0 ) THEN BEGIN
      WriteLn(' *** KSIM PROCESSOR ERROR 0004 ***',CRLF,
              '     SUB: SCHDUL  VAR: DT < 0.');
      ERCODE(4);
    END {IF};

    WITH Ksim2,Ksim3 DO BEGIN
      MemProtection(TRUE);                       {Return NIL if no memory available}
      New(p,Init(nvent,dt));                     {Get memory for event}
      IF ( p <> NIL ) AND ( matrib > 0 ) THEN BEGIN
        New(p^.atb,Init);                        {initialize List of attributes}
        FOR i := 1 TO matrib DO BEGIN
          New(q,Init(atb[i]));                   {Get memory for attribute}
          IF ( q <> NIL ) THEN
                   {if memory available}
            p^.atb^.Insert(q)                    {store attribute on List}
          ELSE
            ERCODE(5);
        END {FOR};

        ListPtr(calendar)^.Insert(p);            {store completed event on calendar}
        Inc(nsched);                             {bump the # of events scheduled}
      END ELSE
      IF ( p = NIL ) THEN BEGIN
        WriteLn(' *** KSIM PROCESSOR ERROR 0005 ***',CRLF,
                '     SUB: SCHEDUL  VAR: MAXCAL : CALENDAR FULL');
        ERCODE(5);
      END {IF};
      MemProtection(FALSE);                      {return to normal memory allocation}
    END {WITH};
{}END {SCHEDUL};




{}PROCEDURE ScheduleOneAttribute( dt:REAL; nvent:INTEGER; a:REAL );
{+H
---------------------------------------------------------------------------
  Purpose     - ScheduleOneAttribute is called by the user to schedule an event
                with the event code of 'nvent' to occur 'dt' time units in the
                future. The event attribute 'a' is stored with the event.

  Declaration - procedure ScheduleOneAttribute(dt:REAL; nvent:INTEGER; a:REAL);

  Remarks     - DT is the number of time units in the future at which the
                event is to occur.
                NVENT is the event number of the event.
                A     is the single user attribute to be saved with the event.

  Revised     - 1992.0122 (KSB) Wrote initial version.
---------------------------------------------------------------------------}
  VAR
    p    : CalObjPtr;
    q    : AtbObjPtr;
  BEGIN
    IF ( dt < 0.0 ) THEN BEGIN
      WriteLn(' *** KSIM PROCESSOR ERROR 0004 ***',CRLF,
              '     SUB: SCHDULEONEATTRIBUTE  VAR: DT < 0.');
      ERCODE(4);
    END {IF};

    WITH Ksim2,Ksim3 DO BEGIN
      MemProtection(TRUE);                       {Return NIL if no memory available}
      New(p,Init(nvent,dt));                     {Get memory for event}
      IF ( p <> NIL ) AND ( matrib > 0 ) THEN BEGIN
        New(p^.atb,Init);                        {initialize List of attributes}
        New(q,Init(a));                          {Get memory for attribute}
        IF ( q <> NIL ) THEN
                     {if memory available}
          p^.atb^.Insert(q)                      {store attribute on List}
        ELSE
          ERCODE(5);

        ListPtr(calendar)^.Insert(p);            {store completed event on calendar}
        Inc(nsched);                             {bump the # of events scheduled}
      END ELSE
      IF ( p = NIL ) THEN BEGIN
        WriteLn(' *** KSIM PROCESSOR ERROR 0005 ***',CRLF,
                '     SUB: SCHEDULEONEATTRIBUTE  VAR: MAXCAL : CALENDAR FULL');
        ERCODE(5);
      END {IF};
      MemProtection(FALSE);                      {return to normal memory allocation}
    END {WITH};
{}END {ScheduleOneAttribute};




{}PROCEDURE SSAVE( jj:INTEGER );
{--------------------------------------------------------------------------
  Purpose     - Save selected values from selected variables during continuous
                simulation for plotting or postprocessing. Uses DATMAN file format.

   Parameters :
           jj - (INT.,INPUT) Control flag.
                If jj=0, write RCD header block.
                If jj=1, write data.

  Written     - 1981.1208 (KSB) Original in VAX-11 Fortran 77
  Revised     - 1982.1020 (KSB) Converted to Harris FORTRAN-77.
              - 1985.0423 (KSB) Modified to write DATMAN file format.
              - 1991.1030 (KSB) Converted to Turbo Pascal.
---------------------------------------------------------------------------}

  CONST
    nameWidth = 15;
    unitWidth = 7;
  VAR
    i    : WORD;
    svar : StoreReal;
  BEGIN
{
      ;---Write an RCD (Run Configuration Data) header block for the
          variables from which values are to be saved...
;$HEADER
; TITLE: data_file_title
; FILE : file_name
; Date : yyyy/mm/dd
; Time : hh:mm:ss
; Remrk: descriptive comment that is ignored
; DATA DEFINITION
; Name : nn
; Unit : nn
; COLUMNS=nnnn: FORMAT=ASCII
; HEADER ------------Title-------  ----Unit----  --Type--  ---Max---  ---Min---
; Cnnnn: column_name_or_descriptn  column_units  dataType  max_value  min_value
;$DATA
}
    WITH Ksim0,Ksim1,Ksim2,Ksim3,Ksim5,Ksim7 DO BEGIN
      IF ( jj = 0 ) THEN BEGIN
        IF ( nvar > 0 ) THEN BEGIN
          WriteLn(iplot, ';$HEADER');
          WriteLn(iplot, '; TITLE: '+nname+' / '+nprjct+' / '+DateStr(cdate)+' / '+Long2Str(nnrns));
          WriteLn(iplot, '; Date : ',FullDateStr(cDate));
          WriteLn(iplot, '; Time : ',TimeStr(CurrentTime));
          WriteLn(iplot, '; DATA DEFINITION');
          WriteLn(iplot, '; Name : ',nameWidth:2);
          WriteLn(iplot, '; Unit : ',unitWidth:2);
          WriteLn(iplot, '; COLUMNS=',nvar:4,': FORMAT=ASCII');
          WriteLn(iplot, '; HEADER -----Title-----  --Unit-  --Type--  ---Max---  ---Min---');
          WriteLn(iplot, '; C0001: TNEXT            TIME     REAL      ',tfin:9,'  ',tstart:9);

          FOR i := 2 TO nvar DO BEGIN
            WriteLn(iplot, '; C',ReplaceAll(Long2LStr(i,4),' ','0'),': ',
            Pad(vvar[i].description,15),'  ',
            VarImage[vvar[i].variable],'(',vvar[i].INDEX:3,')',
                           '  REAL      ',0:9,'  ',0:9);
          END {FOR};
          WriteLn(iplot, ';$DATA');
        END {IF};
{
          write the values from dd(.), ss(.) or xx(.) variables...
}
      END ELSE
      IF ( jj = 1 ) THEN BEGIN
        WriteLn(tnow:16:6);
        IF ( nvar > 0 ) THEN BEGIN
          svar[1] := tnow;
          FOR i := 2 TO nvar DO BEGIN
            CASE vvar[i].variable OF
              _ss : svar[i] := ss[vvar[i].INDEX];
              _dd : svar[i] := dd[vvar[i].INDEX];
              _xx : svar[i] := xx[vvar[i].INDEX];
            END {CASE};
          END {FOR};

          FOR i := 1 TO nvar DO
            Write(iplot, '  ',svar[i]:13:6);

          WriteLn(iplot);
          Flush(iplot);
        END {IF};
      END {IF};

    END {WITH};
{}END {SSAVE};




{$I Ksim_A.IN1}


{}PROCEDURE OneTimeInitialize;
{--------------------------------------------------------------------------
  Purpose     - Set the default values for the number of runs, the user name,
                the project title, the activation switches, etc. (one time
                initializations) ...
---------------------------------------------------------------------------}
  BEGIN
    FillChar(Ksim0,SizeOf(Ksim0),0);
    FillChar(Ksim1,SizeOf(Ksim1),0);
    FillChar(Ksim2,SizeOf(Ksim2),0);
    FillChar(Ksim3,SizeOf(Ksim3),0);
    FillChar(Ksim4,SizeOf(Ksim4),0);
    FillChar(Ksim5,SizeOf(Ksim5),0);
    FillChar(Ksim6,SizeOf(Ksim6),0);
    FillChar(Ksim7,SizeOf(Ksim7),0);

    WITH Ksim3 DO BEGIN
      matrib  := 0;
      calendar:= NIL;
    END {WITH};

    WITH Ksim1 DO BEGIN
      nname   := '---NAME---';
      nprjct  := 'PROJECT/NO';
      ilist   := TRUE;
      ixqt    := TRUE;
      isimrep := TRUE;
      reInit  := TRUE;
      lpsym   := _FIFO;
      cdate   := Today;
    END {WITH};

    WITH Ksim0 DO BEGIN
      maxruns := 1;
      nnrns   := 1;
    END {WITH};

    WITH Ksim4 DO BEGIN
      depf    := TRUE;
      neqst   := 0;
      neqdt   := 0;
      neqtt   := 0;
    END {WITH};

    WITH Ksim5 DO BEGIN
    END {WITH};

    WITH Ksim6 DO BEGIN
      sepf   := TRUE;
      nthres := 0;
     {FillChar(svent,SizeOf(svent),0);}
    END {WITH};

    WITH Ksim7 DO BEGIN
      nvar   := 0;
      kkvent := TRUE;
     {FillChar(vvar, SizeOf(vvar), 0);}
    END {WITH};

    {$I-}
    Assign(nprnt,RprtFileName);
    Rewrite(nprnt);
    IF IOresult <> 0 THEN
      ERCODE(8);

    Assign(iplot,DataFileName);
    Rewrite(iplot);
    IF IOresult <> 0 THEN
      ERCODE(9);
    {$I+}
{}END {OneTimeInitialize};




{}PROCEDURE KSIM;
{+H
---------------------------------------------------------------------------
  Purpose     - Calls I/O and initialization routines for discrete and
                continuous simulation.

  Declaration - procedure KSIM;

  Remarks     - Call Ksim to start the actual simulation after the model is
                set up.

  Written     - 1980.1113 (KSB) Original in VAX-11 Fortran 77.
  Revised     - 1982.0312
              - 1982.1020 (KSB) Converted to Harris FORTRAN-77.
              - 1983.0126 (KSB) Made corrections to "control" and "rkf" (integration algorithm)
              - 1985.0418 V:4.2  (KSB) Revised menus and added comments.
              - 1986.0227 V:4.21 (KSB) Added code to conditionally print
                discrete event and state event release messages.  In the
                process added two new logicals to the common block, 'depf' and 'sepf'.
              - 1986.0605 V:4.30 (KSB) Converted Harris version back to VAX-11.
              - 1991.1025 V:5.00 (KSB) Converted to Turbo Pascal.
---------------------------------------------------------------------------}
  BEGIN
    OneTimeInitialize;

{     ;---Execute the simulation routines...
          Set up the default values;
          Get the user's input values;
          Echo the input values (if required);
          Execute the simulation (if required);
          Write a report on what happened (if required).
}
    WriteLn('A  ',MemAvail);
    WHILE (Ksim0.nnrns <= Ksim0.maxruns) DO BEGIN
      IF ( Ksim0.nnrns = 1 ) OR ( Ksim1.reInit ) THEN BEGIN
        IINZ;
        INTLC;
        WITH Ksim4 DO
          neqtt := neqdt + neqst;
      END {IF};
      WriteLn('B  ',MemAvail);

      DFLT;
      CallUINPUT;

      IF ( Ksim1.ilist  ) THEN
        PRNTIN;

      IF ( Ksim1.ixqt   ) THEN
        CONTROL;

      IF ( Ksim1.isimrep) THEN
        REPORT;

      Inc(Ksim0.nnrns);
      WriteLn('C  ',MemAvail);
    END {WHILE};

    WriteLn('CalendarPtr -> $',HexPtr(Ksim3.Calendar));
    ListPtr(Ksim3.calendar)^.Done;
    WriteLn('D  ',MemAvail);

    Close(nprnt);
    Close(iplot);
    WriteLn('E  ',MemAvail);
{}END {KSIM};




BEGIN
END {BEGIN}.
