{ STZLIB.PAS -- String handling library (zero-ended strings)

  Title   : STZLIB
  Version : 2.2
  Language: Borland Pascal 7.0 (all targets)
            Borland Delphi 1.0 through 3.0
  Date    : Dec 23,1997
  Author  : J.R. Ferguson
  Usage   : Unit

  The routines in this library use a string type of array[0..n] of char or
  PChar. A zero-byte is used as end-of-string marker. The strings
  themselves can therefore never include a character with the ordinal value
  0. The maximum string length is 65535 characters.

  Character positions are numbered starting with 0. This results in some
  differences with likewise-named routines in the libraries StpLib and
  StfLib.

  String pointers having the value nil are not supported.
  The routines that copy, insert or append characters into strings do not
  perform any length checking. The receiving string variables must have
  enough room for the (extra) characters they will receive.

  The declarations of StzPtr, StzInd and StpTyp can be found in DEFLIB.PAS.
}

{$X+ : extended syntax }
{$B- : short-circuit Boolean expression generation }

{$ifndef WIN32}
{$DEFINE USEASM}  { Define to use inline assembler statements }
{$endif}

UNIT StzLib;

INTERFACE
Uses DefLib;

const
  StzNOTFOUND = $FFFF;

function StzAfter(const src,pat: StzPtr): StzPtr;
{ Returns a pointer to the first character in string src after pattern 
  pat, or to the end-of-string marker if pat is not a part of src or if 
  pat is an empty string.

  See also: <StzBefore>, <StzRight>, <StzSub>
}

function StzAlloc(const s: StzPtr): StzPtr;
{ Allocates memory storage for StzLen(s)+1 characters and copies the
  contents of string s to this new allocated block of memory. 
  Returns a pointer to this allocated string.

  See also: <StzFree>, <StzReAlloc>
}

function StzBefore(dst: StzPtr; const src,pat: StzPtr): StzPtr;
{ Extracts into dst the part from string src that comes before pattern pat,
  and adds an end-of-string marker.
  If pat is not a part of src, string src is copied to dst without change.
  Returns dst.

  See also: <StzAfter>, <StzRight>, <StzSub>
}

function StzCat(dst: StzPtr; const src: StzPtr): StzPtr;
{ Appends string src to string dst.
  Returns dst.
  src and dst must be separate string variables.

  See also: <StzECat>, <StzNCat>, <StzcCat>, <StzcECat>
}

function StzCenter(s: StzPtr; n: StzInd): StzPtr;
{ Centers string s to be printed on a line of n characters wide by
  inserting spaces in front of it. 
  Returns s. 
}

function StzCmp(const s1, s2: StzPtr): integer;
{ Compares strings s1 and s2, based on the ordinal value of their
  characters.
  The result is negative if s1<s2, zero if s1=s2 and positive if s1>s2.

  See also:
  <StzLexCmp>, <StzUppCmp>, <StzNCmp>, <StzLexNCmp>, <StzUppNCmp>,
  <StzSoundEx>, <StzSoundAlike>
}

function StzCpy(dst: StzPtr; const src: StzPtr): StzPtr;
{ Copies string src to dst.
  Returns dst.

  See also:
  <StzECpy>, <StzNCpy>, <StzcCpy>, <StzcECpy>, <StzStpCpy>
}

function StzCreate(s: StzPtr): StzPtr;
{ Creates an empty string s, containing an end-of-string marker only.
  Returns s.

  See also: <StzMake>, <StzFill>, <StzLFill>
}

function StzcCat(dst: StzPtr; c: char): StzPtr;
{ Appends character c to string dst.
  If c = #0, string dst is not altered.
  Returns dst.

  See also: <StzCat>, <StzECat>, <StzNCat>, <StzcECat>
}

function StzcCpy(dst: StzPtr; c: char): StzPtr;
{ Copies character c to the first position of string dst.
  If c <> #0, an end-of-string marker is added.
  Returns dst.

  See also:
  <StzCpy>, <StzECpy>, <StzNCpy>, <StzcECpy>, <StzStpCpy>
}

function StzcECat(dst: StzPtr; c: char): StzPtr;
{ Appends characters c to the end of string dst.
  If c = #0, string dst is not altered.
  Returns a pointer to the end-of-string marker of dst.

  See also: <StzCat>, <StzECat>, <StzNCat>, <StzcCat>
}

function StzcECpy(dst: StzPtr; c: char): StzPtr;
{ Copies character c to the first position of string dst.
  If c <> #0, an end-of-string marker is added.
  Returns a pointer to the end-of-string marker of dst.

  See also:
  <StzCpy>, <StzNCpy>, <StzcCpy>, <StzcECpy>, <StzStpCpy>
}

function StzcGet(s: StzPtr): char;
{ Returns the first character from string s and removes that character 
  from string s. If s is an empty string, a character with the ordinal 
  value 0 is returned.

  See also: <StzcRet>, <StzGtw>
}

function StzcIns(dst: StzPtr; c: char; i: StzInd): StzPtr;
{ Inserts character c into string dst at position i.
  If i >= StzLen(s), the character is appended to string dst.
  If c = #0, string dst is not altered.
  Returns dst.

  See also: <StzIns>, <StzNIns>
}

function StzcIPos(const s: StzPtr; c: char): StzInd;
{ Returns the first position of character c in string s
  If c=#0, the value of StzLen(s) will be returned.
  If c cannot be found in s, the value StzNOTFOUND is returned.

  See also: <StzPos>, <StzIPos>, <StzRPos>, <StzRIPos>,
            <StzUppPos>, <StzUppIPos>, <StzUppRPos>, <StzUppRIPos>,
            <StzcPos>, <StzcRPos>, <StzcRIPos>,
            <StzcUppPos>, <StzcUppIPos>, <StzcUppRPos>, <StzcUppRIPos>
}

function StzcPos(const s: StzPtr; c: char): StzPtr;
{ Returns a pointer to the first position of character c in string s,
  or nil if c cannot be found in s.
  If c=#0, an pointer to the end-of-string marker of s is returned.

  See also: <StzPos>, <StzIPos>, <StzRPos>, <StzRIPos>,
            <StzUppPos>, <StzUppIPos>, <StzUppRPos>, <StzUppRIPos>,
            <StzcIPos>, <StzcRPos>, <StzcRIPos>,
            <StzcUppPos>, <StzcUppIPos>, <StzcUppRPos>, <StzcUppRIPos>
}

function StzcRet(const s: StzPtr; i: StzInd): char;
{ Returns the character at position i in string s. String s will not be
  altered.
  If i >= StzLen(s), a character with ordinal value 0 will be returned.

  See also: <StzcGet>, <StzGtw>
}

function StzcRIPos(const s: StzPtr; c: char): StzInd;
{ Returns the last position of character c in string s, or the value
  StzNOTFOUND if c cannot be found in s.
  If c=#0, the value of StzLen(s) will be returned.

  See also: <StzPos>, <StzIPos>, <StzRPos>, <StzRIPos>,
            <StzUppPos>, <StzUppIPos>, <StzUppRPos>, <StzUppRIPos>,
            <StzcPos>, <StzcIPos>, <StzcRPos>,
            <StzcUppPos>, <StzcUppIPos>, <StzcUppRPos>, <StzcUppRIPos>
}

function StzcRPos(const s: StzPtr; c: char): StzPtr;
{ Returns a pointer to the last position of character c in string s,
  or nil if c cannot be found in s.
  If c = #0, a pointer to the end-of-string marker of s is returned.

  See also: <StzPos>, <StzIPos>, <StzRPos>, <StzRIPos>,
            <StzUppPos>, <StzUppIPos>, <StzUppRPos>, <StzUppRIPos>,
            <StzcPos>, <StzcIPos>, <StzcRIPos>,
            <StzcUppPos>, <StzcUppIPos>, <StzcUppRPos>, <StzcUppRIPos>
}

function StzcUpd(s: StzPtr; c: char; i: StzInd): StzPtr;
{ Replaces the character at position i in string s with character c.
  If i >= StzLen(s), string s is left unchanged.
  Returns s.

  See also: <StzRepl>, <StzNRepl>
}

function StzcUppIPos(const s: StzPtr; c: char): StzInd;
{ Returns the first position of character c in string s, or the value
  StzNOTFOUND if c cannot be found in s.
  If c=#0, the value of StzLen(s) will be returned.
  Upper and lower case characters are seen as equal.

  See also: <StzPos>, <StzIPos>, <StzRPos>, <StzRIPos>,
            <StzUppPos>, <StzUppIPos>, <StzUppRPos>, <StzUppRIPos>,
            <StzcPos>, <StzcIPos>, <StzcRPos>, <StzcRIPos>,
            <StzcUppPos>, <StzcUppRPos>, <StzcUppRIPos>
}

function StzcUppPos(const s: StzPtr; c: char): StzPtr;
{ Returns a pointer to the first position of character c in string s,
  or nil if c cannot be found in s.
  If c=#0, an pointer to the end-of-string marker of s is returned.
  Upper and lower case characters are seen as equal.

  See also: <StzPos>, <StzIPos>, <StzRPos>, <StzRIPos>,
            <StzUppPos>, <StzUppIPos>, <StzUppRPos>, <StzUppRIPos>,
            <StzcPos>, <StzcIPos>, <StzcRPos>, <StzcRIPos>,
            <StzcUppIPos>, <StzcUppRPos>, <StzcUppRIPos>
}

function StzcUppRIPos(const s: StzPtr; c: char): StzInd;
{ Returns the last position of character c in string s, or the value
  StzNOTFOUND if c cannot be found in s.
  If c=#0, the value of StzLen(s) will be returned.
  Upper and lower case characters are seen as equal.

  See also: <StzPos>, <StzIPos>, <StzRPos>, <StzRIPos>,
            <StzUppPos>, <StzUppIPos>, <StzUppRPos>, <StzUppRIPos>,
            <StzcPos>, <StzcIPos>, <StzcRPos>, <StzcRIPos>,
            <StzcUppPos>, <StzcUppIPos>, <StzcUppRPos>
}

function StzcUppRPos(const s: StzPtr; c: char): StzPtr;
{ Returns a pointer to the last position of character c in string s,
  or nil if c cannot be found in s.
  If c = #0, a pointer to the end-of-string marker of s is returned.
  Upper and lower case characters are seen as equal.

  See also: <StzPos>, <StzIPos>, <StzRPos>, <StzRIPos>,
            <StzUppPos>, <StzUppIPos>, <StzUppRPos>, <StzUppRIPos>,
            <StzcPos>, <StzcIPos>, <StzcRPos>, <StzcRIPos>,
            <StzcUppPos>, <StzcUppIPos>, <StzcUppRIPos>
}

function StzDel(s: StzPtr; i,n : StzInd): StzPtr;
{ Deletes n characters from string s, starting at position i. 
  Returns s.
}

function StzDetab(dst: StzPtr; const src: StzPtr; n: StzInd): StzPtr;
{ Expands tabs in string src into space groups, using n as the tab field
  width. The result is in dst, and the function also returns dst.
  This function recognizes the following control characters:
    HT : Horizontal tab : expand to spaces.
    BS : Back space     : decrement column position by 1.
    CR : Carriage return: reset column position to 1.
  Other characters with an ordinal value in the range 0..31 are considered
  as non-printable. They are copied without change, but don't alter the
  current column position.
  Characters with an ordinal value in the range 128..255 are considered to
  be printable, so they are copied and increment the column position by 1.
  Remarks:
  - The column positioning may be at fault when string src containts BS
    characters immediately following HT or other control characters.
  - If n=0, string src is copied to dst without change.
  - src and dst must be separate string variables.

  See also: <StzEntab>
}

function StzECat(dst: StzPtr; const src: StzPtr): StzPtr;
{ Appends string src to string dst.
  Returns a pointer to the end-of-string marker of dst.
  src and dst must be separate string variables.

  See also: <StzCat>, <StzNCat>, <StzcCat>, <StzcECat>
}

function StzECpy(dst: StzPtr; const src: StzPtr): StzPtr;
{ Copies string src to dst.
  Returns a pointer to the end-of-string marker of dst.

  See also:
  <StzCpy>, <StzNCpy>, <StzcCpy>, <StzcECpy>, <StzStpCpy>
}

function StzEmpty(const s: StzPtr): boolean;
{ Tests if string s is empty. }

function StzEnd(const s: StzPtr): StzPtr;
{ Returns a pointer to the end-of-string marker of s. }

function StzEntab(dst: StzPtr; const src: StzPtr; n: StzInd): StzPtr;
{ Replaces space groups in src by horizontal tab characters, using
  multiples of n as tab columns. Single spaces are never replaced by tabs.
  The result is in dst, and the function also returns dst.
  This function recognizes the following control characters:
    HT : Horizontal tab : expand to spaces.
    BS : Back space     : decrement column position by 1.
    CR : Carriage return: reset column position to 1.
  Other characters with an ordinal value in the range 0..31 are considered
  as non-printable. They are copied without change, but don't alter the
  current column position.
  Characters with an ordinal value in the range 128..255 are considered to
  be printable, so they are copied and increment the column position by 1.
  Remarks:
  - The column positioning may be at fault when string src containts BS
    characters immediately following a space group, a HT or another control
    character.
  - If n=0 string src is copied to dst without change.
  - src and dst must be separate string variables.

  See also: <StzDetab>
}

function StzFill(s: StzPtr; c: char; n: StzInd): StzPtr;
{ Fills (lengthens) string s to a length of n, by appending characters c
  at the end of string s (also if c = #0), and then appending an
  end-of-string marker to s.
  If n <= StzLen(s), string s is left unchanged.
  Returns s.

  See also: <StzLFill>, <StzCreate>, <StzMake>
}

procedure StzFree(var p: StzPtr);
{ Frees the memory block used for string p, previously allocated with
  StzAlloc or StzRealloc (not regarding the current length of s), and sets
  p to nil.

  See also: <StzAlloc>, <StzReAlloc>
}

function StzGtw(dst,src: StzPtr): StzPtr;
{ Returns the first word from string src into dst and removes this word
  from src. 
  If string src does not contain a word, dst and src are both made empty.

  The folowing ASCII characters are seen as word separators:
  carriage return, line feed, form feed, horizontal tab, vertical tab and
  space.

  See also: <StzcGet>, <StzcRet>
}

function StzIns(dst: StzPtr; const src: StzPtr; i: StzInd): StzPtr;
{ Inserts string src at position i into string dst.
  If i >= StzLen(src), string src will be appended to dst.
  Returns dst

  src and dst must be separate string variables.

  See also: <StzNIns>, <StzcIns>
}

function StzIPos(const src: StzPtr; const pat: StzPtr): StzInd;
{ Returns the first position of string pattern pat in string src,
  or the value StzNOTFOUND if pat cannot be found in src.

  See also: <StzPos>, <StzRPos>, <StzRIPos>,
            <StzUppPos>, <StzUppIPos>, <StzUppRPos>, <StzUppRIPos>,
            <StzcPos>, <StzcIPos>, <StzcRPos>, <StzcRIPos>,
            <StzcUppPos>, <StzcUppIPos>, <StzcUppRPos>, <StzcUppRIPos>
}

function StzLen(const s: StzPtr): StzInd;
{ Returns the number of characters i string s, not counting the end-of-
  string marker. }

function StzLexCmp(const s1, s2: StzPtr): integer;
{ Compares strings s1 and s2, based on the ChrLib.Lexorder character order.
  The result is negative if s1<s2, zero if s1=s2 and positive if s1>s2.

  See also:
  <StzLexCmp>, <StzUppCmp>, <StzNCmp>, <StzLexNCmp>, <StzUppNCmp>
}

function StzLexNCmp(const s1, s2: StzPtr; n: StzInd): integer;
{ Compares a maximum of n characters from strings s1 and s2, based on the
  ChrLib.Lexorder character order.
  The result is negative if s1<s2, zero if s1=s2 and positive if s1>s2.

  See also:
  <StzCmp>, <StzLexCmp>, <StzUppCmp>, <StzNCmp>, <StzUppNCmp>,
  <StzSoundEx>, <StzSoundAlike>
}

function StzLexSrt(dst: StzPtr; const src: StzPtr; n: StzInd): StzPtr;
{ Like StzSrt, using the ChrLib.LexOrder character order.

  See also: <StzSrt>, <StzUppSrt>
}

function StzLFill(s: StzPtr; c: char; n: StzInd): StzPtr;
{ Fills (lengthens) string s to a length of n, by inserting characters c
  at the start of string s (also if c = #0).
  If n <= StzLen(s), string s is left unchanged.
  Returns s.

  See also: <StzFill>, <StzCreate>, <StzMake>
}

function StzLow(s: StzPtr): StzPtr;
{ Converts all upper case letters in string s to lower case.
  Returns s.

  See also: <StzUpp>
}

function StzMake(s: StzPtr; c: char; n: StzInd): StzPtr;
{ Fills string s with n chacarters c and appends an end-of-string-marker to
  it. Returns s.

  See also: <StzCreate>, <StzFill>, <StzLFill>
}

function StzNCat(dst: StzPtr; const src: StzPtr; n: StzInd): StzPtr;
{ Appends a maximum of n characters from string src to string dst. 
  Returns dst. 
  src and dst must be separate string variables.

  See also: <StzCat>, <StzECat>, <StzcCat>, <StzcECat>
}

function StzNCmp(const s1, s2: StzPtr; n: StzInd): integer;
{ Compares a maximum of n characters from strings s1 and s2, based on the
  ordinal value of their characters. 
  The result is negative if s1<s2, zero if s1=s2 and positive if s1>s2.

  See also:
  <StzCmp>, <StzLexCmp>, <StzUppCmp>, <StzLexNCmp>, <StzUppNCmp>,
  <StzSoundEx>, <StzSoundAlike>
}

function StzNCpy(dst: StzPtr; const src: StzPtr; n: StzInd): StzPtr;
{ Copies a maximum of n characters from string src to dst, and appends an
  end-of-string marker to dst. Returns dst.

  See also:
  <StzCpy>, <StzECpy>, <StzcCpy>, <StzcECpy>, <StzStpCpy>
}

function StzNIns(dst: StzPtr; const src: StzPtr; i,n: StzInd): StzPtr;
{ Inserts a maximum of n characters from string src at position i into
  string dst. 
  If i >= StzLen(src), the characters from string src will be appended to
  dst. 
  Returns dst

  src and dst must be separate string variables.

  See also: <StzIns>, <StzcIns>
}

function StzNRepl(dst: StzPtr; const src: StzPtr; i,n: StzInd): StzPtr;
{ Like StzRepl, replacing a maximum of n characters in string dst.

  See also: <StzRepl>, <StzcUpd>
}

function StzPos(const src, pat: StzPtr): StzPtr;
{ Returns a pointer to the first position of string pattern pat in string
  src, or nil if pat is empty or cannot be found in src.

  See also: <StzIPos>, <StzRPos>, <StzRIPos>,
            <StzUppPos>, <StzUppIPos>, <StzUppRPos>, <StzUppRIPos>,
            <StzcPos>, <StzcIPos>, <StzcRPos>, <StzcRIPos>,
            <StzcUppPos>, <StzcUppIPos>, <StzcUppRPos>, <StzcUppRIPos>
}

function StzRAS(s: StzPtr): StzPtr;
{ Remove All Spaces: Removes all carriage return, line feed, form feed,
  horizontal tab, vertical tab and space characters from string s.
  Returns s.

  See also: <StzRLS>, <StzRTS>
}

function StzReAlloc(var p: StzPtr; const s: StzPtr): StzPtr;
{ Frees the memory block used for string p, previously allocated with
  StzAlloc or StzRealloc, disregarding the current length of p; allocates
  new memory storage for StzLen(s)+1 characters and copies the contents of
  string s to this new allocated block of memory. 
  Returns a pointer to this newly allocated string and also sets p to this
  pointer value.

  See also: <StzAlloc>, <StzFree>
}

function StzRepl(dst: StzPtr; const src: StzPtr; i: StzInd): StzPtr;
{ Replaces characters from dst by those of string src, starting at position
  i in dst. The resulting string dst may be longer than its original value.
  If i>StzLen(dst), spaces are added to dst until its length is i, and then
  string src is appended to it.
  Returns dst.

  src and dst must be separate string variables.

  See also: <StzNRepl>, <StzcUpd>
}

function StzRev(s: StzPtr): StzPtr;
{ Reverses the character order of string s. Returns dst. }

function StzRight(const s: StzPtr; n: StzInd): StzPtr;
{ Returns a pointer to the the n rightmost characters of string s.
  If n=0, a pointer to the end-of-string marker of s is returned.

  See also: <StzAfter>, <StzBefore>, <StzSub>
}

function StzRIPos(const src: StzPtr; const pat: StzPtr): StzInd;
{ Returns the last position of string pat in string src. 
  If pat is empty or cannot be found in src, the value StzNOTFOUND is
  returned.

  See also: <StzPos>, <StzIPos>, <StzRPos>,
            <StzUppPos>, <StzUppIPos>, <StzUppRPos>, <StzUppRIPos>,
            <StzcPos>, <StzcIPos>, <StzcRPos>, <StzcRIPos>,
            <StzcUppPos>, <StzcUppIPos>, <StzcUppRPos>, <StzcUppRIPos>
}

function StzRLS(s: StzPtr): StzPtr;
{ Removes Leading Spaces: Remove all leading carriage return, line feed,
  form feed, horizontal tab, vertical tab and space characters from string
  s. Returns s.

  See also: <StzRAS>, <StzRTS>
}

function StzRPos(const src: StzPtr; const pat: StzPtr): StzPtr;
{ Returns a pointer to the last position of string pattern pat in string
  src, or nil if pat is empty or cannot be found in src.

  See also: <StzPos>, <StzIPos>, <StzRIPos>,
            <StzUppPos>, <StzUppIPos>, <StzUppRPos>, <StzUppRIPos>,
            <StzcPos>, <StzcIPos>, <StzcRPos>, <StzcRIPos>,
            <StzcUppPos>, <StzcUppIPos>, <StzcUppRPos>, <StzcUppRIPos>
}

function StzRTS(s: StzPtr): StzPtr;
{ Removes Trailing Spaces: Remove all trailing carriage return, line feed,
  form feed, horizontal tab, vertical tab and space characters from string
  s. Returns s.

  See also: <StzRAS>, <StzRLS>
}

function StzSoundAlike(const s1, s2: StzPtr; i: StzInd): boolean;
{ Tests if StzSoundEx(d1,s1,i) and StzSoundEx(d2,s2,i) yield the same
  results d1 and d2.

  See also:
  <StzSoundEx>, <StzLexCmp>, <StzUppCmp>, <StzNCmp>, <StzLexNCmp>,
  <StzUppNCmp>
}

function  StzSoundEx(dst: StzPtr; const src: StzPtr; i: StzInd): StzPtr;
{ Sound Expression: Creates into dst a string that is derived from string
  src by copying the first i characters, translating lower case letters to
  upper case, and then adding a 'sound code' for the remaining characters. 
  Two strings that yield the same StzSoundEx result will probably sound
  alike in English. This function can therefore be used when searching a
  name or another string value in a database where the correct spelling is
  not certain. 
  Returns dst.

  The exact algorithm for deriving the function result is as follows:
  1. Translate the first i characters to upper case.
  2. Translate the remaining characters to upper case, then code them
     as follows:
       'B','F','P','V'                 become '1'
       'C','G','J','K','Q','S','X','Z' become '2'
       'D','T'                         become '3'
       'L'                             become '4'
       'M','N'                         become '5'
       'R'                             become '6'
       alle other characters           are skipped.
     Moreover, never append the same code digit twice in this coding
     process.

  See also:
  <StzSoundAlike>, <StzLexCmp>, <StzUppCmp>, <StzNCmp>, <StzLexNCmp>,
  <StzUppNCmp>
}

function StzSrt(dst: StzPtr; const src: StzPtr; n: StzInd): StzPtr;
{ Sorts substring fields of string src, having a fixed field length n,
  comparing the ord-value of the characters, and puts the result in dst. 
  If n=0, string src is copied to dst without change.
  Returns dst.

  Example:  src = 'IF    THEN  BEGIN END   WHILE  REPEATDO    ', n=6
            dst = 'BEGIN DO    END   IF    REPEAT THEN  WHILE '

  src and dst must be separate string variables.

  See also: <StzLexSrt>, <StzUppSrt>
}

function StzStpCpy(dst: StzPtr; const src: StpTyp): StzPtr;
{ Converts the type StpTyp string src into type StzPtr string dst appending
  an end-of-string marker. 
  Returns dst.

  See also:
  <StzCpy>, <StzECpy>, <StzNCpy>, <StzcCpy>, <StzcECpy>
}

function StzSub(dst: StzPtr; const src: StzPtr; i,n: StzInd): StzPtr;
{ Returns into dst a substring from s, consisting of a maximum of n
  characters starting at position i. 
  Returns dst.

  src and dst must be separate string variables.

  See also: <StzAfter>, <StzBefore>, <StzRight>
}

function StzTrunc(s: StzPtr; i: StzInd): StzPtr;
{ Replaces the character at position i in string s with an end-of-string
  marker, effectively truncating string s to a length of i.
  If i >= StzLen(s), string s will not be altered. 
  Returns s. }

function StzUpp(s: StzPtr): StzPtr;
{ Converts all lower case letters in string s to upper case.
  Returns s.

  See also: <StzLow>
}

function StzUppCmp(const s1, s2: StzPtr): integer;
{ Compares strings s1 and s2, seeing upper and lower case characters as
  equal. 
  The result is negative if s1<s2, zero if s1=s2 and positive if s1>s2.

  See also:
  <StzCmp>, <StzLexCmp>, <StzNCmp>, <StzLexNCmp>, <StzUppNCmp>,
  <StzSoundEx>, <StzSoundAlike>
}

function StzUppIPos(const src: StzPtr; const pat: StzPtr): StzInd;
{ Returns the first position of string pattern pat in string src,
  or the value StzNOTFOUND if pat cannot be found in src.
  Upper and lower case characters are seen as equal.

  See also: <StzPos>, <StzIPos>, <StzRPos>, <StzRIPos>,
            <StzUppPos>, <StzUppRPos>, <StzUppRIPos>,
            <StzcPos>, <StzcIPos>, <StzcRPos>, <StzcRIPos>,
            <StzcUppPos>, <StzcUppIPos>, <StzcUppRPos>, <StzcUppRIPos>
}

function StzUppNCmp(const s1, s2: StzPtr; n: StzInd): integer;
{ Compares a maximum of n characters of strings s1 and s2, seeing upper 
  and lower case characters as equal. 
  The result is negative if s1<s2, zero if s1=s2 and positive if s1>s2.

  See also:
  <StzCmp>, <StzLexCmp>, <StzUppCmp>, <StzNCmp>, <StzLexNCmp>,
  <StzSoundEx>, <StzSoundAlike>
}

function StzUppPos(const src, pat: StzPtr): StzPtr;
{ Returns a pointer to the first position of string pattern pat in string
  src, or nil if pat is empty or cannot be found in src.
  Upper and lower case characters are seen as equal.

  See also: <StzPos>, <StzIPos>, <StzRPos>, <StzRIPos>,
            <StzUppPos>, <StzUppIPos>, <StzUppRPos>, <StzUppRIPos>,
            <StzcPos>, <StzcIPos>, <StzcRPos>, <StzcRIPos>,
            <StzcUppIPos>, <StzcUppRPos>, <StzcUppRIPos>
}

function StzUppRIPos(const src: StzPtr; const pat: StzPtr): StzInd;
{ Returns the last position of string pattern pat in string src, or the
  value StzNOTFOUND if pat cannot be found in src.
  Upper and lower case characters are seen as equal.

  See also: <StzPos>, <StzIPos>, <StzRPos>, <StzRIPos>,
            <StzUppPos>, <StzUppIPos>, <StzUppRPos>,
            <StzcPos>, <StzcIPos>, <StzcRPos>, <StzcRIPos>,
            <StzcUppPos>, <StzcUppIPos>, <StzcUppRPos>, <StzcUppRIPos>
}

function StzUppRPos(const src: StzPtr; const pat: StzPtr): StzPtr;
{ Returns a pointer to the last position of string pattern pat in string
  src, or nil if pat is empty or cannot be found in src.
  Upper and lower case characters are seen as equal.

  See also: <StzPos>, <StzIPos>, <StzRPos>, <StzRIPos>,
            <StzUppPos>, <StzUppIPos>, <StzUppRPos>,
            <StzcPos>, <StzcIPos>, <StzcRPos>, <StzcRIPos>,
            <StzcUppPos>, <StzcUppIPos>, <StzcUppRPos>, <StzcUppRIPos>
}

function StzUppSrt(dst: StzPtr; const src: StzPtr; n: StzInd): StzPtr;
{ Like StzSrt, seeing upper and lower case characters as equal.

  See also: <StzSrt>, <StzLexSrt>
}


IMPLEMENTATION
uses
  StpLib,
{$IFDEF VER70}  {BP70}
  Strings,
{$else}         {DELPHI}
  SysUtils,
{$endif}
  ChrLib;

const
  BS  = chr(AsciiBS) ; { #008 }
  HT  = chr(AsciiHT) ; { #009 }
  CR  = chr(AsciiCR) ; { #013 }

function StzAfter(const src,pat: StzPtr): StzPtr;
var p: PChar;
begin
  p:= StzPos(src,pat);
  if p=nil then StzAfter:= StzEnd(src)
           else StzAfter:= p + StrLen(pat);
end;

function StzAlloc(const s: StzPtr): StzPtr;
var p: StzPtr;
begin
  if s=nil then StzAlloc:= nil
  else begin
    GetMem(p,StrLen(s)+1);
    if p<>nil then StzCpy(p,s);
    StzAlloc:= p;
  end;
end;

function  StzBefore(dst: StzPtr; const src,pat: StzPtr): StzPtr;
var p: PChar;
begin
  p:= StzPos(src,pat);
  if p=nil then StzBefore:= StrCopy(dst,src)
           else StzBefore:= StrLCopy(dst,src,p-Src);
end;

function  StzCat(dst: StzPtr; const src: StzPtr): StzPtr;
begin StzCat:= StrCat(dst,src); end;

function StzCenter(s: StzPtr; n: StzInd): StzPtr;
var l: StzInd; p: PChar;
begin
  StzRLS(s); StzRTS(s);
  l:= StrLen(s);
  if (l > 0) and (n > l+1) then begin
    p:= s + (n - l) div 2;
    Move(s^,p^,l+1);
    FillChar(s^,p-s,' ');
  end;
  StzCenter:= s;
end;

function StzCmp(const s1, s2: StzPtr): integer;
begin StzCmp:= StrComp(s1,s2); end;

function StzCpy(dst: StzPtr; const src: StzPtr): StzPtr;
begin StzCpy:= StrCopy(dst,src); end;

function StzCreate(s: StzPtr): StzPtr;
begin s^:= #0; StzCreate:= s; end;

function StzcCat(dst: StzPtr; c: char): StzPtr;
var p: PChar;
begin
  if c <> #0 then begin
    p:= StrEnd(dst);
    p[0]:= c; p[1]:= #0;
  end;
  StzcCat:= dst;
end;

function StzcCpy(dst: StzPtr; c: char): StzPtr;
begin dst[0]:= #0; StzcCpy:= StzcCat(dst,c); end;

function StzcECat(dst: StzPtr; c: char): StzPtr;
var p: PChar;
begin
  p:= StrEnd(dst);
  if c <> #0 then begin p^:= c; Inc(p); p^:= #0; end;
  StzcECat:= p;
end;

function StzcECpy(dst: StzPtr; c: char): StzPtr;
begin
  dst^:= #0; StzcECpy:= StzcECat(dst,c);
end;

function StzcGet(s: StzPtr): char;
var c: char;
begin
  c:= s[0]; if c<>#0 then StrCopy(s,s+1);
  StzcGet:=c
end;

function StzcIns(dst:StzPtr; c:char; i:StzInd): StzPtr;
var tmp: array[0..1] of char;
begin
  if c <> #0 then begin
    tmp[0]:= c; tmp[1]:= #0;
    StzcIns:= StzIns(dst,tmp,i);
  end
  else StzcIns:= dst;
end;

function StzcIPos(const s: StzPtr; c: char): StzInd;
var p: StzPtr;
begin
  p:= StrScan(s,c);
  if p=nil then StzcIPos:= StzNOTFOUND
  else StzcIPos:= p-s;
end;

function StzcPos(const s: StzPtr; c: char): StzPtr;
begin StzcPos:= StrScan(s,c); end;

function StzcRet(const s: StzPtr; i: StzInd): char;
begin if i > StrLen(s) then StzcRet:= #0 else StzcRet:=s[i]; end;

function StzcRIPos(const s: StzPtr; c: char): StzInd;
var p: StzPtr;
begin
  p:= StrRScan(s,c);
  if p=nil then StzcRIPos:= StzNOTFOUND
  else StzcRIPos:= p-s;
end;

function StzcRPos(const s: StzPtr; c: char): StzPtr;
begin StzcRPos:= StrRScan(s,c); end;

function StzcUpd(s: StzPtr; c: char; i: StzInd): StzPtr;
begin
  if i < StrLen(s) then s[i]:= c;
  StzcUpd:= s;
end;

function StzcUppIPos(const s: StzPtr; c: char): StzInd;
var p: StzPtr;
begin
  p:= StzcUppPos(s,c);
  if p=nil then StzcUppIPos:= StzNOTFOUND
  else StzcUppIPos:= p-s;
end;

function StzcUppPos(const s: StzPtr; c: char): StzPtr;
{$ifndef USEASM}
var p: StzPtr; notfound: boolean; c1: char;
begin
  c:= ToUpper(c); p:= s;
  c1:= ToUpper(p^); notfound:= c1 <> c;
  while notfound and (c1<>#0) do begin
    Inc(p); c1:= ToUpper(p^); notfound:= c1 <> c;
  end;
  if notfound then StzcUppPos:= nil else StzcUppPos:= p;
end;
{$else}
assembler;
asm
	push	ds
	lds	si,s
	mov	ah,c
	cmp	ah,'a'
	jb	@1
	cmp	ah,'z'
	ja	@1
	sub	ah,'a'-'A'
@1:	cld

@2:	lodsb
	cmp	al,'a'
	jb	@3
	cmp	al,'z'
	ja	@3
	sub	al,'a'-'A'
@3:	cmp	al,ah
	je	@4		{ match found }
	and	al,al
	jne	@2

	xor	ax,ax		{ no match }
	mov	dx,ax
	jmp	@5

@4:	dec	si
	mov	dx,ds
	mov	ax,si
@5:	pop	ds
end;
{$endif}

function StzcUppRIPos(const s: StzPtr; c: char): StzInd;
var p: StzPtr;
begin
  p:= StzcUppRPos(s,c);
  if p=nil then StzcUppRIPos:= StzNOTFOUND
  else StzcUppRIPos:= p-s;
end;

function StzcUppRPos(const s: StzPtr; c: char): StzPtr;
{$ifndef USEASM}
var p: StzPtr; notfound: boolean; c1: char;
begin
  c:= ToUpper(c); p:= s+StrLen(s);
  c1:= ToUpper(p^); notfound:= c1 <> c;
  while notfound and (p>s) do begin
    Dec(p); c1:= ToUpper(p^); notfound:= c1 <> c;
  end;
  if notfound then StzcUppRPos:= nil else StzcUppRPos:= p;
end;
{$else}
assembler;
asm
	mov	ah,c		{ ah = ToUpper(c) }
	cmp	ah,'a'
	jb	@1
	cmp	ah,'z'
	ja	@1
	sub	ah,'a'-'A'
@1:	cld			{ es:di = StzEnd(s) + 1 }
	les	di,s
	mov	cx,$FFFF
	xor	al,al
	repne	scasb
	not	cx		{ cx = StrLen(s) + 1 }

@2:	dec	di
	mov	al,es:[di]
	cmp	al,'a'
	jb	@3
	cmp	al,'z'
	ja	@3
	sub	al,'a'-'A'
@3:	cmp	al,ah
	je	@5		{ match found }
	loop	@2

	xor	ax,ax		{ no match }
	mov	dx,ax
	jmp	@6

@5:	mov	dx,es
	mov	ax,di
@6:
end;
{$endif}

function StzDel(s: StzPtr; i,n : StzInd): StzPtr;
var l: word;
begin
  if n>0 then begin
    l:= StrLen(s);
    if i<l then begin
      if i+n >= l then s[i]:= #0 else StrCopy(s+i,s+i+n);
    end;
  end;
  StzDel:= s;
end;

function StzDetab(dst: StzPtr; const src: StzPtr; n: StzInd): StzPtr;
{$ifndef USEASM}
var k : integer;              { current display column }
    p1: PChar;                { src pointer            }
    p2: PChar;                { dst pointer            }
begin
  if n=0 then StzDetab:= StrCopy(dst,src)
  else begin
    p1:= src; p2:= dst; k:= 0;
    while p1^ <> #0 do begin
      case p1^ of
        HT : repeat p2^:= ' '; Inc(p2); Inc(k); until (k mod n) = 0;
        CR : begin  p2^:= CR ; Inc(p2); k:= 0; end;
        BS : begin  p2^:= BS ; Inc(p2); if k>0 then Dec(k); end;
        else begin  p2^:= p1^; Inc(p2); if not IsCntrl(p1^) then Inc(k); end;
      end;
      Inc(p1);
    end;
    p2^:= #0; StzDetab:= dst;
  end;
end;
{$else}
assembler;
asm
	mov	cx,n		{ cx = n }
	jcxz	@copy

	push	ds
        lds	si,src		{ ds:si	= p1 (source pointer) }
	les	di,dst		{ es:di	= p2 (destination pointer }
	push	es		{ save result pointer }
	push	di
	cld

@0:	xor	bx,bx		{ bx	= k  (current display column) }
@1:	lodsb
	and	al,al
	je	@5

	cmp	al,HT		{ HT }
	jne	@2
@1a:	mov	al,' '
	stosb
	inc	bx
	mov	ax,bx
	xor	dx,dx
	div	cx
	and	dx,dx
	jnz	@1a
	jmp	@1

@2:	cmp	al,CR		{ CR }
	jne	@3
	stosb
	jmp	@0

@3:	cmp	al,BS		{ BS }
	jne	@4
	stosb
	and	bx,bx
	jz	@1
	dec	bx
	jmp	@1

@4:	stosb			{ other characters }
	cmp	al,1fh		{ IsCntrl ($00..$1F, $7F) }
	jbe	@1
	cmp	al,7fh
	je	@1
	inc	bx
	jmp	@1


@5:	stosb			{ append ending 0-byte }
	pop	ax		{ restore result pointer }
	pop	dx
	pop	ds
	jmp	@ret

@copy:	push    dst.Word[2]
	push	dst.Word[0]
	push	src.Word[2]
	push	src.Word[0]
	call	far ptr StrCopy
@ret:
end;
{$endif}

function StzECat(dst: StzPtr; const src: StzPtr): StzPtr;
begin StzECat:= StrECopy(StrEnd(dst),src); end;

function StzECpy(dst: StzPtr; const src: StzPtr): StzPtr;
begin StzECpy:= StrECopy(dst,src); end;

function StzEmpty(const s: StzPtr): boolean;
begin StzEmpty:= s^ = #0; end;

function StzEnd(const s: StzPtr): StzPtr;
begin StzEnd:= StrEnd(s); end;

function StzEntab(dst: StzPtr; const src: StzPtr; n: StzInd): StzPtr;
{$ifndef USEASM}
var
  tmp: StzPtr;
  k0 : integer;              { starting column of last spacegroup }
  k1 : integer;              { current display column             }
  c  : char;                 { current character                  }
  p1 : PChar;                { tmp pointer                        }
  p2 : PChar;                { dst pointer                        }
begin
  if (n=0) or (src[0]=#0) then StzEntab:= StrCopy(dst,src)
  else begin
    { pass 1, src -> tmp : replace tabs by space groups }
    StzDetab(dst,src,n); tmp:= StzAlloc(dst);
    if tmp=nil then StzEntab:= StrCopy(dst,src)
    else begin
    { pass 2, tmp -> dst : replace space groups by tabs }
      k0:= 0; k1:= 0; p1:= tmp; p2:= dst;
      c:= p1^;
      while c <> #0 do begin
        case c of
          ' ' : begin
                  Inc(k1);
                  if (k1 mod n = 0) then begin
                    if k1 - k0 > 1 then c:= HT;
                    p2^:= c; Inc(p2); k0:= k1
                  end;
                end;
          CR  : begin
                  p2^:= CR; Inc(p2);
                  k0:= 0; k1:= 0;
                end;
          BS  : begin
                  p2^:= BS; Inc(p2);
                  if k1 > 0  then Dec(k1);
                  if k0 > k1 then k0:= k1;
                end;
          else  begin
                  while k0 < k1 do begin p2^:= ' '; Inc(p2); Inc(k0) end;
                  p2^:= c; Inc(p2);
                  if not IsCntrl(c) then Inc(k0);
                  k1:= k0
                end;
        end;
        Inc(p1); c:= p1^;
      end;
      while k0 < k1 do begin p2^:= ' '; Inc(p2); Inc(k0) end;
      p2^:= #0;
      dispose(tmp); StzEntab:= dst;
    end;
  end;
end;
{$else}
var tmp: StzPtr;
begin
  if (n=0) or (src[0]=#0) then StzEntab:= StrCopy(dst,src)
  else begin
    { pass 1, src -> tmp : replace tabs by space groups }
    StzDetab(dst,src,n); tmp:= StzAlloc(dst);
    if tmp=nil then StzEntab:= StrCopy(dst,src)
    { pass 2, tmp -> dst : replace space groups by tabs }
    else begin
      asm
	push	ds
	push	bp
	lds	si,tmp
	les	di,dst
	mov	cx,n
	cld
	mov	ah,' '
@0:	xor	bp,bp		{ bp= k0: starting column last space group }
	xor	bx,bx		{ bx= k1: current display column }

@1:	lodsb
	and	al,al
	je	@5

	cmp	al,ah 		{ ' ' }
	jne	@2
	inc	bx		{ ah=al=' ' guaranteed }
	mov	ax,bx
	xor	dx,dx
	div	cx
	and	dx,dx
	mov	ax,'  '		{ restore ah,al }
	jnz	@1
	dec	bx
	cmp	bx,bp
	jna	@1a
	mov	al,HT
@1a:	inc	bx
	stosb
	mov	bp,bx
	jmp	@1

@2:	cmp	al,CR		{ CR }
	jne	@3
	stosb
	jmp	@0

@3:	cmp	al,BS		{ BS }
	jne	@4
	stosb
	and	bx,bx
	jz	@3a
	dec	bx
@3a:	cmp	bp,bx
	jna	@3b
	mov	bp,bx
@3b:	jmp	@1

@4:	xchg	al,ah		{ other characters }
@4a:	cmp	bp,bx
	jnb	@4b
	stosb
	inc	bp
	jmp	@4a
@4b:	xchg	al,ah
	stosb
	cmp	al,1fh		{ IsCntrl ($00..$1F, $7F) }
	jbe	@4c
	cmp	al,7fh
	je	@4c
	inc	bp
@4c:	mov	bx,bp
	jmp	@1

@5:	xchg	al,ah		{ append remaining spaces }
@5a:	cmp	bp,bx
	jnb	@5b
	stosb
	inc	bp
	jmp	@5a
@5b:	xchg	al,ah		{ append ending 0-byte }
	stosb
	pop	bp
	pop	ds
      end;
    dispose(tmp); StzEntab:= dst;
    end;
  end;
end;
{$endif}

function StzFill(s: StzPtr; c: char; n: StzInd): StzPtr;
var i: word;
begin
  i:= StrLen(s);
  if n>i then begin
    FillChar(s[i],n-i,c);
    s[n]:= #0;
  end;
 StzFill:= s;
end;

procedure StzFree(var p: StzPtr);
begin
  if p<>nil then begin
    dispose(p);
    p:= nil;
  end;
end;

function StzGtw(dst,src: StzPtr): StzPtr;
{$ifndef USEASM}
var p1,p2: PChar;
begin
  p1:= src; p2:= dst;
  while IsSpace(p1^) do Inc(p1);
  while not IsSpace(p1^) and (p1^ <> #0) do begin
    p2^:= p1^; Inc(p1); Inc(p2);
  end;
  p2^:= #0;
  StrCopy(src,p1);
  StzGtw:= dst;
end;
{$else}
assembler;
asm
	mov	dx,ds
	lds	si,src
	les	di,dst
	cld
@1:	lodsb
	cmp	al,' '
	je	@1
	cmp	al,13h
	ja	@2
	cmp	al,09h
	jnb	@1
@2:	and	al,al
	jz	@4
	cmp	al,' '
	je	@4
	cmp	al,13h
	ja	@3
	cmp	al,09h
	jnb	@4
@3:	stosb
	lodsb
	jmp	@2
@4:	dec	si
	xor	al,al
	stosb
	push	src.Word[2]
	push	src.Word[0]
	push	ds
	push	si
	mov	ds,dx
	call	far ptr StrCopy
	mov	dx,dst.Word[2]
	mov	ax,dst.Word[0]
end;
{$endif}

function StzIns(dst: StzPtr; const src: StzPtr; i: StzInd): StzPtr;
var l: word; p1,p2: PChar;
begin
  if i >= StrLen(dst) then StzIns:= StrCat(dst,src)
  else begin
    l:= StrLen(src); p1:= dst+i; p2:= p1+l;
    Move(p1^,p2^,StrLen(p1)+1);
    Move(src^,p1^,l);
    StzIns:= dst;
  end;
end;

function StzIPos(const src: StzPtr; const pat: StzPtr): StzInd;
var p: StzPtr;
begin
  p:= StrPos(src,pat);
  if p=nil then StzIPos:= StzNOTFOUND
  else StzIPos:= p-src;
end;

function StzLen(const s: StzPtr): StzInd;
begin StzLen:= StrLen(s); end;

function StzLexCmp(const s1, s2: StzPtr): integer;
{$ifndef USEASM}
var p1,p2: PChar;
begin
  p1:= s1; p2:= s2;
  while (p1^ <> #0) and (UpCase(p1^) = UpCase(p2^)) do begin
    Inc(p1); Inc(p2);
  end;
  StzLexCmp:= LexOrder(p1^,p2^);
end;
{$else}
assembler;
asm
	mov	dx,ds
	lds	si,s1
	les	di,s2
	cld
@1:	lodsb
	mov	bl,es:[di]
	inc	di
	and	al,al
	jz	@4
	cmp	al,'a'
	jb	@2
	cmp	al,'z'
	ja	@2
	sub	al,'a'-'A'
@2:	cmp	bl,'a'
	jb	@3
	cmp	bl,'z'
	ja	@3
	sub	bl,'a'-'A'
@3:	cmp	al,bl
	je	@1
@4:	mov	ds,dx
	xor	ah,ah
	xor	bh,bh
	push	ax
	push	bx
	call	far ptr LexOrder
end;
{$endif}

function StzLexNCmp(const s1, s2: StzPtr; n: StzInd): integer;
{$ifndef USEASM}
var i: word; p1,p2: PChar;
begin
  if n=0 then StzLexNCmp:= 0
  else begin
    p1:= s1; p2:= s2; i:= 1;
    while (i < n) and (p1^ <> #0) and (UpCase(p1^) = UpCase(p2^)) do begin
      Inc(i); Inc(p1); Inc(p2);
    end;
    StzLexNCmp:= LexOrder(p1^,p2^);
  end;
end;
{$else}
assembler;
asm
	xor	ax,ax
	mov	cx,n
	jcxz	@ret

	mov	dx,ds
	lds	si,s1
	les	di,s2
	cld
@1:	lodsb
	mov	bl,es:[di]
	inc	di
	and	al,al
	jz	@4
	cmp	al,'a'
	jb	@2
	cmp	al,'z'
	ja	@2
	sub	al,'a'-'A'
@2:	cmp	bl,'a'
	jb	@3
	cmp	bl,'z'
	ja	@3
	sub	bl,'a'-'A'
@3:	cmp	al,bl
	jne	@4
	loop	@1
@4:	mov	ds,dx
	xor	bh,bh
	push	ax
	push	bx
	call	far ptr LexOrder
@ret:
end;
{$endif}

function StzLexSrt(dst: StzPtr; const src: StzPtr; n: StzInd): StzPtr;
var i,k,l: StzInd; tmp: PChar;
begin
  l:= StrLen(src);
  if (n = 0) or (n >= l) then StzLexSrt:= StrCopy(dst,src) else begin
    tmp := StrNew(src); dst^:= #0;
    while tmp^ <> #0 do begin
      k:=0; i:=n;
      while i <= l do begin
        if StzLexNCmp(tmp+i, tmp+k, n) < 0 then k:= i; Inc(i,n);
      end;
      StzNCat(dst, tmp+k, n); StzDel(tmp, k, n); Dec(l,n);
    end;
    StrDispose(tmp); StzLexSrt:= dst;
  end
end;

function StzLFill(s: StzPtr; c: char; n: StzInd): StzPtr;
var OldLen,InsLen: StzInd;
begin
  OldLen:= StrLen(s);
  if n > OldLen then begin
    InsLen:= n - OldLen;
    StrMove(s+InsLen,s,OldLen+1);
    FillChar(s^,InsLen,c);
  end;
  StzLFill:= s;
end;

function StzLow(s: StzPtr): StzPtr;
begin StzLow:= StrLower(s); end;

function StzMake(s: StzPtr; c: char; n: StzInd): StzPtr;
begin s^:= #0; Stzmake:= StzFill(s,c,n) end;

function StzNCat(dst: StzPtr; const src: StzPtr; n: StzInd): StzPtr;
begin StzNCat:= StrLCat(dst,src,n+StrLen(dst)); end;

function StzNCmp(const s1, s2: StzPtr; n: StzInd): integer;
begin StzNCmp:= StrLComp(s1,s2,n); end;

function StzNCpy(dst: StzPtr; const src: StzPtr; n: StzInd): StzPtr;
begin StzNCpy:= StrLCopy(dst,src,n) end;

function StzNIns(dst: StzPtr; const src: StzPtr; i,n: StzInd): StzPtr;
var l: word; p1,p2: PChar;
begin
  l:= StrLen(dst);
  if i >= l then StzNIns:= StrLCat(dst,src,l+n)
  else begin
    l:= StrLen(src); if l>n then l:= n; p1:= dst+i; p2:= p1+l;
    Move(p1^,p2^,StrLen(p1)+1);
    Move(src^,p1^,l);
    StzNIns:= dst;
  end;
end;

function StzNRepl(dst: StzPtr; const src: StzPtr; i,n: StzInd): StzPtr;
var l: word;
begin
  l:= StrLen(src); if n>l then n:= l;
  if i+n > 1 then StzFill(dst,' ',i+n-1);
  StzNRepl:= StzNIns(StzDel(dst,i,n),src,i,n);
end;

function StzPos(const src, pat: StzPtr): StzPtr;
begin StzPos:= StrPos(src,pat); end;

function StzRAS(s: StzPtr): StzPtr;
{$ifndef USEASM}
var src,dst: PChar;
begin
  src:= s; dst:= s;
  while (src^ <> #0) do begin
    if not IsSpace(src^) then begin dst^:= src^; Inc(dst); end;
    Inc(src);
  end;
  dst^:= #0;
  StzRAS:= s;
end;
{$else}
assembler;
asm
	push	ds
	cld
	lds	si,s
	les	di,s
@1:	lodsb
	cmp	al,00h
	je	@3
	cmp	al,' '
	je	@1
	cmp	al,0dh
	ja	@2
	cmp	al,09h
	jnb	@1
@2:	stosb
	jmp	@1
@3:	stosb
	mov	dx,s.Word[2]
	mov	ax,s.Word[0]
	pop	ds
end;
{$endif}

function StzReAlloc(var p: StzPtr; const s: StzPtr): StzPtr;
begin StzFree(p); p:= StzAlloc(s); StzReAlloc:= p; end;

function StzRepl(dst: StzPtr; const src: StzPtr; i: StzInd): StzPtr;
begin StzRepl:= StzNRepl(dst,src,i,StrLen(src)) end;

function StzRev(s: StzPtr): StzPtr;
{$ifndef USEASM}
var lft,rgt: PChar; c: char;
begin
 lft:= s;
 if lft^ <> #0 then begin
   rgt:= StrEnd(s)-1;
   while lft < rgt do begin
     c:=lft^; lft^:= rgt^; rgt^:= c; Inc(lft); Dec(rgt);
   end;
 end;
 StzRev:= s;
end;
{$else}
assembler;
asm
	mov	dx,ds		{ save ds }
	lds	si,s		{ ds:si = s }
	les	di,s		{ es:di = StzEnd(s) - 1 }
	mov	cx,$FFFF
	xor	al,al
	cld
	repne	scasb
	dec	di
	dec	di
@1:	cmp	si,di
	jnb	@2
	mov	al,[si]
	xchg	es:[di],al
	mov	[si],al
	inc	si
	dec	di
	jmp	@1
@2:	mov	ds,dx		{ restore ds }
	mov	dx,s.Word[2]
	mov	ax,s.Word[0]
end;
{$endif}

function StzRight(const s: StzPtr; n: StzInd): StzPtr;
var l: word;
begin
  if n=0 then StzRight:= StrEnd(s)
  else begin
   l:= StrLen(s);
   if n >= l then StzRight:= s else StzRight:= s + (l-n);
  end;
end;

function StzRIPos(const src: StzPtr; const pat: StzPtr): StzInd;
var p: StzPtr;
begin
  p:= StzRPos(src,pat);
  if p=nil then StzRIPos:= StzNOTFOUND
  else StzRIPos:= p-src;
end;

function StzRLS(s: StzPtr): StzPtr;
{$ifndef USEASM}
var p: PChar;
begin
  p:= s;
  while IsSpace(p^) do Inc(p);
  StzRLS:= StrCopy(s,p);
end;
{$else}
assembler;
asm
	les	di,s
	push	es		{ dst parm for StrCopy }
	push	di
@1:	mov	al,es:[di]	{ skip white space }
	inc	di
	cmp	al,00h
	je	@2
	cmp	al,' '
	je	@1
	cmp	al,0dh
	ja	@2
	cmp	al,09h
	jnb	@1
@2:	dec	di
	push	es		{ src parm for StrCopy }
	push	di
	call	far ptr StrCopy
end;
{$endif}


function StzRPos(const src: StzPtr; const pat: StzPtr): StzPtr;
{$ifndef USEASM}
var PatLen: StzInd; p: StzPtr; notfound: boolean;
begin
  notfound:= true;
  PatLen:= StrLen(pat);
  if PatLen > 0 then begin
    p:= src+StrLen(src)-PatLen+1;
    while notfound and (p > src) do begin
      Dec(p);
      notfound:= StrLComp(p,pat,PatLen) <> 0;
    end;
  end;
  if notfound then StzRPos:= nil else StzRPos:= p;
end;
{$else}
assembler;
asm
	push	ds
	cld

	les	di,pat		{ dx := StrLen(pat) }
	mov	cx,$FFFF
	xor	al,al
	repne	scasb
	mov	dx,-2
	sub	dx,cx
	jz	@x0		{ exit if dx = 0 }

	les	di,src
	mov	cx,$FFFF	{ bx := StrLen(src) }
	xor	al,al
	repne	scasb
	mov	bx,-2
	sub	bx,cx
(*PM
	jz	@x0		{ exit if bx = 0 }
*)
	sub	bx,dx		{ bx := StrLen(src) - Strlen(pat) }
	jb	@x0		{ exit if bx < 0 }

	lds	si,pat		{ ds:si := pat }
	les	di,src		{ es:di := src + bx }
	add	di,bx

@1:	mov	ax,si
	mov	bp,di
	mov	cx,dx
	repe	cmpsb
	mov	di,bp
	mov	si,ax
	je	@x1
	dec	di
	dec	bx
	jns	@1


@x0:	xor	ax,ax		{ result = nil }
	mov	dx,ax
	jmp	@ret
@x1:	mov	dx,es
	mov	ax,di
@ret:	pop	ds
end;
{$endif}

function StzRTS(s: StzPtr): StzPtr;
{$ifndef USEASM}
var i: word;
begin
  i:= StrLen(s);
  while (i > 0) and IsSpace(s[i-1]) do Dec(i);
  s[i]:= #0;
  StzRTS:= s;
end;
{$else}
assembler;
asm
	cld		{ look for end of string }
	les	di,s
	mov	cx,$FFFF
	xor	al,al
	repne	scasb	{ es:di = StzEnd(s) }
	not	cx
	dec	cx	{ cs - StzLen(s) }
	jcxz	@x

	std
	dec	di
@1:	dec	di
	mov	al,es:[di]
	cmp	al,' '
	je	@2
	cmp	al,0dh
	ja	@3
	cmp	al,09h
	jb	@3
@2:	loop	@1
	jmp	@4
@3:	inc	di
@4:	xor	al,al
	mov	es:[di],al
@x:	mov	dx,s.Word[2]
	mov	ax,s.Word[0]
end;
{$endif}

function StzSoundAlike(const s1, s2: StzPtr; i: StzInd): boolean;
begin StzSoundAlike:= StpSoundAlike(StrPas(s1),StrPas(s2),i); end;

function  StzSoundEx(dst: StzPtr; const src: StzPtr; i: StzInd): StzPtr;
var s1,s2: StpTyp;
begin
  StpStzCpy(s1,src);
  StpSoundEx(s2,s1,i);
  StzSoundEx:= StzStpCpy(dst,s2);
end;

function StzSrt(dst: StzPtr; const src: StzPtr; n: StzInd): StzPtr;
var i,k,l: StzInd; tmp: PChar;
begin
  l:= StrLen(src);
  if (n = 0) or (n >= l) then StzSrt:= StrCopy(dst,src) else begin
    tmp := StrNew(src); dst^:= #0;
    while tmp^ <> #0 do begin
      k:=0; i:=n;
      while i <= l do begin
        if StrLComp(tmp+i, tmp+k, n) < 0 then k:= i; Inc(i,n);
      end;
      StzNCat(dst, tmp+k, n); StzDel(tmp, k, n); Dec(l,n);
    end;
    StrDispose(tmp); StzSrt:= dst;
  end
end;

function StzStpCpy(dst: StzPtr; const src: StpTyp): StzPtr;
begin StzStpCpy:= StrPCopy(dst,src); end;

function StzSub(dst: StzPtr; const src: StzPtr; i,n: StzInd): StzPtr;
begin
  if i >= StrLen(src) then StzSub:= StrCopy(dst,'')
                      else StzSub:= StrLCopy(dst,src+i,n);
end;

function StzTrunc(s: StzPtr; i: StzInd): StzPtr;
begin
  if i < StrLen(s) then s[i]:= #0;
  StzTrunc:= s;
end;

function StzUppSrt(dst: StzPtr; const src: StzPtr; n: StzInd): StzPtr;
var i,k,l: StzInd; tmp: PChar;
begin
  l:= StrLen(src);
  if (n = 0) or (n >= l) then StzUppSrt:= StrCopy(dst,src) else begin
    tmp := StrNew(src); dst^:= #0;
    while tmp^ <> #0 do begin
      k:=0; i:=n;
      while i <= l do begin
        if StrLIComp(tmp+i, tmp+k, n) < 0 then k:= i; Inc(i,n);
      end;
      StzNCat(dst, tmp+k, n); StzDel(tmp, k, n); Dec(l,n);
    end;
    StrDispose(tmp); StzUppSrt:= dst;
  end
end;

function StzUpp(s: StzPtr): StzPtr;
begin StzUpp:= StrUpper(s); end;

function StzUppCmp(const s1, s2: StzPtr): integer;
begin StzUppCmp:= StrIComp(s1,s2); end;

function StzUppIPos(const src: StzPtr; const pat: StzPtr): StzInd;
var p: StzPtr;
begin
  p:= StzUppPos(src,pat);
  if p=nil then StzUppIPos:= StzNOTFOUND
  else StzUppIPos:= p-src;
end;

function StzUppNCmp(const s1, s2: StzPtr; n: StzInd): integer;
begin StzUppNCmp:= StrLIComp(s1,s2,n); end;

function StzUppPos(const src, pat: StzPtr): StzPtr;
{$ifndef USEASM}
var p,p1: StzPtr; notfound: boolean; Patlen: StzInd;
begin
  notfound:= true;
  PatLen:= StrLen(pat);
  if PatLen > 0 then begin
    p:= src-1; p1:= src+StrLen(src)-PatLen;
    while p < p1 do begin
      Inc(p);
      notfound:= StrLIComp(p,pat,PatLen) <> 0;
    end;
  end;
  if notfound then StzUppPos:= nil else StzUppPos:= p;
end;
{$else}
assembler;
asm
	push	ds
	cld

	lds	si,src		{ ds:si = src ptr }
	les	di,pat		{ es:di = pat ptr }

	mov	ah,es:[di]	{ ah = pat[0] }
	and	ah,ah
	jz	@nfd		{ empty pat : not found }
   	cmp	ah,'a'		{ ah = ToUpper(pat[0]) }
	jb	@1
	cmp	ah,'z'
	ja	@1
	sub	ah,'a'-'A'

@1:	lodsb			{ compare first char of pat }
	and	al,al
	jz	@nfd		{ end of src : no match }
	cmp	al,'a'
	jb	@1a
	cmp	al,'z'
	ja	@1a
	sub	al,'a'-'A'
@1a:	cmp	al,ah
	jne	@1

	mov	bx,si		{ save ptr offsets }
	mov	dx,di
@2:	lodsb			{ compare remaining chars of pat }
   	cmp	al,'a'		{ al = ToUpper([src++]) }
	jb	@2a
	cmp	al,'z'
	ja	@2a
	sub	al,'a'-'A'
@2a:	inc	di		{ cl = ToUpper([++pat]) }
	mov	cl,es:[di]
	and	cl,cl
	stc
	jz	@3		{ end pat: set C,Z flags }
	cmp	cl,'a'
        jb	@2b
	cmp	cl,'z'
	ja	@2b
	sub	cl,'a'-'A'
@2b:	cmp	al,cl
	je	@2
	and	al,al		{ no match: set Z flag if end of string }
@3:	mov	di,dx		{ restore ptr offsets }
	mov	si,bx
	jnz	@1
	jc	@match

@nfd:	xor	ax,ax
	mov	dx,ax
	jmp	@x

@match:	dec	si		{ match }
	mov	ax,si
	mov	dx,ds
@x:	pop	ds
end;
{$endif}

function StzUppRIPos(const src: StzPtr; const pat: StzPtr): StzInd;
var p: StzPtr;
begin
  p:= StzUppRPos(src,pat);
  if p=nil then StzUppRIPos:= StzNOTFOUND
  else StzUppRIPos:= p-src;
end;

function StzUppRPos(const src: StzPtr; const pat: StzPtr): StzPtr;
{$ifndef USEASM}
var PatLen: StzInd; p: StzPtr; notfound: boolean;
begin
  notfound:= true;
  PatLen:= StrLen(pat);
  if PatLen > 0 then begin
    p:= src+StrLen(src)-PatLen+1;
    while notfound and (p > src) do begin
      Dec(p);
      notfound:= StrLIComp(p,pat,PatLen) <> 0;
    end;
  end;
  if notfound then StzUppRPos:= nil else StzUppRPos:= p;
end;
{$else}
assembler;
asm
	push	ds
	cld

	les	di,pat		{ dx := StrLen(pat) }
	mov	cx,$FFFF
	xor	al,al
	repne	scasb
	mov	dx,-2
	sub	dx,cx
	jz	@nfd		{ exit if dx = 0 }

	les	di,src
	mov	cx,$FFFF	{ bx := StrLen(src) }
	xor	al,al
	repne	scasb
	mov	bx,-2
	sub	bx,cx
	jz	@nfd		{ exit if bx = 0 }

	sub	bx,dx		{ bx := StrLen(src) - Strlen(pat) }
	jb	@nfd		{ exit if bx < 0 }

	lds	si,pat		{ ds:si := pat }
	les	di,src		{ es:di := src + bx }
	add	di,bx

@1:	push	si
	push	di
	mov	cx,dx
@2:	lodsb
	cmp	al,'a'
	jb	@2a
	cmp	al,'z'
	ja	@2a
	sub	al,'a'-'A'
@2a:	mov	ah,es:[di]
	inc	di
	cmp	ah,'a'
	jb	@2b
	cmp	ah,'z'
	ja	@2b
	sub	ah,'a'-'A'
@2b:	cmp	al,ah
	jne	@3
	loop	@2
@3:	pop	di
	pop	si
	je	@match
	dec	di
	dec	bx
	jns	@1

@nfd:	xor	ax,ax		{ result = nil }
	mov	dx,ax
	jmp	@ret
@match:	mov	dx,es
	mov	ax,di
@ret:	pop	ds
end;
{$endif}

END.
