{ FLTLIB.PAS : Text filter object library

  Title   : FLTLIB
  Version : 1.2
  Date    : Feb 19, 2000
  Author  : J R Ferguson
  Language: Borland Pascal v7.0 with Objects
  Usage   : Unit
  Remarks : See FLTTST.PAS for examples of the use of this library
}

{$V-  Var-string checking off}
{$B-  Short-circuit boolean expression generation }
{$R+  Range check on}

{$UNDEF OUTBUFHEAP}   { UNDEF to work around a BP 7.0 bug resulting in
                        erroneous file output }

Unit FltLib;

INTERFACE
uses DefLib, OpoLib, RngLib, Objects, Dos;

const
  C_Flt_DflMsg    = 'CON';  { Default message output destination }
  C_Flt_HlpMsg    = '';     { Help message output destination }
  C_Flt_MaxBuf    = 65534;  { Max file buffer size in bytes }
  C_Flt_InpBufSiz = 4096;   { Input buffer size in bytes }
  C_Flt_OutBufSiz = 4096;   { Output buffer size in bytes }
  C_Flt_OptChrHlp = 'H';

  { Error codes and messages: }
  C_Flt_ErrOK     = 0;
  C_Flt_ErrArg    = 1;
  C_Flt_ErrFnf    = 2;
  C_Flt_ErrCre    = 3;
  C_Flt_ErrRea    = 4;
  C_Flt_ErrWri    = 5;

  C_Flt_ErrMsg    : array[C_Flt_ErrFnf..C_Flt_ErrWri] of StpTyp =
 ('File not found',
  'File creation error',
  'Read error',
  'Write error'
 );

type
  P_Flt_OutBuf  = ^T_Flt_OutBuf;
  P_Flt_InpFile = ^T_Flt_InpFile;
  P_Flt_MaskList= ^T_Flt_MaskList;
  P_Flt_PipeApp = ^T_Flt_PipeApp;
  P_Flt_FileApp = ^T_Flt_FileApp;
  P_Flt_WildApp = ^T_Flt_WildApp;

  T_Flt_BufInd  = 0..C_Flt_MaxBuf;
  T_Flt_OutBuf  = array[1..C_Flt_OutBufSiz] of char;

  T_Flt_InpFile = Object(TObject)
    opn : boolean;      { - open flag }
    pbc : boolean;      { - push-back character flag }
    efl : boolean;      { - end-of-file flag (not valid until AFTER a read attempt) }
    ind : T_Flt_BufInd; { - current buffer index }
    top : T_Flt_BufInd; { - current buffer top }
    siz : T_Flt_BufInd; { - input buffer size }
    buf : Pointer;      { - input buffer }
    fil : word;         { - file handle }
    function    ReadBuffer: boolean;
    Constructor Init(V_BufSiz: word);
    Destructor  Done; virtual;
    function    OpenFile(V_Path: StpTyp): boolean; virtual;
    procedure   CloseFile; virtual;
    function    GetChar(var V_Char: char): boolean; virtual;
    function    GetLine(var V_Line: StpTyp): boolean; virtual;
    function    EndFile: boolean; virtual; { not valid until AFTER GetChar/GetLine }
  end;

  T_Flt_MaskList= Object(TCollection) { of input file mask string pointers }
    procedure   FreeItem(Item: Pointer); virtual;
  end;

  T_Flt_PipeApp = Object(TObject)
  { Pipe text filter application.
    Used with command format "EXENAME <inp >out [/option...]"
    or with the | command pipe symbol.
  }
{$IFDEF OUTBUFHEAP}
    OutBuf      : P_Flt_OutBuf;
{$ELSE}
    OutBuf      : T_Flt_OutBuf;
{$ENDIF}
    OutOpn      : boolean;
    Msg,
    Out         : Text;
    Inp         : P_Flt_InpFile;
    InpFnm      : StpTyp;
    OutFnm      : StpTyp;
    Options     : P_Opo_Options;
    ErrCod      : integer;
    Ident       : StpPtr;
    Version     : StpPtr;
    Title       : StpPtr;
    Constructor Init(V_Ident, V_Version, V_Title: StpTyp);
    Destructor  Done;               virtual;
    procedure   Run;                virtual;
    procedure   Wi(V_Int: integer); virtual;
    procedure   Ws(V_Stp: StpTyp);  virtual;
    procedure   Wl(V_Stp: StpTyp);  virtual;
    procedure   Help;               virtual;
    procedure   HelpIdent;          virtual;
    procedure   HelpDescription;    virtual;
    procedure   HelpUsage;          virtual;
    procedure   HelpOptions;        virtual;
    procedure   HelpRemarks;        virtual;
    procedure   DefineOptions;      virtual;
    function    ErrOk: boolean;     virtual;
    procedure   SetErrCod(V_ErrCod: integer); virtual;
    procedure   ReadArgs;           virtual;
    function    ProcessArg(V_ArgC: integer; V_Arg: StpTyp): boolean; virtual;
    function    EvaluateArgs: boolean;                    virtual;
    procedure   OpenMsg;  virtual;
    procedure   CloseMsg; virtual;
    procedure   OpenInp;  virtual;
    procedure   CloseInp; virtual;
    procedure   OpenOut;  virtual;
    procedure   CloseOut; virtual;
    procedure	ErrorMsg; virtual;
    procedure   ProcessFile;                              virtual;
    function    GetLine(var V_Line: StpTyp): boolean;     virtual;
    function    PutLine(V_Line: StpTyp): boolean;         virtual;
    function    ProcessLine(var V_Line: StpTyp): boolean; virtual;
  end;

  T_Flt_FileApp = Object(T_Flt_PipeApp)
  { File text filter application, using 1 input file and 1 output file.
    Command format "EXENAME inpfile [outfile] [/option...]"
  }
    DflInpExt   : StpPtr;
    DflOutExt   : StpPtr;
    Constructor Init(V_Ident, V_Version, V_Title,
                     V_DflInpExt, V_DflOutExt: StpTyp);
    Destructor  Done;                  virtual;
    procedure   HelpUsage;             virtual;
    procedure   ReadArgs;              virtual;
    function    ProcessArg(V_ArgC: integer; V_Arg: StpTyp): boolean; virtual;
    function    EvaluateArgs: boolean; virtual;
    procedure   AdjustFileNames;       virtual;
    procedure   Identify;              virtual;
    procedure   ProcessFile;           virtual;
  end;

  T_Flt_WildApp = Object(T_Flt_FileApp)
  { Wildcard text filter application, using one or more input file masks and
    a separate output file for each matching input file.
    Command format "EXENAME inpmask [...] [/option...]"
  }
    MaskList    : P_Flt_MaskList;
    Constructor Init(V_Ident, V_Version, V_Title, V_OutExt: StpTyp);
    Destructor  Done;            virtual;
    procedure   Run;             virtual;
    procedure   HelpUsage;       virtual;
    function    ProcessArg(V_ArgC: integer; V_Arg: StpTyp): boolean; virtual;
    function    EvaluateArgs: boolean; virtual;
  end;


IMPLEMENTATION
uses ArgLib, ChrLib, StpLib, Strings;

{
--- T_Flt_InpFilemethods ---
}

Constructor T_Flt_InpFile.Init(V_BufSiz: word);
begin
  Inherited Init;
  opn:= false; pbc:= false; efl:= true; ind:= 0; top:= 0; fil:= 0;
  if V_BufSiz > C_Flt_MaxBuf then siz:= C_Flt_MaxBuf else siz:= V_BufSiz;
  GetMem(buf,siz);
end;

Destructor  T_Flt_InpFile.Done;
begin
  CloseFile;
  FreeMem(buf,siz);
  Inherited Done;
end;

function    T_Flt_InpFile.OpenFile(V_Path: StpTyp): boolean;
var PathStr: array[0..80] of char;
    cpu: Dos.Registers;
begin if opn then OpenFile:= false else begin
  if StpEmpty(V_Path) then begin fil:= 0; opn:= true; end {standard input}
  else begin
    StpTrunc(V_Path,80); StrPCopy(PathStr,V_Path);
    cpu.ax:= $3D00; { open read-only file handle }
    cpu.ds:= Seg(PathStr); cpu.dx:= Ofs(PathStr);
    MsDos(cpu);
    opn:= (cpu.Flags and FCarry) = 0;
    if opn then fil:= cpu.AX else fil:= 0;
  end;
  pbc:= false; efl:= false; ind:= 0; top:= 0;
  OpenFile:= opn;
end; end;

procedure   T_Flt_InpFile.CloseFile;
var cpu: Dos.Registers;
begin if opn and (fil > 4) then begin
  cpu.ah:= $3E; { close file handle }
  cpu.bx:= fil;
  MsDos(cpu);
  opn:= false; pbc:= false; efl:= true; ind:= 0; top:= 0; fil:= 0;
end; end;


function    T_Flt_InpFile.ReadBuffer: boolean;
var cpu: Dos.Registers;
begin if not opn then ReadBuffer:= false else begin
  cpu.ah:= $3F;   { read via file handle }
  cpu.bx:= fil; cpu.cx:= siz;
  cpu.ds:= Seg(buf^); cpu.dx:= Ofs(buf^);
  MsDos(cpu);
  if (cpu.Flags and FCarry) = 1 then ReadBuffer:= false
  else begin
    top:= cpu.ax; ind:= 0; efl:= top = 0;
    ReadBuffer:= not efl;
  end;
end; end;

function    T_Flt_InpFile.GetChar(var V_Char: char): boolean;
const EOFCHR = #026;
type  BufArr = array[1..C_Flt_MaxBuf] of char;
      BufPtr = ^BufArr;
begin
  if pbc then begin
    pbc:= false;
    V_Char:= BufPtr(buf)^[ind];
    GetChar:= true;
  end
  else begin
    if efl then GetChar:= false
    else begin
      if (ind < top) or ReadBuffer then begin
        Inc(ind);
        V_Char:= BufPtr(buf)^[ind];
        if V_Char = EOFCHR then begin
          efl:= true;
          GetChar:= false;
        end
        else GetChar:= true;
      end
      else GetChar:= false;
    end;
  end;
end;

function    T_Flt_InpFile.GetLine(var V_Line: StpTyp): boolean;
const
  CR      = #013; { Carriage Return }
  SR      = #141; { Soft Return (WordStar) }
  LF      = #010; { Line Feed }
  FF      = #012; { Form Feed }
var
  EndLine : boolean;
  c       : char;
begin
  V_Line:= '';
  if GetChar(c) then begin
    EndLine:= false;
    repeat
      case c of
        CR,SR : begin
                  EndLine:= true;
                  StpcCat(V_Line,c);
                  if GetChar(c) then begin
                    if c = LF then StpcCat(V_Line,c) else pbc:= true;
                  end;
                end;
        LF,FF : begin
                  EndLine:= true;
                  StpcCat(V_Line,c);
                end;
        else    StpcCat(V_Line,c);
      end;
    until EndLine or not GetChar(c);
    GetLine:= true;
  end
  else GetLine:= false;
end;

function    T_Flt_InpFile.EndFile: boolean;
begin EndFile:= efl; end;


{
--- T_Flt_MaskList ---
}

procedure   T_Flt_MaskList.FreeItem(Item: Pointer);
begin StpFree(StpPtr(Item)); end;


{
--- T_Flt_PipeApp methods ---
}


Constructor T_Flt_PipeApp.Init(V_Ident, V_Version, V_Title: StpTyp);
begin
  Inherited Init;
  ErrCod  := C_Flt_ErrOK;
  StpCreate(InpFnm);
  StpCreate(OutFnm);
  OutOpn  := false;
  Ident   := StpAlloc(V_Ident);
  Version := StpAlloc(V_Version);
  Title   := StpAlloc(V_Title);
  New(Inp,Init(C_Flt_InpBufSiz));
  New(Options,Init);
  DefineOptions;
end;

Destructor  T_Flt_PipeApp.Done;
begin
  Dispose(Options,Done);
  Dispose(Inp,Done);
  StpFree(Title);
  StpFree(Version);
  StpFree(Ident);
  Inherited Done;
end;

procedure   T_Flt_PipeApp.DefineOptions;
begin Options^.DefSw(C_Flt_OptChrHlp); end;

procedure   T_Flt_PipeApp.Run;
begin
  ReadArgs;
  OpenMsg;
  if ErrOK then begin
    OpenInp; OpenOut;
    if ErrOK then ProcessFile;
    CloseInp; CloseOut;
  end;
  ErrorMsg;
  CloseMsg;
end;

procedure   T_Flt_PipeApp.ProcessFile;
var line: StpTyp;
begin
  while GetLine(line) do
    if ProcessLine(line) then
      if PutLine(line) then;
end;

function    T_Flt_PipeApp.GetLine(var V_Line: StpTyp): boolean;
begin if not ErrOK then GetLine:= false else begin
  if Inp^.GetLine(V_Line) then GetLine:= true else begin
    if not Inp^.EndFile then SetErrCod(C_Flt_ErrRea);
    GetLine:= false;
  end;
end; end;

function    T_Flt_PipeApp.PutLine(V_Line: StpTyp): boolean;
begin
  if not ErrOK then PutLine:= false
  else begin
    {$I-} write(Out,V_Line); {$I+}
    if IOresult <> 0 then begin
      SetErrCod(C_Flt_ErrWri);
      PutLine:= false;
    end
    else PutLine:= true;
  end;
end;

function    T_Flt_PipeApp.ProcessLine(var V_Line: StpTyp): boolean;
begin ProcessLine:= true; end;

procedure T_Flt_PipeApp.Wi(V_Int: integer); begin write  (Msg,V_Int) end;
procedure T_Flt_PipeApp.Ws(V_Stp: StpTyp);  begin write  (Msg,V_Stp) end;
procedure T_Flt_PipeApp.Wl(V_Stp: StpTyp);  begin writeln(Msg,V_Stp) end;

procedure T_Flt_PipeApp.Help;
begin
  HelpIdent;
  HelpDescription;
  HelpUsage;
  HelpOptions;
  HelpRemarks;
end;

procedure T_Flt_PipeApp.HelpIdent;
begin
Wl(Ident^+' v'+Version^+' - '+Title^);
end;

procedure T_Flt_PipeApp.HelpDescription;
begin end;

procedure T_Flt_PipeApp.HelpUsage;
begin
Wl('Usage  : '+Ident^+' [<inp] [>out] [/option[...] [...]]');
end;

procedure T_Flt_PipeApp.HelpOptions;
begin
Wl('Options: '+C_Flt_OptChrHlp+'     Send this help text to (redirected) output.');
end;

procedure T_Flt_PipeApp.HelpRemarks;
begin end;

function  T_Flt_PipeApp.ErrOK: boolean;
begin ErrOK:= ErrCod = C_Flt_ErrOK; end;

procedure T_Flt_PipeApp.SetErrCod(V_ErrCod: integer);
begin if ErrOK then ErrCod:= V_ErrCod; end;

procedure T_Flt_PipeApp.ReadArgs;
var i,n : ArgInd;
    arg : StpTyp;
    ok  : boolean;
begin
  GetArgs; i:= 0; n:= 0;
  while ErrOK and (i < ArgC) do begin
    Inc(i); StpCpy(arg,ArgV[i]); StpUpp(arg);
    if Options^.Parse(arg,ok) then begin
      if not ok then SetErrCod(C_Flt_ErrArg);
    end
    else begin
      Inc(n);
      if not ProcessArg(n,arg) then SetErrCod(C_Flt_ErrArg);
    end;
  end;
  if not EvaluateArgs then SetErrCod(C_Flt_ErrArg);
  if Options^.ValSw(C_Flt_OptChrHlp) then SetErrCod(C_Flt_ErrArg);
end;

function T_Flt_PipeApp.ProcessArg(V_ArgC: integer; V_Arg: StpTyp): boolean;
begin ProcessArg:= false; end;

function T_Flt_PipeApp.EvaluateArgs: boolean;
begin EvaluateArgs:= true; end;

procedure T_Flt_PipeApp.OpenMsg;
begin
  if Options^.ValSw(C_Flt_OptChrHlp) then Assign(Msg,C_Flt_HlpMsg)
  else Assign(Msg,C_Flt_DflMsg);
  rewrite(Msg);
end;

procedure T_Flt_PipeApp.CloseMsg;
begin Close(Msg) end;

procedure T_Flt_PipeApp.OpenInp;
begin if ErrOK then begin
  if not Inp^.OpenFile(InpFnm) then SetErrCod(C_Flt_ErrFnf);
end; end;

procedure T_Flt_PipeApp.CloseInp;
begin Inp^.CloseFile; end;

procedure T_Flt_PipeApp.OpenOut;
begin if ErrOK then begin
  Assign(Out,OutFnm);
{$IFDEF OUTBUFHEAP}
  new(OutBuf); SetTextBuf(Out,OutBuf^);
{$ELSE}
  SetTextBuf(Out,OutBuf);
{$ENDIF}
  {$I-} rewrite(Out); {$I+}
  if IOresult <> 0 then SetErrCod(C_Flt_ErrCre) else OutOpn:= true;
end; end;

procedure T_Flt_PipeApp.CloseOut;
begin if OutOpn then begin
{$IFDEF OUTBUFHEAP}
  dispose(OutBuf);
{$ENDIF}
  {$I-} Close(Out); {$I+}
  if IOresult = 0 then OutOpn:= false else SetErrCod(C_Flt_ErrWri);
end; end;

procedure T_Flt_PipeApp.ErrorMsg;
begin if not ErrOK then begin
  if (ErrCod=C_Flt_ErrArg) then Help
  else begin
    write(Msg,C_Flt_ErrMsg[ErrCod]);
    case ErrCod of
      C_Flt_ErrFnf,C_Flt_ErrRea: writeln(Msg,': ',InpFnm);
      C_Flt_ErrCre,C_Flt_ErrWri: writeln(Msg,': ',OutFnm);
    end;
  end;
end; end;


{
--- T_Flt_FileApp methods ---
}


Constructor T_Flt_FileApp.Init(V_Ident, V_Version, V_Title,
                               V_DflInpExt, V_DflOutExt: StpTyp);
begin
  Inherited Init(V_Ident,V_Version,V_Title);
  DflInpExt:= StpAlloc(V_DflInpExt);
  DflOutExt:= StpAlloc(V_DflOutExt);
end;

destructor  T_Flt_FileApp.Done;
begin
  StpFree(DflOutExt);
  StpFree(DflInpExt);
  Inherited Done;
end;

procedure   T_Flt_FileApp.HelpUsage;
begin
Wl('Usage  : '+Ident^+' inpfile [outfile] [/option[...] [...]]');
Wl(' where   inpfile = [inppath]inpname[.ext]');
Wl('           default inppath = current directory');
Ws('           default .ext    = '); Wl(DflInpExt^);
Wl('         outfile = [outpath]outname[.ext]');
Wl('           default outpath = same as inppath');
Wl('           default outname = inpname');
Ws('           default .ext    = '); Wl(DflOutExt^);
end;

procedure   T_Flt_FileApp.ReadArgs;
begin
  Inherited ReadArgs;
  if ErrOK then Identify;
end;

function    T_Flt_FileApp.ProcessArg(V_ArgC: integer; V_Arg: StpTyp): boolean;
begin case V_ArgC of
  1  : begin StpCpy(InpFnm,V_Arg); ProcessArg:= true; end;
  2  : begin StpCpy(OutFnm,V_Arg); ProcessArg:= true; end;
  else ProcessArg:= false;
end; end;

function    T_Flt_FileApp.EvaluateArgs: boolean;
begin
  if StpEmpty(InpFnm) then EvaluateArgs:= false
  else begin
    AdjustFileNames;
    EvaluateArgs:= true;
  end;
end;

procedure T_Flt_FileApp.Identify;
begin writeln(Ident^+' v'+Version^); end;

procedure T_Flt_FileApp.AdjustFileNames;
var Dir: DirStr; InpName, OutName: NameStr; Ext: Extstr;
begin
  FSplit(FExpand(InpFnm),Dir,InpName,Ext);
  if StpEmpty(Ext) then StpCpy(Ext,DflInpExt^);
  InpFnm:= Dir + InpName + Ext;
  FSplit(FExpand(OutFnm),Dir,OutName,Ext);
  if StpEmpty(OutName) then StpCpy(OutName,InpName);
  if StpEmpty(Ext)     then StpCpy(Ext,DflOutExt^);
  if ErrOK then OutFnm:= Dir + OutName + Ext;
end;

procedure   T_Flt_FileApp.ProcessFile;
begin
  writeln(InpFnm,' ==> ',OutFnm);
  Inherited ProcessFile;
end;


{
--- T_Flt_WildApp methods ---
}


Constructor T_Flt_WildApp.Init(V_Ident,V_Version,V_Title,V_OutExt: StpTyp);
begin
  Inherited Init(V_Ident,V_Version,V_Title,'',V_OutExt);
  New(MaskList,Init(10,10));
end;

destructor  T_Flt_WildApp.Done;
begin
  Dispose(MaskList,Done);
  Inherited Done;
end;

procedure   T_Flt_WildApp.Run;
var
  DirInfo: SearchRec; rc: integer;
  Mask,Path: PathStr; Name: NameStr; Ext: ExtStr;
  i: StpInd;
begin
  ReadArgs;
  OpenMsg;
  while ErrOK and (MaskList^.Count > 0) do begin
    Mask:= FExpand(StpPtr(MaskList^.At(0))^); MaskList^.AtFree(0);
    FSplit(Mask,Path,Name,Ext);
    FindFirst(Mask,Archive+ReadOnly,DirInfo); rc:= DosError;
    case rc of
      00 : {ok};
      02 : writeln(Msg,'Directory not found: ',Mask);
      18 : writeln(Msg,'No files: ',Mask);
      else writeln(Msg,'Dos error ',rc,' on FindFirst: ',Mask);
    end;
    while ErrOK and (rc=0) do begin
      StpCpy(InpFnm,Path); StpCat(InpFnm,DirInfo.Name);
      i:= StpcRPos(InpFnm,'.'); if i=0 then i:= StpLen(InpFnm);
      StpSub(OutFnm,InpFnm,1,i-1);
      StpCat(OutFnm,DflOutExt^);
      if StpCmp(InpFnm,OutFnm) = 0 then
        writeln(Msg,InpFnm+' skipped, input = output')
      else begin
        OpenInp; OpenOut;
        if ErrOK then ProcessFile;
        CloseInp; CloseOut;
      end;
      FindNext(DirInfo); rc:= DosError;
    end;
  end;
  ErrorMsg;
  CloseMsg;
end;

procedure   T_Flt_WildApp.HelpUsage;
begin
Wl('Usage  : '+Ident^+' inpmask [...] [/option[...] [...]]');
Wl(' where   inpmask = [path]name.ext');
Wl('         Wildcards * and ? allowed in name and ext.');
Wl('         For each input file matching the specified mask, an output file is');
Ws('         created with the same path and name as the inputfile and .ext=');
Wl(DflOutExt^);
end;

function    T_Flt_WildApp.ProcessArg(V_ArgC: integer; V_Arg: StpTyp): boolean;
begin MaskList^.Insert(StpAlloc(V_Arg)); end;

function    T_Flt_WildApp.EvaluateArgs: boolean;
begin EvaluateArgs:= MaskList^.Count > 0; end;


END.
