{ CVTLIB.PAS : string <-> integer conversion routine library

  Title   : CVTLIB
  Version : 6.0
  Language: Borland Pascal 7.0 (all targets)
            Delphi 1.0 through 3.0
  Date    : Dec 22,1997
  Author  : J.R. Ferguson
  Usage   : Unit
  Remark  : Programs or units that use this unit must be compiled
            with the $N+ compiler directive (use or emulate
            numerical coprocessor).
}

{$IFNDEF WIN32}
{$DEFINE USEASM}  { if defined: use assembler routines }
{$ENDIF}

{$Q- overflow checking off}
{$N+ use or emulate numerical coprocessor}

UNIT CvtLib;

INTERFACE
uses DefLib, ChrLib;


procedure ItoA(n: integer; var s: StpTyp);
{ Convert signed decimal integer n (-32768..32767) to string s. }

procedure ItoAB(n: integer; var s: StpTyp; g: integer);
{ Convert unsigned integer n (0..FFFFh) with base g (1<g<37) to string s.
  See also: <ItoABL>, <ItoABLS>. }

procedure ItoABL(n: integer; var s: StpTyp; g,l: integer);
{ Like <ItoAB>, with fixed field length l, using leading zeroes. }

procedure ItoABLS(n: integer; var s: StpTyp; g,l: integer);
{ Like <ItoAB>, with fixed field length l, using leading spaces. }


function  StfItoA(n: integer): StpTyp;
{ Return string representation of signed decimal integer n 
  (-32768..32767). }

function  StfItoAB(n: integer; g: integer): StpTyp;
{ Return string representation of unsigned integer (0..FFFFh) n with base
  g (1<g<37).
  See also: <StfItoABL>, <StfItoABLS>. }

function  StfItoABL(n: integer; g,l: integer): StpTyp;
{ Like <StfItoAB>, with fixed field length l, using leading zeroes. }

function  StfItoABLS(n: integer; g,l: integer): StpTyp;
{ Like <StfItoAB>, with fixed field length l, using leading spaces. }


function AtoI(s: StpTyp): integer;
{ Read signed decimal integer (-32768..32767) from string s.
  Leading spaces are skipped. The plus or minus sign, if present, must
  be immediately followed by the first digit.
  Trailing mum-mumeric characters are allowed.
  This function does not check for overflow. It simply wraps around.
}

function AtoIB(s: StpTyp; g: integer): integer;
{ Read unsigned integer (0..FFFFh) with base g (1<g<37) from string s.
  Leading spaces are skipped.
  Trailing mum-mumeric (base g) characters are allowed.
  This function does not check for overflow. It simply wraps around.
}


function IfAtoI(s: StpTyp; var n: integer): boolean;
{ Read signed decimal integer n (-32768..32767) from string s.
  Return true if successful, false if s is not the string representation
  of a valid integer.
  Leading spaces are skipped. The plus or minus sign, if present, must
  be immediately followed by the first digit.
  Trailing mum-mumeric characters are not allowed.
}

function IfAtoIB(s: StpTyp; g: integer; var n: integer): boolean;
{ Read unsigned integer n (0..FFFFh) with base g (1<g<37) from string s.
  Return true if successful, false if s is not the string representation
  of a valid unsiged integer in base g.
  Leading spaces are skipped.
  Trailing mum-mumeric (base g) characters are not allowed.
}



function GetIfromA(var s: StpTyp; var n: integer): boolean;
{ Get signed decimal integer n (-32768..32767) from string s and remove
  it from s.
  Any leading spaces are skipped and deleted from s.
  The plus or minus sign, if present, must be immediately followed by
  the first digit. Trailing mum-mumeric characters are left in string s.
  Return true if successful.
  Return false and do not alter s if s does not start with the string
  representation of a valid integer.
}

function GetIBfromA(var s: StpTyp; g: integer; var n: integer): boolean;
{ Get unsigned integer n (0..FFFFh) with base g (1<g<37) from string s
  and remove it from s.
  Any leading spaces are skipped and deleted from s.
  Trailing mum-mumeric (base g) characters are left in string s.
  Return true if successful.
  Return false and do not alter s if s does not start with the string
  representation of a valid integer in base g.
}




procedure WtoA(n: word; var s: StpTyp);
{ Convert unsigned decimal word n (0..65535) to string s. }

procedure WtoAB(n: word; var s: StpTyp; g: integer);
{ Convert unsigned word n (0..FFFFh) with base g (1<g<37) to string s.
  See also: <WtoABL>, <WtoABLS>. }

procedure WtoABL(n: word; var s: StpTyp; g,l: integer);
{ Like <WtoAB>, with fixed field length l, using leading zeroes. }

procedure WtoABLS(n: word; var s: StpTyp; g,l: integer);
{ Like <WtoAB>, with fixed field length l, using leading spaces. }


function  StfWtoA(n: word): StpTyp;
{ Return string representation of unsigned decimal word (0..65535). }

function  StfWtoAB(n: word; g: integer): StpTyp;
{ Return string representation of unsigned word (0..FFFFh) n with base
  g (1<g<37). 
  See also: <StfWtoABL>, <StfWtoABLS>. }

function  StfWtoABL(n: word; g,l: integer): StpTyp;
{ Like <StfWtoAB>, with fixed field length l, using leading zeroes. }


function  StfWtoABLS(n: word; g,l: integer): StpTyp;
{ Like <StfWtoAB>, with fixed field length l, using leading spaces. }


function AtoW(s: StpTyp): word;
{ Read unsigned decimal word (0..65535) from string s.
  Leading spaces are skipped.
  Trailing mum-mumeric characters are allowed.
  This function does not check for overflow. It simply wraps around.
}

function AtoWB(s: StpTyp; g: integer): word;
{ Read unsigned word (0..FFFFh) with base g (1<g<37) from string s.
  Leading spaces are skipped.
  Trailing mum-mumeric (base g) characters are allowed.
  This function does not check for overflow. It simply wraps around.
}


function IfAtoW(s: StpTyp; var n: word): boolean;
{ Read unsigned decimal word n (0..65535) from string s.
  Return true if successful, false if s is not the string representation
  of a valid unsiged decimal integer.
  Leading spaces are skipped.
  Trailing mum-mumeric characters are not allowed.
}

function IfAtoWB(s: StpTyp; g: integer; var n: word): boolean;
{ Read unsigned word n (0..FFFFh) with base g (1<g<37) from string s.
  Return true if successful, false if s is not the string representation
  of a valid unsiged integer in base g.
  Leading spaces are skipped.
  Trailing mum-mumeric (base g) characters are not allowed.
}


function GetWfromA(var s: StpTyp; var n: word): boolean;
{ Get unsigned decimal word n (0..65535) from string s and remove it
  from s.
  Any leading spaces are skipped and deleted from s.
  Trailing mum-mumeric characters are left in string s.
  Return true if successful.
  Return false and do not alter s if s does not start with the string
  representation of a valid unsigned decimal word.
}

function GetWBfromA(var s: StpTyp; g: integer; var n: word): boolean;
{ Get unsigned word n (0..FFFFh) with base g (1<g<37) from string s
  and remove it from s.
  Any leading spaces are skipped and deleted from s.
  Trailing mum-mumeric (base g) characters are left in string s.
  Return true if successful.
  Return false and do not alter s if s does not start with the string
  representation of a valid unsigned word in base g.
}




procedure LtoA(n: longint; var s: StpTyp);
{ Convert signed decimal longint n (-2147483648..2147483647) to string s. }

procedure LtoAB(n: longint; var s: StpTyp; g: integer);
{ Convert unsigned longint n (0..FFFF FFFFh) with base g (1<g<37) to
  string s.
  See also: <LtoABL>, <LtoABLS>. }


procedure LtoABL(n: longint; var s: StpTyp; g,l: integer);
{ Like <LtoAB>, with fixed field length l, using leading zeroes. }

procedure LtoABLS(n: longint; var s: StpTyp; g,l: integer);
{ Like <LtoAB>, with fixed field length l, using leading spaces. }


function  StfLtoA(n: longint): StpTyp;
{ Return string representation of signed decimal longint
  (-2147483648..2147483647). }

function  StfLtoAB(n: longint; g: integer): StpTyp;
{ Return string representation of unsigned longint n (0..FFFF FFFFh)
  with base g (1<g<37).
  See also: <StfLtoABL>, <StfLtoABLS>. }


function  StfLtoABL(n: longint; g,l: integer): StpTyp;
{ Like <StfLtoAB>, with fixed field length l, using leading zeroes. }

function  StfLtoABLS(n: longint; g,l: integer): StpTyp;
{ Like <StfLtoAB>, with fixed field length l, using leading spaces. }


function AtoL(s: StpTyp): longint;
{ Read signed decimal longint (-2147483648..2147483647) from string s.
  Leading spaces are skipped. The plus or minus sign, if present, must
  be immediately followed by the first digit.
  Trailing mum-mumeric characters are allowed.
  This function does not check for overflow. It simply wraps around.
}

function AtoLB(s: StpTyp; g: integer): longint;
{ Read unsigned longint with (0..FFFF FFFF) base g (1<g<37) from string s.
  Leading spaces are skipped.
  Trailing mum-mumeric (base g) characters are allowed.
  This function does not check for overflow. It simply wraps around.
}


function IfAtoL(s: StpTyp; var n: longint): boolean;
{ Read signed decimal longint n (-2147483648..2147483647) from string s.
  Return true if successful, false if s is not the string representation
  of a valid longint.
  Leading spaces are skipped. The plus or minus sign, if present, must
  be immediately followed by the first digit.
  Trailing mum-mumeric characters are not allowed.
}

function IfAtoLB(s: StpTyp; g: integer; var n: longint): boolean;
{ Read unsigned longint n (0..8FFF FFFF) with base g (1<g<37) from 
  string s.
  Return true if successful, false if s is not the string representation
  of a valid unsiged integer in base g.
  Leading or trailing mum-mumeric (base g) characters are not allowed.
}


function GetLfromA(var s: StpTyp; var n: longint): boolean;
{ Get signed decimal longint n (-2147483648..2147483647) from string s
  and remove it from s.
  Any leading spaces are skipped and deleted from s.
  The plus or minus sign, if present, must be immediately followed by
  the first digit. Trailing mum-mumeric characters are left in string s.
  Return true if successful.
  Return false and do not alter s if s does not start with the string
  representation of a valid longint.
}

function GetLBfromA(var s: StpTyp; g: integer; var n: longint): boolean;
{ Get unsigned longint n (0..8FFF FFFF) with base g (1<g<37) from string
  s and remove it from s.
  Any leading spaces are skipped and deleted from s.
  Trailing mum-mumeric (base g) characters are left in string s.
  Return true if successful.
  Return false and do not alter s if s does not start with the string
  representation of a valid integer in base g.
}





IMPLEMENTATION

{ --- Local routines --- }

function CvtDigChr(i: integer): char;
begin if i>9 then CvtDigChr:= chr(i+55) else CvtDigChr:= chr(i+48) end;

procedure CvtUDiv(t,n: integer; var q,r: integer); { n <> 0 }
{$IFDEF USEASM}
begin asm
  mov	ax,t	{ DX:AX = t }
  xor	dx,dx
  div	n	{ divide t/n, quotient in AX, remainder in DX }
  les	di,q
  mov	[es:di],ax
  les	di,r
  mov	[es:di],dx
end; end;
{$ELSE}
var i : 0..15;
begin
  r:= 0; q:= t;
  for i:= 0 to 15 do begin
    r:= r shl 1; if q<0 then Inc(r); q:= q shl 1;
    if r>=n then begin Dec(r,n); Inc(q); end
  end
end;
{$ENDIF}

procedure CvtULDiv(    t: longint;     { numerator }
                       n: integer;     { denominator  <> 0 }
                   var q: longint;     { quotient  }
                   var r: integer);    { renainder }
{$IFDEF USEASM}
begin asm
	xor	bx,bx		{ BX   =r := 0; }
	mov	ax,word(t)	{ DX:AX=q := t; }
	mov	dx,word(t+2)
	mov	di,n		{ DI   =n }

	mov	cx,32		{ for i:= 0 to 31 do begin }
@1:	shl	bx,1		{   r:= r shl 1; }
	and	dx,dx		{   if q < 0 then }
	jns	@2
	inc	bx		{     Inc(r); }
@2:     shl	ax,1		{   q:= q shl 1; }
	rcl	dx,1
	cmp	bx,di		{   if r >= n then begin }
	jb	@3
	sub	bx,di		{     Dec(r,n); }
	add	ax,1		{     Inx(q); }
	adc	dx,0
@3:				{   end; }
	loop	@1		{ end; }

	les	di,q
	mov	[es:di],ax
	mov	[es:di+2],dx
	les	di,r
	mov	[es:di],bx
end; end;
{$ELSE}
var i : 0..31;
begin
  r:= 0; q:= t;
  for i:= 0 to 31 do begin
    r:= r shl 1; if q<0 then Inc(r); q:= q shl 1;
    if r>=n then begin Dec(r,n); Inc(q) end
  end
end;
{$ENDIF}

function DigVal(c:char; g: integer; var d: integer): boolean;
begin
  if IsAlNum(c) then begin
    if c > '9' then d:= ord(UpCase(c))-55 else d:= ord(c) - 48;
    DigVal:= d < g
  end
  else DigVal:= false
end;


{ --- Library routines --- }


procedure ItoA(n: integer; var s: StpTyp);
begin
  if n<0 then begin ItoAB(-n,s,10); Insert('-',s,1) end else ItoAB(n,s,10)
end;

procedure ItoAB(n: integer; var s: StpTyp; g: integer);
var r: integer;
begin
  s:= '';
  if (g>1) and (g<37) then repeat
    CvtUDiv(n,g,n,r); Insert(CvtDigChr(r),s,1);
  until n=0;
end;

procedure ItoABL(n: integer; var s: StpTyp; g,l: integer);
var r: integer;
begin
  s:= '';
  if (g>1) and (g<37) then while l>0 do begin
    CvtUDiv(n,g,n,r); Insert(CvtDigChr(r),s,1); l:= l-1;
  end;
end;

procedure ItoABLS(n: integer; var s: StpTyp; g,l: integer);
var i: StpInd;
begin
  ItoABL(n,s,g,l);
  i:= 1;
  while (i < l) and (s[i] = '0') do begin s[i]:= ' '; Inc(i); end;
end;


function  StfItoA(n: integer): StpTyp;
var s: Stptyp;
begin ItoA(n,s); StfItoA:= s; end;

function  StfItoAB(n,g: integer): StpTyp;
var s: Stptyp;
begin ItoAB(n,s,g); StfItoAB:= s; end;

function  StfItoABL(n: integer; g,l: integer): StpTyp;
var s: Stptyp;
begin ItoABL(n,s,g,l); StfItoABL:= s; end;

function  StfItoABLS(n: integer; g,l: integer): StpTyp;
var s: Stptyp;
begin ItoABLS(n,s,g,l); StfItoABLS:= s; end;


function AtoI(s: StpTyp): integer;
begin AtoI:= integer(AtoL(s) and $FFFF) end;

function AtoIB(s: StpTyp; g: integer): integer;
begin AtoIB:= integer(AtoLB(s,g) and $FFFF) end;


function IfAtoI(s: StpTyp; var n: integer): boolean;
var s0: StpTyp; n0: longint; ok: boolean;
begin
  s0:= s; ok:= false;
  if GetLfromA(s0,n0) then
    if (s0 = '') and (n0 <= 32767) and (n0 >= -32768) then begin
      n:= n0;
      ok:= true;
    end;
  IfAtoI:= ok;
end;

function IfAtoIB(s: StpTyp; g: integer; var n: integer): boolean;
var w: word; ok: boolean;
begin
  w:= word(n);
  ok:= IfAtoWB(s,g,w);
  n:= integer(w);
  IfAtoIB:= ok;
end;

function GetIfromA(var s: StpTyp; var n: integer): boolean;
var s0: StpTyp; n0: longint; ok: boolean;
begin
  s0:= s; ok:= false;
  if GetLfromA(s0,n0) then
    if (n0 <= 32767) and (n0 >= -32768) then begin
      n:= n0; s:= s0;
      ok:= true;
    end;
  GetIfromA:= ok;
end;

function GetIBfromA(var s: StpTyp; g: integer; var n: integer): boolean;
var w: word; ok: boolean;
begin
  w:= word(n);
  ok:= GetWBfromA(s,g,w);
  n:= integer(w);
  GetIBfromA:= ok;
end;




procedure WtoA(n: word; var s: StpTyp);
begin ItoAB(integer(n),s,10) end;

procedure WtoAB(n: word; var s: StpTyp; g: integer);
begin ItoAB(integer(n),s,g) end;

procedure WtoABL(n: word; var s: StpTyp; g,l: integer);
begin ItoABL(integer(n),s,g,l) end;

procedure WtoABLS(n: word; var s: StpTyp; g,l: integer);
begin ItoABLS(integer(n),s,g,l) end;


function  StfWtoA(n: word): StpTyp;
var s: Stptyp;
begin ItoAB(integer(n),s,10); StfWtoA:= s; end;


function  StfWtoAB(n: word; g: integer): StpTyp;
var s: Stptyp;
begin ItoAB(integer(n),s,g); StfWtoAB:= s; end;


function  StfWtoABL(n: word; g,l: integer): StpTyp;
var s: Stptyp;
begin ItoABL(integer(n),s,g,l); StfWtoABL:= s; end;


function  StfWtoABLS(n: word; g,l: integer): StpTyp;
var s: Stptyp;
begin ItoABLS(integer(n),s,g,l); StfWtoABLS:= s; end;



function AtoW(s: StpTyp): word;
begin AtoW:= word(AtoLB(s,10) and $FFFF) end;


function AtoWB(s: StpTyp; g: integer): word;
begin AtoWB:= word(AtoLB(s,g) and $FFFF) end;



function IfAtoW(s: StpTyp; var n: word): boolean;
begin IfAtoW:= IfAtoWB(s,10,n) end;

function IfAtoWB(s: StpTyp; g: integer; var n: word): boolean;
var s0: StpTyp; n0: longint; ok: boolean;
begin
  s0:= s; ok:= false;
  if GetLBfromA(s0,g,n0) then
    if (s0 = '') and (n0 <= 65535) and (n0 >= 0) then begin
      n:= n0;
      ok:= true;
    end;
  IfAtoWB:= ok;
end;



function GetWfromA(var s: StpTyp; var n: word): boolean;
begin GetWfromA:= GetWBfromA(s,10,n) end;

function GetWBfromA(var s: StpTyp; g: integer; var n: word): boolean;
var s0: StpTyp; n0: longint; ok: boolean;
begin
  s0:= s; ok:= false;
  if GetLBfromA(s0,g,n0) then
    if (n0 <= 65535) and (n0 >= 0) then begin
      n:= n0; s:= s0;
      ok:= true;
    end;
  GetWBfromA:= ok;
end;





procedure LtoA(n: longint; var s: StpTyp);
begin
  if n<0 then begin LtoAB(-n,s,10); Insert('-',s,1) end else LtoAB(n,s,10)
end;

procedure LtoAB(n: longint; var s: StpTyp; g: integer);
var r: integer;
begin
  s:= '';
  if (g>1) and (g<37) then repeat
    CvtULDiv(n,g,n,r); Insert(CvtDigChr(r),s,1);
  until n=0;
end;

procedure LtoABL(n: longint; var s: StpTyp; g,l: integer);
var r: integer;
begin
  s:= '';
  if (g>1) and (g<37) then while l>0 do begin
    CvtULDiv(n,g,n,r); Insert(CvtDigChr(r),s,1); Dec(l);
  end;
end;

procedure LtoABLS(n: longint; var s: StpTyp; g,l: integer);
var i: StpInd;
begin
  LtoABL(n,s,g,l);
  i:= 1;
  while (i < l) and (s[i] = '0') do begin s[i]:= ' '; Inc(i); end;
end;


function StfLtoA(n: longint): StpTyp;
var s: StpTyp;
begin LtoA(n,s); StfLtoA:= s; end;

function StfLtoAB(n: longint; g: integer): StpTyp;
var s: StpTyp;
begin LtoAB(n,s,g); StfLtoAB:= s; end;

function StfLtoABL(n: longint; g,l: integer): StpTyp;
var s: StpTyp;
begin LtoABL(n,s,g,l); StfLtoABL:= s; end;

function StfLtoABLS(n: longint; g,l: integer): StpTyp;
var s: StpTyp;
begin LtoABLS(n,s,g,l); StfLtoABLS:= s; end;


function AtoL(s: StpTyp): longint;
var c   : char;
    i   : StpInd;
    l   : StpInd absolute s;
    n   : longint;
    plus: boolean;
  function NxtChr(var c: char): char;
  begin if i=l then c:= #0 else begin i:= i+1; c:= s[i] end; NxtChr:= c end;
begin {AtoL}
  i:= 0; n:= 0; plus:= true;
  while IsSpace(NxtChr(c)) do ;
  if c in ['+','-'] then begin plus:= c='+'; c:= NxtChr(c) end;
  while IsDigit(c) do begin
    n:= 10*n-(ord(c)-48);
    c:= NxtChr(c);
  end;
  if plus then AtoL:= -n else AtoL:= n
end;

function AtoLB(s: StpTyp; g: integer): longint;
var c : char;
    i : StpInd;
    l : StpInd absolute s;
    d : integer;
    n : longint;
  function NxtChr(var c: char): char;
  begin if i=l then c:=#0 else begin i:=i+1; c:=s[i] end; NxtChr:=c end;
begin {AtoLB}
  if (g>1) and (g<37) then begin
    i:= 0; n:= 0;
    while IsSpace(NxtChr(c)) do ;
    while DigVal(c,g,d) do begin n:= g*n + d; c:= NxtChr(c) end;
    AtoLB:= n;
  end
  else AtoLB:= 0;
end;


function IfAtoL(s: StpTyp; var n: longint): boolean;
var s0: StpTyp; n0: longint; ok: boolean;
begin
  s0:= s; ok:= false;
  if GetLfromA(s0,n0) then if s0 = '' then begin
    n:= n0;
    ok:= true;
  end;
  IfAtoL:= ok;
end;

function IfAtoLB(s: StpTyp; g: integer; var n: longint): boolean;
var s0: StpTyp; n0: longint; ok: boolean;
begin
  s0:= s; ok:= false;
  if GetLBfromA(s0,g,n0) then if s0 = '' then begin
    n:= n0;
    ok:= true;
  end;
  IfAtoLB:= ok;
end;


function GetLfromA(var s: StpTyp; var n: longint): boolean;
var c    : char;
    i    : StpInd;
    l    : StpInd absolute s;
    n0   : longint;
    plus : boolean;
    ok   : boolean;
  function NxtChr(var c: char): char;
  begin if i=l then c:= #0 else begin i:= i+1; c:= s[i] end; NxtChr:= c end;
begin {GetLfromA}
  i:= 0; plus:= true;
  while IsSpace(NxtChr(c)) do ;
  if c in ['+','-'] then begin plus:= c='+'; c:= NxtChr(c) end;
  n0:= 0; ok:= IsDigit(c);
  while ok and IsDigit(c) do begin
    n0:= 10*n0-(ord(c)-48);
    ok:= n0 <= 0; {test overflow}
    c:= NxtChr(c);
  end;
  if ok then begin
    if plus then begin if (n0=$80000000) then ok:= false else n:= -n0 end
    else n:= n0;
    if ok then if (c = #0) then s:= '' else Delete(s,1,i-1);
  end;
  GetLfromA:= ok;
end;

function GetLBfromA(var s: StpTyp; g: integer; var n: longint): boolean;
var c    : char;
    i    : StpInd;
    l    : StpInd absolute s;
    d    : integer;
    n0   : comp;
    ok   : boolean;
  function NxtChr(var c: char): char;
  begin if i=l then c:= #0 else begin i:= i+1; c:= s[i] end; NxtChr:= c end;
begin {GetLBfromA}
  ok:= (g>1) and (g<37);
  if ok then begin
    i:= 0; while IsSpace(NxtChr(c)) do ;
    n0:= 0.0; ok:= DigVal(c,g,d);
    while ok and DigVal(c,g,d) do begin
      n0:= g*n0 + d;
      c:= NxtChr(c);
    end;
    if ok then begin
      if (n0 >= 0.0) and (n0 <= 4294967295.0) then begin
        if n0 <= 2147483647.0
          then n:= trunc(n0)
          else n:= trunc(n0-4294967296.0);
        if c=#0 then s:= '' else Delete(s,1,i-1);
      end
      else ok:= false;
    end;
  end;
  GetLBfromA:= ok;
end;


END.
