{ Title   : PBMLIB.PAS - PibMusic routine library
  Language: Borland Pascal v7.0 (all targets), Delphi 1 through 3
  Version : 1.0
  Date    : April 30,1999
  Author  : Philip R. Burns, J.R.Ferguson
  Usage   : Unit

  History :
  Apr 30,1999 Pascal Unit PBMLIB v1.0 by J.R. Ferguson.
              - Use of unit SndLib by J.R. Ferguson for Windows support.
              - Renamed interfaced identifiers and added a common prefix Pbm_
                to them, in order to avoid name conflicts.
              - Added comment corrections [JRF ...]
  Jan 25,1985 Program PIBMUSIC.PAS v1.0 by Philip R. Burns.
}
{ ------------------------------------------------------------------------- }
{                                                                           }
{   Program:  PibMusic [JRF: Unit PbmLib]                                   }
{                                                                           }
{   Purpose:  Demonstrates the enclosed routine PibPlay, which emulates     }
{             the Microsoft Basic PLAY statement.  (See the Basic manual    }
{             for details.)                                                 }
{                                                                           }
{   Author:   Philip R. Burns                                               }
{   Date:     January 25, 1985                                              }
{   Version:  1.0                                                           }
{                                                                           }
{   Use:                                                                    }
{                                                                           }
{      Call PibPlaySet to initialize global play variables.[JRF: Pbm_Reset] }
{      Call PibPlay to play a line of music.               [JRF: Pbm_Play]  }
{                                                                           }
{   Remarks:  You are free to use this routine is your own code.  If you    }
{             find any bugs or have suggestions for improvements, please    }
{             leave them for me on one of the following two Chicago BBSs:   }
{                                                                           }
{                Gene Plantz's IBBS    (312) 882 4227                       }
{                Ron Fox's RBBS        (312) 940 6496                       }
{                                                                           }
{             Thanks.                                                       }
{                                                                           }
{             Note: This code ignores requests for buffered music.          }
{                                                                           }
{ ------------------------------------------------------------------------- }

{$V-} {relaxed string var length checking}

Unit PbmLib;

Interface
{Uses Crt;} {JRF:} Uses SndLib;

Const
  Pbm_MaxSoundStr  = 254;             { Max length of Pbm_Play sound string  }

  { The following typed constants are used as initial values by Pbm_Reset:   }
  Pbm_Dfl_Octave   : Integer = 4;     { Initial octave sets note A to 440 Hz }
  Pbm_Dfl_Fraction : Real    = 0.875; { Initial sustain is semi-legato       }
  Pbm_Dfl_Length   : Real    = 0.25;  { Initial length is quarter note       }
  Pbm_Dfl_Quarter  : Real    = 500.0; { Initial tempo is moderato            }

Type
  Pbm_SoundStr     = String[Pbm_MaxSoundStr];

Procedure Pbm_Reset;
{ Reset global variables to their default values }

Procedure Pbm_Play(const SoundStr: Pbm_SoundStr);
{ Play music though PC's speaker.
  SoundStr: The string containing the encoded music to be played.
  The format is the same as that of the MicroSoft Basic PLAY Statement.
  The string must be <= 254 characters in length.

  The characters accepted by this routine are:
    A - G   Musical Notes
    # or +  Following A - G note,  indicates sharp
    -       Following A - G note,  indicates flat
    <       Move down one octave
    >       Move up one octave
    .       Dot previous note (extend note duration by 3/2)
    MN      Normal duration (7/8 of interval between notes)
    MS      Staccato duration
    ML      Legato duration
    Ln      Length of note (n=1-64; 1=whole note, 4=quarter note, etc.)
    Pn      Pause length (same n values as Ln above)
    Tn      Tempo, n=notes/minute (n=32-255, default n=120)
    On      Octave number (n=0-6, default n=4)
    Nn      Play note number n (n=0-84)

  The following two commands are IGNORED by Pbm_Play:
    MF      Complete note before continuing
    MB      Another process may begin before speaker is finished playing note
}

Implementation

Var
  Note_Octave      : Integer;      { Current Octave for note                 }
  Note_Fraction    : Real;         { Fraction of duration given to note      }
  Note_Length      : Real;         { Length of note                          }
  Note_Quarter     : Real;         { Length of quarter note (principal beat) }

Procedure Pbm_Reset;
begin
  Note_Octave     := Pbm_Dfl_Octave;
  Note_Fraction   := Pbm_Dfl_Fraction;
  Note_Length     := Pbm_Dfl_Length;
  Note_Quarter    := Pbm_Dfl_Quarter;
end;

{#JRF# addition begin}
Procedure Sound(Hz: Word);   begin SndSOundOn(Hz) end;
Procedure NoSound;           begin SndSoundOff    end;
Procedure Delay(msec: Word); begin SndDelay(msec) end;
{#JRF# addition end}


Procedure Pbm_Play(const SoundStr: Pbm_SoundStr);

Const
   Quarter_Note = 0.25;                    { Length of a quarter note }
   Note_Offset: Array['A'..'G'] Of Integer { Offsets in octave of natural notes }
              = (9,11,0,2,4,5,7);
   Note_Freqs: Array[0..84] Of Integer =   { Frequencies for 7 octaves }
   ( 0,
     65,   69,   73,   78,   82,   87,   92,   98,  104,  110,  116,  123,
    131,  139,  147,  156,  165,  175,  185,  196,  208,  220,  233,  247,
    262,  278,  294,  312,  330,  350,  370,  392,  416,  440,  466,  494,
    524,  556,  588,  624,  660,  700,  740,  784,  832,  880,  932,  988,
   1048, 1112, 1176, 1248, 1320, 1400, 1480, 1568, 1664, 1760, 1864, 1976,
   2096, 2224, 2352, 2496, 2640, 2800, 2960, 3136, 3328, 3520, 3728, 3952,
   4192, 4448, 4704, 4992, 5280, 5600, 5920, 6272, 6656, 7040, 7456, 7904);
{     C    C#     D    D#     E     F    F#     G    G#     A    A#     B }

Type
  PlayStr       = String[Pbm_MaxSoundStr+1];
Var
  S             : PlayStr;        { Copy of SoundStr with space added }
  Play_Freq     : Integer;        { Frequency of note to be played    }
  Play_Duration : Integer;        { Duration to sound note            }
  Rest_Duration : Integer;        { Duration of rest after a note     }
  I             : Integer;        { Offset in Music string            }
  C             : Char;           { Current character in music string }
  Freq          : Array[0..6,0..11] Of Integer ABSOLUTE Note_Freqs;
  N             : Integer;
  XN            : Real;

  Function GetInt : Integer;
  { Get integer from music string }
  Var N: Integer;
  Begin { GetInt }
    N:= 0;
    While(S[I] In ['0'..'9']) Do Begin
      N:= N * 10 + ORD(S[I]) - ORD('0');
      I:= I + 1;
    End;
    I     := I - 1;
    GetInt:= N;
  End; { GetInt }

Begin { Pbm_Play }
  S:= SoundStr + ' ';            { Append blank to end of music string }
  I:= 1;                         { Point to first character in music   }
  While(I<LENGTH(S)) Do Begin    { Interpret Music                     }
    C:= Upcase(S[I]);            { Get next character in music string  }
    Case C Of                    { Interpret it                        }
      'A'..'G' : Begin { note }
                   N        := Note_Offset[ C ];
                   Play_Freq:= Freq[Note_Octave,N];
                   XN:= Note_Quarter * (Note_Length / Quarter_Note);
                   Play_Duration:= Trunc(XN * Note_Fraction);
                   Rest_Duration:= Trunc(XN * (1.0 - Note_Fraction));

                   If S[I+1] In ['#','+','-' ] Then Begin { sharp/flat }
                     I:= I + 1;
                     Case S[I] Of
                        '#' : Play_Freq:= Freq[Note_Octave, N+1];
                        '+' : Play_Freq:= Freq[Note_Octave, N+1];
                        '-' : Play_Freq:= Freq[Note_Octave, N-1];
                         Else ;
                     End { Case };
                   End;

                   If S[I+1] In ['0'..'9'] Then Begin { note length }
                     I := I+1;
                     N := GetInt;
                     XN:= (1.0 / N) / Quarter_Note;
                     Play_Duration:= Trunc(Note_Fraction * Note_Quarter * XN);
                     Rest_Duration:=
                       Trunc((1.0 - Note_Fraction) * Xn * Note_Quarter);
                   End;

                   If S[I+1] = '.' Then Begin { dotting }
                     XN:= 1.0;
                     While(S[I+1] = '.') Do Begin
                       XN:= XN * 1.5;
                       I := I+1;
                     End;
                     Play_Duration:= Trunc(Play_Duration * XN);
                   End;

                   Sound(Play_Freq);      { play the note }
                   Delay(Play_Duration);
                   NoSound;
                   Delay(Rest_Duration);
                 End { note };

      'M'      : Begin { 'M' Commands }
                   I:= I+1;
                   C:= S[I];
                   Case C Of
                     'F' : ;
                     'B' : ;
                     'N' : Note_Fraction:= 0.875;
                     'L' : Note_Fraction:= 1.000;
                     'S' : Note_Fraction:= 0.750;
                     Else ;
                   End { Case };
                 End   { 'M' Commands };

      'O'      : Begin { Set Octave }
                   I:= I + 1;
                   N:= ORD(S[I]) - ORD('0');
                   If (N < 0) OR (N > 6) Then N:= 4;
                   Note_Octave:= N;
                 End   { Set Octave };

      '<'      : Begin { Drop an octave }
                   If Note_Octave > 0 Then Note_Octave:= Note_Octave - 1;
                 End   { Drop an octave };

      '>'      : Begin { Ascend an octave }
                   If Note_Octave < 6 Then Note_Octave:= Note_Octave + 1;
                 End   { Ascend an octave };

      'N'      : Begin { Play Note N }
                   I:= I + 1;
                   N:= GetInt;
                   If (N > 0) AND (N <= 84) Then Begin
                     Play_Freq   := Note_Freqs[ N ];
                     XN          := Note_Quarter * (Note_Length/Quarter_Note);
                     Play_Duration:= Trunc(XN * Note_Fraction);
                     Rest_Duration:= Trunc(XN * (1.0 - Note_Fraction));
                   End
                   Else If (N = 0) Then Begin
                     Play_Freq    := 0;
                     Play_Duration:= 0;
                     Rest_Duration:= Trunc(Note_Fraction * Note_Quarter *
                                      (Note_Length/Quarter_Note));
                   End;
                   Sound(Play_Freq);
                   Delay(Play_Duration);
                   NoSound;
                   Delay(Rest_Duration);
                 End   { Play Note N };

      'L'      : Begin { Set Length of Notes }
                   I:= I + 1;
                   N:= GetInt;
                   If N > 0 Then Note_Length:= 1.0 / N;
                 End   { Set Length of Notes };

      'T'      : Begin { # of quarter notes in a minute }
                   I:= I + 1;
                   N:= GetInt;
                   Note_Quarter:= (1092.0 / 18.2 / N) * 1000.0;
                 End   { # of quarter notes in a minute };

      'P'      : Begin { Pause }
                   I:= I + 1;
                   N:= GetInt;
                   If (N < 1) Then N:= 1
                   Else If (N > 64) Then N:= 64;
                   Play_Freq    := 0;
                   Play_Duration:= 0;
                   Rest_Duration:= Trunc(((1.0/N)/Quarter_Note) * Note_Quarter);
                   Sound(Play_Freq);
                   Delay(Play_Duration);
                   NoSound;
                   Delay(Rest_Duration);
                 End   { Pause };

      Else { Ignore other stuff };
    End { Case };

    I:= I + 1;
  End  { Interpret Music };

  NoSound;                        { Make sure sound turned off when through }
End; { Pbm_Play }


Begin { Unit initialization }
  Pbm_Reset;
End.
