{ **********************************************************************
  *                         Program SYSEQLU.PAS                        *
  *                             Version 1.0                            *
  *                     (c) J. Debord, January 1997                    *
  **********************************************************************
  This program solves a system of linear equations (A * X = B) with
  several constant vectors by LU decomposition. The system is stored in
  a data file with the following structure :

    Line  1                   : dimension of the matrix (N)
    Lines 2 to (N + 1)        : matrix
    Line  N + 2               : number of constant vectors (P)
    Lines (N + 3) to (2N + 2) : constant vectors (one by column)

  The file MATRIX2.DAT is an example data file with N = 4 and P = 5

  LU decomposition factors the square matrix A as a product L * U, where
  L is a lower triangular matrix (with unit diagonal terms) and U is an
  upper triangular matrix. Then the system is solved as two subsystems,
  L * Z = B and U * X = Z, taking advantage of the triangular nature of
  the matrices.
  ********************************************************************** }

uses
  Crt, FMath, Matrices;

var
  A  : PMatrix;  { System matrix }
  B  : PMatrix;  { Constant vectors }
  X  : PMatrix;  { Solutions }
  N  : Integer;  { Dimension of matrix }
  P  : Integer;  { Number of constant vectors }
  I  : Integer;  { Loop variable }
  Ch : Char;     { Key pressed to exit program }

  procedure ReadMatrices(FileName : String; var A, B : PMatrix;
                         var N, P : Integer);
{ ----------------------------------------------------------------------
  Reads data from file. Note that matrices are passed as VAR parameters
  because they are dimensioned inside the procedure.
  ---------------------------------------------------------------------- }
  var
    F    : Text;     { Data file }
    I, J : Integer;  { Loop variable }
  begin
    Assign(F, FileName);
    Reset(F);

    { Read matrix }
    Read(F, N);
    DimMatrix(A, N, N);
    for I := 1 to N do
      for J := 1 to N do
        Read(F, A^[I]^[J]);

    { Read constant vectors }
    Read(F, P);
    DimMatrix(B, P, N);
    for J := 1 to N do
      for I := 1 to P do
        Read(F, B^[I]^[J]);  { B^[I] is the I-th constant vector }
    Close(F);
  end;

  procedure WriteMatrix(Title : String; A : PMatrix; N : Integer);
{ ----------------------------------------------------------------------
  Writes system matrix on screen
  ---------------------------------------------------------------------- }
  var
    I, J : Integer;
  begin
    WriteLn(Title, ' :', #10);
    for I := 1 to N do
      begin
        for J := 1 to N do
          Write(A^[I]^[J]:12:6);
        WriteLn;
      end;
    WriteLn;
  end;

  procedure WriteVectors(Title : String; B : PMatrix; N, P : Integer);
{ ----------------------------------------------------------------------
  Writes constant vectors or solution vectors
  so that they appear on columns.
  ---------------------------------------------------------------------- }
  var
    I, J : Integer;
  begin
    WriteLn(Title, ' :', #10);
    for J := 1 to N do
      begin
        for I := 1 to P do
          Write(B^[I]^[J]:12:6);
        WriteLn;
      end;
    WriteLn;
  end;

begin
  ClrScr;

  { Read and display data }
  ReadMatrices('MATRIX2.DAT', A, B, N, P);
  WriteMatrix('System matrix', A, N);
  WriteVectors('Constant vectors', B, N, P);

  { Dimension solution matrix }
  DimMatrix(X, P, N);

  { Perform LU decomposition of A. If successful,
    solve system for each constant vector }
  case LU_Decomp(A, 1, N) of
    MAT_OK     : begin
                   for I := 1 to P do
                     LU_Solve(A, B^[I], 1, N, X^[I]);
                   WriteVectors('Solution vectors', X, N, P);
                 end;
    MAT_SINGUL : Write('Singular matrix!');
  end;

  GotoXY(1, 25);
  Write('Press a key ... ');
  Ch := ReadKey;
end.

