{ **********************************************************************
  *                         Program SYSEQC.PAS                         *
  *                             Version 1.1                            *
  *                     (c) J. Debord, August 1997                     *
  **********************************************************************
  This program solves a system of linear equations with complex
  coefficients (A * X = B) by LU decomposition. The system is stored
  in a data file with the following structure :

    Line  1            : dimension of the matrix (N)
    Lines 2 to (N + 1) : matrix, followed by constant vector

  Complex numbers are given in rectangular form : real part, followed
  by imaginary part.

  The file MATRIX3.DAT is an example data file with N = 2

  LU decomposition factors the square matrix A as a product L * U, where
  L is a lower triangular matrix (with unit diagonal terms) and U is an
  upper triangular matrix. Then the system is solved as two subsystems,
  L * Z = B and U * X = Z, taking advantage of the triangular nature of
  the matrices.
  ********************************************************************** }

uses
  Crt, FMath, FComp, MatComp, PaString;

var
  A  : PCompMatrix;  { System matrix }
  B  : PCompVector;  { Constant vector }
  X  : PCompVector;  { Solution vector }
  N  : Integer;      { Dimension of matrix }
  P  : Integer;      { Number of constant vectors }
  I  : Integer;      { Loop variable }
  Ch : Char;         { Key pressed to exit program }

  procedure ReadSystem(FileName : String; var A : PCompMatrix;
                       var B : PCompVector; var N : Integer);
{ ----------------------------------------------------------------------
  Reads data from file. Note that matrices and vectors are passed as VAR
  parameters because they are dimensioned inside the procedure.
  ---------------------------------------------------------------------- }
  var
    F    : Text;     { Data file }
    I, J : Integer;  { Loop variables }
    X, Y : Float;    { Real and imaginary parts }
  begin
    Assign(F, FileName);
    Reset(F);

    { Read matrix and constant vector }
    Read(F, N);
    DimCompMatrix(A, N, N);
    DimCompVector(B, N);

    for I := 1 to N do
      begin
        { Read line I of matrix }
        for J := 1 to N do
          begin
            Read(F, X, Y);
            CSet(A^[I]^[J], X, Y, Rec);
          end;
        { Read element I of constant vector }
        Read(F, X, Y);
        CSet(B^[I], X, Y, Rec);
      end;

    Close(F);
  end;

  procedure WriteMatrix(Title : String; A : PCompMatrix; N : Integer);
{ ----------------------------------------------------------------------
  Writes system matrix on screen
  ---------------------------------------------------------------------- }
  var
    I, J : Integer;
  begin
    WriteLn(Title, ' :', #10);
    for I := 1 to N do
      begin
        for J := 1 to N do
          Write(CompToStr(A^[I]^[J]));
        WriteLn;
      end;
    WriteLn;
  end;

  procedure WriteVector(Title : String; B : PCompVector; N : Integer);
{ ----------------------------------------------------------------------
  Writes constant vector or solution vector.
  ---------------------------------------------------------------------- }
  var
    I : Integer;
  begin
    WriteLn(Title, ' :', #10);
    for I := 1 to N do
      WriteLn(CompToStr(B^[I]));
    WriteLn;
  end;

begin
  ClrScr;

  { Read and display data }
  ReadSystem('MATRIX3.DAT', A, B, N);
  WriteMatrix('System matrix', A, N);
  WriteVector('Constant vector', B, N);

  { Dimension solution vector }
  DimCompVector(X, N);

  { Perform LU decomposition of A. If successful, solve system }
  case C_LU_Decomp(A, 1, N) of
    MAT_OK     : begin
                   C_LU_Solve(A, B, 1, N, X);
                   WriteVector('Solution vector', X, N);
                 end;
    MAT_SINGUL : Write('Singular matrix!');
  end;

  GotoXY(1, 25);
  Write('Press a key ... ');
  Ch := ReadKey;
end.

