{ **********************************************************************
  *                         Program MINFUNC.PAS                        *
  *                             Version 1.3                            *
  *                     (c) J. Debord, August 1997                     *
  **********************************************************************
  This program demonstrates the minimization of a function of several
  variables by a combination of the Simplex and Marquardt algorithms.

  If f is a function of n independent variables defined by :

                       y = f(x(1), x(2),... x(n))

  and if X is an approximation of the vector of variables at the
  minimum, a better approximation may be computed by the Newton-Raphson
  algorithm :
                         X' = X - inv(H(X)).G(X)

  where H(X) is the hessian matrix and G(X) the gradient vector of f,
  evaluated at X

  In order to improve the convergence of the Newton-Raphson algorithm,
  Marquardt multiplies each diagonal term of H by (1 + lambda), where
  lambda is a scalar which is given an initial value (e.g. 1) at the
  beginning of the iterations, then divided by 10 if the iteration is
  successful (i.e. if f decreases) or otherwise multiplied by 10. The
  value of lambda thus approaches zero as f comes to a minimum.

  Marquardt's method converges rapidly but requires good initial
  estimates of X. By contrast, the Simplex method of Nelder and Mead,
  which makes no use of derivatives, converges more slowly but can start
  from more crude estimates.

  In this program the Simplex method is run first to approach the
  minimum, which is then refined by Marquardt's method.

  The maximum number of iterations allowed to each method is fixed by
  the constants ITSIMP and ITMARQ. The precision required for locating
  the minimum is fixed by the constants TOLSIMP and TOLMARQ.
  ********************************************************************** }

program MinFunc;

uses
  Crt, FMath, Matrices, Optim;

const
  NVAR    = 2;       { Number of variables }
  ITSIMP  = 100;     { Number of iterations allowed to Simplex }
  TOLSIMP = 1.0E-2;  { Required precision for Simplex estimation }
  ITMARQ  = 100;     { Number of iterations allowed to Marquardt }
  TOLMARQ = 1.0E-6;  { Required precision for Marquardt estimation }

var
  X     : PVector;  { Coordinates of minimum }
  H_inv : PMatrix;  { Inverse hessian matrix }
  F_min : Float;    { Function value at minimum }

{ ----------------------------------------------------------------------
  Define your function here - Must be compiled in FAR mode ($F+)
  Don't modify the number or type of parameters, nor the type returned
  by the function
  ---------------------------------------------------------------------- }

  {$F+}
  function Func(X : PVector) : Float;
  { Example taken from 'Numerical Recipes'
    True minimum is at (-2.0, +/-0.89442719) }
  var
    A, AA, B, BB   : Float;
  begin
    A := X^[2] * X^[2] * (3.0 - X^[1]) - X^[1] * X^[1] * (3.0 + X^[1]);
    B := 2.0 + X^[1];
    AA := Sqr(A);
    BB := Sqr(B);
    Func := 10.0 * AA + BB / (1.0 + BB);
  end;
  {$F-}

  procedure InitVar;
  { Set initial estimates for the coordinates of the minimum }
  begin
    X^[1] := 0.1;
    X^[2] := 4.2;
  end;

  procedure WriteResult;
  { Output results to screen }
  var
    I, J : Integer;
  begin
    WriteLn('Coordinates of minimum :', #10);
    for I := 1 to NVAR do
      WriteLn(X^[I]:14:8);
    WriteLn(#10, 'Function value :', F_min:14);
    WriteLn(#10, 'Inverse Hessian matrix :', #10);
    for I := 1 to NVAR do
      begin
        for J := 1 to NVAR do
          Write(H_inv^[I]^[J]:14:8);
        WriteLn;
      end;
  end;

begin
  ClrScr;
  { Allocate arrays }
  DimVector(X, NVAR);
  DimMatrix(H_inv, NVAR, NVAR);

  { Store the address of your function in the global variable ObjFuncAddr }
  ObjFuncAddr := @Func;

  { Marquardt's method requires a procedure to compute the Gradient and
    Hessian of the objective function. The address of this procedure is
    stored in the global variable HessProcAddr (defined in OPTIM.PAS).
    By default, this variable points to the procedure HessGrad (also
    defined in OPTIM.PAS), which uses numerical differentiation. If you
    wish to use another procedure, you will need to add the instruction
    HessProcAddr := @Your_procedure; }

  { Set initial estimates }
  InitVar;

  { Start with Simplex method, then switch to Marquardt }
  case Simplex(X, 1, NVAR, ITSIMP, TOLSIMP, F_min) of
    MAT_OK : case Marquardt(X, 1, NVAR, ITMARQ, TOLMARQ, F_min, H_inv) of
               MAT_OK : WriteResult;
               MAT_SINGUL : WriteLn('Singular Hessian matrix.');
               BIG_LAMBDA : WriteLn('Lambda too high.');
               NON_CONV : WriteLn('Too many iterations in Marquardt.');
             end;
    NON_CONV : WriteLn('Too many iterations in Simplex.');
  end;

  { Deallocate arrays }
  DelVector(X, NVAR);
  DelMatrix(H_inv, NVAR, NVAR);
end.
