{ **********************************************************************
  *                         Program FITSVD.PAS                         *
  *                             Version 1.0                            *
  *                     (c) J. Debord, January 1997                    *
  **********************************************************************
  This program performs unweighted multiple linear regression by
  singular value decomposition (See SYSEQSVD.PAS for details on this
  technique, and FITMULT.PAS for a more standard linear least squares
  treatment).

  The name of the input file (default extension = .DAT) is passed on the
  command line. Input files are ASCII files with the following structure :

    Line 1     : Title of study
    Line 2     : Number of variables (must be >= 2 here !)
    Next lines : Names of variables x1, x2, ..., y
    Next line  : Number of observations (must be > number of variables !)

    The next lines contain the coordinates (x1, x2, ..., y) of the
    observations (1 observation by line). The coordinates must be
    separated by spaces or tabulations.

  The file INHIB.DAT is an example of data relating the inhibition of an
  enzyme to the physico-chemical properties of the inhibitors. The program
  parameter is : INHIB

  The program may be executed from Turbo Pascal's integrated environment,
  in which case the file name is entered through the "Parameters" option
  of the menu, or from DOS (after compilation into an executable file),
  in which case the file name is entered on the command line (e.g.
  FITSVD INHIB).
  ********************************************************************** }

uses
  FMath, Matrices, Regress, PaString;

const
  TOL = 1.0E-5;  { A singular value will be set to zero if it is   }
                 { lower than TOL times the highest singular value }
var
  InFName : String;      { Name of input file }
  Title   : String;      { Title of study }
  XName   : PStrVector;  { Names of independent variables }
  YName   : String;      { Name of dependent variable }
  N       : Integer;     { Number of observations }
  Nvar    : Integer;     { Number of independent variables }
  X       : PMatrix;     { Matrix of independent variables }
  Y       : PVector;     { Vector of dependent variable }
  Ycalc   : PVector;     { Expected Y values }
  B       : PVector;     { Regression parameters }
  V       : PMatrix;     { Variance-covariance matrix of parameters }
  ErrCode : Integer;     { Error code }

  procedure ReadCmdLine;
  { Reads command line parameter }
  begin
    InFName := ParamStr(1);
    if Pos('.', InFName) = 0 then
      InFName := InFName + '.DAT';
  end;

  function FuncName : String;
  { Returns the name of the regression function }
  var
    Name, S : String;
    I : Integer;
  begin
    Name := 'y = b0 + b1.x1';
    for I := 2 to Nvar do
      begin
        Str(I, S);
        Name := Name + ' + b' + S + '.x' + S;
      end;
    FuncName := Name;
  end;

  function ParamName(I : Integer) : String;
  { Returns the name of the I-th parameter }
  var
    S : String;
  begin
    Str(I, S);
    ParamName := 'b' + S;
  end;

  procedure ReadInputFile;
  var
    InF  : Text;     { Input file }
    I, K : Integer;  { Loop variables }
  begin
    ErrCode := 0;
    Assign(InF, InFName);
    Reset(InF);
    ReadLn(InF, Title);
    ReadLn(InF, Nvar);  { Total number of variables }
    if Nvar < 2 then
      begin
        WriteLn('Data file must contain at least 2 variables !');
        ErrCode := - 1;
        Exit;
      end;

    Nvar := Pred(Nvar);         { Number of independent variables }
    DimStrVector(XName, Nvar);
    for I := 1 to Nvar do
      ReadLn(InF, XName^[I]);
    ReadLn(InF, YName);

    ReadLn(InF, N);
    DimMatrix(X, Nvar, N);
    DimVector(Y, N);
    DimVector(Ycalc, N);

    for K := 1 to N do
      begin
        for I := 1 to Nvar do
          Read(InF, X^[I]^[K]);
        Read(InF, Y^[K]);
      end;
    Close(InF);
  end;

  procedure FitModel;
  var
    U       : PMatrix;  { Matrix of independent variables for SVD }
    Z       : PVector;  { Vector of dependent variables for SVD }
    S       : PVector;  { Singular values }
    S2inv   : PVector;  { Inverses of squared singular values }
    W       : PMatrix;  { Orthogonal matrix from the SV decomposition }
    N1      : Integer;  { Number of observations - 1 }
    I, J, K : Integer;  { Loop variables }
    Sum     : Float;    { Element of variance-covariance matrix }
  begin
    { Dimension arrays }
    N1 := Pred(N);
    DimVector(B, Nvar);
    DimMatrix(V, Nvar, Nvar);
    DimMatrix(U, N1, Nvar);
    DimVector(Z, N1);
    DimVector(S, Nvar);
    DimVector(S2inv, Nvar);
    DimMatrix(W, Nvar, Nvar);

    WriteLn(#10, 'Linear least squares fit of function :');
    WriteLn(FuncName);
    WriteLn('to the data in file : ', InFName);

    { Prepare arrays for SVD }
    for I := 0 to N1 do
      begin
        U^[I]^[0] := 1.0;
        for J := 1 to Nvar do
          U^[I]^[J] := X^[J]^[I + 1];
        Z^[I] := Y^[I + 1];
      end;

    { Perform singular value decomposition }
    ErrCode := SV_Decomp(U, 0, N1, Nvar, S, W);

    if ErrCode <> MAT_OK then
      begin
        WriteLn('Unable to perform singular value decomposition !');
        Exit;
      end;

    { Set the lowest singular values to zero }
    SV_SetZero(S, 0, Nvar, TOL);

    { Solve the system }
    SV_Solve(U, S, W, Z, 0, N1, Nvar, B);

    { Compute variance-covariance matrix }
    for I := 0 to Nvar do
      if S^[I] > 0.0 then
        S2inv^[I] := 1.0 / Sqr(S^[I])
      else
        S2inv^[I] := 0.0;
    for I := 0 to Nvar do
      for J := 0 to I do
        begin
          Sum := 0.0;
          for K := 0 to Nvar do
            Sum := Sum + W^[I]^[K] * W^[J]^[K] * S2inv^[K];
          V^[I]^[J] := Sum;
          V^[J]^[I] := Sum;
        end;

    { Compute estimated Y values }
    for K := 1 to N do
      begin
        Ycalc^[K] := B^[0];
        for I := 1 to Nvar do
          Ycalc^[K] := Ycalc^[K] + B^[I] * X^[I]^[K];
        end;
  end;

  procedure WriteOutputFile;
  var
    OutFName : String;    { Name of output file }
    OutF     : Text;      { Output file }
    Line1,
    Line2    : String;    { Separating lines }
    Delta    : Float;     { Residual }
    Test     : TRegTest;  { Regression tests }
    Sr       : Float;     { Residual error }
    S        : PVector;   { Standard deviations of parameters }
    T        : PVector;   { Student's t }
    Prob     : PVector;   { Probabilities }
    I, K     : Integer;   { Loop variables }
  begin
    DimVector(S, Nvar);
    DimVector(T, Nvar);
    DimVector(Prob, Nvar);

    K := Pos('.', InFName);
    OutFName := Copy(InFName, 1, Pred(K)) + '.OUT';
    Assign(OutF, OutFName);
    Rewrite(OutF);
    Line1 := StrChar(73, '-');
    Line2 := StrChar(73, '=');
    WriteLn(OutF, Line2);
    WriteLn(OutF, 'Data file  : ', InFName);
    WriteLn(OutF, 'Study name : ', Title);
    for I := 1 to Nvar do
      WriteLn(OutF, 'x', I:1, '         : ', XName^[I]);
    WriteLn(OutF, 'y          : ', YName);
    WriteLn(OutF, 'Function   : ', FuncName);

    { Update var/cov matrix and compute regression tests
      Note : RegTest must always be called before ParamTest }
    RegTest(Y, Ycalc, N, 0, Nvar, V, Test);
    ParamTest(B, V, N, 0, Nvar, S, T, Prob);

    WriteLn(OutF, Line1);
    WriteLn(OutF, 'Parameter    Est.value         Std.dev.        t Student       Prob(>|t|)');
    WriteLn(OutF, Line1);
    for I := 0 to Nvar do
      WriteLn(OutF, ParamName(I):5, B^[I]:17:4, S^[I]:17:4, T^[I]:17:2, Prob^[I]:17:4);
    WriteLn(OutF, Line1);
    WriteLn(OutF, 'Number of observations            : n   = ', N:5);
    with Test do
      begin
        Sr := Sqrt(Vr);
        WriteLn(OutF, 'Residual error                    : s   = ', Sr:10:4);
        if (R2 >= 0.0) and (R2 <= 1.0) then
          WriteLn(OutF, 'Coefficient of determination      : r2  = ', R2:10:4);
        if (R2a >= 0.0) and (R2a <= 1.0) then
          WriteLn(OutF, 'Adjusted coeff. of determination  : r2a = ', R2a:10:4);
        Write(OutF, 'Variance ratio (explained/resid.) : F   = ', F:10:4);
        WriteLn(OutF, '    Prob(>F) = ', Prob:6:4);
      end;
    WriteLn(OutF, Line1);
    WriteLn(OutF, '  i        Y obs.       Y calc.      Residual      Std.dev.      Std.res.');
    WriteLn(OutF, Line1);
    for K := 1 to N do
      begin
        Delta := Y^[K] - Ycalc^[K];
        WriteLn(OutF, K:3, Y^[K]:14:4, Ycalc^[K]:14:4, Delta:14:4, Sr:14:4, (Delta / Sr):14:4);
      end;
    WriteLn(OutF, Line2);
    Close(OutF);
    WriteLn('Results written to file ', OutFName);
    DelVector(S, Nvar);
    DelVector(T, Nvar);
    DelVector(Prob, Nvar);
  end;

begin
  ReadCmdLine;
  ReadInputFile;
  if ErrCode = 0 then
    FitModel;
  if ErrCode = MAT_OK then
    WriteOutputFile;
end.

