{ **********************************************************************
  *                        Program FITPOLY.PAS                         *
  *                            Version 1.2                             *
  *                      (c) J. Debord, July 1997                      *
  **********************************************************************
  This program performs a weighted least squares fit of a straight line
  or polynomial :

                     y = b0 + b1 * x + b2 * x^2 + ...

  The following parameters are passed on the command line :

    1st parameter = name of input file (default extension = .DAT)
                    The structure of the input file is described in
                    REG_IN.INC
    2nd parameter = degree of polynomial (default = 1, i.e. straight line)

  The files LINE.DAT and POLYNOM.DAT are examples of 2- and 5th-order
  polynomials, respectively. The program parameters are :

                              LINE 2
                          or  POLYNOM 5

  The program may be executed from Turbo Pascal's integrated environment,
  in which case the parameters are entered through the "Parameters" option
  of the menu, or from DOS (after compilation into an executable file),
  in which case the parameters are entered on the command line (e.g.
  FITPOLY LINE 2 or FITPOLY POLYNOM 5).
  ********************************************************************** }

program FitPoly;

uses
  Crt, Graph, FMath, Matrices, Polynom, Regress, PaString, Plot;

var
  InFName : String;   { Name of input file }
  Title   : String;   { Title of study }
  XName,
  YName   : String;   { Variable names }
  N       : Integer;  { Number of points }
  X, Y    : PVector;  { Point coordinates }
  Ycalc   : PVector;  { Expected Y values }
  W       : PVector;  { Weights }
  Deg     : Integer;  { Degree of polynomial }
  Lbound,
  Ubound  : Integer;  { Indices of first & last fitted parameters }
  B       : PVector;  { Regression parameters }
  V       : PMatrix;  { Variance-covariance matrix of parameters }
  ErrCode : Integer;  { Error code }

{ ----------------------------------------------------------------------
  Define here the function used to compute the variance of an observed
  y value. The true variance will be : V(y) = Vr * VarFunc(y), where Vr
  is the residual variance (estimated by the program).

  Ex. : VarFunc(y) = Sqr(y)  for a variance proportional to y

  For unweighted regression, set VarFunc to 1.0 or use the unweighted
  version of the regression routines (LinFit, PolFit, and RegTest).
  ---------------------------------------------------------------------- }

  function VarFunc(Y : Float) : Float;
  begin
    VarFunc := 1.0;  { Here we assume a constant variance }
  end;

  procedure ReadCmdLine;
  { Reads command line parameters }
  var
    ErrCode : Integer;
  begin
    { Name of input file }
    InFName := ParamStr(1);
    if Pos('.', InFName) = 0 then InFName := InFName + '.DAT';

    { Degree of polynomial }
    Deg := 0;
    Val(ParamStr(2), Deg, ErrCode);
    if (ErrCode <> 0) or (Deg < 1) then Deg := 1;
  end;

  function FuncName : String;
  { Returns the name of the regression function }
  var
    Name, S : String;
    I : Integer;
  begin
    Name := 'y = b0 + b1.x';
    for I := 2 to Deg do
      begin
        Str(I, S);
        Name := Name + ' + b' + S + '.x^' + S;
      end;
    FuncName := Name;
  end;

  function FirstParam : Integer;
  { Returns the index of the first parameter to be fitted }
  begin
    FirstParam := 0;
  end;

  function LastParam : Integer;
  { Returns the index of the last parameter to be fitted }
  begin
    LastParam := Deg;
  end;

  function ParamName(I : Integer) : String;
  { Returns the name of the I-th parameter }
  var
    S : String;
  begin
    Str(I, S);
    ParamName := 'b' + S;
  end;

  {$F+}
  function PlotRegFunc(X : Float) : Float;
  { Defines the function to be plotted }
  begin
    PlotRegFunc := Poly(X, B, Deg);
  end;
  {$F-}

  {$I REG_IN.INC}  { Read input file }

  procedure FitModel;
  var
    K : Integer;
    Ch : Char;
  begin
    { Dimension arrays }
    Lbound := FirstParam;
    Ubound := LastParam;
    DimVector(B, Ubound);
    DimMatrix(V, Ubound, Ubound);
    WriteLn(#10, 'Linear least squares fit of function :');
    WriteLn(FuncName);
    WriteLn('to the data in file : ', InFName);

    { Compute weights }
    for K := 1 to N do
      W^[K] := 1.0 / VarFunc(Y^[K]);

    { Perform fit }
    if Deg = 1 then
      ErrCode := WLinFit(X, Y, W, N, B, V)
    else
      ErrCode := WPolFit(X, Y, W, N, Deg, B, V);

    if ErrCode = MAT_OK then
      for K := 1 to N do
        Ycalc^[K] := Poly(X^[K], B, Deg)  { Compute Y values }
    else
      WriteLn('Singular matrix !');
    Write(#10'Press a key...');
    Ch := ReadKey;
  end;

  {$I REG_OUT.INC}   { Write output file }

  {$I REG_PLOT.INC}  { Plot fitted function }

{ *************************** Main program ***************************** }

begin
  ReadCmdLine;
  ReadInputFile;
  if ErrCode = 0 then
    FitModel;
  if ErrCode = MAT_OK then
    begin
      { Path to the graphic drivers (Default = C:\BP\BGI) }
      { BGIPath := 'C:\BP\BGI'; }
      GraphTitle.Text := Title;
      PlotFuncAddr := @PlotRegFunc;
      PlotGraph;
      WriteOutputFile;
    end;
end.
