{ **********************************************************************
  *                        Program FITMULT.PAS                         *
  *                            Version 1.2                             *
  *                     (c) J. Debord, July 1997                       *
  **********************************************************************
  This program performs a weighted multiple linear least squares fit :

                     y = b0 + b1 * x1 + b2 * x2 + ...

  The following parameters are passed on the command line :

    1st parameter = name of input file (default extension = .DAT)
    2nd parameter = 1 if the equation includes a constant term b0

  Input files are ASCII files with the following structure :

    Line 1     : Title of study
    Line 2     : Number of variables (must be >= 2 here !)
    Next lines : Names of variables x1, x2, ..., y
    Next line  : Number of observations (must be > number of variables !)

    The next lines contain the coordinates (x1, x2, ..., y) of the
    observations (1 observation by line). The coordinates must be
    separated by spaces or tabulations.

  The file INHIB.DAT is an example of data relating the inhibition of an
  enzyme to the physico-chemical properties of the inhibitors (J. DEBORD,
  P. N'DIAYE, J. C. BOLLINGER et al, J. Enzyme Inhib., 1997, 12, 13-26).
  The program parameters are : INHIB 1

  The program may be executed from Turbo Pascal's integrated environment,
  in which case the parameters are entered through the "Parameters" option
  of the menu, or from DOS (after compilation into an executable file),
  in which case the parameters are entered on the command line (e.g.
  FITMULT INHIB 1).
  ********************************************************************** }

program FitMult;

uses
  FMath, Matrices, Regress, PaString;

var
  InFName  : String;      { Name of input file }
  Title    : String;      { Title of study }
  Nvar     : Integer;     { Number of independent variables }
  XName    : PStrVector;  { Names of independent variables }
  YName    : String;      { Name of dependent variable }
  N        : Integer;     { Number of observations }
  X        : PMatrix;     { Matrix of independent variables }
  Y        : PVector;     { Vector of dependent variable }
  Ycalc    : PVector;     { Expected Y values }
  W        : PVector;     { Weights }
  ConsTerm : Boolean;     { Flags the presence of a constant term b0 }
  Lbound,
  Ubound   : Integer;     { Indices of first & last fitted parameters }
  B        : PVector;     { Regression parameters }
  V        : PMatrix;     { Variance-covariance matrix of parameters }
  ErrCode  : Integer;     { Error code }

{ ----------------------------------------------------------------------
  Define here the function used to compute the variance of an observed
  y value. The true variance will be : V(y) = Vr * VarFunc(y), where Vr
  is the residual variance (estimated by the program).

  Ex. : VarFunc(y) = Sqr(y)  for a variance proportional to y

  For unweighted regression, set VarFunc to 1.0 or use the unweighted
  version of the regression routines (MulFit and RegTest).
  ---------------------------------------------------------------------- }

  function VarFunc(Y : Float) : Float;
  begin
    VarFunc := 1.0;  { Here we assume a constant variance }
  end;

  procedure ReadCmdLine;
  { Reads command line parameters }
  var
    ErrCode, I : Integer;
  begin
    { Name of input file }
    InFName := ParamStr(1);
    if Pos('.', InFName) = 0 then InFName := InFName + '.DAT';

    { Presence of constant term }
    I := 0;
    Val(ParamStr(2), I, ErrCode);
    ConsTerm := (I = 1);
  end;

  function FuncName : String;
  { Returns the name of the regression function }
  var
    Name, S : String;
    I : Integer;
  begin
    Name := 'y = ';
    if ConsTerm then
      Name := Name + 'b0 + ';
    Name := Name + 'b1.x1';
    for I := 2 to Nvar do
      begin
        Str(I, S);
        Name := Name + ' + b' + S + '.x' + S;
      end;
    FuncName := Name;
  end;

  function FirstParam : Integer;
  { Returns the index of the first parameter to be fitted }
  begin
    if ConsTerm then
      FirstParam := 0
    else
      FirstParam := 1;
  end;

  function LastParam : Integer;
  { Returns the index of the last parameter to be fitted }
  begin
    LastParam := Nvar;
  end;

  function ParamName(I : Integer) : String;
  { Returns the name of the I-th parameter }
  var
    S : String;
  begin
    Str(I, S);
    ParamName := 'b' + S;
  end;

  procedure ReadInputFile;
  var
    InF : Text;      { Input file }
    I, K : Integer;  { Loop variables }
  begin
    ErrCode := 0;
    Assign(InF, InFName);
    Reset(InF);
    ReadLn(InF, Title);
    ReadLn(InF, Nvar);  { Total number of variables }
    if Nvar < 2 then
      begin
        WriteLn('Data file must contain at least 2 variables !');
        ErrCode := - 1;
        Exit;
      end;
    Nvar := Pred(Nvar);         { Number of independent variables }
    DimStrVector(XName, Nvar);
    for I := 1 to Nvar do
      ReadLn(InF, XName^[I]);
    ReadLn(InF, YName);
    ReadLn(InF, N);
    DimMatrix(X, Nvar, N);
    DimVector(Y, N);
    DimVector(Ycalc, N);
    DimVector(W, N);
    for K := 1 to N do
      begin
        for I := 1 to Nvar do
          Read(InF, X^[I]^[K]);
        Read(InF, Y^[K]);
      end;
    Close(InF);
  end;

  procedure FitModel;
  var
    I, K : Integer;
  begin
    { Dimension arrays }
    Lbound := FirstParam;
    Ubound := LastParam;
    DimVector(B, Ubound);
    DimMatrix(V, Ubound, Ubound);

    WriteLn(#10, 'Linear least squares fit of function :');
    WriteLn(FuncName);
    WriteLn('to the data in file : ', InFName);

    for K := 1 to N do
      W^[K] := 1.0 / VarFunc(Y^[K]);

    ErrCode := WMulFit(X, Y, W, N, Nvar, ConsTerm, B, V);

    if ErrCode = MAT_OK then
      { Compute Y values }
      for K := 1 to N do
        begin
          if ConsTerm then
            Ycalc^[K] := B^[0]
          else
            Ycalc^[K] := 0.0;
          for I := 1 to Nvar do
            Ycalc^[K] := Ycalc^[K] + B^[I] * X^[I]^[K];
        end
    else
      WriteLn('Singular matrix !');
  end;

  procedure WriteOutputFile;
  var
    OutFName : String;    { Name of output file }
    OutF     : Text;      { Output file }
    Line1,
    Line2    : String;    { Separating lines }
    Delta    : Float;     { Residual }
    Test     : TRegTest;  { Regression tests }
    Sr       : Float;     { Residual error }
    S        : PVector;   { Standard deviations of parameters }
    T        : PVector;   { Student's t }
    Prob     : PVector;   { Probabilities }
    SY       : Float;     { Standard deviation of observation }
    I, K     : Integer;   { Loop variables }
  begin
    DimVector(S, Ubound);
    DimVector(T, Ubound);
    DimVector(Prob, Ubound);
    K := Pos('.', InFName);
    OutFName := Copy(InFName, 1, Pred(K)) + '.OUT';
    Assign(OutF, OutFName);
    Rewrite(OutF);
    Line1 := StrChar(73, '-');
    Line2 := StrChar(73, '=');
    WriteLn(OutF, Line2);
    WriteLn(OutF, 'Data file  : ', InFName);
    WriteLn(OutF, 'Study name : ', Title);
    for I := 1 to Nvar do
      WriteLn(OutF, 'x', I:1, '         : ', XName^[I]);
    WriteLn(OutF, 'y          : ', YName);
    WriteLn(OutF, 'Function   : ', FuncName);

    { Update var/cov matrix and compute regression tests
      Note : RegTest or WRegTest must always be called before ParamTest }
    WRegTest(Y, Ycalc, W, N, Lbound, Ubound, V, Test);
    ParamTest(B, V, N, Lbound, Ubound, S, T, Prob);

    WriteLn(OutF, Line1);
    WriteLn(OutF, 'Parameter    Est.value         Std.dev.        t Student       Prob(>|t|)');
    WriteLn(OutF, Line1);
    for I := Lbound to Ubound do
      WriteLn(OutF, ParamName(I):5, B^[I]:17:4, S^[I]:17:4, T^[I]:17:2, Prob^[I]:17:4);
    WriteLn(OutF, Line1);
    WriteLn(OutF, 'Number of observations            : n   = ', N:5);
    with Test do
      begin
        Sr := Sqrt(Vr);
        WriteLn(OutF, 'Residual error                    : s   = ', Sr:10:4);
        if (R2 >= 0.0) and (R2 <= 1.0) then
          WriteLn(OutF, 'Coefficient of determination      : r2  = ', R2:10:4);
        if (R2a >= 0.0) and (R2a <= 1.0) then
          WriteLn(OutF, 'Adjusted coeff. of determination  : r2a = ', R2a:10:4);
        Write(OutF, 'Variance ratio (explained/resid.) : F   = ', F:10:4);
        WriteLn(OutF, '    Prob(>F) = ', Prob:6:4);
      end;

    WriteLn(OutF, Line1);
    WriteLn(OutF, '  i        Y obs.       Y calc.      Residual      Std.dev.      Std.res.');
    WriteLn(OutF, Line1);

    for K := 1 to N do
      begin
        Delta := Y^[K] - Ycalc^[K];
        SY := Sr / Sqrt(W^[K]);
        WriteLn(OutF, K:3, Y^[K]:14:4, Ycalc^[K]:14:4, Delta:14:4, SY:14:4, (Delta / SY):14:4);
      end;
    WriteLn(OutF, Line2);
    Close(OutF);
    WriteLn('Results written to file ', OutFName);
    DelVector(S, Ubound);
    DelVector(T, Ubound);
    DelVector(Prob, Ubound);
  end;

{ *************************** Main program ***************************** }

begin
  ReadCmdLine;
  ReadInputFile;
  if ErrCode = 0 then
    FitModel;
  if ErrCode = MAT_OK then
    WriteOutputFile;
end.
