{ **********************************************************************
  *                        Program FITFRAC.PAS                         *
  *                            Version 1.2                             *
  *                     (c) J. Debord, July 1997                       *
  **********************************************************************
  This program performs a nonlinear least squares fit of a rational
  fraction :
                           p0 + p1.x + p2.x^2 + ...
                       y = ------------------------
                           1 + q1.x + q2.x^2 + ...

  The following parameters are passed on the command line :

    1st parameter = Name of input file (default extension = .DAT)
                    The structure of the input file is described in
                    NLFIT.INC
    2nd parameter = Degree of numerator (default = 1)
    3rd parameter = Degree of denominator (default = 1)
    4th parameter = 1 if the function includes a constant term (p0)

  The file ENZYME.DAT is an example of enzyme kinetics (unpublished data
  from the author's laboratory). The best fit is obtained with 2nd-degree
  numerator and denominator, with no constant term, so that the program
  parameters are :

                              ENZYME 2 2

  The program may be executed from Turbo Pascal's integrated environment,
  in which case the parameters are entered through the "Parameters" option
  of the menu, or from DOS (after compilation into an executable file),
  in which case the parameters are entered on the command line (e.g.
  FITFRAC ENZYME 2 2).
  ********************************************************************** }

program FitFrac;

uses
  Crt, Graph, FMath, Matrices, Optim,
  Polynom, Stat, Regress, PaString, Plot;

const
  ITMARQ  = 500;     { Number of iterations allowed to Marquardt }
  TOLMARQ = 1.0E-4;  { Required precision for Marquardt estimation }

var
  InFName  : String;   { Name of input file }
  Title    : String;   { Title of study }
  XName,
  YName    : String;   { Variable names }
  N        : Integer;  { Number of points }
  X, Y     : PVector;  { Point coordinates }
  Ycalc    : PVector;  { Expected Y values }
  W        : PVector;  { Weights }
  Deg1,
  Deg2     : Integer;  { Degrees of numerator and denominator }
  ConsTerm : Boolean;  { Flags the presence of a constant term p0 }
  Lbound,
  Ubound   : Integer;  { Indices of first & last fitted parameters }
  B        : PVector;  { Regression parameters }
  V        : PMatrix;  { Variance-covariance matrix of parameters }
  ErrCode  : Integer;  { Error code }

{ ----------------------------------------------------------------------
  Define here the function used to compute the variance of an observed
  y value. The true variance will be : V(y) = Vr * VarFunc(y), where Vr
  is the residual variance (estimated by the program).

  Ex. : VarFunc(y) = Sqr(y)  for a variance proportional to y^2

  For unweighted regression, simply set VarFunc to 1.0
  ---------------------------------------------------------------------- }

  function VarFunc(Y : Float) : Float;
  begin
    { Here we assume a constant variance }
    VarFunc := 1.0;
  end;

  procedure ReadCmdLine;
  { Reads command line parameters }
  var
    I, ErrCode : Integer;
  begin
    { Name of input file }
    InFName := ParamStr(1);
    if Pos('.', InFName) = 0 then InFName := InFName + '.DAT';

    { Degree of numerator }
    Deg1 := 0;
    Val(ParamStr(2), Deg1, ErrCode);
    if (ErrCode <> 0) or (Deg1 < 1) then Deg1 := 1;

    { Degree of denominator }
    Deg2 := 0;
    Val(ParamStr(3), Deg2, ErrCode);
    if (ErrCode <> 0) or (Deg2 < 1) then Deg2 := 1;

    { Presence of constant term }
    I := 0;
    Val(ParamStr(4), I, ErrCode);
    ConsTerm := (I = 1);
  end;

  function FuncName : String;
  { Returns the name of the regression function }
  var
    Name, S : String;
    I : Integer;
  begin
    Name := 'y = (';
    if ConsTerm then
      Name := Name + 'p0 + ';
    Name := Name + 'p1.x';
    for I := 2 to Deg1 do
      begin
        Str(I, S);
        Name := Name + ' + p' + S + '.x^' + S;
      end;
    Name := Name + ') / (1 + q1.x';
    for I := (Deg1 + 2) to (Deg1 + Deg2) do
      begin
        Str(I - Deg1, S);
        Name := Name + ' + q' + S + '.x^' + S;
      end;
    Name := Name + ')';
    FuncName := Name;
  end;

  function FirstParam : Integer;
  { Returns the index of the first parameter to be fitted }
  begin
    if ConsTerm then
      FirstParam := 0
    else
      FirstParam := 1;
  end;

  function LastParam : Integer;
  { Returns the index of the last parameter to be fitted }
  begin
    LastParam := Deg1 + Deg2;
  end;

  function ParamName(I : Integer) : String;
  { Returns the name of the I-th parameter }
  var
    S : String;
  begin
    if I <= Deg1 then
      begin
        Str(I, S);
        ParamName := 'p' + S;
      end
    else
      begin
        Str(I - Deg1, S);
        ParamName := 'q' + S;
      end;
  end;

  function RegFunc(X : Float; B : PVector) : Float;
  { --------------------------------------------------------------------
    Computes the regression function
    B is the vector of parameters, such that :
      B^[0] = p0
      B^[1] = p1     B^[Deg1+1] = q1
      B^[2] = p2     B^[Deg1+2] = q2
      ..........     ...............
    -------------------------------------------------------------------- }
  begin
    RegFunc := RFrac(X, B, Deg1, Deg2);
  end;

  procedure DerivProc(X : Float; B, D : PVector);
  { --------------------------------------------------------------------
    Computes the derivatives of the regression function with respect to
    the parameters, for a given data point
    Input  : X     = point abscissa
             Ycalc = computed Y value
             B     = regression parameters
    Output : D     = vector of derivatives, d(i) = df(x, b) / db(i)
    -------------------------------------------------------------------- }
  var
    I : Integer;
    Den, Y : Float;
  begin
    Den := 0.0;
    for I := (Deg1 + Deg2) downto Succ(Deg1) do
      Den := (Den + B^[I]) * X;
    Den := 1.0 + Den;
    Y := Poly(X, B, Deg1) / Den;

    { Derivatives for numerator parameters }
    D^[0] := 1.0 / Den;
    for I := 1 to Deg1 do
      D^[I] := D^[I - 1] * X;

    { Derivatives for denominator parameters }
    D^[Deg1 + 1] := - Y * X / Den;
    for I := (Deg1 + 2) to (Deg1 + Deg2) do
      D^[I] := D^[I - 1] * X;
  end;

  function ApproxFit : Integer;
  { --------------------------------------------------------------------
    Approximate fit of the rational fraction by linear regression :
    y = p0 + p1.x + p2.x^2 + ... - q1.(x.y) - q2.(x^2.y) - ...
    -------------------------------------------------------------------- }
  var
    I, K : Integer;  { Loop variables }
    M : Integer;     { Index of last parameter }
    U : PMatrix;     { Matrix of independent variables }
  begin
    M := LastParam;
    DimMatrix(U, M, N);
    for K := 1 to N do
      begin
        U^[1]^[K] := X^[K];
        for I := 2 to Deg1 do
          U^[I]^[K] := U^[I - 1]^[K] * X^[K];
        U^[Deg1 + 1]^[K] := - X^[K] * Y^[K];
        for I := (Deg1 + 2) to M do
          U^[I]^[K] := U^[I - 1]^[K] * X^[K];
      end;
    ApproxFit := MulFit(U, Y, N, M, ConsTerm, B, V);
    DelMatrix(U, M, N);
  end;

  {$F+}
  function PlotRegFunc(X : Float) : Float;
  { Defines the function to be plotted }
  begin
    PlotRegFunc := RegFunc(X, B);
  end;
  {$F-}

  {$I REG_IN.INC}    { Read input file }

  {$I REG_NL.INC}    { Nonlinear regression }

  {$I REG_OUT.INC}   { Write output file }

  {$I REG_PLOT.INC}  { Plot function }

{ *************************** Main program ***************************** }

begin
  ReadCmdLine;
  ReadInputFile;
  if ErrCode = 0 then
    FitModel;
  if ErrCode = MAT_OK then
    begin
      { Path to the graphic drivers (Default = C:\BP\BGI) }
      { BGIPath := 'C:\BP\BGI'; }
      GraphTitle.Text := Title;
      PlotFuncAddr := @PlotRegFunc;
      PlotGraph;
      WriteOutputFile;
    end;
end.
