{ **********************************************************************
  *                           Program GLS.PAS                          *
  *                             Version 1.0                            *
  *             (c) J. Debord & K. Suchaud (December 1996)             *
  **********************************************************************
  This program fits an equation to a set of data points by minimizing
  the pseudo-likelihood function :

                     m  ( (y(k) - ycalc(k))^2            )
               PL = Sum ( ------------------- + Ln(v(k)) )
                    k=1 (        v(k)                    )

  where m        = number of observations
        y(k)     = observed value of the dependent variable
        ycalc(k) = estimated value of the dependent variable
        v(k)     = variance of y(k)

  ycalc(k) is a function of the independent variable x(k) and a vector
           of regression parameters B = [b(0), b(1) ...] :

                          ycalc(k) = f(x(k), B)

  v(k) is a function of ycalc(k) and a vector of variance parameters
       Theta = [theta(0), theta(1)...] such that :

               v(k) = theta(0) * g(ycalc(k), theta(1)...)

  where g is a user-specified function, e.g. a power function :

              g(ycalc(k), theta(1)...) = ycalc(k)^theta(1)

  The generalized least squares (GLS) method uses a set of N curves to
  estimate both the parameter vectors B(i) (one for each curve) and Theta
  (common to all curves) by an iterative process such that, at the current
  iteration :

  1) Variance parameters are estimated by minimizing the sum of pseudo-
     likelihoods with respect to Theta, using the current values of the
     regression parameters B(i) :

           N                  m(i) ( (y(i,k) - ycalc(i,k))^2              )
     PL = Sum PL(i),  PL(i) = Sum  ( ----------------------- + Ln(v(i,k)) )
          i=1                 k=1  (          v(i,k)                      )

     where m(i)       = number of observations for curve i
           y(i,k)     = observed value of the dependent variable,
                        for the k-th observation in curve i
           ycalc(i,k) = estimated value of the dependent variable,
                      = f(x(i,k), B(i))
           v(i,k)     = variance of y(i,k)
                      = theta(0) * g(ycalc(i,k), theta(1)...)

  2) Regression parameters are determined, for each curve, by minimizing
     the pseudo-likelihood PL(i) with respect to B(i), using the variance
     parameters from the previous step.

  These two steps are iterated until convergence of the variance parameters.

  Initial estimates of the regression parameters B(i) are obtained by
  weighted nonlinear regression. The variance is approximated by :

                     v(i,k) ~ g(y(i,k), theta(1),...)

  where theta(1),... are fixed to values chosen by the user (e.g. 1)

  The linear parameter theta(0) can then be estimated from the results of
  this weighted regression, by minimizing the sum of pseudo-likelihoods
  with respect to theta(0) alone, giving :

                            N   m(i)  (y(i,k) - ycalc(i,k))^2
         theta(0) = (1/Mt) Sum  Sum  --------------------------
                           i=1  k=1  g(ycalc(i,k), theta(1)...)

                                                      N
  where Mt is the total number of observations, Mt = Sum m(i)
                                                     i=1

  The method is illustrated here with the Michaelis-Menten equation :

                            Ymax . x
              y = f(x, B) = --------          B = [Ymax, Km]
                             Km + x

  and a power variance function. All minimizations are performed by the
  simplex method of Nelder and Mead.

  The program operates on a data file containing the coordinates (x,y)
  of a series of N curves. The name of this file (default extension .DAT)
  is passed on the command line. Its structure is as follows :

    Line  1            : Title of study
    Line  2            : Number of variables (N + 1)
    Line  3            : Name of variable x
    Lines 4 to (N + 3) : Names of variables y (1 for each curve)
    Line  (N + 4)      : Number of points

    The next lines contain the coordinates (x, y) of the points.
    The first column contains the x coordinates (common to all curves).
    The other columns contain the y coordinates (one column per curve).
    Missing y values are denoted by the value of the constant MISSING,
    defined at the beginning of the program.

    File ARYL.DAT is an example data file from the authors' laboratory.
    The data represent the activity of the serum enzyme arylesterase in
    several subjects, as a function of substrate concentration.

  The program generates an output file with the same name as the data
  file and extension .OUT; this file contains the parameter estimates
  for each fitted curve. Its structure is as follows :

    Line  1            : Title of study
    Line  2            : Number of fitted parameters (P) + 1
    Line  3            : The word "Index"
    Lines 4 to (P + 3) : Names of parameters
    Line  (P + 4)      : Number of fitted curves

    The next lines contain, for each fitted curve, its index
    followed by the values of the fitted parameters.

  The program produces three types of plots :

  1) A plot of the regression function, for all fitted curves.
  2) A plot of the standardized residuals, (y(k) - ycalc(k)) / s(k), where
     s(k) = sqrt(v(k)), vs ycalc(k). These residuals are supposed to be
     normally distributed with 95% of their values in the interval [-2, 2].
  3) A plot of the coefficient of variation, CV = 100 * s(k) / ycalc(k)
     vs ycalc(k).

  The program may be run from Turbo Pascal's integrated environment, in
  which case the name of the data file (e.g. ARYL) is entered through the
  "Parameters" option of the menu, or from DOS (after compilation into an
  executable file), in which case the name of the data file is entered on
  the command line (e.g. GLS ARYL).
  ********************************************************************** }

program GLS;

uses
  { Turbo units }
  Crt,
  Graph,
  { TP Math units }
  FMath,
  Matrices,
  Optim,
  Stat,
  Regress,
  Plot;

const
  MISSING = -1;      { Code for missing y value }
  MAXITER = 500;     { Maximum number of iterations }
  TOL     = 1.0E-4;  { Tolerance on fitted parameters }

var
  InFName : String;      { Name of input file }
  Title   : String;      { Title of study }
  XName   : String;      { Variable names }
  YName   : PStrVector;  { Variable names }
  N       : Integer;     { Number of curves }
  Nobs    : Integer;     { Number of observations }
  M       : PIntVector;  { Number of points in each curve }
  X, Y, W : PMatrix;     { Coordinates & weights }
  Ycalc   : PMatrix;     { Expected Y values }
  B       : PMatrix;     { Regression parameters }
  Theta   : PVector;     { Variance parameters }
  ErrCode : PIntVector;  { Error codes }
  Icalc   : Integer;     { Index of current curve }

  procedure ReadCmdLine(var InFName : String);
{ ----------------------------------------------------------------------
  Reads command line parameter
  ---------------------------------------------------------------------- }
  begin
    InFName := ParamStr(1);
    if Pos('.', InFName) = 0 then
      InFName := InFName + '.DAT';
  end;

{ **********************************************************************
                Routines related to the variance function
  ********************************************************************** }

  function VarFunc(Y : Float; Theta : PVector) : Float;
{ ----------------------------------------------------------------------
  Defines the variance function g(y, theta(1), theta(2)...)
  ---------------------------------------------------------------------- }
  begin
    VarFunc := Pow(Y, Theta^[1]);
  end;

  procedure InitVarParam(Theta : PVector);
{ ----------------------------------------------------------------------
  Initializes the parameters of the variance function
  ---------------------------------------------------------------------- }
  begin
    Theta^[1] := 1.0;
  end;

  function LastVarParam : Integer;
{ ----------------------------------------------------------------------
  Returns the index of the last variance parameter (upper bound of Theta)
  ---------------------------------------------------------------------- }
  begin
    LastVarParam := 1;
  end;

{ **********************************************************************
                Routines related to the regression function
  ********************************************************************** }

  function FuncName : String;
{ ----------------------------------------------------------------------
  Returns the name of the regression function
  ---------------------------------------------------------------------- }
  begin
    FuncName := 'y = Ymax . x / (Km + x)';
  end;

  function FirstParam : Integer;
{ ----------------------------------------------------------------------
  Returns the index of the first parameter to be fitted
  ---------------------------------------------------------------------- }
  begin
    FirstParam := 1;
  end;

  function LastParam : Integer;
{ ----------------------------------------------------------------------
  Returns the index of the last parameter to be fitted
  ---------------------------------------------------------------------- }
  begin
    LastParam := 2;
  end;

  function ParamName(I : Integer) : String;
{ ----------------------------------------------------------------------
  Returns the name of the I-th parameter
  ---------------------------------------------------------------------- }
  begin
    case I of
      1 : ParamName := 'Ymax';
      2 : ParamName := 'Km  ';
    end;
  end;

  function RegFunc(X : Float; B : PVector) : Float;
{ ----------------------------------------------------------------------
  Computes the regression function for a given data point
  ----------------------------------------------------------------------
  Input  : X = point abscissa
           B = regression parameters (Ymax, Km)
  ---------------------------------------------------------------------- }
  begin
    RegFunc := B^[1] * X / (B^[2] + X);
  end;

  function ApproxFit(X, Y : PVector; M : Integer; B : PVector) : Integer;
{ ----------------------------------------------------------------------
  Approximate fit of Michaelis equation by weighted linear regression
  ----------------------------------------------------------------------
  Input  : X, Y = Point coordinates
           M    = Number of points
  Output : B    = Regression parameters (Ymax, Km)
  ----------------------------------------------------------------------
  Possible results : MAT_OK
                     MAT_SINGUL
  ---------------------------------------------------------------------- }
  var
    X1, Y1  : PVector;  { Transformed coordinates }
    W1      : PVector;  { Weights }
    A       : PVector;  { Linear regression parameters }
    V       : PMatrix;  { Variance-covariance matrix }
    P       : Integer;  { Number of points used for linear regression }
    K       : Integer;  { Loop variable }
    ErrCode : Integer;  { Error code }
  begin
    DimVector(X1, M);
    DimVector(Y1, M);
    DimVector(W1, M);
    DimVector(A, 1);
    DimMatrix(V, 1, 1);

    { Linear regression : 1/Y = 1/B[1] + (B[2]/B[1]) * (1/X) }

    P := 0;
    for K := 1 to M do
      if (X^[K] > 0.0) and (Y^[K] > 0.0) then
        begin
          Inc(P);
          X1^[P] := 1.0 / X^[K];
          Y1^[P] := 1.0 / Y^[K];
          W1^[P] := Sqr(Sqr(Y^[K]));  { Var(1/Y) ~ Var(Y)/Y^4 }
        end;

    ErrCode := WLinFit(X1, Y1, W1, P, A, V);

    { Compute Ymax and Km from linear regression parameters }
    if ErrCode = MAT_OK then
      begin
        B^[1] := 1.0 / A^[0];
        B^[2] := A^[1] / A^[0];
      end;

    ApproxFit := ErrCode;

    DelVector(X1, M);
    DelVector(Y1, M);
    DelVector(W1, M);
    DelVector(A, 1);
    DelMatrix(V, 1, 1);
  end;

{ **********************************************************************
  Define here the objective functions to be minimized, according to the
  declaration in OPTIM.PAS. These functions must be compiled in FAR mode
  ($F+). The global variable Icalc is used to pass the index of the
  current curve.
  ********************************************************************** }

  {$F+}
  function WLS_ObjFunc(B : PVector) : Float;
{ ----------------------------------------------------------------------
  Objective function for weighted least squares
  ---------------------------------------------------------------------- }
  var
    K : Integer;  { Loop variable }
  begin
    for K := 1 to M^[Icalc] do
      Ycalc^[Icalc]^[K] := RegFunc(X^[Icalc]^[K], B);
    WLS_ObjFunc := SumWSqrDifVect(Y^[Icalc], Ycalc^[Icalc],
                                  W^[Icalc], 1, M^[Icalc]);
  end;

  function GLS_ObjFunc(B : PVector) : Float;
{ ----------------------------------------------------------------------
  Objective function for generalized least squares
  ---------------------------------------------------------------------- }
  var
    S : Float;    { Sum }
    V : Float;    { Variance }
    K : Integer;  { Loop variable }
  begin
    S := 0.0;
    for K := 1 to M^[Icalc] do
      begin
        Ycalc^[Icalc]^[K] := RegFunc(X^[Icalc]^[K], B);
        V := Theta^[0] * VarFunc(Ycalc^[Icalc]^[K], Theta);
        S := S + Sqr(Y^[Icalc]^[K] - Ycalc^[Icalc]^[K]) / V + Ln(V);
      end;
    GLS_ObjFunc := S;
  end;

  function Var_ObjFunc(Theta : PVector) : Float;
{ ----------------------------------------------------------------------
  Objective function for variance estimation
  ---------------------------------------------------------------------- }
  var
    S    : Float;    { Sum }
    V    : Float;    { Variance }
    I, K : Integer;  { Loop variables }
  begin
    S := 0.0;
    for I := 1 to N do
      if ErrCode^[I] = MAT_OK then
        for K := 1 to M^[I] do
          begin
            V := Theta^[0] * VarFunc(Ycalc^[I]^[K], Theta);
            S := S + Sqr(Y^[I]^[K] - Ycalc^[I]^[K]) / V + Ln(V);
          end;
    Var_ObjFunc := S;
  end;
  {$F-}

{ **********************************************************************
  Define here the functions to be plotted, according to the declaration
  in PLOT.PAS. These functions must be compiled in FAR mode ($F+). The
  global variable Icalc is used to pass the index of the current curve.
  ********************************************************************** }

  {$F+}
  function Reg_PlotFunc(X : Float) : Float;
  { Regression function }
  begin
    Reg_PlotFunc := RegFunc(X, B^[Icalc]);
  end;

  function CV_PlotFunc(Y : Float) : Float;
  { Coefficient of variation }
  begin
    if Y = 0.0 then
      CV_PlotFunc := 100.0
    else
      CV_PlotFunc := Sqrt(Theta^[0] * VarFunc(Y, Theta)) / Y * 100.0;
  end;
  {$F-}

{ **********************************************************************
        Routines to read data, perform fit, write and plot results
  ********************************************************************** }

  procedure ReadInputFile(InFName          : String;
                          var Title, XName : String;
                          var YName        : PStrVector;
                          var N, Nobs      : Integer;
                          var M            : PIntVector;
                          var X, Y         : PMatrix);
{ ----------------------------------------------------------------------
  Reads input file and dimensions arrays
  ---------------------------------------------------------------------- }
  var
    InF    : Text;     { Input file }
    Nvar   : Integer;  { Number of variables }
    X1, Y1 : Float;    { Values read }
    I, K   : Integer;  { Loop variables }
  begin
    Assign(InF, InFName);
    Reset(InF);
    ReadLn(InF, Title);
    ReadLn(InF, Nvar);
    ReadLn(InF, XName);

    N := Nvar - 1;
    DimIntVector(M, N);

    DimStrVector(YName, N);
    for I := 1 to N do
      ReadLn(InF, YName^[I]);

    ReadLn(InF, Nobs);
    DimMatrix(X, N, Nobs);
    DimMatrix(Y, N, Nobs);

    for K := 1 to Nobs do
      begin
        Read(InF, X1);
        for I := 1 to N do
          begin
            Read(InF, Y1);
            if Y1 <> MISSING then
              begin
                Inc(M^[I]);
                X^[I]^[M^[I]] := X1;
                Y^[I]^[M^[I]] := Y1;
              end;
          end;
      end;
    Close(InF);
  end;

  function TotalObs(N : Integer; M : PIntVector;
                    ErrCode : PIntVector) : Integer;
{ ----------------------------------------------------------------------
  Returns the total number of observations for the fitted curves
  ---------------------------------------------------------------------- }
  var
    Mt : Integer;  { Total }
    I  : Integer;  { Loop variable }
  begin
    Mt := 0;
    for I := 1 to N do
      if ErrCode^[I] = MAT_OK then
        Mt := Mt + M^[I];
    TotalObs := Mt;
  end;

  procedure WLSFit(N       : Integer;
                   M       : PIntVector;
                   X, Y, W : PMatrix;
                   Theta   : PVector;
                   B       : PMatrix;
                   ErrCode : PIntVector);
{ ----------------------------------------------------------------------
  Fits the regression function to all curves by weighted least squares
  ---------------------------------------------------------------------- }
  var
    I, K  : Integer;  { Loop variables }
    F_min : Float;    { Objective function at minimum }
  begin
    { Set address of objective function }
    ObjFuncAddr := @WLS_ObjFunc;

    { Compute weights }
    for I := 1 to N do
      for K := 1 to M^[I] do
        W^[I]^[K] := 1.0 / VarFunc(Y^[I]^[K], Theta);

    { Fit each curve. The global variable Icalc is used to pass
      the index of the current curve to the minimization routine }
    for Icalc := 1 to N do
      begin
        { Compute initial parameter estimates }
        ErrCode^[Icalc] := ApproxFit(X^[Icalc], Y^[Icalc], M^[Icalc],
                                     B^[Icalc]);
        { If estimation successful, refine parameters }
        if ErrCode^[Icalc] = MAT_OK then
          ErrCode^[Icalc] := Simplex(B^[Icalc], FirstParam, LastParam,
                                     MAXITER, TOL, F_min);
      end;
  end;

  procedure GLSFit(N       : Integer;
                   M       : PIntVector;
                   Y       : PMatrix;
                   Theta   : PVector;
                   B       : PMatrix;
                   ErrCode : PIntVector);
{ ----------------------------------------------------------------------
  Fits the regression function to all curves by generalized least squares
  ---------------------------------------------------------------------- }
  var
    S         : Float;    { Sum of weighted squared residuals }
    F_min     : Float;    { Objective function at minimum }
    Code      : Integer;  { Error code }
    Iter      : Integer;  { Iteration number }
    Old_Theta : PVector;  { Variance parameters from previous iteration }
    Corr      : Float;    { Parameter correction }
    MaxCorr   : Float;    { Highest correction }
    I, K      : Integer;  { Loop variables }
  begin
    DimVector(Old_Theta, LastVarParam);

    { Estimate theta(0) from weighted least squares results }
    S := 0.0;
    for I := 1 to N do
      if ErrCode^[I] = MAT_OK then
        S := S + SumWSqrDifVect(Y^[I], Ycalc^[I], W^[I], 1, M^[I]);
    Theta^[0] := S / TotalObs(N, M, ErrCode);

    { Perform GLS estimation }
    Iter := 0;
    repeat
      Inc(Iter);
      CopyVector(Old_Theta, Theta, 0, LastVarParam);

      { Fit variance function to the whole set of curves }
      ObjFuncAddr := @Var_ObjFunc;
      Code := Simplex(Theta, 0, LastVarParam, MAXITER, TOL, F_min);
      if Code = MAT_OK then
        begin
          WriteLn(#10'Iter.', Iter:3, ' *** Estimated variance parameters :');
          for K := 0 to LastVarParam do
            Write('Theta(', K, ') = ', Theta^[K]:10:4, '  ');
          WriteLn;

          { Fit regression function to each curve }
          Write(#10'Fitting equation to curve : ');
          ObjFuncAddr := @GLS_ObjFunc;
          for Icalc := 1 to N do
            if ErrCode^[Icalc] = MAT_OK then
              begin
                ErrCode^[Icalc] := Simplex(B^[Icalc], FirstParam, LastParam,
                                           MAXITER, TOL, F_min);
                if ErrCode^[Icalc] = MAT_OK then
                  Write(Icalc:3);
              end;
        end;
      WriteLn;

      { Test convergence }
      MaxCorr := 0.0;
      for I := 0 to LastVarParam do
        begin
          Corr := Abs(1.0 - Theta^[I] / Old_Theta^[I]);
          if Corr > MaxCorr then MaxCorr := Corr;
        end;
    until (Code <> MAT_OK) or (MaxCorr < TOL) or (Iter = MAXITER);

    DelVector(Old_Theta, LastVarParam);
  end;

  procedure WriteOutputFile(InFName : String;
                            Title   : String;
                            N       : Integer;
                            B       : PMatrix;
                            ErrCode : PIntVector);
{ ----------------------------------------------------------------------
  Writes fitted parameters to output file (*.OUT)
  ---------------------------------------------------------------------- }
  var
    OutFName : String;   { Name of output file }
    OutF     : Text;     { Output file }
    Ncalc    : Integer;  { Number of fitted curves }
    I, K     : Integer;  { Loop variables }
  begin
    K := Pos('.', InFName);
    OutFName := Copy(InFName, 1, Pred(K)) + '.OUT';
    Assign(OutF, OutFName);
    Rewrite(OutF);

    WriteLn(OutF, Title);
    WriteLn(OutF, LastParam - FirstParam + 2);
    WriteLn(OutF, 'Index');
    for K := FirstParam to LastParam do
      WriteLn(OutF, ParamName(K));

    Ncalc := 0;
    for I := 1 to N do
      if ErrCode^[I] = 0 then
        Inc(Ncalc);

    WriteLn(OutF, Ncalc);
    for I := 1 to N do
      begin
        if ErrCode^[I] = MAT_OK then
          begin
            Write(OutF, I:3);
            for K := FirstParam to LastParam do
              Write(OutF, ' ', B^[I]^[K]:12:6);
            WriteLn(OutF);
          end;
      end;
    Close(OutF);
    WriteLn('Results written to file ', OutFName);
  end;

  procedure PlotCurves(N       : Integer;
                       M       : PIntVector;
                       X, Y    : PMatrix;
                       ErrCode : PIntVector);
{ ----------------------------------------------------------------------
  Plots the fitted curves
  ---------------------------------------------------------------------- }
  var
    Mt      : Integer;  { Total number of points }
    XX, YY  : PVector;  { Point coordinates }
    I, J, K : Integer;  { Loop variables }
    Ch      : Char;     { Key pressed }
  begin
    Mt := TotalObs(N, M, ErrCode);

    DimVector(XX, Mt);
    DimVector(YY, Mt);

    J := 0;
    for I := 1 to N do
      if ErrCode^[I] = MAT_OK then
        for K := 1 to M^[I] do
          begin
            Inc(J);
            XX^[J] := X^[I]^[K];
            YY^[J] := Y^[I]^[K];
          end;

    { Determine scale, setting origin at (0,0) for Michaelis equation }
    AutoScale(XX, 1, Mt, XAxis);
    AutoScale(YY, 1, Mt, YAxis);
    if XAxis.Min <> 0.0 then XAxis.Min := 0.0;
    if YAxis.Min <> 0.0 then YAxis.Min := 0.0;

    { Set the address of the function to be plotted }
    PlotFuncAddr := @Reg_PlotFunc;

    Write(#10'Press a key...');
    Ch := ReadKey;

    if GraphOk then
      begin
        SetClipping(True);

        { Plot experimental points }
        PlotCurve(XX, YY, 1, Mt, 1, 0);

        { Plot fitted curves. The global variable Icalc is used to pass
          the index of the current curve to the plotting function }
        for Icalc := 1 to N do
          if ErrCode^[Icalc] = MAT_OK then
            PlotFunc(XAxis.Min, XAxis.Max, 1);

        { Press a key to leave graphic }
        Ch := ReadKey;
        CloseGraph;
      end;

    DelVector(XX, Mt);
    DelVector(YY, Mt);
  end;

  procedure PlotCV(N       : Integer;
                   M       : PIntVector;
                   Ycalc   : PMatrix;
                   ErrCode : PIntVector);
{ ----------------------------------------------------------------------
  Plots the coefficient of variation as a function of Ycalc
  ---------------------------------------------------------------------- }
  var
    Mt      : Integer;  { Total number of points }
    Yc, CV  : PVector;  { Ycalc and coefficient of variation }
    I, J, K : Integer;  { Loop variables }
    Ch      : Char;     { Key pressed to exit }
  begin
    Mt := TotalObs(N, M, ErrCode);

    DimVector(Yc, Mt);
    DimVector(CV, Mt);

    J := 0;
    for I := 1 to N do
      if ErrCode^[I] = MAT_OK then
        for K := 1 to M^[I] do
          begin
            Inc(J);
            Yc^[J] := Ycalc^[I]^[K];
            CV^[J] := CV_PlotFunc(Yc^[J]);
          end;

    AutoScale(Yc, 1, Mt, XAxis);
    AutoScale(CV, 1, Mt, YAxis);
    if XAxis.Min <> 0.0 then XAxis.Min := 0.0;
    if YAxis.Min <> 0.0 then YAxis.Min := 0.0;

    PlotFuncAddr := @CV_PlotFunc;

    XTitle.Text := 'Y';
    YTitle.Text := 'CV (%)';
    Grid := BOTH_GRID;

    if GraphOk then
      begin
        SetClipping(True);
        PlotGrid;
        PlotFunc(XAxis.Min, XAxis.Max, 1);
        Ch := ReadKey;
        CloseGraph;
      end;

    DelVector(Yc, Mt);
    DelVector(CV, Mt);
  end;

  procedure PlotResiduals(N        : Integer;
                          M        : PIntVector;
                          Y, Ycalc : PMatrix;
                          Theta    : PVector;
                          ErrCode  : PIntVector);
{ ----------------------------------------------------------------------
  Plots the standardized residuals as a function of Ycalc
  ---------------------------------------------------------------------- }
  var
    Mt      : Integer;  { Total number of points }
    Yc, R   : PVector;  { Ycalc and standardized residuals }
    V       : Float;    { Estimated variance of an observation }
    I, J, K : Integer;  { Loop variables }
    Ch      : Char;     { Key pressed to exit }
  begin
    Mt := TotalObs(N, M, ErrCode);

    DimVector(Yc, Mt);
    DimVector(R, Mt);

    J := 0;
    for I := 1 to N do
      if ErrCode^[I] = MAT_OK then
        for K := 1 to M^[I] do
          begin
            Inc(J);
            V := Theta^[0] * VarFunc(Y^[I]^[K], Theta);
            Yc^[J] := Ycalc^[I]^[K];
            R^[J] := (Y^[I]^[K] - Ycalc^[I]^[K]) / Sqrt(V);
          end;

    AutoScale(Yc, 1, Mt, XAxis);
    AutoScale(R, 1, Mt, YAxis);

    with YAxis do
      if Abs(Min) > Max then
        Max := - Min
      else
        Min := - Max;

    XTitle.Text := 'Y calc';
    YTitle.Text := 'Normalized residual';
    Grid := HORIZ_GRID;

    if GraphOk then
      begin
        PlotGrid;
        PlotCurve(Yc, R, 1, Mt, 1, 0);
        Ch := ReadKey;
        CloseGraph;
      end;

    DelVector(Yc, Mt);
    DelVector(R, Mt);
  end;

{ **********************************************************************
                               Main program
  ********************************************************************** }

begin
  { Read command line parameter }
  ReadCmdLine(InFName);

  { Read input file and dimension arrays }
  ReadInputFile(InFName, Title, XName, YName, N, Nobs, M, X, Y);

  { Dimension additional arrays }
  DimMatrix(Ycalc, N, Nobs);
  DimMatrix(W, N, Nobs);
  DimMatrix(B, N, LastParam);
  DimVector(Theta, LastVarParam);
  DimIntVector(ErrCode, N);

  { Initialize variance parameters }
  InitVarParam(Theta);

  { Compute first estimates of regression parameters by weighted
    least squares, using initial variance parameters }
  WLSFit(N, M, X, Y, W, Theta, B, ErrCode);

  { Estimate regression and variance parameters
    by generalized least squares }
  GLSFit(N, M, Y, Theta, B, ErrCode);

  { Plot fitted curves, coefficient of variation, and residuals }
  PlotCurves(N, M, X, Y, ErrCode);
  PlotResiduals(N, M, Y, Ycalc, Theta, ErrCode);
  PlotCV(N, M, Ycalc, ErrCode);

  { Write results to output file }
  WriteOutputFile(InFName, Title, N, B, ErrCode);
end.