{ ******************************************************************* }
{ Turbo Pascal Version 4.0 --> 7.0 Date Functions and Procedures      }
{ ------------------------------------------------------------------- }
{ Copyright 1988, 1995 Roger E. Donais        <rdonais@southeast.net> }
{ ------------------------------------------------------------------- }

UNIT DATES;
{ ====================================================================}
{ - signifies options set by IDE or command line                      }
{-O+}    { Overlay compatiblity                - OFF  + ON            }
{-F-}    { Force far calls                     - OFF  + ON            }
{-R-}    { Range checking                      - OFF  + ON            }
{-S-}    { Stack checking                      - OFF  + ON            }
{-N-}    { Numeric coprocessor                 - OFF  + ON            }
{-E-}    { 8087 Emulation                      - OFF  + ON            }
{-D+}    { Dedug                               - OFF  + ON            }
{-L+}    { Local label information             - OFF  + ON            }
{$I+}    { I/O Error Checking is bracketed when needed w/in code      }
{$V+}    { Strict VAR-Strings are bracketed when needed w/in code     }
{$B-}    { Complete Boolean eval is bracketed when needed w/in code   }
{-X-}    { Extended syntax is not required by this unit               }
{ ------------------------------------------------------------------- }

INTERFACE
USES  Dos,Util;

CONST TimePerDay = 86400.0;    { Number of seconds in a 24 hour day        }
      BaseDate: LongInt = 0;   { User defined julian base (zero adjust)    }
      Century : Integer = 1900;{ Added to dates with year less than 99     }
      Rollover: Integer =    0;{ CENTURY+100 for RollOver <= Year <= 99    }
                               { EXAMPLE:                                  }
                               { RollOver = 0 and Century = 1900 forces    }
                               { 01/01/00 thru 12/31/99 to 1900 thru 1999  }
                               {                                           }
                               { RollOver = 50 and Century = 1900 forces   }
                               { 01/01/00 thru 12/31/49 to 2000 thru 2049  }
                               { 01/01/50 thru 12/31/99 to 1950 thru 1999  }

{ ------------------------------------------------------------------- }
{ The DATES Unit can be aligned with different spreadsheet programs,  }
{ operating systems and/or applications by setting BaseDate to the    }
{ match the other program's BaseDate.  The initial BaseDate can be    }
{ set directly using something like:                                  }
{                                                                     }
{                         BaseDate := JDate('01/01/1960');            }
{                                                                     }
{ Once set, it should be cleared before being reset, or the current   }
{ value should be added into the new value:                           }
{                                                                     }
{                         BaseDate := JDate('01-01-1980') + BaseDate; }
{                                                                     }
{ Default date format is MM/DD/YY.  See SetDateFormat()               }
{ JDate conversions require a non-alphanumeric separator.  Dates in   }
{ the form YYYYMMDD will not convert w/o separating the components.   }
{ Conversions JDate -> CDate can me made w/o separating componments.  }
{ ------------------------------------------------------------------- }


FUNCTION  Jdate(GregorianDate: String): LongInt;
{ =================================================================== }
{ Converts a Gregorian calandar date in the default (M/D/Y) format    }
{ to a julian date as the number of days since "BaseDate".            }
{ ------------------------------------------------------------------- }
{ NOTE: When the year is less than 99 it is summed with "Century"     }
{ =================================================================== }

FUNCTION  Cdate(JulianDate: LongInt): String;
{ =================================================================== }
{ Converts a Julian date to a Gregorian date in the default format.   }
{ ------------------------------------------------------------------- }
{ (See DateFormat)                                                    }
{ =================================================================== }

FUNCTION DateFormat(JulianDate: Longint; Fmt: String): String;
{ =================================================================== }
{ Returns a formated date --                                          }
{ ------------------------------------------------------------------- }
{ D = Day, M = Month, Y = Year, W = Day of the Week.                  }
{ number of M,D,Y characters specify width: 1=As required, 2..4 will  }
{   be right justified with leading spaces if lower case letter, or   }
{   zero filled if all uppercase.                                     }
{ When m > 2 specifies the text month in upper and lower case if any  }
{   m is lower case, otherwise all upper case.  Three M's will give   }
{   an abbreviated month (Jan, Feb, etc.), four or more will return   }
{   the full spelling (January,February,etc.)                         }
{ One W will return an abbreviated day (Mon, Tues, etc.), two or more }
{ W's will return the full spelling (Monday, Tuesday, etc.)           }
{                                                                     }
{     DD MMM YY           >>-- yields -->       02 JAN 89             }
{     Mmmm d, yyyy                              January 2, 1989       }
{                                                                     }
{ =================================================================== }

PROCEDURE SetDateFormat(Fmt: String);
{ =================================================================== }
{ Establishes the default date format - (See DateFormat)              }
{ =================================================================== }

FUNCTION  Day(JulianDate: LongInt): Integer;
{ =================================================================== }
{ Returns the Day of the month [1..31] for the Julian date            }
{ =================================================================== }

FUNCTION  Month(JulianDate: LongInt): Integer;
{ =================================================================== }
{ Returns the Month of the year [1..12] for the Julian date           }
{ =================================================================== }

FUNCTION  Year(JulianDate: LongInt): Integer;
{ =================================================================== }
{ Returns the year portion of the Julian date (e.g. 1987)             }
{ =================================================================== }

FUNCTION  Dow(JulianDate: LongInt): Integer;
{ =================================================================== }
{ Returns the numeric day of the week for a Julian date as --         }
{      (1 =Mon, 2 =Tues, 3 =Wed, 4 =Thus, 5 =Fri, 6 =Sat, 7 =Sun)     }
{ ------------------------------------------------------------------- }
{ HINT:  Succ(Dow(JuilianDate)) and 7 yields 0=Sun, 1=Mon, ..., 6=Sat }
{ =================================================================== }

FUNCTION  SystemDate: LongInt;
{ =================================================================== }
{ Returns the system date as a Julian date                            }
{ ------------------------------------------------------------------- }
{ See DOS Procedure GetDate(VAR Year, Month, Day, DayOfWeek: Integer);}
{ =================================================================== }

FUNCTION  SystemTime: Real;
{ =================================================================== }
{ Returns the system time as the number of seconds since mid-night    }
{ ------------------------------------------------------------------- }
{ See DOS Procedure GetTime(VAR Hour, Min, Sec, Sec100: Integer);     }
{ =================================================================== }

FUNCTION  FileStamp(VAR F): LongInt;
{ =================================================================== }
{ Returns the packed date and time that a file was last written       }
{ ------------------------------------------------------------------- }
{ See DOS Procedure GetFtime(VAR F; VAR PackedTime: LongInt);         }
{ =================================================================== }

FUNCTION  TimeOfStamp(FileStamp: LongInt): Real;
{ =================================================================== }
{ Returns the time portion of a packed date/time stamp returned by    }
{ FileStamp, GetFtime, FindFirst, or FindNext                         }
{ ------------------------------------------------------------------- }
{ See DOS Procedure UnpackTime(PackedTime: LongInt; VAR DT: DateTime);}
{ =================================================================== }

FUNCTION  DateOfStamp(FileStamp: LongInt): LongInt;
{ =================================================================== }
{ Returns the date portion of a packed date/time stamp returned by    }
{ FileStamp, GetFtime, FindFirst, or FindNext                         }
{ ------------------------------------------------------------------- }
{ See DOS Procedure UnpackTime(PackedTime: LongInt; VAR DT: DateTime);}
{ =================================================================== }

{-IFOPT N+}
PROCEDURE StampFile(VAR F; JulianDate: LongInt; Time: Real);   {-ENDIF}
{ =================================================================== }
{ Forces the date and time of the last file update.                   }
{ ------------------------------------------------------------------- }
{ See DOS Procedure SetFtime(VAR F; PackedTime: LongInt);             }
{ =================================================================== }

{ ******************************************************************* }
IMPLEMENTATION

{ ******************************************************************* }
{ ....................... U D A T E   L O G ......................... }
{                                                                     }
{ REV  DATE       DESCRIPTION of CHANGES                          BY  }
{                                                                     }
{       4 Dec 88  Kludged DOS System and File Date/Time Functions     }
{      12 Jan 88  Original program                             ...red }
{ ******************************************************************* }
CONST  Months: Array[1..12] of String[9]
            =  ('January','February','March','April','May','June','July',
                'August','September','October','November','December');

       Days  : Array[1..7] of String[9]
           =   ('Monday','Tuesday','Wednesday',
                'Thursday','Friday','Saturday','Sunday');

     Picture : String[16] = 'MM/DD/YY';
         MDY : Byte =  0;  { 0=MDY, 1=DMY, 2=YMD }

FUNCTION DateFormat(JulianDate: Longint; Fmt: String): String;
{ =================================================================== }
CONST mm : array[0..12] of Word = (000,031,061,092,122,153,184,214,245,275,306,337,666);
                                  {MAR APR MAY JUN JUL AUG SEP OCT NOV DEC JAN FEB}
      Letter: Array[1..4] of Char = 'dymw';
       Short: Array[1..7] of Char = #3#4#3#5#3#3#3;

VAR y400,y100,y4,y1: LongInt;
       id,i,j,k,Dow: Integer;
           Rslt,s,t: String;
           n: Array[1..4] of Integer;

BEGIN{DateFormat}
    Inc(JulianDate,BaseDate); y100 := 0; y4 := 0; y1 := 0;
    Dow := Succ(Succ(JulianDate) mod 7);

    y400 := JulianDate div 146097; Dec(JulianDate,y400 * 146097);
    If JulianDate < 1 Then Begin Inc(JulianDate,366); Dec(y1); End
    Else Begin
       y100 := JulianDate div 36524; Dec(JulianDate,y100 * 36524);
       If JulianDate < 1 Then Begin Inc(JulianDate,365); Dec(y1); End
       Else Begin
          y4   := JulianDate div 1461; Dec(JulianDate,y4 * 1461);
          If JulianDate < 1 Then Begin Inc(JulianDate,366); Dec(y1); End
          Else Begin
             y1   := JulianDate div 365; Dec(JulianDate,y1 * 365);
             If JulianDate < 1 Then Begin Inc(JulianDate,365); Dec(y1); End
          End;
       End;
    End;
    i := 1; While JulianDate > mm[i] Do Inc(i); Dec(JulianDate,mm[Pred(i)]);
    If i > 10 Then Begin Dec(i,12); Inc(y1); End;
    Y1 := y400*400 +y100*100 +y4*4 +y1;
    y4 := i+2;

    Rslt := Fmt;
    s := LowerCase(Rslt) + ' ';
    For id := 1 to 4 Do Begin
        i := Pos(Letter[id],s);
        If i <> 0 Then Begin
           j := Succ(i); While (s[j] = Letter[id]) Do Inc(j); Dec(j,i);
           n[id] := j; Rslt[i] := Char(id);
        End Else n[id] := 0;
    End;
    For id := 1 to 4 do Begin
        If n[id] <> 0 Then Begin
           i := Pos(Char(id),Rslt);
           k := Pos(Letter[id],Fmt);
           j := n[id];
           Case id of
                1: { 'd' == numeric day }
                   BEGIN t := Ascii(JulianDate,j);
                         If k = 0 Then t := Zfill(t);
                   END;

                2: { 'y' == year        }
                   BEGIN t := Rset(Ascii(Abs(y1),0),j);
                         If y1 < 0 Then t := t + ' BC';
                   END;

                3: { 'm' == month       }
                   Case j of
                      1,2: BEGIN t := Ascii(y4,j);
                                 If k = 0 Then t := Zfill(t);
                            END;
                       Else t := Months[y4];
                            If k = 0 Then t := UpperCase(t);
                            If j = 3 Then t[0] := #3;
                   End{Case};

                4: { w == week day      }
                   BEGIN t := Days[Dow];
                         If k = 0 Then t := UpperCase(t);
                         If j = 1 Then t[0] := Short[Dow];
                   End;
           End{Case};
           Delete(Rslt,i,j); Insert(t,Rslt,i);
        End;
    End;
    DateFormat := Rslt;
END{DateFormat};


PROCEDURE SetDateFormat(Fmt: String);
{ =================================================================== }
VAR S: String;
BEGIN
    Picture := Fmt; s := UpperCase(FMT);
    MDY := Ord(Pos('D',s) < Pos('M',s));
    If Pos('Y',s) < Pos('M',s) Then Inc(MDY,1);
END;


FUNCTION  Juliandate(GregorianDate: String; MDY: Integer): LongInt;
{ =================================================================== }
CONST mm : array[1..12] of Word = (306,337,000,031,061,092,122,153,184,214,245,275);
                                  {MAR APR MAY JUN JUL AUG SEP OCT NOV DEC JAN FEB}
VAR   d,m,y,i,j,k: Word;
      v: Array[1..3] of Integer;
            z: LongInt;

BEGIN{Jdate}
    Juliandate := 0; { Assume error in conversion }
    GregorianDate := WordCase(Ftrim(GregorianDate))+' ';

    m := 0; d := 0; y := 0; k := 0; FillChar(v,Sizeof(v),0);
    While GregorianDate[0] <> #0 Do Begin
        Case GregorianDate[1] of
           'A'..'Z': BEGIN j := 1;
                         REPEAT Inc(j); UNTIL NOT(GregorianDate[j] in ['a'..'z']);
                         For i := 1 to 12 Do
                             If Copy(Months[i],1,Pred(j)) = Copy(GregorianDate,1,Pred(j)) Then Begin
                                {If mdy = 0 Then v[1] := i Else v[2] := i;}
                                v[3] := i;
                                mdy := 3;
                             End;
                         REPEAT
                             Delete(GregorianDate,1,1);
                         UNTIL (GregorianDate[0] = #0) or NOT(GregorianDate[1] in ['a'..'z']);
                      END;
           '0'..'9':  BEGIN j := 1; Inc(k);
                            While GregorianDate[j] in ['0'..'9'] Do Inc(j);
                            v[k] := Binary(Copy(GregorianDate,1,Pred(j)));
                            Delete(GregorianDate,1,Pred(j));
                      END;
                 Else Delete(GregorianDate,1,1);
        End{Case};
    End;
    Case MDY of
         0: BEGIN m := v[1]; d := v[2]; y := v[3]; END;
         1: BEGIN d := v[1]; m := v[2]; y := v[3]; END;
         2: BEGIN y := v[1]; m := v[2]; d := v[3]; END;
         3: BEGIN d := v[1]; m := v[3]; y := v[2]; END;
    End;
    If y < 99 Then
       If y < RollOver Then
          Inc(y,Century+100)
       Else Inc(y, Century);

    If (m < 1) or (m > 12) or (d < 1) or (d > 31) Then Exit;
    If m = 2 Then Begin
       If m > 28 Then Begin
          If m > 29 Then Exit;
          If (y and 3) <> 0 Then Exit;
          If (y mod 100 = 0) and (y mod 400 <> 0) Then Exit;
       End;
    End Else If (d > 30) and Odd($0A54 Shr m) Then Exit;

    If m < 3 Then Dec(y);
    Juliandate := LongInt(y) * 365 + y div 4 - y div 100 + y div 400 + mm[m] + d - BaseDate;
END{Jdate};


FUNCTION  Day(JulianDate: LongInt): Integer;
{ =================================================================== }
BEGIN  Day := Binary(DateFormat(JulianDate,'D'));  END;


FUNCTION  Month(JulianDate: LongInt): Integer;
{ =================================================================== }
BEGIN  Month := Binary(DateFormat(JulianDate,'M'));  END;


FUNCTION  Year(JulianDate: LongInt): Integer;
{ =================================================================== }
BEGIN  Year := Binary(DateFormat(JulianDate,'YYYY'));  END;


FUNCTION  Dow(JulianDate: LongInt): Integer;
{ =================================================================== }
BEGIN Dow := Succ(Succ(JulianDate + BaseDate) mod 7); END;


FUNCTION  SystemDate: LongInt;
{ =================================================================== }
VAR Year, Month, Day, DayOfWeek: Word;

BEGIN
    GetDate(Year, Month, Day, DayOfWeek);
    SystemDate := Juliandate(Ascii(Month,1) +'/'+Ascii(Day,1) + '/'+ Ascii(Year,1),0);
END;


FUNCTION  SystemTime: Real;
{ =================================================================== }
VAR Hour, Min, Sec, Sec100: Word;

BEGIN
    GetTime(Hour, Min, Sec, Sec100);
    SystemTime := 3600.0 * Hour + 60.0 * Min + Sec + Sec100 / 100.0;
END;


FUNCTION  FileStamp(VAR F): LongInt;
{ =================================================================== }
VAR PackedTime: LongInt;

BEGIN
    GetFtime(F,PackedTime);
    FIleStamp := PackedTime;
END;


FUNCTION  TimeOfStamp(FileStamp: LongInt): Real;
{ =================================================================== }
VAR DT: DateTime;

BEGIN
    UnpackTime(FileStamp, DT);
    TimeOfStamp := 3600.0 * DT.Hour + 60.0 * DT.Min + DT.Sec;
END;


FUNCTION  DateOfStamp(FileStamp: LongInt): LongInt;
{ =================================================================== }
VAR DT: DateTime;

BEGIN
    UnpackTime(FileStamp,DT);
    DateOfStamp := Juliandate(Ascii(DT.Month,1) +'/'+ Ascii(DT.Day,1) +'/'+ Ascii(DT.Year,1),0);
END;

{-IFOPT N+}
PROCEDURE StampFile(VAR F; JulianDate: LongInt; Time: Real);
{ =================================================================== }
VAR DT: DateTime;
    PackedTime: LongInt;

BEGIN
    DT.Month := Month(JulianDate);
    DT.Day   := Day(JulianDate);
    DT.Year  := Year(JulianDate);
    DT.Hour  := Trunc(Time / 3600); Time := Time - (DT.Hour * 3600);
    DT.Min   := Trunc(Time / 60);   Time := Time - (DT.Min  * 60);
    DT.Sec   := Trunc(Time);
    PackTime(DT,PackedTime);
    SetFtime(F,PackedTime);
END;
{-ENDIF}

FUNCTION  Cdate(JulianDate: LongInt): String;
{ =================================================================== }
BEGIN
    Cdate := DateFormat(JulianDate,Picture);
END;

FUNCTION  Jdate(GregorianDate: String): Longint;
{ =================================================================== }
BEGIN
    Jdate := JulianDate(GregorianDate,MDY);
END;

END{Dates}.

{ DATES UNIT ENHANCEMENTS }
USES DATES,UTIL;

BEGIN
    Writeln(#10#10#10#10#10#10#10#10#10#10#10#10);
    Writeln(DateFormat(SystemDate,'DD MMM YY'));
    Writeln(DateFormat(SystemDate,'mm/dd/yyyy'));
    Writeln(DateFormat(SystemDate,'dD Mmm YY'));
    Writeln(DateFormat(SystemDate,'ww, mmmmmm dd, YYYY'));

    Writeln('Tomorrow will be ',DateFormat(Succ(SystemDate),'ww'));

END.
