{$A-} {Align data}
{$B-} {Boolean complete evaluation}
{$D-} {Debug information}
{$E-} {Emulation}
{$F-} {Far calls}
{$I-} {Input/Output checking}
{$L-} {Local symbol information}
{$N-} {Numeric processing}
{$O+} {Overlay code generation}
{$R+} {Range checking}
{$S+} {Stack overflow checking}
{$V-} {Var string checking}

UNIT FORTRAN;
{+H
---------------------------------------------------------------------------
  Version     - 0.10

  File        - FORTRAN.PAS

  Copyright   - None. Public Domain.

  Author      - Keith S. Brown (except where otherwise noted)
                713-483-8952
                Surface mail:              Email:(brown@smd4.jsc.nasa.gov)
                  K.Brown
                  Code:NASA/JSC/ES64
                  Houston, TX 77058 (USA)  Voice:(713)483-8952

  Purpose     - Provide FORTRAN formatted read for Pascal.

  Remarks     - See usage notes following.

  Requires    - No auxiliary units required.

  Revised     - 1988.1014 V:0.00 (KSB) Wrote first version.
              - 1990.1001 V:0.10 (KSB) Updated.
              - 1993.0803 Reformatted using KPPF.  Updated documentation.
---------------------------------------------------------------------------
  Usage Notes -
  . IOcheck :
     Check for I/O error and print an error message if needed.
     Calls IOresult directly, therefore for use with external I/O.

  . IOstatusCheck :
     Check for I/O error by examining the IOstatus as set in OpenFile or
     OpenText and print an error message if needed.  For use with FORTRAN
     internal I/O.

  . FileExists :
     Checks for file existance by attempting to open the file.  Function
     returns true and closes file if it exists, else it returns false.

  . OpenFile :
     Opens an untyped file for either read or write access. OpenFile does
     no checking to ensure that the file exists.

  . OpenText :
     Opens a text file for either read or write access. OpenFile does no
     checking to ensure that the file exists.

  . FIOerror :
     Returns the index of the character in the FMT string that caused
     a FormatRead error 2; or returns the index of the character of the
     INDATA string that caused a FormatRead error 1.

  . FormatRead :
     FormatRead is simplified FORTRAN FORMATTED read in which repetition
     counts are allowed, but parentheses are not.  The following field
     descriptors are defined:

     Field        Field          Field     Pascal      Turbo Pascal
     Descriptor   Reprentation   Example   Example     List Element
     ----------   ------------   -------   -------     ------------
        A            Aw            A10     STRING[10]  w+1 byte Turbo String
        E            Ew.d          E6.1    REAL          6 byte Real
        F            Fw.d          F6.1    REAL          6 byte Real
        I            Iw            I3      LongINT       4 byte Long INT
        L            Lw            L5      BOOLEAN       1 byte Boolean
        T            Tn            T37     ---           positional tabulation
        X            nX            20X     ---           positional specifier

     FormatRead interprets an INDATA string according to the provided FMT
     statement.  The interpreted values are returned, concatenated together
     in OUTDATA, which is assumed to be an externally declared array or
     record of the proper size.

     STAT returns the status of the interpretation:

     If STAT is zero then no error was encountered.
     If STAT is 1, then a format error was found. FIOerror returns
       the location of the bad symbol in the format.
     If STAT is 2, then a data conversion error occured. FIOerror
       returns the location of the bad data symbol.
     If STAT is 3, then the size of OutData is too small to contain
       all the data specified by the format and converted from
       the input data.

   Example:

USES
  FORTRAN;

VAR
  MGTD   : RECORD
    days : LongINT;
    hour : LongINT;
    mins : LongINT;
    secs : REAL;
  END;

  s,fmt  : STRING;
  i,Err  : BYTE;
BEGIN
  s := '';
  FOR i := 1 TO ParamCount DO s := s + ParamStr(i);
  fmt := 'i3,1x,i2,1x,i2,1x,f6.3';
  WriteLn(s);
  FormatRead(s,fmt,MGTD,SizeOf(MGTD),Err);
  WriteLn('Bias data from GMT of ',MGTD.days,':',MGTD.hour,':',MGTD.mins,':',MGTD.secs:6:3);
END.
}

{.$DEFINE debug}


INTERFACE
{$IFDEF debug}
USES
  TPcrt,
{$ENDIF}


{}PROCEDURE IOCheck;
{}PROCEDURE IOStatusCheck;
{}FUNCTION  FileExists(Fname : STRING) : BOOLEAN;
{}FUNCTION  OpenFile(VAR FV:FILE; fn:STRING; op:STRING):BOOLEAN;
{}FUNCTION  OpenText(VAR FV:TEXT; fn:STRING; op:STRING):BOOLEAN;
{}FUNCTION  FIOerror:BYTE;
{}PROCEDURE FormatRead(
     indata   : STRING; {text line to be processed}
     fmt      : STRING; {format statement}
 VAR outdata;           {processed data}
     dataSize : WORD;   {size of outData}
 VAR stat     : BYTE);  {status of FormatRead}

     {====================================================================}

IMPLEMENTATION
VAR
  FIO__error  : BYTE;
  IOerr  : BOOLEAN;
  IOstatus    : INTEGER;


{}FUNCTION  IOErrorMessage:STRING;
{---------------------------------------------------------------------------
  Purpose     - Return a description for an error number.
---------------------------------------------------------------------------}
{}{}FUNCTION Long2Str(L:LongINT):STRING;
    VAR
      s  : STRING[11];
    BEGIN
      Str(L,s);
      Long2Str := s;
{}{}END {Long2Str};

  BEGIN
    IF IOerr THEN
      CASE IOstatus OF
        2   : IOErrorMessage := 'File not found.';
        3   : IOErrorMessage := 'Path not found.';
        4   : IOErrorMessage := 'Too many open files.';
        5   : IOErrorMessage := 'File access denied.';
        6   : IOErrorMessage := 'Invalid file handle.';
        12  : IOErrorMessage := 'Invalid file access code.';
        15  : IOErrorMessage := 'Invalid drive number.';
        16  : IOErrorMessage := 'Cannot remove current directory.';
        17  : IOErrorMessage := 'Cannot rename across drives.';
        100 : IOErrorMessage := 'Disk read error.';
        101 : IOErrorMessage := 'Disk write error.';
        102 : IOErrorMessage := 'File not assigned.';
        103 : IOErrorMessage := 'File not open.';
        104 : IOErrorMessage := 'File not open for input.';
        105 : IOErrorMessage := 'File not open for output.';
        106 : IOErrorMessage := 'Invalid numeric format.';
        ELSE
          IOErrorMessage := ^G+'DOS error number = '+Long2Str(IOstatus)+'. See manual.';
      END {CASE};
{}END {IOErrorMessage};




{}PROCEDURE Error(Msg : STRING);
{---------------------------------------------------------------------------
  Purpose     - Write an error message.
---------------------------------------------------------------------------}
  BEGIN
    WriteLn;
    Write(^G); { Beep! }
    WriteLn(Msg);
    WriteLn;
{}END {Error};




{}PROCEDURE IOCheck;
{+H
---------------------------------------------------------------------------
  Purpose     - Tests for I/O error.

  Declaration - procedure IOCheck;

  Remarks     - Calls IOresult directly, therefore for use with external I/O.
                Prints an error message if needed.
---------------------------------------------------------------------------}
  BEGIN
    IOstatus := IOresult;
    IOerr    := IOstatus <> 0;
    IF IOerr THEN
      Error(IOErrorMessage);
    IOstatus := 0;
{}END {IOCheck};




{}PROCEDURE IOStatusCheck;
{+H
---------------------------------------------------------------------------
  Purpose     - Check for I/O error by examining the internal IOstatus
                variable set by a prior function call.

  Declaration - procedure IOStatusCheck;

  Remarks     - Check for I/O error by examining the IOstatus as set in
                OpenFile or OpenText and print an error message if needed.
                For use with FORTRAN internal I/O.
                Prints an error message if needed.
---------------------------------------------------------------------------}
  BEGIN
    IOerr := IOstatus <> 0;
    IF IOerr THEN
      Error(IOErrorMessage);
    IOstatus := 0;
{}END {IOStatusCheck};




{}FUNCTION FileExists(Fname : STRING) : BOOLEAN;
{+H
---------------------------------------------------------------------------
  Purpose     - Determine if the named file exists.

  Declaration - function FileExists(fname:STRING):BOOLEAN;

  Remarks     - FNAME is a string type expression representing the name of a
                DOS file.  The function returns TRUE if the named file
                exists, FALSE otherwise.  If found, the file is closed before
                the function exits.
---------------------------------------------------------------------------}
  VAR
    CheckFile : FILE;
  BEGIN
    Assign(CheckFile, Fname);
    {$I-} Reset(CheckFile); {$I+}
    IF IOresult = 0 THEN BEGIN
      FileExists := TRUE;
      Close(CheckFile)
    END ELSE
      FileExists := FALSE;
{}END {FileExists};




{}FUNCTION OpenFile(VAR FV:FILE; fn:STRING; op:STRING):BOOLEAN;
{+H
---------------------------------------------------------------------------
  Purpose     - Open an untyped file for Read or Write access

  Declaration - function OpenFile(VAR FV:FILE; fn:STRING; op:STRING):BOOLEAN;

  Remarks     - Opens the named file as an untyped file for either read or
                write access. OpenFile does no checking to ensure that the
                file exists.

                FV is an untyped file variable for the named file.

                FN is a string expression representing the name of a valid
                DOS file.

                OP is a string expression naming the operation to be performed.
                If the first character of OP is "r" or "R" then the file is
                opened for reading.  If the first character of OP is "w" or
                "W" then the file is opened for writing.  The file is not
                opened for any other character.

                The function returns TRUE if the file could be opened.

  Revised     - 1989.0505 (KSB) Wrote initial version.
              - 1993.0803 (KSB) Guarded against bad OP code.
---------------------------------------------------------------------------}
  BEGIN
    Assign(FV,fn);

    {$I-}
    CASE UpCase(Op[1]) OF
    'R' : Reset(FV,1);
    'W' : Rewrite(FV,1);
    END {CASE};

    IF UpCase(op[1]) IN ['R','W'] THEN
      IOstatus := IOresult
    ELSE
      IOstatus := 12;

    OpenFile := IOstatus = 0;
    {$I+}
{}END {OpenFile};




{}FUNCTION OpenText(VAR FV:TEXT; fn:STRING; op:STRING):BOOLEAN;
{+H
---------------------------------------------------------------------------
  Purpose     - Open a text file for Read or Write access

  Declaration - function OpenText(VAR FV:TEXT; fn:STRING; op:STRING):BOOLEAN;

  Remarks     - Opens the named file as an untyped file for either read or
                write access. OpenFile does no checking to ensure that the
                file exists.

                FV is a text file variable for the named file.

                FN is a string expression representing the name of a valid
                DOS file.

                OP is a string expression naming the operation to be performed.
                If the first character of OP is "r" or "R" then the file is
                opened for reading.  If the first character of OP is "w" or
                "W" then the file is opened for writing.  The file is not
                opened for any other character.

                The function returns TRUE if the file could be opened.

  Revised     - 1989.0505 (KSB) Wrote initial version.
              - 1993.0803 (KSB) Guarded against bad OP code.
---------------------------------------------------------------------------}
  BEGIN
    Assign(FV,fn);

    {$I-}
    CASE UpCase(Op[1]) OF
    'R' : Reset(FV);
    'W' : Rewrite(FV);
    END {CASE};

    IF UpCase(op[1]) IN ['R','W'] THEN
      IOstatus := IOresult
    ELSE
      IOstatus := 12;

    OpenText := IOstatus = 0;
    {$I+}
{}END {OpenText};




{}FUNCTION  FIOerror:BYTE;
{+H
---------------------------------------------------------------------------
  Purpose     - Returns the index of the offending format character.

  Declaration - function FIOerror:BYTE;

  Remarks     - If an error occured during a FormatRead call, FIOerror can
                be called to return the index of the offending format
                character.
                FIOerror returns zero if the format was not at falt.
---------------------------------------------------------------------------}
  BEGIN
    FIOerror   := FIO__error;
    FIO__error := 0;
{}END {FIOerror};




{}PROCEDURE FormatRead(
       indata   : STRING; {text line to be processed}
       fmt:STRING;        {format statement}
   VAR outdata;           {processed data}
       dataSize:WORD;     {size of outData}
   VAR stat     : BYTE);  {status of FormatRead}
{+H
---------------------------------------------------------------------------
  Purpose     - Provide a FORTRAN like formatted read.

  Declaration - procedure FormatRead(indata:STRING; fmt:STRING; VAR outdata;
                                     dataSize:WORD; VAR stat:BYTE);

  Remarks
     INDATA is a string expression containing FORTRAN formatted data.

     FMT is a string expression containing a FORTRAN format specification to
       be used to interpret the data contained in INDATA.

     OUTDATA is an untyped variable (usually a record or an array) in which
       the data read from INDATA is to be placed.

     DATASIZE is the size of OUTDATA in bytes.

     STAT returns the status code resulting from interpreting the data
       contained in INDATA.

     If STAT is zero then no error was encountered.
     If STAT is 1, then a format error was found. A call to FIOerror will
       returns the location of the bad symbol in the format.
     If STAT is 2, then a data conversion error occured. A call to FIOerror
       will return the location of the bad data symbol.
     If STAT is 3, then the size of OutData is too small to contain-
       all the data specified by the format and converted from
       the input data.
     Parentheses are not allowed and will generate an error.

     A Field Descriptor      Aw           w+1 byte Turbo String
     E Field Descriptor      Ew.d           6 byte Real
     F Field Descriptor      Fw.d           6 byte Real
     I Field Descriptor      Iw             4 byte Long INT
     L Field Descriptor      Lw             1 byte Boolean
     T Field Descriptor      Tn             positional tabulation
     X Field Descriptor      nX             positional specifier

  Revised     - 1988.1014 (KSB) Wrote initial version.
---------------------------------------------------------------------------}
  TYPE
    ByteArray = ARRAY[0..0] OF BYTE;
  CONST
    RealLen   = 6;      {no of bytes in a real}
    IntLen    = 4;      {no of bytes in an integer}
    LogLen    = 1;      {no of bytes in a boolean}
  VAR
    fmtPtr    : BYTE;   {points at current char in fmt}
    datPtr    : BYTE;   {points at current char in Data string}
    fmtLen    : BYTE;   {length of fmt}
    datLen    : BYTE;   {length of data string}
    repCnt    : BYTE;   {repetition count}
    dataPos   : WORD;   {current position in OutData}
    charToken : CHAR;   {current character}
    fld,dcml  : WORD;   {field and decimal format values.}
    r    : REAL;        {floatpt  value read from inData.}
    Li   : LongINT;     {integer  value read from inData.}
    L    : BOOLEAN;     {Logical  value read from inData.}
    s    : STRING;      {character data read from inData.}
    n    : WORD;        {counter}

{$IFDEF debug}
{}{}PROCEDURE SnapShot;
    VAR
      key: CHAR;
    BEGIN
      WriteLn;
      WriteLn('fmt = ',fmt);
      IF fmtPtr > 0 THEN
        WriteLn('      ',CharStr(' ',Pred(fmtPtr)),'^');
      WriteLn('charToken  . . . ',charToken);
      WriteLn('fmtPtr . . . . . ',fmtPtr);
      WriteLn('fmtLen . . . . . ',fmtLen);
      WriteLn('repCnt . . . . . ',repCnt);
      WriteLn;
      WriteLn('dat = ',inData);
      IF datPtr > 0 THEN
        WriteLn('      ',CharStr(' ',Pred(datPtr)),'^');
      WriteLn('datPtr . . . . . ',datPtr);
      WriteLn('stat   . . . . . ',stat);

      key := ReadKey;
      CASE key OF
       '1' : Halt;
      END {CASE};
{}{}END {SnapShot};


{$ENDIF}


{}{}PROCEDURE GetUnsigned(VAR n:WORD);
{---------------------------------------------------------------------------
  Purpose     - Return an unsigned integer value.
---------------------------------------------------------------------------}
    BEGIN
      n := 0;
      WHILE (fmt[fmtPtr] IN ['0'..'9']) AND (fmtPtr <= fmtLen) DO BEGIN
        n := n*10 + (Ord(fmt[fmtPtr]) - $30);
        Inc(fmtPtr);
      END {WHILE};
      Dec(fmtPtr);
{}{}END {GetUnsigned};


{}{}PROCEDURE FixRep;
{---------------------------------------------------------------------------
  Purpose     - Make a repetition count at least 1.
---------------------------------------------------------------------------}
    BEGIN
      IF repCnt = 0 THEN
        repCnt := 1
{}{}END {FixRep};


{}{}PROCEDURE GetFloatFmt(VAR fld,dcml:WORD);
{---------------------------------------------------------------------------
  Purpose     - Interpret F field descriptors.
---------------------------------------------------------------------------}
    BEGIN
      fld  := 0;
      dcml := 0;

      GetUnsigned(fld);  Inc(fmtPtr);

      IF fmt[fmtPtr] <> '.' THEN BEGIN
        stat := fmtPtr;
        WriteLn('NOT "." :',fmt[fmtPtr]);
        Exit;
      END {IF};

      Inc(fmtPtr);
      GetUnsigned(dcml);
{}{}END {GetFloatFmt};


{}{}PROCEDURE GetIntFmt(VAR fld:WORD);
{---------------------------------------------------------------------------
  Purpose     - Interpret I field descriptors.
---------------------------------------------------------------------------}
    BEGIN
      fld  := 0;
      dcml := 0;

      GetUnsigned(fld);
{}{}END {GetIntFmt};


{}{}PROCEDURE ReadFloat(VAR r:REAL; fld,Dec:BYTE; VAR stat:BYTE);
{---------------------------------------------------------------------------
  Purpose     - Use format to convert value from incoming data
---------------------------------------------------------------------------}
    VAR
      i  : BYTE;
      t  : STRING;
      start   : BOOLEAN;
      Token   : CHAR;
      status  : INTEGER;
    BEGIN
      r     := 0;
      t     := '';
      i     := 1;
      start := TRUE;
      stat  := 0;
      status:= 0;

      WHILE (i <= fld) AND (datPtr+i <= datLen) DO BEGIN
        token := UpCase(inData[datPtr+i]);
        CASE token OF
        ' ' : {leading spaces are ignored, embedded spaces are treated as 0}
          IF NOT start THEN
            t := t + '0';

   '0'..'9',  {legal symbols include the numbers}
        '-',  {signs and decimals}
        '+',  {as well as exponents}
        '.',
        'E' :
          BEGIN
            t     := t + token;
            start := FALSE;
          END {BEGIN};

              {all other symbols are illegal}
          ELSE BEGIN
            FIO__error := datPtr+i;
            stat       := 2;
            datPtr := datPtr + fld;
            Exit;
          END {BEGIN};
        END {CASE};

        Inc(i);
      END {WHILE};

      datPtr := datPtr + fld;

      IF Length(t) = 0 THEN
        r := 0.0                    {treat blankfield as zero, per FORTRAN}
      ELSE
      IF Pos('.',t) > 0 THEN
        Val(t,r,status)             {explicit decimal}
      ELSE
      IF Pos('E',t) > 0 THEN BEGIN  {explicit exponent, no decimal}
        IF (t[1] = 'E') OR          {  check for proper format}
           ((t[1] IN ['+','-']) AND (t[2] = 'E')) THEN BEGIN
          status := 1;
          Exit;
        END {IF};

        i := Pos('E',t);            {  insert .0 for proper VAL operation}
        Insert('.0',t,i);
        Val(t,r,status);
      END ELSE BEGIN
        Insert('.',t,fld-Dec);      {  insert decimal}
        t := Copy(t,1,fld);         {  trim excess characters}
        Val(t,r,status);            {  convert to real}
      END {BEGIN};

      IF status <> 0 THEN BEGIN     {return error status}
        stat       := 2;
        FIO__error := datPtr-fld+status;
      END {IF};
{}{}END {ReadFloat};


{}{}PROCEDURE ReadInt(VAR Li:LongINT; fld:BYTE; VAR stat:BYTE);
{---------------------------------------------------------------------------
  Purpose     - Use format to convert value from incoming data
---------------------------------------------------------------------------}
    VAR
      i  : BYTE;
      t  : STRING;
      start   : BOOLEAN;
      Token   : CHAR;
      status  : INTEGER;
    BEGIN
      Li    := 0;
      t     := '';
      i     := 1;
      start := TRUE;
      stat  := 0;
      status:= 0;

      WHILE (i <= fld) AND (datPtr+i <= datLen) DO BEGIN
        token := UpCase(inData[datPtr+i]);
        CASE token OF
        ' ' : {leading spaces are ignored, embedded spaces are treated as 0}
          IF NOT start THEN
            t := t + '0';

   '0'..'9',  {legal symbols include the numbers and signs}
        '-',
        '+' :
          BEGIN
            t     := t + token;
            start := FALSE;
          END {BEGIN};

          ELSE BEGIN                {all other symbols are illegal}
            FIO__error := datPtr+i;
            stat       := 2;
            datPtr     := datPtr + fld;
            Exit;
          END {BEGIN};
        END {CASE};

        Inc(i);
      END {WHILE};

      datPtr := datPtr + fld;

      IF Length(t) = 0 THEN
        Li := 0                     {treat blankfield as zero, per FORTRAN}
      ELSE
        Val(t,Li,status);

      IF status <> 0 THEN BEGIN     {return error status}
        stat       := 2;
        FIO__error := datPtr-fld+status;
      END {IF};
{}{}END {ReadInt};


{}{}PROCEDURE ReadHollerith(VAR s:STRING; fld:BYTE; VAR stat:BYTE);
{---------------------------------------------------------------------------
  Purpose     - Use format to convert value from incoming data
---------------------------------------------------------------------------}
    VAR
      i  : WORD;
    BEGIN
      s := '';
      i := 1;
      stat := 0;
      WHILE (i <= fld) AND (datPtr+i <= datLen) DO BEGIN
        s := s + inData[datPtr+i];
        Inc(i);
      END {WHILE};
      datPtr := datPtr + fld;
{}{}END {ReadHollerith};


{}{}PROCEDURE ReadLogical(VAR L:BOOLEAN; fld:BYTE; VAR stat:BYTE);
{---------------------------------------------------------------------------
  Purpose     - Use format to convert value from incoming data
---------------------------------------------------------------------------}
    VAR
      i  : WORD;
      s  : STRING;
      token   : CHAR;
      start   : BOOLEAN;
    BEGIN
      s := '';
      i := 1;
      start := TRUE;
      stat  := 0;

      WHILE (i <= fld) AND (datPtr+i <= datLen) DO BEGIN
        token := UpCase(inData[datPtr+i]);
        CASE token OF
         ' ' : {ignore leading blanks}
          IF NOT start THEN
            i := fld+1;

         '.' : {ignore 1st leading period}
          IF start THEN
            start := FALSE
          ELSE
            i := fld+1;

          ELSE BEGIN                {build a string for test}
            start := FALSE;
            s := s + token;
          END {IF};
        END {CASE};
        Inc(i);
      END {WHILE};

      WriteLn('ReadLog:',s);
      IF s[1] = 'T' THEN
        L := TRUE
      ELSE
      IF s[1] = 'F' THEN
        L := FALSE
      ELSE
        stat := 2;

      datPtr := datPtr + fld;
{}{}END {ReadLogical};


{}{}PROCEDURE Skip;
    BEGIN
      Inc(datPtr,repCnt);
      repCnt := 0;
{}{}END {Skip};


{}{}PROCEDURE OutPutFloat(r:REAL);
{---------------------------------------------------------------------------
  Purpose     - Store the real value R in OUTDATA.
  Remarks     - Increments dataPos to the next available data slot.
---------------------------------------------------------------------------}
    BEGIN
      IF dataPos+RealLen > dataSize THEN BEGIN
        stat := 3;
        Exit;
      END {IF};
      {$R-}Move(r,ByteArray(outData)[dataPos],RealLen);{$R+}
      Inc(dataPos,RealLen);
{}{}END {OutPutFloat};


{}{}PROCEDURE OutPutInteger(Li:LongINT);
{---------------------------------------------------------------------------
  Purpose     - Store the integer value LI in OUTDATA.
  Remarks     - Increments dataPos to the next available data slot.
---------------------------------------------------------------------------}
    BEGIN
      IF dataPos+IntLen > dataSize THEN BEGIN
        stat := 3;
        Exit;
      END {IF};
      {$R-}Move(Li,ByteArray(outData)[dataPos],IntLen);{$R+}
      Inc(dataPos,IntLen);
{}{}END {OutPutInteger};


{}{}PROCEDURE OutPutLogical(L:BOOLEAN);
{---------------------------------------------------------------------------
  Purpose     - Store the boolean value LI in OUTDATA.
  Remarks     - Increments dataPos to the next available data slot.
---------------------------------------------------------------------------}
    BEGIN
      IF dataPos+LogLen > dataSize THEN BEGIN
        stat := 3;
        Exit;
      END {IF};
      {$R-}Move(L,ByteArray(outData)[dataPos],LogLen);{$R+}
      Inc(dataPos,LogLen);
{}{}END {OutPutLogical};


{}{}PROCEDURE OutPutString(s:STRING;n:BYTE);
{---------------------------------------------------------------------------
  Purpose     - Store the string value S in OUTDATA.
  Remarks     - Increments dataPos to the next available data slot.
---------------------------------------------------------------------------}
    BEGIN
      Inc(n);   { account for string 0 byte }
      IF dataPos+n > dataSize THEN BEGIN
        stat := 3;
        Exit;
      END {IF};
      {$R-}Move(s[0],ByteArray(outData)[dataPos],n);{$R+}
      Inc(dataPos,n);
{}{}END {OutPutString};


  BEGIN
    fmtPtr := 1;
    datPtr := 0;
    fmtLen := Length(fmt);
    datLen := Length(inData);
    repCnt := 0;
    stat   := 0;

    FillChar(outData,dataSize,0);
    dataPos  := 0;

    REPEAT
      charToken := UpCase(fmt[fmtPtr]);

{$IFDEF debug} Write('"',charToken,'" '); {$ENDIF}

      CASE charToken OF
        ' ' : ; {ignore spaces}

        ',' :   {commas}
        repCnt := 0;

        'A' :   {character strings}
        BEGIN
          FixRep;                     {repetition count at least 1}
          Inc(fmtPtr);                {bypass descriptor field}
          GetUnsigned(n);             {get size of character string}
          IF (n>255) THEN BEGIN       {->return err if too big}
            stat := 1;
            FIO__error := fmtPtr;
            Exit;
          END {IF};
          WHILE repCnt > 0 DO BEGIN   {Read REPCNT strings of length n}
            ReadHollerith(s,n,stat);

            {$IFDEF debug}
            WriteLn('STRING: ',s,'  fld:',n,'  stat:',stat);
            {$ENDIF}

            IF stat <> 0 THEN
              Exit;                   {check for errors}
            OutPutString(s,n);        {->pack string into output record}
            Dec(repCnt);              {Decrement REPCNT}
          END {WHILE};
        END {BEGIN};

        'E',  {exponential format}
        'F' : {floating point values}
        BEGIN
          FixRep;

          {$IFDEF debug}
          WriteLn('>F< ');
          SnapShot;
          {$ENDIF}

          Inc(fmtPtr);
          GetFloatFmt(fld,dcml);

          WHILE repCnt > 0 DO BEGIN
            ReadFloat(r,fld,dcml,stat);

            {$IFDEF debug}
            WriteLn('REAL  : ',r:fld:dcml,'  fld:dcml = ',fld,':',dcml,'  stat:',stat);
            {$ENDIF}

            IF stat <> 0 THEN
              Exit;
            OutPutFloat(r);
            Dec(repCnt);
          END {WHILE};
        END {BEGIN};

        'I' : {integer values}
        BEGIN
          FixRep;

          {$IFDEF debug}
          WriteLn('>I<');
          SnapShot;
          {$ENDIF}

          Inc(fmtPtr);
          GetIntFmt(fld);
          WHILE repCnt > 0 DO BEGIN
            ReadInt(Li,fld,stat);

            {$IFDEF debug}
            WriteLn(repCnt:2,' INTEGR:',Li:fld,' fld = ',fld,'  stat:',stat);
            {$ENDIF}

            IF stat <> 0 THEN
              Exit;
            OutPutInteger(Li);
            Dec(repCnt);
          END {WHILE};
        END {BEGIN};

        'L' : {logicals = booleans}
        BEGIN
          FixRep;

          {$IFDEF debug}
          WriteLn('>L<');
          SnapShot;
          {$ENDIF}

          Inc(fmtPtr);
          GetUnsigned(n);
          IF (n<1) OR (n>7) THEN BEGIN
            stat := 1;
            FIO__error := fmtPtr;
            Exit;
          END {IF};
          WHILE repCnt > 0 DO BEGIN
            ReadLogical(L,n,stat);

            {$IFDEF debug}
            WriteLn('LOGICL: ',L,'  fld:',n,'  stat:',stat);
            {$ENDIF}

            IF stat <> 0 THEN
              Exit;
            OutPutLogical(L);
            Dec(repCnt);
          END {WHILE};
        END {BEGIN};

        'T' : {tabs}
        BEGIN
          {$IFDEF debug}
          WriteLn('>T<');
          {$ENDIF}

          Inc(fmtPtr);
          GetUnsigned(n);
          IF n > 0 THEN
            datPtr := Pred(n)
          ELSE BEGIN
            stat := 1;
            FIO__error := fmtPtr;
            Exit;
          END {IF};

          {$IFDEF debug}
          WriteLn('Tab resets data Ptr to ',datPtr);
          {$ENDIF}
        END {BEGIN};

        'X' : {spacers}
        BEGIN
          {$IFDEF debug}
          WriteLn('>X< ');
          {$ENDIF}

          FixRep;
          Skip;
        END {BEGIN};

   '0'..'9' : {repetition count}
        repCnt := repCnt*10 + (Ord(charToken) - $30);

        ELSE BEGIN
          stat       := 1;
          FIO__error := fmtPtr;
          Exit;
        END {BEGIN};
      END {CASE};

      IF stat <> 0 THEN
        Exit;
      Inc(fmtPtr);
    UNTIL (fmtPtr > fmtLen);
{}END {FormatRead};




END {UNIT}.
