{$N+,E+}
UNIT Primitives;   {Graphics Primitives}

  {Copyright (C) 1985, 1992 by Earl F. Glynn, Overland Park, KS.
   All Rights Reserved.}


INTERFACE

  USES
    GRAPH,
    GraphFix,
    MathLibrary;

  TYPE
    ProjectionType = (orthoXY,orthoXZ,orthoYZ,perspective);
    MoveType       = (MoveAbsolute,MoveRelative);

    GraphicsWindow =
      OBJECT
        uSaveW     :  vector; {last vector -- world coordinates}
        uSaveX     :  vector; {last vector -- transformed}
        xEast      :  DOUBLE; {Window parameters}
        xWest      :  DOUBLE;
        yNorth     :  DOUBLE;
        ySouth     :  DOUBLE;
        iEast      :  INTEGER;{ViewPort parameters}
        iWest      :  INTEGER;
        jSouth     :  INTEGER;
        jNorth     :  INTEGER;
        xPixel     :  DOUBLE; {pixels per world unit horizontally}
        yPixel     :  DOUBLE; {pixels per world unit vertically}
        xform2D    :  matrix; {default 2D transformation matrix}
        xform3D    :  matrix; {default 3D transformation matrix}
        vcx,vcy    :  DOUBLE; {3D projection center-size viewport parms}
        vsx,vsy    :  DOUBLE;
        projection :  ProjectionType;
        ClipFlag   :  BOOLEAN;
        AbsRel     :  MoveType;
        coord      :  coordinates;
        tag        :  STRING[7];

        PROCEDURE Init;
                                       { LineTo / MoveTo / PointAt }
        PROCEDURE AbsoluteMove;
        PROCEDURE RelativeMove;
        PROCEDURE SetCurrent;

        PROCEDURE LineTo  (u:  vector);
        PROCEDURE MoveTo  (u:  vector);
        PROCEDURE PointAt (u:  vector);

        PROCEDURE LineTo2D  (x,y:    DOUBLE);
        PROCEDURE MoveTo2D  (x,y:    DOUBLE);
        PROCEDURE PointAt2D (x,y:    DOUBLE);

        PROCEDURE LineTo3D  (x,y,z:  DOUBLE);
        PROCEDURE MoveTo3D  (x,y,z:  DOUBLE);
        PROCEDURE PointAt3D (x,y,z:  DOUBLE);

                                       { Clipping }
        PROCEDURE Clip (VAR u1,u2:  vector; VAR visible:  BOOLEAN);
        PROCEDURE SetClipping (flag: BOOLEAN);

                                       { 3D-to-2D Projection }
        PROCEDURE Project (u:  vector; VAR v:  vector);
        PROCEDURE SetProjectionType (PrjType:  ProjectionType);

                                       { Default Transformation Matrices}
        PROCEDURE ClearTransform  (d:  dimension);
        PROCEDURE GetTransform    (d:  dimension; VAR a:  matrix);
        PROCEDURE SetTransform    (a:  matrix);
        PROCEDURE VectorTransform (u:  vector; VAR v:  vector);

                                       { Windows / Viewports }
        PROCEDURE WorldCoordinatesRange (xMin,xMax,yMin,yMax:  DOUBLE);
        PROCEDURE ViewPort (xMin,xMax,yMin,yMax:  DOUBLE);
        PROCEDURE ShowOutline;

                                       {World-to-Pixel Coordinate Conversion}
        PROCEDURE WorldToPixel (u:  vector; VAR i,j:  INTEGER);

                                       {Set coordinate type}
        PROCEDURE SetCoordType (c:  coordinates);

      END;


IMPLEMENTATION


  PROCEDURE GraphicsWindow.Init;
  BEGIN
    SetClipping (TRUE);
    AbsoluteMove;
    ClearTransform (TwoD);
    ClearTransform (ThreeD);
    SetProjectionType (perspective);
    coord := cartesian;

    xWest  := 0.0;
    xEast  := 1.0;
    ySouth := 0.0;
    yNorth := 1.0;
    vcx := 0.5;
    vcy := 0.5;
    vsx := 0.5;
    vsy := 0.5;

    ViewPort (0.0,1.0, 0.0,1.0);
    Vector3D (0,0,0, uSaveW)
  END {Init};


{  LineTo / MoveTo / PointAt    }

  PROCEDURE GraphicsWindow.AbsoluteMove;
  BEGIN
    AbsRel := MoveAbsolute
  END {AbsoluteMove};


  PROCEDURE GraphicsWindow.RelativeMove;
  BEGIN
    AbsRel := MoveRelative
  END {RelativeMove};


  PROCEDURE GraphicsWindow.SetCurrent;
  BEGIN
    MoveTo (uSaveW)
  END {RelativeMove};


  PROCEDURE GraphicsWindow.LineTo (u:  vector);

    {"LineTo" draws a straight line from the current screen position to
    a new point and resets the current screen position.  Transformation
    of the point can automatically occur (see note for "MoveTo" above).
    Pixels traced over by the line segement are automatically selected.
    Clipping of the line segment to the view boundary can also automatically
    occur.}

    VAR
      flag     :  BOOLEAN;
      i        :  INTEGER;
      j        :  INTEGER;
      uNew,uOld:  vector;
      visible  :  BOOLEAN;

  BEGIN
    IF   AbsRel = MoveRelative
    THEN VectorAdd (uSaveW,u, u);
    uSaveW := u; {'uSaveW' needed by next relative MoveTo/LineTo/PointAt}
    ToCartesian (coord,u);
    VectorTransform (u,u);
    IF   ClipFlag
    THEN BEGIN
      uOld := uSaveX;
      uNew := u;
      clip (uOld,uNew, visible);
      IF   visible
      THEN BEGIN
        flag := (uOld.x <> uSaveX.x)  OR  (uOld.y <> uSaveX.y);
        IF   uOld.Size = 4
        THEN flag := flag OR (uOld.z <> uSaveX.z);
        IF   flag
        THEN BEGIN
          WorldToPixel (uOld, i,j);
          MoveToPixel (i,j)
        END;
        WorldToPixel (uNew, i,j);
        LineToPixel (i,j)
      END
    END
    ELSE BEGIN
      WorldToPixel (u, i,j);
      LineToPixel (i,j)
    END;
    uSaveX := u
  END {LineTo};


  PROCEDURE GraphicsWindow.MoveTo (u:  vector);

    {"MoveTo" sets the current screen position.  This position is defined
    in user-defined world units.  Transformation of the point automatically
    occurs if a transformation matrix was defined for the dimensionality
    ("TwoD" or "ThreeD") of the point.}

    VAR
      i,j:  INTEGER;

  BEGIN
    IF   AbsRel = MoveRelative
    THEN VectorAdd (uSaveW,u, u);
    uSaveW := u; {'uSaveW' needed by next relative MoveTo/LineTo/PointAt}
    ToCartesian (coord,u);
    VectorTransform (u,uSaveX);
    WorldToPixel (uSaveX, i,j);
    IF   ClipFlag
    THEN BEGIN
      IF   (i >= iWest) AND (i <= iEast) AND (j >= jSouth) AND (j <= jNorth)
      THEN MoveToPixel (i,j)
    END
    ELSE MoveToPixel (i,j)
  END {MoveTo};


  PROCEDURE GraphicsWindow.PointAt (u:  vector);

    { A 'point' could be considered an extremely 'short' line segment.}

    VAR
      i,j:  INTEGER;

  BEGIN
    IF   AbsRel = MoveRelative
    THEN VectorAdd (uSaveW,u, u);
    uSaveW := u; {'uSaveW' needed by next relative MoveTo/LineTo/PointAt}
    ToCartesian (coord,u);
    VectorTransform (u,uSaveX);
    WorldToPixel (uSaveX, i,j);
    IF   ClipFlag
    THEN BEGIN
      IF   (i >= iWest) AND (i <= iEast) AND (j >= jSouth) AND (j <= jNorth)
      THEN PointAtPixel (i,j)
    END
    ELSE PointAtPixel (i,j)
  END {PointAt};


  PROCEDURE GraphicsWindow.LineTo2D  (x,y:  DOUBLE);
    VAR
      u:  vector;
  BEGIN
    Vector2D (x,y, u);
    LineTo (u)
  END {LineTo2D};


  PROCEDURE GraphicsWindow.MoveTo2D  (x,y:  DOUBLE);
    VAR
      u:  vector;
  BEGIN
    Vector2D (x,y, u);
    MoveTo (u)
  END {MoveTo2D};


  PROCEDURE GraphicsWindow.PointAt2D (x,y:  DOUBLE);
    VAR
      u:  vector;
  BEGIN
    Vector2D (x,y, u);
    PointAt (u)
  END {PointAt2D};


  PROCEDURE GraphicsWindow.LineTo3D (x,y,z:  DOUBLE);
    VAR
      u:  vector;
  BEGIN
    Vector3D (x,y,z, u);
    LineTo (u)
  END {LineTo3D};


  PROCEDURE GraphicsWindow.MoveTo3D (x,y,z:  DOUBLE);
    VAR
      u:  vector;
  BEGIN
    Vector3D (x,y,z, u);
    MoveTo (u)
  END {MoveTo3D};


  PROCEDURE GraphicsWindow.PointAt3D (x,y,z:  DOUBLE);
    VAR
      u:  vector;
  BEGIN
    Vector3D (x,y,z, u);
    PointAt (u)
  END {PointAt3D};


{    C l i p p i n g   }


  PROCEDURE GraphicsWindow.Clip (VAR u1,u2:  vector; VAR visible:  BOOLEAN);

    {"Clip" integrates both two- and three-dimensional clipping into a
    single procedure.  The input parameters are of the type "vector" and
    internally have their dimensionality defined.  These clipping algorithms
    were adapted from Newman and Sproull, "Principles of Interactive
    Computer Graphics", Second Edition, pp. 66-67 for 2D and p. 345 for 3D.
    The Newman and Sproull internal "code" procedure was replaced by a
    "region" procedure for clarity.  For example, the code '1001' is
    replaced by a set of Regions [North West]. See diagram below.  Also,
    see pp. 144-148 and 295-297 of "Fundamentals of Interactive Computer
    Graphics" by Foley and Van Dam.}

    TYPE
      Regions   = (North,South,East,West);
      RegionSet = SET OF Regions;

    VAR
      deltaX,deltaY:  DOUBLE;
      reg,reg1,reg2:  RegionSet;
      u            :  vector;

    PROCEDURE Region (u:  vector; VAR reg: RegionSet);
      {This procedure is internal to "clip".  "Region" defines the regions
      a given point is in.  If the set of regions is the null set [], then
      the point is in the viewing area.  If the union of regions from both
      points is the empty set, the segment is entirely visible. If the
      intersection of regions from two different points is not the empty
      set, the segment must lie entirely off the screen.}
    BEGIN
      reg := [];
      CASE u.size OF                {Picture and surrounding regions:}
        3:  BEGIN
          IF   u.x < xWest          {[North West]   [North]   [North East]}
          THEN reg := [West]        {}
          ELSE                      {    [West]       []        [East]    }
            IF  u.x > xEast         {}
            THEN reg := [East];     {[South West]   [South]   [South East]}
          IF   u.y < ySouth
          THEN reg := reg + [South]
          ELSE
            IF  u.y > yNorth
            THEN reg := reg + [North]
        END {3};
        4:  BEGIN
          IF   u.x < -u.z
          THEN reg := [West]
          ELSE
            IF  u.x > u.z
            THEN reg := [East];
            IF   u.y < -u.z
            THEN reg := reg + [South]
            ELSE
              IF  u.y > u.z
              THEN reg := reg + [North]
        END {4}
      END {CASE}
    END {Region};

    PROCEDURE Clip2D;

      {This code is used only for clipping two-dimensional vectors.}

      VAR
        deltaX, deltaY:  DOUBLE;

    BEGIN
      deltaX := u2.x-u1.x;             { x2 - x1 }
      deltaY := u2.y-u1.y;             { y2 - y1 }
      IF   West IN reg               {crosses West edge}
      THEN Vector2D (xWest, u1.y + deltaY*(xWest-u1.x)/deltaX, u)
      ELSE
        IF   East IN reg             {crosses East edge}
        THEN Vector2D (xEast, u1.y + deltaY*(xEast-u1.x)/deltaX, u)
        ELSE
          IF   South IN reg          {crosses South edge}
          THEN Vector2D (u1.x + deltaX*(ySouth-u1.y)/deltaY, ySouth, u)
          ELSE
            IF   North IN reg        {crosses North edge}
            THEN Vector2D (u1.x + deltaX*(yNorth-u1.y)/deltaY, yNorth, u);
    END {Clip2D};

    PROCEDURE Clip3D;

      {This code is used only for clipping three-dimensional vectors.}

      VAR
        t,z:  DOUBLE;

    BEGIN
      IF   West IN reg                 {crosses West edge}
      THEN BEGIN
        t := (u1.z+u1.x) / ((u1.x-u2.x)-(u2.z-u1.z));
        z := t*(u2.z-u1.z)+u1.z;
        Vector3D (-z, t*(u2.y-u1.y)+u1.y, z, u)
      END
      ELSE
        IF   East IN reg               {crosses East edge}
        THEN BEGIN
          t := (u1.z-u1.x) / ((u2.x-u1.x)-(u2.z-u1.z));
          z := t*(u2.z-u1.z)+u1.z;
          Vector3D (z, t*(u2.y-u1.y)+u1.y, z, u)
        END
        ELSE
          IF  South IN reg             {crosses South edge}
          THEN BEGIN
            t := (u1.z+u1.y) / ((u1.y-u2.y)-(u2.z-u1.z));
            z := t*(u2.z-u1.z)+u1.z;
            Vector3D (t*(u2.x-u1.x)+u1.x, -z, z, u)
          END
          ELSE
            IF  North IN reg           {crosses North edge}
            THEN BEGIN
              t := (u1.z-u1.y) / ((u2.y-u1.y)-(u2.z-u1.z));
              z := t*(u2.z-u1.z)+u1.z;
              Vector3D (t*(u2.x-u1.x)+u1.x, z, z, u)
            END
    END {Clip3D};

  BEGIN {clip}
    Region (u1,reg1);
    Region (u2,reg2);
    visible := reg1*reg2 = [];
    WHILE ((reg1 <> []) OR (reg2 <> [])) AND visible DO BEGIN
      reg := reg1;
      IF   reg = []
      THEN reg := reg2;
      CASE u1.size OF
        3:  Clip2D;
        4:  Clip3D
      END;
      IF   reg = reg1
      THEN BEGIN
        u1 := u;
        Region (u,reg1)
      END
      ELSE BEGIN
        u2 := u;
        Region (u,reg2)
      END;
      visible :=  reg1*reg2 = []
    END {WHILE}
  END {clip};


  PROCEDURE GraphicsWindow.SetClipping (flag:  BOOLEAN);
    {This procedure sets the clipping flag to a specified value.  When TRUE
    is specified, the "Clip" procedure will be used following any "LineTo"
    calls.}
  BEGIN
    ClipFlag := flag
  END {SetClipping};


{    3D-to-2D Projection   }

  PROCEDURE GraphicsWindow.Project (u:  vector; VAR v:  vector);
    {A three-dimensional vector is projected into two dimensions with this
    procedure. Orthograpic or perspective projections can be specified with
    the "SetProjectionType" procedure.}
  BEGIN
    IF  u.size = 4
    THEN BEGIN
      CASE projection OF
        orthoXY:
          BEGIN
            v.x := u.x;
            v.y := u.y
          END;
        orthoXZ:
          BEGIN
            v.x := u.x;
            v.y := u.z
          END;
        orthoYZ:
          BEGIN
            v.x := u.y;
            v.y := u.z
          END;
        perspective:
          BEGIN
            IF   defuzz(u.z) = 0.0
            THEN BEGIN       {avoid division by zero}
              v.x := vcx;    {visible viewing screen is only a point}
              v.y := vcy
            END
            ELSE BEGIN
              v.x := vcx + vsx*u.x/u.z;
              v.y := vcy + vsy*u.y/u.z
            END
          END
      END {CASE};
      v.size := 3;           {now a 2D vector}
      v.z := 1.0             {last component of 2D homogenous coordinate}
    END
    ELSE
      v.size := 1              {not a 3D vector}
  END {Project};


  PROCEDURE GraphicsWindow.SetProjectionType (PrjType:  ProjectionType);
  BEGIN
    projection := PrjType
  END {SetProjectionType};


{    Default Transformation Matrices   }

  PROCEDURE GraphicsWindow.ClearTransform (d:  dimension);
    {"ClearTransform" sets the "size" of the "TwoD" or "ThreeD"
    transformation matrix in the parameter prefix to 1.  When the "Size" of the
    matrix is 1, it is not automatically used when "MoveTo" or "LineTo" is
    invoked.  "ClearTransform" removes the effect of the "SetTransform"
    matrix.}
  BEGIN
    CASE d OF
      TwoD  :  xform2D.size := 1;
      ThreeD:  xform3D.size := 1
    END
  END {ClearTransform};


  PROCEDURE GraphicsWindow.GetTransform (d:  dimension; VAR a:  matrix);
    {The default "TwoD" or "ThreeD" transformation matrix can be retreived
    using "GetTransform" for inspection or further modification using other
    matrix operations.}
  BEGIN
    CASE d OF
      TwoD  :  a := xform2D;
      ThreeD:  a := xform3D
    END
  END  {GetTransform};


  PROCEDURE GraphicsWindow.SetTransform (a:  matrix);
    {"SetTransform" establishes a default "TwoD" or "ThreeD" transformation
    matrix which is used every time "MoveTo" or "LineTo" is invoked.  The
    transformation matrix can be changed any number of times.  "ClearTransform"
    removes the effect of the matrix.  Alternately, an identity transformation
    matrix could be specified.}
  BEGIN
    CASE a.size OF
      3:  xform2D := a;
      4:  xform3D := a
    END
  END {SetTransfrom};


  PROCEDURE GraphicsWindow.VectorTransform  (u:  vector; VAR v:  vector);
    {"VectorTransform" is used by "MoveTo" and "LineTo" to automatically
    multiply a "vector" by a default transformation matrix if one has
    been defined.}
  BEGIN
    CASE u.size OF
      1:  v.size := 1; {undefined indicator}
      3:  IF   xform2D.size = 3
          THEN Transform (u,xform2D, v)
          ELSE v := u;
      4:  IF   xform3D.size = 4
          THEN Transform (u,xform3D, v)
          ELSE v := u
    END {CASE}
  END {VectorTransform};


{    Windows / Viewports   }


  PROCEDURE vwParmError (title:  STRING; xMin,xMax,yMin,yMax:  DOUBLE);
    {This procedure reports definition errors in setting the "View" or
    "Window".}
  BEGIN
    WRITELN (title);
    WRITELN ('    xMin =',xMin:7:2,', xMax =',xMax:7:2,
      ', yMin =',yMin:7:2,', yMax =',yMax:7:2);
  END {vwParmError};


  PROCEDURE GraphicsWindow.WorldCoordinatesRange (xMin,xMax,yMin,yMax:  DOUBLE);
  BEGIN
    IF  (xMax <= xMin) OR (yMax <= yMin)
    THEN BEGIN
      vwParmError ('Window parameter error(s):',xMin,xMax,yMin,yMax);
      EXIT
    END;

    xWest  := xMin;
    xEast  := xMax;
    ySouth := yMin;
    yNorth := yMax;
    vcx := 0.5 * (xEast  + xWest ); {3D projection parameters}
    vcy := 0.5 * (yNorth + ySouth);
    vsx := 0.5 * (xEast  - xWest );
    vsy := 0.5 * (yNorth - ySouth);

    xPixel := (iEast - iWest)   / (xEast - xWest);
    yPixel := (jNorth - jSouth) / (yNorth - ySouth);
    Vector2D (xWest,ySouth, uSaveX);
    MoveToPixel (iWest,jSouth);
    AbsoluteMove

  END {Window};


  PROCEDURE GraphicsWindow.ViewPort (xMin,xMax,yMin,yMax:  DOUBLE);
  BEGIN
    IF  (xMin < 0.0)  OR  (xMax > 1.0)  OR  (yMin < 0.0)  OR
        (yMax > 1.0)  OR  (xMax <= xMin) OR (yMax <= yMin)
    THEN BEGIN
      vwParmError ('View parameter error(s):',xMin,xMax,yMin,yMax);
      EXIT
    END;

    iWest  := ROUND(xMin * GetMaxX);
    iEast  := ROUND(xMax * GetMaxX);
    jSouth := ROUND(yMin * GetMaxY);
    jNorth := ROUND(yMax * GetMaxY);

    xPixel := (iEast - iWest)   / (xEast - xWest);
    yPixel := (jNorth - jSouth) / (yNorth - ySouth);
    Vector2D (xWest,ySouth, uSaveX);
    MoveToPixel (iWest,jSouth);
    AbsoluteMove

  END {ViewPort};


  PROCEDURE GraphicsWindow.ShowOutline;
  BEGIN
    MoveToPixel (iWest,jSouth);
    LineToPixel (iEast,jSouth);
    LineToPixel (iEast,jNorth);
    LineToPixel (iWest,jNorth);
    LineToPixel (iWest,jSouth)
  END {ShowOutline};


{   World-to-Pixel Coordinate Conversion   }

  PROCEDURE GraphicsWindow.WorldToPixel (u: vector; VAR i,j:  INTEGER);

    {This procedure converts a 2D/3D 'vector' into pixel indices.  A three-
    dimensional vector is projected into two dimensions automatically.
    The user-defined world coordinates are used in specifying the 'vector'.
    The pixel indices are determined by the definition of the Viewport and
    Window.}

    VAR temp:  DOUBLE;
  BEGIN
    IF   u.size = 4
    THEN Project (u,u);
    IF   u.x < xWest
    THEN i := iWest-1
    ELSE BEGIN
      temp :=  xPixel * (u.x - xWest);
      i := iWest  + ROUND(temp)
    END;
    IF   u.y < ySouth
    THEN j := jSouth - 1
    ELSE BEGIN
      temp :=  yPixel * (u.y - ySouth);
      j := jSouth + ROUND(temp)
    END
  END {WorldToPixel};


{  Set Type of Coordinates  }

  PROCEDURE GraphicsWindow.SetCoordType (c:  coordinates);
  BEGIN
    IF   coord <> c
    THEN BEGIN
      ToCartesian   (coord,uSaveW);
      FromCartesian (c,    uSaveW);
      coord := c
    END
  END {SetCoordType};


END {Primitives UNIT}.
