{$N+,E+}
UNIT MathLibrary;  {Matrix/Vector Operations}

  {Copyright (C) 1985, 1992 by Earl F. Glynn, Overland Park, KS}


INTERFACE

  TYPE
    axis        = (Xaxis,Yaxis,Zaxis);
    coordinates = (cartesian,spherical,cylindrical);
    dimension   = (TwoD,ThreeD);  {two- or three-dimensional TYPE}
    index       = 1..4;           {index of 'matrix' and 'vector' TYPEs}
    matrix =                      {transformation 'matrix'}
      RECORD
        size:  index;
        mtrx:  ARRAY[index,index] OF DOUBLE
      END;
    rotation = (cw,ccw);          {cw = clockwise, ccw = counterclockwise}
    vector =                      {'vector' TYPE used to define points}
      RECORD
        size : index;
        CASE INTEGER OF
          0:  (vctr:  ARRAY[index] OF DOUBLE);
          1:  (x:  DOUBLE;
               y:  DOUBLE;
               z:  DOUBLE;
               h:  DOUBLE)
        END;

  VAR
    fuzz : DOUBLE;
                                                {Vector Operations}
  PROCEDURE Vector2D  (x,y:  DOUBLE; VAR u:  vector);
  PROCEDURE Vector3D  (x,y,z:  DOUBLE; VAR u:  vector);
  PROCEDURE Transform (u:  vector; a:  matrix; VAR v:  vector);
  PROCEDURE VectorAdd (u,v:  vector; VAR w: vector);

                                                {Basic Matrix Operations}
  PROCEDURE Matrix2D (m11,m12,m13, m21,m22,m23, m31,m32,m33:  DOUBLE;
                      VAR a:  matrix);
  PROCEDURE Matrix3D (m11,m12,m13,m14, m21,m22,m23,m24,
                      m31,m32,m33,m34, m41,m42,m43,m44:  DOUBLE;
                      VAR a:  matrix);
  PROCEDURE MatrixMultiply (a,b:  matrix; VAR c:  matrix);

                                                {Transformation Matrices}
  PROCEDURE RotateMatrix    (d:  dimension; xyz:  axis; angle: DOUBLE;
        direction:  rotation; VAR a:  matrix);
  PROCEDURE ScaleMatrix     (u:  vector; VAR a:  matrix);
  PROCEDURE TranslateMatrix (u:  vector; VAR a:  matrix);

                                                {miscellaneous}
  FUNCTION Defuzz(x:  DOUBLE):  REAL;
  FUNCTION Radians(angle {degrees}:  DOUBLE):  REAL {radians};
  PROCEDURE FromCartesian (coord:  coordinates; VAR u:  vector);
  PROCEDURE ToCartesian (coord:  coordinates; VAR u:  vector);

  PROCEDURE PrintVector (VAR out:  TEXT;
    indent,w,d:  INTEGER; title:  STRING; u:  vector);
  PROCEDURE PrintMatrix    (VAR out:  TEXT;
       indent,w,d:  INTEGER; title:  STRING; a:  matrix);

  PROCEDURE MatrixInverse  (a: matrix; VAR b: matrix; VAR determinant:  DOUBLE);
  PROCEDURE ViewTransformMatrix (viewtype:  coordinates;
       azimuth {or x}, elevation {or y}, distance {or z}:  DOUBLE;
       ScreenX, ScreenY, ScreenDistance:  DOUBLE; VAR a:  matrix);


IMPLEMENTATION

{ * * * * * * * * * * * * *  Vector Operations * * * * * * * * * * * * * }

  PROCEDURE Vector2D (x,y:  DOUBLE; VAR u:  vector);
    {This procedure defines two-dimensional homogeneous coordinates (x,y,1)
    as a single 'vector' data element 'u'.  The 'size' of a two-dimensional
    homogenous vector is 3.}
  BEGIN
    u.x := x;  u.y := y;  u.z := 1.0;
    u.size := 3
  END {Vector2D};


  PROCEDURE Vector3D (x,y,z:  DOUBLE; VAR u:  vector);
    {This procedure defines three-dimensional homogeneous coordinates
    (x,y,z,1) as a single 'vector' data element 'u'.  The 'size' of a
    three-dimensional homogenous vector is 4.}
  BEGIN
    u.x := x;  u.y := y;  u.z := z;  u.h := 1.0;
    u.size := 4
  END {Vector3D};


  PROCEDURE Transform (u:  vector; a: matrix; VAR v: vector);
    {'Transform' multiplies a row 'vector' by a transformation 'matrix'
    resulting in a new row 'vector'.  The 'size' of the 'vector' and 'matrix'
    must agree (if not, the transformed vector is given a 'size' of 1 but none
    of the components are defined).  To save execution time, the vectors are
    assumed to contain a homogeneous coordinate of 1.}
  VAR
    i,k :  index;
    temp:  DOUBLE;
  BEGIN
    v.size := a.size;
    IF  a.size = u.size
    THEN BEGIN
      FOR i := 1 TO a.size-1 DO BEGIN
        temp := 0;
        FOR k := 1 TO a.size DO
          temp := temp + u.vctr[k]*a.mtrx[k,i];
          v.vctr[i] := Defuzz(temp)
        END;
        v.vctr[a.size] := 1 {assume homogeneous coordinate}
      END
    ELSE BEGIN
      WRITELN ('MAT03: Ignoring attempt to multiply a vector of ',
        'dimension ',u.size,' by a square matrix of dimension ',a.size,'.');
      v.size := 1  {signal error by setting dimension of 'v' to 1}
    END
  END {Transform};


  PROCEDURE VectorAdd (u,v:  vector; VAR w: vector);
    {VectorAdd adds two vectors defined with homogenous coordinates.}
  VAR
    i: index;
  BEGIN
    IF  (u.size IN [3..4])  AND  (v.size IN [3..4])   {2D or 3D vector}
    THEN BEGIN
      IF   u.size = v.size
      THEN BEGIN                               { u + v  = w}
        FOR i := 1 TO u.size-1 DO              {2D + 2D = 2D  or  3D + 3D = 3D}
          w.vctr[i] := u.vctr[i] + v.vctr[i];
          w.vctr[u.size] := 1.0;               {homogeneous coordinate}
          w.size := u.size
        END
      ELSE BEGIN
        FOR i := 1 TO 2 DO
          w.vctr[i] := u.vctr[i] + v.vctr[i];
          IF   u.size = 3
          THEN w.z := v.z                      {2D + 3D = 3D}
          ELSE w.z := u.z;                     {3D + 2D = 3D}
          w.h := 1.0;
          w.size := 4
        END
      END
    ELSE
      w.size :=1;                              {invalid vector indicator}
  END {VectorAdd};


{ * * * * * * * * * * * * Basic Matrix Operations * * * * * * * * * * * * }

  PROCEDURE Matrix2D (m11,m12,m13, m21,m22,m23, m31,m32,m33:  DOUBLE;
                      VAR a:  matrix);
  BEGIN
    WITH a DO BEGIN
      mtrx[1,1] := m11; mtrx[1,2] := m12; mtrx[1,3] := m13;
      mtrx[2,1] := m21; mtrx[2,2] := m22; mtrx[2,3] := m23;
      mtrx[3,1] := m31; mtrx[3,2] := m32; mtrx[3,3] := m33;
      size := 3
    END
  END {Matrix2D};


  PROCEDURE Matrix3D (m11,m12,m13,m14, m21,m22,m23,m24,
                      m31,m32,m33,m34, m41,m42,m43,m44:  DOUBLE;
                      VAR a:  matrix);
  BEGIN
    WITH a DO BEGIN
      mtrx[1,1] := m11; mtrx[1,2] := m12; mtrx[1,3] := m13; mtrx[1,4] := m14;
      mtrx[2,1] := m21; mtrx[2,2] := m22; mtrx[2,3] := m23; mtrx[2,4] := m24;
      mtrx[3,1] := m31; mtrx[3,2] := m32; mtrx[3,3] := m33; mtrx[3,4] := m34;
      mtrx[4,1] := m41; mtrx[4,2] := m42; mtrx[4,3] := m43; mtrx[4,4] := m44;
      size := 4
    END
  END {Matrix3D};


  PROCEDURE MatrixMultiply (a,b:  matrix; VAR c:  matrix);

    {Compound geometric transformation matrices can be formed by multiplying
    simple transformation matrices.  This procedure only multiplies together
    matrices for two- or three-dimensional transformations, i.e., 3x3 or 4x4
    matrices.  The multiplier and multiplicand must be of the same dimension.}

    VAR
      i,j,k:  index;
      temp :  DOUBLE;

  BEGIN
    c.size := a.size;
    IF  a.size = b.size
    THEN
      FOR i := 1 TO a.size DO
        FOR j := 1 TO a.size DO BEGIN
          temp := 0;
          FOR k := 1 TO a.size DO
            temp := temp + a.mtrx[i,k]*b.mtrx[k,j];
            c.mtrx[i,j] := Defuzz(temp)
          END
    ELSE BEGIN
      WRITELN ('MAT01: Ignoring attempt to multiply square matrices of ',
        'different dimensions:  ',a.size, ' and ',b.size,'.');
      c.size := 1  {signal error by setting dimension of 'c' to 1}
    END
  END {MatrixMultiply};


{ * * * * * * * * * * * *  Transformation Matrices  * * * * * * * * * * * }

  PROCEDURE RotateMatrix (d:  dimension;  { TwoD or ThreeD }
      xyz:  axis;                         { Xaxis, Yaxis or Zaxis }
      angle :  DOUBLE;                      { radians }
      direction:  rotation;               { cw or ccw }
      VAR a:  matrix);

    {This procedure defines a matrix for a two- or three-dimensional rotation.
    To avoid possible confusion in the sense of the rotation, 'cw' for
    clockwise or 'ccw' for counter-clockwise must always be specified along
    with the axis of rotation. Two-dimensional rotations are assumed to
    be about the z-axis in the x-y plane.

    A rotation about an arbitrary axis can be performed with the following
    steps:
      (1) Translate the object into a new coordinate system where (x,y,z)
          maps into the origin (0,0,0).
      (2) Perform appropriate rotations about the x and y axes of the
          coordinate system so that the unit vector (a,b,c) is mapped into
          the unit vector along the z axis.
      (3) Perform the desired rotation about the z-axis of the new
          coordinate system.
      (4) Apply the inverse of step (2).
      (5) Apply the inverse of step (1).}

    VAR
      cosx:  DOUBLE;
      sinx:  DOUBLE;

  BEGIN
    IF  direction = ccw   {ccw is -cw}
    THEN angle := -angle;
    cosx := Defuzz( COS(angle) );
    sinx := Defuzz( SIN(angle) );
    CASE d OF
      TwoD:
        CASE xyz OF
          Xaxis,Yaxis:
            WRITELN ('MAT02:  FOR 2D rotation in x-y plane, ',
              'specify ''Zaxis''');
          Zaxis:  Matrix2D (cosx,-sinx,0, sinx,cosx,0, 0,0,1, a)
        END;
      ThreeD:
        CASE xyz OF
          Xaxis:  Matrix3D (1,0,0,0, 0,cosx,-sinx,0, 0,sinx,cosx,0, 0,0,0,1, a);
          Yaxis:  Matrix3D (cosx,0,sinx,0, 0,1,0,0, -sinx,0,cosx,0, 0,0,0,1, a);
          Zaxis:  Matrix3D (cosx,-sinx,0,0, sinx,cosx,0,0, 0,0,1,0, 0,0,0,1, a);
        END
    END
  END {RotateMatrix};


  PROCEDURE ScaleMatrix (u:  vector; VAR a:  matrix);
    {'ScaleMatrix' accepts a 'vector' containing the scaling factors for
    each of the dimensions and creates a scaling matrix.  The size
    of the vector dictates the size of the resulting matrix.}
  BEGIN
    CASE u.size OF
      3: Matrix2D (u.x,0,0,   0,u.y,0,      0,0,1,    a);
      4: Matrix3D (u.x,0,0,0, 0,u.y,0,0, 0,0,u.z,0, 0,0,0,1,  a)
    END
  END {ScaleMatrix};


  PROCEDURE TranslateMatrix (u:  vector; VAR a:  matrix);
    {'TranslateMatrix' defines a translation transformation matrix.  The
    components of the vector 'u' determine the translation components.}
  BEGIN
    CASE u.size OF
      3: Matrix2D (1,0,0,   0,1,0,       u.x,u.y,1,      a);
      4: Matrix3D (1,0,0,0, 0,1,0,0, 0,0,1,0, u.x,u.y,u.z,1,  a)
    END
  END {TranslateMatrix};


{ * * * * * * * * * * * * * *  Miscellaneous  * * * * * * * * * * * * * * }

  FUNCTION Defuzz (x: DOUBLE): REAL;
    {'Defuzz' is used for comparisons and to avoid propagation of 'fuzzy',
    nearly-zero values.  DOUBLE calculations often result in 'fuzzy' values.}
  BEGIN
    IF  ABS(x) < fuzz
    THEN Defuzz := 0.0
    ELSE Defuzz := x
  END {Defuzz};


  FUNCTION Radians (angle:  DOUBLE):  REAL;
    {Convert angle in degrees to radians.}
  BEGIN
    radians := 1.745329252E-2 * angle    { (pi/180) * angle }
  END {Radians};


  PROCEDURE FromCartesian (coord:  coordinates; VAR u: vector);

    {This procedure converts the vector parameter from Cartesian
    coordinates to 'coord' coordinates.}

    VAR
      phi  :  DOUBLE;
      r    :  DOUBLE;
      temp :  DOUBLE;
      theta:  DOUBLE;

  BEGIN
    IF  coord <> cartesian  {do not modify cartesian coordinates}
    THEN BEGIN
      IF   (u.size = 4) AND (coord = spherical)
      THEN BEGIN  {spherical 3D}
        temp := SQR(u.x)+SQR(u.y);    { (x,y,z) -> (r,theta,phi) }
        r := SQRT(temp+SQR(u.z));
        IF   Defuzz(u.x) = 0.0
        THEN theta := PI/4
        ELSE theta := ARCTAN(u.y/u.x);
        IF   Defuzz(u.z) = 0.0
        THEN phi := PI/4
        ELSE phi := ARCTAN(SQRT(temp)/u.z);
        u.x := r;
        u.y := theta;
        u.z := phi
      END
      ELSE BEGIN  {cylindrical 2D/3D or spherical 2D}
                    { (x,y) -> (r,theta)  or  (x,y,z) -> (r,theta,z) }
        r := SQRT( SQR(u.x) + SQR(u.y) );
        IF   Defuzz(u.x) = 0.0
        THEN theta := PI/4
        ELSE theta := ARCTAN(u.y/u.x);
        u.x := r;
        u.y := theta
      END
    END
  END {FromCartesian};


  PROCEDURE ToCartesian (coord:  coordinates; VAR u: vector);

    {This procedure converts the vector parameter from 'coord' coordinates
    into cartesian coordinates.}

    VAR
      phi   :  DOUBLE;
      r     :  DOUBLE;
      sinphi:  DOUBLE;
      theta :  DOUBLE;

  BEGIN
    IF  coord <> cartesian  {do not modify cartesian coordinates}
    THEN BEGIN
      IF   (u.size = 4) AND (coord = spherical)
      THEN BEGIN  {spherical 3D}
        r :=  u.x;     { (r,theta,phi) -> (x,y,z) }
        theta := u.y;
        phi := u.z;
        sinphi := SIN(phi);
        u.x := r * COS(theta) * sinphi;
        u.y := r * SIN(theta) * sinphi;
        u.z := r * COS(phi)
      END
      ELSE BEGIN  {cylindrical 2D/3D or spherical 2D}
        r :=  u.x;  { (r,theta) -> (x,y)  or  (r,theta,z) -> (x,y,z) }
        theta := u.y;
        u.x := r * COS(theta);
        u.y := r * SIN(theta)
      END
    END
  END {ToCartesian};


  PROCEDURE PrintVector (VAR out:  TEXT;
     indent,w,d:  INTEGER; title: STRING; u:  vector);

    VAR
      j: index;
      k: INTEGER;

  BEGIN
    WRITELN (out);
    WRITELN (out,title);
    IF  u.size = 1
    THEN WRITELN (out,'  vector is undefined')
    ELSE BEGIN
      FOR k := 1 TO indent DO
        WRITE (out,' ');
      FOR j := 1 TO u.size DO
        WRITE (out,u.vctr[j]:w:d);
      WRITELN (out)
    END
  END {PrintVector};


  PROCEDURE PrintMatrix  (VAR out:  TEXT;
    indent,w,d:  INTEGER; title:  STRING; a:  matrix);

    VAR
      i,j:  index;
      k  :  INTEGER;

  BEGIN
    WRITELN (out);
    WRITELN (out,title);
    IF  a.size = 1
    THEN WRITELN (out,'  matrix is undefined')
    ELSE
      FOR i := 1 TO a.size DO BEGIN
        FOR k := 1 TO indent DO
          WRITE (out,' ');
        FOR j := 1 TO a.size DO
          WRITE (out,a.mtrx[i,j]:w:d);
        WRITELN (out)
      END
  END {PrintMatrix};


  PROCEDURE ViewTransformMatrix  (viewtype:  coordinates;
      azimuth {or x}, elevation {or y}, distance {or z}: DOUBLE;
      ScreenX, ScreenY, ScreenDistance:  DOUBLE;
      VAR a:  matrix);

    {'ViewTransformMatrix' creates a transformation matrix for changing
    from world coordinates to eye coordinates. The location of the 'eye'
    from the 'object' is given in spherical (azimuth,elevation,distance)
    coordinates or cartesian (x,y,z) coordinates.  The size of the screen
    is 'ScreenX' units horizontally and 'ScreenY' units vertically.  The
    eye is 'ScreenDistance' units from the viewing screen.  A large ratio
    'ScreenDistance/ScreenX (or ScreenY)' specifies a narrow aperature
    -- a telephoto view.  Conversely, a small ratio specifies a large
    aperature -- a wide-angle viw.  This view transform matrix is very
    useful as the default three-dimensional transformation matrix.  Once
    set, all points are automatically transformed.}

    CONST
      half_pi   =  PI / 2.0;

    VAR
      b         :  matrix;
      cosm      :  DOUBLE;        {COS(-angle)}
      hypotenuse:  DOUBLE;
      sinm      :  DOUBLE;        {SIN(-angle)}
      temporary :  DOUBLE;
      u         :  vector;
      x,y,z     :  DOUBLE;

  BEGIN
    CASE viewtype OF
      cartesian:
        BEGIN
          x := azimuth;   {The parameters are renamed to avoid confusion.}
          y := elevation;
          z := distance;
          Vector3D (-x,-y,-z, u)
        END;
      spherical:
        BEGIN
          temporary := -distance * COS(elevation);
          Vector3D (temporary * COS(azimuth-half_pi),
                    temporary * SIN(azimuth-half_pi),
                    -distance * SIN(elevation),  u);
        END
    END;
    TranslateMatrix (u, a);     {translate origin to 'eye'}
    RotateMatrix (ThreeD,Xaxis,half_pi,cw, b);
    MatrixMultiply (a,b, a);
    CASE viewtype OF
      cartesian:
        BEGIN
          temporary := SQR(x) + SQR(y);
          hypotenuse := SQRT(temporary);
          cosm := -y/hypotenuse;
          sinm :=  x/hypotenuse;
          Matrix3D (cosm,0,sinm,0, 0,1,0,0, -sinm,0,cosm,0, 0,0,0,1, b);
          MatrixMultiply (a,b, a);
          cosm := hypotenuse;
          hypotenuse := SQRT(temporary + SQR(z));
          cosm := cosm/hypotenuse;
          sinm := -z/hypotenuse;
          Matrix3D (1,0,0,0, 0,cosm,-sinm,0, 0,sinm,cosm,0, 0,0,0,1, b)
        END;
      spherical:
        BEGIN
          RotateMatrix (ThreeD,Yaxis,-azimuth,ccw, b);
          MatrixMultiply (a,b, a);
          RotateMatrix (ThreeD,Xaxis,elevation,ccw, b);
        END
    END {CASE};
    MatrixMultiply (a,b, a);
    Vector3D (ScreenDistance/(0.5*ScreenX),
              ScreenDistance/(0.5*ScreenY),-1.0, u);
    ScaleMatrix (u, b);  {reverse sense of z-axis; screen transformation}
    MatrixMultiply (a,b, a)
  END {ViewTransformMatrix};


  PROCEDURE MatrixInverse
              (a:  matrix; VAR b:  matrix; VAR determinant:  DOUBLE);

    {This procedure inverts a general transformation matrix.  The user need
    not form an inverse geometric transformation by keeping a product of
    the inverses of simple geometric transformations:  translations, rotations
    and scaling.  A determinant of zero indicates no inverse is possible for
    a singular matrix.}

    VAR
      i,i_pivot: index;
      i_flag   : PACKED ARRAY[index] OF BOOLEAN;
      j,j_pivot: index;
      j_flag   : PACKED ARRAY[index] OF BOOLEAN;
      modulus  : DOUBLE;
      n      : index;
      pivot    : DOUBLE;
      pivot_col: PACKED ARRAY[index] OF index;
      pivot_row: PACKED ARRAY[index] OF index;
      temporary: DOUBLE;

  BEGIN                             {The matrix inversion algorithm used here}
    WITH a DO BEGIN                 {is similar to the "maximum pivot strategy"}
      FOR i := 1 TO size DO BEGIN   {described in "Applied Numerical Methods"}
        i_flag[i] := TRUE;          {by Carnahan, Luther and Wilkes,}
        j_flag[i] := TRUE           {pp. 282-284.}
      END;
      modulus := 1.0;
      FOR n := 1 TO size DO BEGIN
        pivot := 0.0;
        IF   ABS(modulus) > 0.0
        THEN BEGIN
          FOR i := 1 TO size DO
            IF  i_flag[i]
            THEN
              FOR j := 1 TO size DO
                IF   j_flag[j]
                THEN
                  IF   ABS(mtrx[i,j]) > ABS(pivot)
                  THEN BEGIN
                    pivot := mtrx[i,j];   {largest value on which to pivot}
                    i_pivot := i;         {indices of pivot element}
                    j_pivot := j
                  END;
          IF   Defuzz(pivot) = 0   {If pivot is too small, consider}
          THEN modulus := 0    {the matrix to be singular}
          ELSE BEGIN
            pivot_row[n] := i_pivot;
            pivot_col[n] := j_pivot;
            i_flag[i_pivot] := FALSE;
            j_flag[j_pivot] := FALSE;
            FOR i := 1 TO size DO
              IF   i <> i_pivot
              THEN
                FOR j := 1 TO size DO  {pivot column unchanged for elements}
                  IF   j <> j_pivot    {not in pivot row or column ...}
                  THEN mtrx[i,j] := (mtrx[i,j]*mtrx[i_pivot,j_pivot] -
                                    mtrx[i_pivot,j]*mtrx[i,j_pivot])
                                    / modulus;  {2x2 minor / modulus}
            FOR j := 1 TO size DO
              IF   j <> j_pivot        {change signs of elements in pivot row}
              THEN mtrx[i_pivot,j] := -mtrx[i_pivot,j];
            temporary := modulus;      {exchange pivot element and modulus}
            modulus := mtrx[i_pivot,j_pivot];
            mtrx[i_pivot,j_pivot] := temporary
          END
        END
      END {FOR n}
    END {WITH};
    determinant := Defuzz(modulus);
    IF  determinant <> 0
    THEN BEGIN
      b.size := a.size;         {The matrix inverse must be unscrambled}
      FOR i := 1 TO a.size DO   {if pivoting was not along main diagonal.}
        FOR j := 1 TO a.size DO
          b.mtrx[pivot_row[i],pivot_col[j]] := Defuzz(a.mtrx[i,j]/determinant)
    END
  END {MatrixInverse};


BEGIN {MathLibrary Initialization}
  fuzz := 1.0E-6
END {MathLibrary UNIT}.
