{$A-}    {- no word alignment}
{$B-}    {- no complete boolean evaluation}
{$D-}    {- no debug}
{$E-}    {- no 80*87 emulation}
{$F+}    {+ do far calls}
{$I-}    {- no I/O checking}
{$L-}    {- no local symbols}
{$N-}    {- no numeric processor code}
{$O-}    {- no overlayed code}
{$R-}    {- no range checking code}
{$S+}    {+ do stack checking}
{$V-}    {- no strict var string type checking}

UNIT KBlkRead;
{+H
---------------------------------------------------------------------------
  Version     - 0.04

  File        - KBLKREAD.PAS

  Copyright   - None. Public Domain.

  Author      - Keith S. Brown (except where otherwise noted)
                713-483-8952
                Surface mail:              Email:(brownk@samnet.jsc.nasa.gov)
                  K.Brown
                  Code:NASA/JSC/ES64
                  Houston, TX 77058 (USA)  Voice:(713)483-8952

  Purpose     - Read a text file quickly by doing block reads.

  Remarks     - Allows a file to be (effectively) read 1 character at a time.
                 Has been used to implement a multi-character look-ahead
                  parser.
                 Undesired characters can be pre-screened (for example in a
                  text file the LF (^J) can be ignored and not seen by the
                  calling program).
                 With some caution (the pushback buffer should be empty), a
                  file can be read randomly by calling JumpTo the desired
                  file offset.

  Requires    - DOS.
                TPdate by TurboPower Software (optional).

  Revised     - 1991.0128 V:0.00 (KSB) Wrote initial version.
              - 1991.0905 V:0.01 (KSB) Made GetChar virtual.
              - 1992.0422 V:0.02 (KSB) Renamed to KBLKREAD from BLK_READ.
              - 1992.0508 V:0.03 (KSB) Added pb_buf & PushBack. Mod'd GetNextChar.
              - 1993.0617 V:0.04 (KSB) Added position,useEOFmark fields & JumpTo method.
---------------------------------------------------------------------------
  Usage Notes -

  . At a minimum the following methods should be overwritten:
    Init
    Done
    OutOfText

  . KBlkRead (and its variants) has been used extensively for source file
    translators as well as for binary (EBCDIC to ASCII, etc) translators.
}

INTERFACE

TYPE
  oBuffer= OBJECT
    signature : LongINT;

    sizeFile  : LongINT;     { file size in bytes         }
    position  : LongINT;     { offset into the file       }

    open      : BOOLEAN;     { file is open               }
    INDEX     : WORD;        { current position in buffer }
    block     : BYTE;        { n'th block of file         }
    LastByte  : WORD;        { last byte in buffer        }

    Lcount    : LongINT;     { number of CR read.         }
    ignore    : SET OF CHAR; { characters to skip over [^J] is default }
    useEOFmark: BOOLEAN;     { if T, #26 (^Z) marks EOF, default is T  }

    pb_buf    : STRING;      { push back stack for look ahead}
    fTime     : LongINT;     { last time file was written }
    fDate     : LongINT;     { last date file was written }
    SFV       : FILE;        { file variable }
    buffer    : POINTER;     { source buffer }

    CONSTRUCTOR Init;
      { allocates memory for buffer
      }
    DESTRUCTOR  Done; VIRTUAL;
      { deallocates memory for buffer
      }
    FUNCTION    OpenSource(fn:STRING): BOOLEAN;
      { opens & reads initial block from file
      }
    PROCEDURE   ReadBlock;
      { reads a block from the file & stores result in buffer
      }
    FUNCTION    GetChar:CHAR; VIRTUAL;
      { raw method for getting a character ** should not be used directly
      }
    FUNCTION    GetNextChar:CHAR;
      { preferred method of getting a character, checks push-back stack first
      }
    PROCEDURE   PushBack(s:STRING);
      { push-back 1 or more characters, may be required for look ahead parsers
      }
    PROCEDURE   OutOfText; VIRTUAL;
      { replace with whatever needs to be done if the last byte has been processed
      }
    FUNCTION    JumpTo(n:LongINT):BOOLEAN;
      { if possible, jump to location N in the current file
      }
  END {OBJECT};


     {====================================================================}

IMPLEMENTATION
{$DEFINE UseTPdate}

USES
  DOS {$IFDEF UseTPdate},
  TPDate      {only used to convert file-date/time to TPRO julian format}
  {$ENDIF};

CONST
  ValidSignature   = 734838952;

CONST
  bufSize= $8000;

TYPE
  tBuffer= ARRAY[0..bufSize-1] OF CHAR;
  pBuffer= ^tBuffer;



{}PROCEDURE ReadError;
{---------------------------------------------------------------------------
  Purpose     - Let'm know that the object was not init'd.
---------------------------------------------------------------------------}
  BEGIN
    WriteLn('FATAL ERROR in [KBLKREAD] --> OBJECT oBuffer not initialized.');
    Halt;
{}END {ReadError};




{}CONSTRUCTOR oBuffer.Init;
{+H
---------------------------------------------------------------------------
  Purpose     - Allocates memory for buffer, establishes defaults.

  Declaration - constructor oBuffer.Init;

  Override    - Never.  Must be called to allocate buffer memory and
                establish defaults.

  Revised     - 1992.0508 (KSB) Added pb_buf.
              - 1993.0617 (KSB) Set POSITION to 0, useEOFmark to T.
---------------------------------------------------------------------------}
  BEGIN
    Signature := ValidSignature;

    New(pBuffer(buffer));
    fDate     := 0;
    fTime     := 0;
    INDEX     := 0;
    block     := 0;
    sizeFile  := 0;
    lastByte  := 0;
    Lcount    := 0;
    position  := 0;
    open      := FALSE;
    useEOFmark:= TRUE;
    ignore    := [^j];
    pb_buf    := '';
{}END {Init};




{}DESTRUCTOR oBuffer.Done;
{+H
---------------------------------------------------------------------------
  Purpose     - Remove the buffer from memory & de-init the object.

  Declaration - destructor oBuffer.Done; VIRTUAL;

  Override    - Never. Must be called to remove allocated memory.
---------------------------------------------------------------------------}
  BEGIN
    Dispose(pBuffer(Buffer));
    signature := 0;
{}END {Done};




{}FUNCTION oBuffer.OpenSource(fn:STRING): BOOLEAN;
{+H
---------------------------------------------------------------------------
  Purpose     - Open the specified file and read the first text block.

  Declaration - function oBuffer.OpenSource(fn:STRING): BOOLEAN;

  Remarks     - Uses GetFtime and UnpackTime from the DOS unit. Uses
                HMStoTime and DMYtoDate from the TPDate unit to convert
                to TP's Julian Date Format for later processing.

  Revised     - 1993.0617 (KSB) Set position & Lcount to 0, cleared pb_buf.
---------------------------------------------------------------------------}
  VAR
    timeDate  : LongINT;
    Ok   : BOOLEAN;
    DT   : DateTime;
  BEGIN
    IF signature <> ValidSignature THEN
      ReadError;

    Assign(SFV,fn);
    {$I-}
    Reset(SFV,1);
    Ok := IOresult = 0;

    IF Ok THEN BEGIN
      GetFTime(SFV,TimeDate);
      UnpackTime(timeDate,DT);

      {$IFDEF UseTPdate}
      fTime    := HMStoTime(DT.hour,DT.min,  DT.sec);
      fDate    := DMYtoDate(DT.day, DT.month,DT.year);
      {$ENDIF}

      open     := TRUE;
      sizeFile := FileSize(SFV);
      block    := 1;
      position := 0;
      Lcount   := 0;
      pb_buf   := '';

      ReadBlock;
    END {IF};
    OpenSource := Ok;
{}END {OpenSource};




{}PROCEDURE oBuffer.ReadBlock;
{+H
---------------------------------------------------------------------------
  Purpose     - Read a text block from the current file

  Declaration - procedure oBuffer.ReadBlock;

  Remarks     - For internal use.  Should not be called directly.
---------------------------------------------------------------------------}
  VAR
    i    : WORD;
  BEGIN
    IF signature <> ValidSignature THEN
      ReadError;

    IF open THEN BEGIN
      BlockRead(SFV, buffer^, bufSize, LastByte);

      INDEX := 0;
      IF LongINT(block)*LongINT(bufSize) >= sizeFile THEN BEGIN
        Close(SFV);
        open := FALSE;
      END {IF};
    END ELSE BEGIN
      FillChar(pBuffer(buffer)^,SizeOf(tBuffer),0);
    END {BEGIN};
{}END {ReadBlock};




{}PROCEDURE oBuffer.OutOfText;
{+H
---------------------------------------------------------------------------
  Purpose     - A virtual method signalling the end of file.
                Should be overwritten by the descendent of oBuffer.

  Declaration - procedure oBuffer.OutOfText; VIRTUAL;

  Override    - Always.  By default OutOfText halts the program.
---------------------------------------------------------------------------}
  BEGIN
    IF signature <> ValidSignature THEN
      ReadError;

    WriteLn;
    WriteLn('No more Text in ',FileRec(SFV).NAME);
    WriteLn('Lines    ... ',LCount);
    WriteLn('Lastbyte ... ',lastByte);
    WriteLn('Index    ... ',INDEX);
    WriteLn('FileSize ... ',sizeFile);
    WriteLn('Block    ... ',block);
    WriteLn('Open     ... ',open);

    Halt;
{}END {OutOfText};




{}FUNCTION oBuffer.GetChar:CHAR;
{+H
---------------------------------------------------------------------------
  Purpose     - Gets the next character from the current file.

  Declaration - function oBuffer.GetChar:CHAR; VIRTUAL;

  Override    - Rarely.

  Remarks     - For internal use.  Should not be called directly.
                The default is that CRs are returned, but LFs are not.

  Revised     - 1993.0617 (KSB) Replaced ^J case branch with ignore test.
---------------------------------------------------------------------------}
  LABEL    Restart;
  VAR
    c    : CHAR;
  BEGIN
    IF signature <> ValidSignature THEN
      ReadError;

    Restart:
    IF INDEX < LastByte THEN BEGIN
      c := pBuffer(buffer)^[INDEX];

      CASE c OF
        #26 :
        IF useEOFmark THEN
          OutOfText;

        ^m  : Inc(Lcount);
      END {CASE};

      IF c IN ignore THEN BEGIN  {1993.0617}
        Inc(INDEX);
        Inc(position);
        GOTO Restart;
      END {IF};

      Inc(INDEX);
      Inc(position);             {1993.0617}
      GetChar := c;
    END ELSE BEGIN

      IF open THEN BEGIN
        Inc(block);
        ReadBlock;
        GOTO Restart;
      END ELSE
        OutOfText;
    END {BEGIN};
{}END {GetChar};




{}FUNCTION oBuffer.GetNextChar:CHAR;
{+H
---------------------------------------------------------------------------
  Purpose     - Gets the next character.

  Declaration - function oBuffer.GetNextChar:CHAR;

  Remarks     - Gets the next character from the push-back buffer (if it is
                not empty) otherwise gets the next character from GetChar.

  Revised     - 1992.0508 (KSB) Removed "S" arg. check "PB_BUF" instead.
---------------------------------------------------------------------------}
  VAR
    Len  : BYTE ABSOLUTE pb_buf;
  BEGIN
    IF signature <> ValidSignature THEN
      ReadError;
    IF Len > 0 THEN BEGIN
      GetNextChar := pb_buf[1];
      Delete(pb_buf,1,1);
    END ELSE
      GetNextChar := GetChar;
{}END {GetNextChar};




{}PROCEDURE oBuffer.PushBack(s:STRING);
{+H
---------------------------------------------------------------------------
  Purpose     - Push "S" onto the push-back buffer.

  Declaration - procedure oBuffer.PushBack(s:STRING);

  Revised     - 1992.0508 (KSB) Added method.
---------------------------------------------------------------------------}
  BEGIN
    pb_buf := s + pb_buf;
{}END {PushBack};




{}FUNCTION  oBuffer.JumpTo(n:LongINT):BOOLEAN;
{+H
---------------------------------------------------------------------------
  Purpose     - Jump to location N in the current file.

  Declaration - function  oBuffer.JumpTo(n:LongINT):BOOLEAN;

  Remarks     - Returns T if the jump has been accomplished.  Returns F if
                the jump is not possible or if an error has occured.

  Warning     - Should only be called if the push-back buffer is empty.

  Revised     - 1993.0617 (KSB) Wrote initial version.
---------------------------------------------------------------------------}
  LABEL    Restart;
  VAR
    c    : CHAR;
    topOfBlock: LongINT;
  BEGIN
    IF signature <> ValidSignature THEN
      ReadError;

    IF (n < 0) OR (n > sizeFile) THEN
      JumpTo := FALSE
    ELSE BEGIN
      IF n > position THEN BEGIN

        Restart:
        REPEAT
          IF INDEX < LastByte THEN BEGIN
            c := pBuffer(buffer)^[INDEX];

            CASE c OF
              #26 :
              IF useEOFmark THEN BEGIN
                JumpTo := FALSE;
                OutOfText;
              END {IF};

              ^m  : Inc(Lcount);
            END {CASE};

            Inc(INDEX);
            Inc(position);
          END ELSE BEGIN

            IF open THEN BEGIN
              Inc(block);
              ReadBlock;
              GOTO Restart;
            END ELSE BEGIN
              JumpTo := FALSE;
              OutOfText;
            END {BEGIN};
          END {BEGIN};
        UNTIL position >= n;
        JumpTo := TRUE;
        pb_buf := '';
      END ELSE
      IF n < position THEN BEGIN
        topOfBlock := LongINT(Pred(block))*LongINT(bufSize);
        IF n >= topOfBlock THEN BEGIN
          INDEX    := n - topOfBlock;
          position := topOfBlock + INDEX;
          JumpTo   := TRUE;
          pb_buf   := '';
        END ELSE
        IF OpenSource(FileRec(SFV).NAME) THEN
          GOTO Restart
        ELSE
          JumpTo := FALSE;
      END ELSE
        JumpTo := TRUE;  {no change}
    END {IF};
{}END {JumpTo};




BEGIN
END {BEGIN}.
