(* ---------------------------------- *)
(* Intrupt                            *)
(* Copyright (c) 1994 by Dado Colussi *)
(* ---------------------------------- *)


UNIT Intrupt;


INTERFACE


TYPE


  InterruptPointer      = ^InterruptClass;
  InterruptClass        =
    OBJECT              (* Public section *)

      CONSTRUCTOR       InstallInterruptHandler(irq : Byte; new_vec : Pointer);
      DESTRUCTOR        RemoveInterruptHandler; VIRTUAL;

      PROCEDURE         AddService(port_number : Byte); VIRTUAL;
      PROCEDURE         RemoveService(port_number : Byte); VIRTUAL;
      FUNCTION          Service(port_number : Byte) : Boolean; VIRTUAL;
      PROCEDURE         EnableIRQ; VIRTUAL;
      PROCEDURE         DisableIRQ; VIRTUAL;

      PRIVATE           (* Private section *)
    
      irq_line          : Byte;
      vector            : Pointer;
      old_vector        : Pointer;
      ports_to_serve    : Longint;

    END; (* InterruptClass *)


FUNCTION  IRQAvailable(irq : Byte) : Boolean;
PROCEDURE CLI;
PROCEDURE STI;
PROCEDURE EOI(irq : Byte);
FUNCTION  RIL : Byte;
PROCEDURE EnableIRQ(irq : Byte);
PROCEDURE DisableIRQ(irq : Byte);


IMPLEMENTATION


USES

  DOS,
  Error,
  Regs,
  StrIO;


(*
  Returns the interrupt number used by 'irq'
*)
FUNCTION InterruptByIRQ(irq : Byte) : Byte;
BEGIN
  CASE irq OF
    0  : InterruptByIRQ := $08;         (* Timer *)
    1  : InterruptByIRQ := $09;         (* Keyboard service required *)
    2  : InterruptByIRQ := $0A;         (* Slave 8259A or EGA/VGA vertical retrace *)
    3  : InterruptByIRQ := $0B;         (* COM2 or COM4 *)
    4  : InterruptByIRQ := $0C;         (* COM1 OR COM3 *)
    5  : InterruptByIRQ := $0D;         (* Fixed disk or data request from LPT2 *)
    6  : InterruptByIRQ := $0E;         (* Floppy diks service requided *)
    7  : InterruptByIRQ := $0F;         (* Data request from LPT1 (Unreliable on IBM mono) *)

(*
  The following irqs are not supported by this unit.
  Perhaps they will someday... who knows?
*)
    8  : InterruptByIRQ := $70;         (* Real Time Clock *)
    9  : InterruptByIRQ := $71;         (* Software redirected to IRQ2 *)
    10 : InterruptByIRQ := $72;         (* Reserved *)
    11 : InterruptByIRQ := $73;         (* Reserved *)
    12 : InterruptByIRQ := $74;         (* Mouse interrupt *)
    13 : InterruptByIRQ := $75;         (* Numeric coprosessor error *)
    14 : InterruptByIRQ := $76;         (* Fixed disk controller *)
    15 : InterruptByIRQ := $77;         (* Reserved *)
  ELSE
     InterruptByIRQ := 0;               (* Invalid IRQ number *)
  END; (* CASE *)
END; (* InterruptByIRQ *)

(*
  Disables all maskable interrupts
*)
PROCEDURE CLI;
BEGIN
  INLINE($FA);
END; (* CLI *)

(*
  Enables all maskable interrupts
*)
PROCEDURE STI;
BEGIN
  INLINE($FB);
END; (* STI *)

(*
  Sends an End Of Interrupt message to the PIC
*)
PROCEDURE EOI(irq : Byte);
BEGIN
  Port[ISR] := SPECIFIC_EOI OR irq AND $F;
END; (* EOI *)

(*
  The following routine is copied from another serial I/O driver. I have
  not read any documentation on this function and the quality of it may
  be questionable.

  However, I believe that every bit in the return value represents an
  IRQ line so that if a specified bit is a logic 1 that IRQ line needs
  to be served and if it's a logic 0 that IRQ line need not to be
  served.

  7 6 5 4 3 2 1 0
  | | | | | | | \- Bit 0 => IRQ 0
  | | | | | | \-- Bit 1 => IRQ 1
  | | | | | \--- Bit 2 => IRQ 2
  | | | | \---- Bit 3 => IRQ 3
  | | | \----- Bit 4 => IRQ 4
  | | \------ Bit 5 => IRQ 5
  | \------- Bit 6 => IRQ 6
  \-------- Bit 7 => IRQ 7
*)
FUNCTION RIL : Byte;
BEGIN

  (* Report Interrupt Level Request *)
  Port[ISR] := $B;

  (* Return Interrupt Level *)
  RIL := Port[ISR];
END; (* RIL *)

(*
  Enables interrupts from a specified IRQ line.
  We must manipulate the Interrupt Mask Register so that the bit that
  represents the irq we want to use is set to logic 0. Let's say that
  irq 3 must be enabled and IMR looks like this:
      11001110
  After manipulation IMR would look like this:
      11000110
*)
PROCEDURE EnableIRQ(irq : Byte);
BEGIN
  Port[IMR] := Port[IMR] AND NOT (1 SHL irq);
END; (* EnableIRQ *)

(*
  Disables interrupts from a specified IRQ line
*)
PROCEDURE DisableIRQ(irq : Byte);
BEGIN
  Port[IMR] := Port[IMR] OR (1 SHL irq);
END; (* DisableIRQ *)

(*
  Returns TRUE if the specified irq line is available for use.
  In other words, if the bit in Interrupt Mask Register that
  represents the irq we want to use is logic 1 (disabled)
  IRQAvailable returns TRUE.
*)
FUNCTION IRQAvailable(irq : Byte) : Boolean;
BEGIN
  IRQAvailable := Port[IMR] AND (1 SHL irq) = 1 SHL irq;
END; (* IRQAvailable *)


(*
  Installs the interrupt handler, but does NOT enable interrupts.
  To enable interrupts, use method EnableIRQ.
*)
CONSTRUCTOR InterruptClass.InstallInterruptHandler(irq : Byte; new_vec : Pointer);
VAR
  int_no        : Byte;
  i             : Byte;

BEGIN
  IF NOT IRQAvailable(irq) THEN
    BEGIN
      PrintError('IRQ #' + Int2Str(irq) + ' is already in use', dead_end);
      Exit;
    END; (* IF *)
  IF error_alarm THEN
    Exit;
  irq_line := irq;
  vector := new_vec;
  int_no := InterruptByIRQ(irq);
  GetIntVec(int_no, old_vector);
  SetIntVec(int_no, vector);
  ports_to_serve := 0;
END; (* InterruptClass.InstallInterruptHandler *)

(*
  Removes interrupt handler and automaticly disables interrupts
*)
DESTRUCTOR InterruptClass.RemoveInterruptHandler;
VAR
  i     : Byte;

BEGIN
  CLI;
  DisableIRQ;
  SetIntVec(InterruptByIRQ(irq_line), old_vector);
  STI;
END; (* InterruptClass.RemoveInterruptHandler *)

(*
  Adds port to be served
*)
PROCEDURE InterruptClass.AddService(port_number : Byte);
BEGIN
  IF port_number <= MAX_PORTS THEN
    ports_to_serve := ports_to_serve OR (1 SHL (port_number - 1));
END; (* InterruptClass.AddService *)

(*
  Removes port from service list
*)
PROCEDURE InterruptClass.RemoveService(port_number : Byte);
BEGIN
  IF port_number <= MAX_PORTS THEN
    ports_to_serve := ports_to_serve AND NOT (1 SHL (port_number - 1));
END; (* InterruptClass.RemoveService *)

(*
  Returns TRUE if the specified port is in service list
*)
FUNCTION InterruptClass.Service(port_number : Byte) : Boolean;
BEGIN
  Service := ports_to_serve AND (1 SHL (port_number - 1)) > 0;
END; (* InterruptClass.Service *)

(*
  Enables interrupt from PIC
*)
PROCEDURE InterruptClass.EnableIRQ;
BEGIN
  CLI;
  Port[IMR] := Port[IMR] AND NOT (1 SHL irq_line);

  (*
    Just to be sure
  *)
  EOI(irq_line);
  STI;
END; (* InterruptClass.EnableIRQ *)

(*
  Disables interrupts from PIC
*)
PROCEDURE InterruptClass.DisableIRQ;
BEGIN
  Port[IMR] := Port[IMR] OR (1 SHL irq_line);
END; (* InterruptClass.DisableIRQ *)


END. (* Intr *)
