{ SHUFFLEV.PAS : Shuffle Game - DOS version, English

  Title   : SHUFFLEV
  Language: Borland Pascal v7.0 with Objects, plus Turbo Vision v2.0
  Version : 1.0
  Date    : Dec 09,1996
  Author  : J R Ferguson
  Download: http://www.xs4all.nl/~ferguson
  E-mail  : j.r.ferguson@iname.com
  Usage   : MS-DOS real mode application

  This program and its source may be used and copied freely whithout
  charge, but only for non-commercial purposes. In no way the author 
  can be held responsible for any damage or loss of data that may be
  caused by using this software.
}


{--- Compiler options ---}

{$B-} { Short-circuit Boolean expression evaluation }
{$V-} { Relaxed var-string checking }
{$X+} { Extended syntax }


PROGRAM SHUFFLEV;

uses App, Dialogs, Drivers, Menus, MsgBox, Objects, Views, Dos;

const
{ Natural constants }
  C_SecPerDay   = 24 * 60 * 60;

{ Program constants }
  C_ProgIdent   = 'SHUFFLEV v1.0';
  C_ProgTitle   = 'Shuffle game';
  C_Copyright   = '(C) 1996, J.R. Ferguson';
  C_MinFields   = 3;       { minimum number of fields }
  C_MaxFields   = 5;       { maximum number of fields }
  C_StepStrMax  = 3;       { string length of step count }
  C_TimeStrMax  = 8;       { string length of playing time }

{ Default values }
  C_DflFields   = 4;       { number of fields (C_MinFields..C_MaxFields) }
  C_DflMustEnd  = true;    { game must end }
  C_DflShowSteps= true;    { show step count }
  C_DflShowTime = true;    { show playing time }

{ Other constants }
  cm_GameStart  = 101;
  cm_GameRestart= 102;
  cm_GamePause  = 103;
  cm_GameSetup  = 104;
  cm_HelpRules  = 111;
  cm_HelpAbout  = 112;
  cm_FieldBase  = 200;

  hc0           = hcNoContext;
  kb0           = kbNoKey;


type
  P_Clock       = ^T_Clock;         { clock to keep track of playing time }
  P_Sequence    = ^T_Sequence;      { number sequence set up initially }
  P_MenuBar     = ^T_MenuBar;
  P_StatusLine  = ^T_StatusLine;
  P_InfoDialog  = ^T_InfoDialog;
  P_SetupBuf    = ^T_SetupBuf;
  P_SetupDlg    = ^T_SetupDlg;
  P_FramedText  = ^T_FramedText;
  P_Frame       = ^T_Frame;
  P_Field       = ^T_Field;
  P_Board       = ^T_Board;
  P_GameWindow  = ^T_GameWindow;
  P_Application = ^T_Application;

  T_StepStr     = String[C_StepStrMax];
  T_TimeStr     = String[C_TimeStrMax];
  T_ScoreStr    = String[C_StepStrMax + 2 + C_TimeStrMax];

  T_Clock       = object(TObject)
    ClockActive : boolean; { timer is active }
    Seconds     : LongInt; { cumulative playing time in seconds }
    ClockTime   : LongInt; { clock time last read in seconds after midnight }
    constructor Init;
    function    Active: boolean;
    procedure   Start;
    procedure   Stop;
    procedure   Pause;
    procedure   Restart;
    procedure   GetTimeStr(var V_TimeStr: T_TimeStr);
    procedure   GetSysTime;
    function    Adjust: boolean;
  end;

  T_Sequence    = object(TObject)
    NumSequence : array[1..C_MaxFields*C_MaxFields] of integer;
    constructor Init(V_Count: integer; V_Random: boolean);
    function    GetNumber(V_Index: integer): integer;
  end;

  T_MenuBar     = Object(TMenuBar)
    procedure   Draw; virtual;
  end;

  T_StatusLine  = Object(TStatusLine)
    ScoreStr    : T_ScoreStr;
    Constructor Init(var V_Rect: TRect; V_Defs: PStatusDef);
    procedure   SetScore(V_ScoreStr: T_ScoreStr);
    procedure   Draw; virtual;
  end;

  T_InfoDialog  = Object(TDialog)
    Constructor Init(V_Width,V_Height: integer; V_Title,V_Message: String);
  end;

  T_SetupBuf    = record
    IO_Count    : Word;
    IO_Options  : Word;
  end;

  T_FramedText  = Object(TStaticText)
    Constructor Init(var V_Rect: TRect; V_Title: TTitleStr);
  end;

  T_Frame       = Object(T_FramedText)
    Constructor Init(V_Rect: TRect);
  end;

  T_SetupDlg    = Object(TDialog)
    Constructor Init(var V_Rect: TRect);
  end;

  T_Field       = Object(TButton)
    Row, Column : integer;
    Number      : integer;
    Constructor Init(var V_Rect: TRect; V_Row,V_Column: integer;
                     V_Number: integer);
    function    AttachNumber(V_Number: integer): boolean;
    procedure   DetachNumber;
    procedure   Show; virtual;
    function    GetPalette: PPalette; virtual;
  end;

  T_Board       = Object(TGroup)
    Count       : integer;       { number of fields horizontal and vertical }
    Field       : array[1..C_MaxFields,1..C_MaxFields] of P_Field;
    InitialPos  : array[1..C_MaxFields,1..C_MaxFields] of integer;
    Constructor Init(var V_Rect: TRect; V_Count: integer; V_Random: boolean);
    Destructor  Done; virtual;
    procedure   SavePosition;
    procedure   RestorePosition;
    function    AllRight: boolean;
    function    EmptyField: P_Field;
  end;

  T_GameWindow  = Object(TDialog)
    Count       : integer;
    MustEnd     : boolean;          { game must end }
    ShowSteps   : boolean;          { show step count }
    ShowTime    : boolean;          { show playing time }
    Steps       : integer;          { step count so far }
    Board       : P_Board;          { game board }
    Clock       : P_Clock;          { timer to keep track of playing time }
    Shuffling   : boolean;          { busy shuffling: do not paint }
    Paused      : boolean;          { game is paused }
    Constructor Init(var V_Rect: TRect);
    Destructor  Done; virtual;
    procedure   Construct;
    procedure   Dismantle;
    procedure   NewGame;
    procedure   SameGame;
    procedure   Shuffle;
    procedure   RecordStep;
    procedure   ShowScore;
    procedure   EndMessage;
    function    ShiftTo(dr,dc: integer): boolean;
    procedure   Draw; virtual;
    procedure   HandleEvent(var V_Event: TEvent); virtual;
    procedure   DoGameSetup;
    procedure   DoGamePause;
    procedure   DoButtonClick(V_Row,V_Column: integer);
    procedure   DoLeft;
    procedure   DoRight;
    procedure   DoUp;
    procedure   DoDown;
  end;

  T_Application  = Object(TApplication)
    GameWindow  : P_GameWindow;
    Constructor Init;
    Destructor  Done; virtual;
    procedure   InitMenuBar; virtual;
    procedure   InitStatusLine; virtual;
    procedure   Idle; virtual;
    procedure   HandleEvent(var V_Event: TEvent); virtual;
    procedure   DoHelpRules;
    procedure   DoHelpAbout;
  end;


{ --- General routines --- }

function IMin(i,j: integer): integer;
begin if i<j then IMin:= i else IMin:= j; end;

function CenterX(var V_Rect: TRect; V_Width : integer): integer;
begin with V_Rect do CenterX:= (B.X - A.X - V_Width ) div 2; end;

function CenterY(var V_Rect: TRect; V_Height: integer): integer;
begin with V_Rect do CenterY:= (B.Y - A.Y - V_Height) div 2; end;

function CommandOf(V_Row, V_Column: integer): Word;
begin CommandOf:= cm_FieldBase + (V_Row-1)*C_MaxFields + (V_Column-1); end;

function RowOf(V_Command: Word): integer;
begin
  if V_Command < cm_FieldBase then RowOf:= 0
  else RowOf  := 1 + (V_Command - cm_FieldBase) div C_MaxFields;
end;

function ColumnOf(V_Command: Word): integer;
begin
  if V_Command < cm_FieldBase then ColumnOf:= 0
  else ColumnOf:= 1 + (V_Command - cm_FieldBase) mod C_MaxFields;
end;


{ --- T_Clock --- }

constructor T_Clock.Init;
begin inherited Init; Stop; end;

function    T_Clock.Active: boolean;
begin Active:= ClockActive end;

procedure   T_Clock.Start;
begin Seconds:= 0; GetSysTime; ClockActive:= true; end;

procedure   T_Clock.Stop;
begin Seconds:= 0; ClockTime:= 0; ClockActive:= false; end;

procedure   T_Clock.Pause;
begin ClockActive:= false; end;

procedure   T_Clock.Restart;
begin GetSysTime; ClockActive:= true; end;

procedure   T_Clock.GetTimeStr(var V_TimeStr: T_TimeStr);
var tmp: LongInt; h,m,s: word; s0: String[2];
begin
  Adjust;
  s := Seconds mod 60; tmp:= Seconds div 60;
  m := tmp     mod 60;
  h := tmp     div 60;
  if h > 0 then begin Str(h:2,s0); V_TimeStr:= s0+':'; end
                else  V_TimeStr:= '';
  Str(m:2,s0); if s0[1]=' ' then s0[1]:= '0'; V_TimeStr:= V_TimeStr+s0+':';
  Str(s:2,s0); if s0[1]=' ' then s0[1]:= '0'; V_TimeStr:= V_TimeStr+s0;
end;

procedure   T_Clock.GetSysTime;
var h,m,s,c: Word;
begin
  GetTime(h,m,s,c);
  ClockTime:= (LongInt(h)*60 + LongInt(m))*60 + LongInt(s);
end;

function    T_Clock.Adjust: boolean;
var PrevTime: LongInt;
begin
  if Active then begin
    PrevTime:= ClockTime; GetSysTime;
    if ClockTime < PrevTime
      then Inc(Seconds, ClockTime - PrevTime + C_SecPerDay)
      else Inc(Seconds, ClockTime - PrevTime);
    Adjust:= ClockTime <> PrevTime;
  end
  else Adjust:= false;
end;


{ --- T_Sequence --- }

constructor T_Sequence.Init(V_Count: integer; V_Random: boolean);
var i,j,k,max: integer; ok: boolean;
begin
  inherited Init;
  i:= 0; max:= V_Count*V_Count;
  if V_Random then begin
    repeat
      for i:= 1 to max do NumSequence[i]:= 0;
      for j:= 0 to max-1 do begin
        for k:= 1 + Random(max-1) downto 0 do begin
          Inc(i); if i>max then i:= 1;
          while NumSequence[i] <> 0 do begin
            Inc(i);
            if i>max then i:= 1;
          end;
        end;
        NumSequence[i]:= j;
      end;
      ok:= true; i:= 0;
      while ok and (i<max) do begin Inc(i); ok:= NumSequence[i]=i; end;
    until not ok;
  end
  else begin
    for i:= 1 to max-1 do NumSequence[i]:= i;
    NumSequence[max]:= 0;
  end;
end;

function    T_Sequence.GetNumber(V_Index: integer): integer;
begin GetNumber:= NumSequence[V_Index]; end;


{ --- T_MenuBar --- }

procedure   T_MenuBar.Draw;
var R: TRect;
begin
  Inherited Draw;
  GetExtent(R);
  WriteStr(CenterX(R,Length(C_ProgTitle)),R.A.Y,C_ProgTitle,1);
end;


{ --- T_StatusLine --- }

Constructor T_StatusLine.Init(var V_Rect: TRect; V_Defs: PStatusDef);
begin
  Inherited Init(V_Rect,V_Defs);
  ScoreStr:= '';
end;

procedure   T_StatusLine.SetScore(V_ScoreStr: T_ScoreStr);
begin ScoreStr:= V_ScoreStr; Draw; end;

procedure   T_StatusLine.Draw;
var R: TRect;
begin
  Inherited Draw;
  GetExtent(R);
  WriteStr(R.B.X-Length(ScoreStr)-1,R.A.Y,ScoreStr,1);
end;


{ ---T_InfoDialog --- }

Constructor T_InfoDialog.Init(V_Width,V_Height: integer;
                              V_Title,V_Message: String);
var R: TRect; X,Y,AX,AY: integer;
begin
  DeskTop^.GetExtent(R);
  X:= (R.B.X - R.A.X) div 2; AX:= V_Width  div 2;
  Y:= (R.B.Y - R.A.Y) div 2; AY:= V_Height div 2;
  R.Assign(X-AX,Y-AY,X+AX,Y+AY);
  Inherited Init(R,V_Title);
  GetExtent(R); X:= R.B.X div 2; Y:= R.B.Y - 3;
  R.Grow(-2,-2);
  Insert(New(PStaticText,Init(R,V_Message)));
  R.Assign(X-5,Y,X+5,Y+2);
  Insert(New(PButton,Init(R,'O~K~',cmOK,bfDefault)));
end;


{ --- T_FramedText --- }

Constructor T_FramedText.Init(var V_Rect: TRect; V_Title: TTitleStr);
begin Inherited Init(V_Rect,V_Title); Options:= Options or ofFramed; end;


{ --- T_Frame --- }

Constructor T_Frame.Init(V_Rect: TRect);
begin Inherited Init(V_Rect,''); end;


{ --- T_SetupDlg --- }

Constructor T_SetupDlg.Init(var V_Rect: TRect);
var R: TRect; p: PView; i: integer;
begin
  Inherited Init(V_Rect,'Setup');

  R.Assign(03,02,29,06); Insert(New(P_FramedText,Init(R,'Field count')));
  R.Assign(03,03,29,06); Insert(New(PRadioButtons,Init(R,
    NewSItem('~3~x3',
    NewSItem('~4~x4',
    NewSItem('~5~x5',nil))))));
  R.Assign(03,08,29,12); Insert(New(P_FramedText,Init(R,'Options')));
  R.Assign(03,09,29,12); Insert(New(PCheckBoxes,Init(R,
    NewSItem('Game must ~e~nd',
    NewSItem('Show ~s~teps',
    NewSItem('Show ~t~ime',nil))))));
  R.Assign(34,04,46,06);
    Insert(New(PButton,Init(R,'O~K~'    ,cmOK    ,bfDefault)));
  R.Assign(34,09,46,11);
    Insert(New(PButton,Init(R,'~C~ancel',cmCancel,bfNormal)));
  SelectNext(false);
end;


{ --- T_Field --- }

Constructor T_Field.Init(var V_Rect: TRect; V_Row,V_Column: integer;
                         V_Number: integer);
var R: TRect;
begin
  Inherited Init(V_Rect,'  ', CommandOf(V_Row,V_Column), 0);
  Row:= V_Row; Column:= V_Column; DetachNumber; AttachNumber(V_Number);
end;

function    T_Field.AttachNumber(V_Number: integer): boolean;
begin
  if Number > 0 then AttachNumber:= false
  else begin
    Number:= V_Number; Str(V_Number:2, Title^); Show;
    AttachNumber:= true;
  end;
end;

procedure  T_Field.DetachNumber;
begin Number:= 0; Title^:= '  '; Hide; end;

procedure  T_Field.Show;
begin if Number > 0 then Inherited Show; end;

function   T_Field.GetPalette: PPalette;
const C_Palette: string[8] = #10#10#10#10#14#14#14#15;
begin GetPalette:= @C_Palette; end;


{ --- T_Board --- }

Constructor T_Board.Init(var V_Rect: TRect; V_Count: integer;
                         V_Random: boolean);
var
  p: PView;
  row,column,i: integer;
  X,Y,Ax,Ay,X0,Y0,Ax0,Ay0: integer;
  R,R0: TRect;
  Sequence: P_Sequence;
begin
  R.Copy(V_Rect);
  X := R.A.X + (R.B.X - R.A.X) div 2;
  Y := R.A.Y + (R.B.Y - R.A.Y) div 2;
  Ax0:= (R.B.X - R.A.X) div (2*V_Count); Ax:= Ax0 * V_Count; X0:= X - Ax;
  Ay0:= (R.B.Y - R.A.Y) div (2*V_Count); Ay:= Ay0 * V_Count; Y0:= Y - Ay;

  R.Assign(X0,Y0,X0+2*Ax0*V_Count,Y0+2*Ay0*V_Count);
  Inc(R.B.X); Inc(R.B.Y);
  Inherited Init(R);
  GetExtent(R); Insert(New(P_Frame,Init(R)));
  Count:= V_Count;

  New(Sequence,Init(Count,V_Random)); i:= 0;
  R0.Assign(1,1,2*Ax0+1,2*Ay0+1);
  for row:= 1 to Count do begin
    R.Copy(R0);
    for column:= 1 to Count do begin
      inc(i);
      New(Field[row,column], Init(R,row,column,Sequence^.GetNumber(i)));
      Insert(Field[row,column]);
      R.Move(2*Ax0,0);
    end;
    R0.Move(0,2*Ay0);
  end;
  Dispose(Sequence,Done);
end;

Destructor  T_Board.Done;
var r,c: integer;
begin
  for r:= 1 to Count do for c:= 1 to Count do begin
    Delete(Field[r,c]); Dispose(Field[r,c],Done);
  end;
  Inherited Done;
end;

procedure   T_Board.SavePosition;
var r,c: integer;
begin
  for r:= 1 to Count do for c:= 1 to Count do
    InitialPos[r,c]:= Field[r,c]^.Number;
end;

procedure   T_Board.RestorePosition;
var r,c: integer;
begin for r:= 1 to Count do for c:= 1 to Count do with Field[r,c]^ do begin
  DetachNumber; AttachNumber(InitialPos[r,c]);
end; end;

function    T_Board.AllRight: boolean;
var i,r,c: integer; ok: boolean;
begin
  ok:= true; r:= Count; c:= Count; i:= Count*Count;
  while ok and (i>1) do begin
    Dec(i); if c>1 then  Dec(c) else begin Dec(r); c:= Count; end;
    ok:= Field[r,c]^.Number = i;
  end;
  AllRight:= ok;
end;

function    T_Board.EmptyField: P_Field;
var r,c: integer; ok: boolean;
begin
  ok:= false; r:= 0;
  while (r < Count) and not ok do begin
    Inc(r); c:= 0;
    while (c < Count) and not ok do begin
      Inc(c); ok:= Field[r,c]^.Number=0;
    end;
  end;
  if ok then EmptyField:= Field[r,c] else EmptyField:= nil;
end;



{ --- T_GameWindow --- }

Constructor T_GameWindow.Init(var V_Rect: TRect);
begin
  Inherited Init(V_Rect,'');
  Palette   := dpBlueDialog;
  Flags     := Flags and not (wfMove or wfClose);
  Count     := C_DflFields;
  MustEnd   := C_DflMustEnd;
  ShowSteps := C_DflShowSteps;
  ShowTime  := C_DflShowTime;
  Shuffling := false;
  Paused    := false;
  New(Clock,Init);
  Construct;
end;

destructor  T_GameWindow.Done;
begin
  Dismantle;
  Dispose(Clock,Done);
  inherited Done;
end;

procedure   T_GameWindow.Construct;
var R: TRect;
begin
  Steps:= 0; Clock^.Stop; Paused:= false;
  GetExtent(R); R.Grow(-14,-1);
  New(Board,Init(R,Count,not MustEnd)); if MustEnd then Shuffle;
  Insert(Board);
  Board^.SavePosition;
end;

procedure   T_GameWindow.Dismantle;
begin
  Delete(Board);
  Dispose(Board,Done);
end;

procedure   T_GameWindow.NewGame;
begin
  Dismantle; Construct;
  Draw;
end;

procedure   T_GameWindow.SameGame;
begin
  Board^.RestorePosition;
  Steps:= 0; Clock^.Stop; Paused:= false;
  Draw;
end;

procedure   T_GameWindow.Shuffle;
var i: integer;
begin
  Shuffling:= true;
  repeat
    i:= 0;
    while i < 100 *Count do case random(4) of
      0: if ShiftTo( 0,-1) then Inc(i);
      1: if ShiftTo( 0,+1) then Inc(i);
      2: if ShiftTo(-1, 0) then Inc(i);
      3: if ShiftTo(+1, 0) then Inc(i);
    end;
  until not Board^.AllRight;
  Shuffling:= false;
end;

procedure   T_GameWindow.RecordStep;
begin if not Shuffling then begin
  if not Clock^.Active then Clock^.Start;
  Clock^.Adjust; Inc(Steps); ShowScore;
end end;

procedure   T_GameWindow.ShowScore;
var StepStr  : T_StepStr;
    TimeStr  : T_TimeStr;
    ScoreStr : T_ScoreStr;
begin
  ScoreStr:= '';
  if ShowSteps or ShowTime then begin
    if ShowTime then begin Clock^.GetTimeStr(TimeStr); ScoreStr:=TimeStr end;
    if ShowSteps then begin
      Str(Steps:3,StepStr);
      if ShowTime then ScoreStr:=ScoreStr+' '+StepStr else ScoreStr:=StepStr;
    end;
  end;
  P_StatusLine(StatusLine)^.SetScore(ScoreStr);
end;

procedure   T_GameWindow.EndMessage;
var result: Word;
begin
  result:= MessageBox(
      #3'All right!'#13#13#3'New initial position ?',
      nil,mfYesButton or mfNoButton or mfInformation
     );
  if (result = cmNo) or (result = cmCancel)
    then SameGame
    else NewGame;
end;

function    T_GameWindow.ShiftTo(dr,dc: integer): boolean;
var p: P_Field;
begin with Board^ do begin
  p:= EmptyField;
  if p = nil then ShiftTo:= false
  else with p^ do begin
    if (Row  -dr < 1) or (Row  -dr > Count) or
       (Column-dc < 1) or (Column-dc > Count)
    then ShiftTo:= false
    else begin
      if Field[Row,Column]^.AttachNumber(Field[Row-dr,Column-dc]^.Number)
      then begin
        Field[Row-dr,Column-dc]^.DetachNumber; RecordStep;
        ShiftTo:= true;
      end
      else ShiftTo:= false;
    end;
  end;
end end;

procedure   T_GameWindow.Draw;
begin
  Inherited Draw;
  ShowScore;
end;

procedure   T_GameWindow.HandleEvent(var V_Event: TEvent);
  var Row, Column: integer;
  procedure Clear; begin ClearEvent(V_Event); end;
begin {HandleEvent}
  Inherited HandleEvent(V_Event);
  with V_Event do case What of
    evCommand: case Command of
      cm_GameSetup   : begin DoGameSetup; Clear; end;
      cm_GameStart   : begin NewGame;     Clear; end;
      cm_GameRestart : begin SameGame;    Clear; end;
      cm_GamePause   : begin DoGamePause; Clear; end;
      else begin
         Row   := RowOf   (Command);
         Column:= ColumnOf(Command);
         if (Row    >= 1) and (Row    <= Count) and
            (Column >= 1) and (Column <= Count)
         then begin DoButtonClick(Row,Column); Clear; end;
      end;
    end;
    evKeyDown: case KeyCode of
      kbLeft         : begin DoLeft;      Clear; end;
      kbRight        : begin DoRight;     Clear; end;
      kbUp           : begin DoUp;        Clear; end;
      kbDown         : begin DoDown;      Clear; end;
    end;
  end;
end;

procedure   T_GameWindow.DoGameSetup;
var R: TRect; SetupBuf: T_SetupBuf; n: integer;
    must_end,show_steps,show_time: boolean;
begin with SetupBuf do begin
  n:= Count;
  must_end  := MustEnd;
  show_steps:= ShowSteps;
  show_time := ShowTime;
  IO_Count  := Count - C_MinFields;
  IO_Options:= Ord(MustEnd)
           or (Ord(ShowSteps) shl 1)
           or (Ord(ShowTime)  shl 2);
  R.Assign(15,4,65,18);
  if Application^.ExecuteDialog(New(P_SetupDlg,Init(R)),@SetupBuf)
     <> cmCancel
  then begin
    n         := IO_Count + C_MinFields;
    must_end  :=(IO_Options and $01) <> 0;
    show_steps:=(IO_Options and $02) <> 0;
    show_time :=(IO_Options and $04) <> 0;
    if (n <> Count) or (must_end <> MustEnd) then begin
      Count:= n; MustEnd:= must_end; NewGame;
    end;
    if (show_steps <> ShowSteps) or (show_time <> ShowTime) then begin
      ShowSteps:= show_steps; ShowTime:= show_time;
      Draw;
    end;
  end;
end end;

procedure   T_GameWindow.DoGamePause;
begin
  Paused:= true; Clock^.Pause;
  Board^.Hide;
  MessageBox(#13#3'PAUSE',nil,mfOKButton or mfInformation);
  Paused:= false; if Steps>0 then Clock^.Restart;
  Board^.Show;
end;

procedure   T_GameWindow.DoButtonClick(V_Row,V_Column: integer);
  function Shift(dr,dc: integer): boolean;
  var r,c: integer;
  begin
    r:= V_Row + dr; c:= V_Column + dc;
    if (r<1) or (r>Count) or (c<1) or (c>Count) then Shift:= false
    else with Board^ do begin
      if Field[r,c]^.AttachNumber(Field[V_Row,V_Column]^.Number) then begin
        Field[V_Row,V_Column]^.DetachNumber; RecordStep;
        Shift:= true;
      end
      else Shift:= false;
    end;
  end;
begin if Board^.Field[V_Row,V_Column]^.Number > 0 then begin {DoButtonClick}
  if Shift(0,-1) or Shift(0,+1) or Shift(-1,0) or Shift(+1,0) then
    if Board^.AllRight then EndMessage;
end; end;

procedure   T_GameWindow.DoLeft;
begin if ShiftTo(0,-1) then if Board^.AllRight then EndMessage; end;

procedure   T_GameWindow.DoRight;
begin if ShiftTo(0,+1) then if Board^.AllRight then EndMessage; end;

procedure   T_GameWindow.DoUp;
begin if ShiftTo(-1,0) then if Board^.AllRight then EndMessage; end;

procedure   T_GameWindow.DoDown;
begin if ShiftTo(+1,0) then if Board^.AllRight then EndMessage; end;


{ --- T_Application --- }

Constructor T_Application.init;
var R: TRect;
begin
  Inherited Init;
  DeskTop^.GetExtent(R);
  New(GameWindow,Init(R));
  InsertWindow(GameWindow);
end;

Destructor  T_Application.Done;
begin
  Dispose(GameWindow,Done); GameWindow:= nil;
  Inherited Done;
end;

procedure   T_Application.InitMenuBar;
var R: TRect;
begin
  GetExtent(R); R.B.Y:= R.A.Y+1;
  MenuBar:= New(P_MenuBar,Init(R,NewMenu(
    NewSubMenu('~G~ame',hc0,NewMenu(
      NewItem('~N~ew game'        ,''     ,kb0    ,cm_GameStart   ,hc0,
      NewItem('~I~nitial position',''     ,kb0    ,cm_GameRestart ,hc0,
      NewItem('~P~ause'           ,'Alt+P',kbAltP ,cm_GamePause   ,hc0,
      NewLine(
      NewItem('~S~etup...'        ,''     ,kb0    ,cm_GameSetup   ,hc0,
      NewLine(
      NewItem('E~x~it'            ,'Alt+X',kbAltX ,cmQuit         ,hc0,
    nil)))))))),
    NewSubMenu('~H~elp',hc0,NewMenu(
      NewItem('~R~ules'           ,'F1'   ,kbF1   ,cm_HelpRules   ,hc0,
      NewItem('~A~bout'           ,''     ,kb0    ,cm_HelpAbout   ,hc0,
    nil))),
  nil)))));
end;

procedure   T_Application.InitStatusLine;
var R: TRect;
begin
  GetExtent(R); R.A.Y:= R.B.Y-1;
  StatusLine:= New(P_StatusLine,Init(R,
    NewStatusDef($0000,$FFFF,
      NewStatusKey('~F1~ Help'        ,kbF1   ,cm_HelpRules,
      NewStatusKey('~F10~ Menu'       ,kbF10  ,cmMenu,
      NewStatusKey('~Alt+P~ Pause'    ,kbAltP ,cm_GamePause,
      NewStatusKey('~Alt+X~ Exit'     ,kbAltX ,cmQuit,
      StdStatusKeys(
    nil))))),
  nil)));
end;

procedure   T_Application.Idle;
begin if GameWindow <> nil then with GameWindow^ do begin
  if Clock^.Active then if Clock^.Adjust then ShowScore;
end end;

procedure   T_Application.HandleEvent(var V_Event: TEvent);
  procedure Clear; begin ClearEvent(V_Event); end;
begin {HandleEvent}
  Inherited HandleEvent(V_Event);
  with V_Event do case What of
    evCommand: case Command of
      cm_HelpRules: begin DoHelpRules; Clear; end;
      cm_HelpAbout: begin DoHelpAbout; Clear; end;
    end;
  end;
end;

procedure  T_Application.DoHelpRules;
begin
  ExecuteDialog(New(P_InfoDialog,Init(50,14,'Rules',
    'Click a number joining the empty field to shift it, or use the '+
    'cursor keys.'#13#13 +
    'Order the numbers from low to high, in rows left to right and columns '+
    'top to bottom. The empty field must be in the lower-right corner.'
  )),nil);
end;

procedure  T_Application.DoHelpAbout;
begin
  MessageBox(
    #3+C_ProgIdent+#13 +
    #3'Borland Pascal + Turbo Vision'#13 +
    #3+C_Copyright,
    nil,mfInformation or mfOKButton);
end;


{ --- Hoofdprogramma --- }

begin
  Randomize;
  Application:= New(P_Application,Init);
  P_Application(Application)^.Run;
  Dispose(P_Application(Application),Done);
end.
