PROGRAM BuffonsNeedle;

  {This Turbo Pascal 5.5 "screen saver" program simulates throwing Buffon's
   needles to provide a Monte Carlo approximation of pi ().

   (C) Copyright 1990, Earl F. Glynn, Overland Park, KS.  Compuserve 73256,3527.
   All rights reserved.  This program may be freely distributed only for
   non-commercial use.

   Execute this program by entering "Buffon" and optionally, a numeric
   parameter.  The parameter defines how many needle throws are made
   before rescaling (and clearing) the view.  The default value is
   2000.  Use a smaller number to keep more black space on the screen.

   Interrupt the program by pressing any key.  On exit a summary will
   display the number of tosses, the number of line crossings, and
   the approximate value of pi.  [If more than 2,147,483,647 needles are
   thrown (longword overflow), the estimated value of pi will be
   inaccurate.  On my 20-MHz 286 system I estimate an overflow
   would occur after about 58 days.]

   This program will work in VGA, EGA or CGA modes and will automatically
   detect the highest resolution available.   To override the auto-
   detection feature, or if autodetection does not work, specify the
   graphics driver and mode with DOS environment variables before
   executing BUFFON:

        CGA:  SET DRIVER=CGA
              SET MODE=320x200
                       640x200

        EGA:  SET DRIVER=EGA
              SET MODE=640x350
                       640x200

        VGA:  SET DRIVER=VGA
              SET MODE=640x480
                       640x350
                       640x200

   For more information about Buffon's needle problem consult either
   of the following:

   Richard W. Hamming, "Introduction to Applied Numerical Analysis,"
        McGraw-Hill, New York (1971), pp. 311-312.

   Anthony Ralston (editor), "Encylopedia of Computer Science and
        Engineering" (second edition), Van Nostrand Reinhold Company,
        New York (1983), pp. 997-998.


  The length of Buffon's needle, L, in this program is a constant 1.}

  USES
    Clocks,     {clock object:  Start, Elapsed, hhmmss}
    CRT,        {Readkey, KeyPressed, ClrScr}
    DOS,        {GetEnv}
    Drivers,    {CGA, EGA and VGA Drivers, DeCodeDriverMode}
    Graph;      {InitGraph, CloseGraph, Line, SetColor, SetLineStyle, ...}

  CONST
    null      = #$00;
    MaxStyles = 4;

  VAR
    cross        :  LongInt;
    DrawLineCount:  LongInt;
    ExitSave     :  POINTER;
    HalfLength   :  REAL;    {L/2 in pixels; L = 1 in throwing space}
    key          :  CHAR;
    Itemp        :  INTEGER;
    Ltemp        :  Longint;
    nLine        :  WORD;
    offset       :  WORD;    {vary position of first parallel line}
    RescaleCount :  LongInt;
    SaveAttr     :  BYTE;
    SaveMode     :  WORD;
    step         :  WORD;    {distance between parallel line in pixels}
    timer        :  Clock;
    toss         :  LongInt;
    VideoDriver  :  INTEGER;
    VideoMode    :  INTEGER;

  PROCEDURE DrawLines;
    VAR
      row:  WORD;
  BEGIN
    SetLineStyle (Random(MaxStyles), 0, 1+2*Random(2));
    SetColor (1+Random(GetMaxColor));  {don't allow 0 (black)}
    row := offset;
    nLine := 0;
    WHILE (row <= GetMaxY) DO BEGIN
      Line (0, row, GetMaxX, row);
      INC (row, step);
      INC (nLine)
    END
  END {DrawLines};

  PROCEDURE SetupLines;
  BEGIN
    ClearViewPort;
    offset := Random(1+GetMaxY DIV 10);
    step := (GetMaxY DIV 15) + Random(1+GetMaxY DIV (1+Random(9)) );
    HalfLength := 0.5*step;
    DrawLines
  END {SetupLines};

  PROCEDURE ThrowNeedle;
    VAR
      alpha        :  REAL;
      CosTheta     :  REAL;
      hypotenuse   :  REAL;
      SinTheta     :  REAL;
      x,y          :  REAL;
      xDelta,yDelta:  INTEGER;    {pixel units}
      xMid,yMid    :  INTEGER;    {pixel position of needle midpoint}
  BEGIN
    {define needle's midpoint distance from parallel line}
    alpha := Random - 0.5;        {-0.5 <= alpha <= 0.5}

    {define direction vector (x,y) to assign needle orientation}
    x := 2.0*(Random - 0.5);      {-1.0 <= x <= 1.0}
    y := Random;                  { 0.0 <= y <= 1.0}
    IF   x = 0.0
    THEN BEGIN
      SinTheta := 1.0;
      CosTheta := 0.0
    END
    ELSE BEGIN
      hypotenuse := SQRT( SQR(x) + SQR(y) );
      SinTheta := y / hypotenuse;
      CosTheta := x / hypotenuse  {needed for plotting needle}
    END;

    {update statistics}
    INC (toss);                   {count the tosses}
    IF   ABS(alpha) <= 0.5*SinTheta  {recall L = 1 in throwing space}
    THEN INC (cross);             {count the crossings}

    {define midpoint of needle in throwing space}
    xMid := Random(GetMaxX+1);    {determine placement of needle midpoint}
    yMid := offset + step*Random(1+nLine) + ROUND(alpha*step);

    xDelta := ROUND(HalfLength*CosTheta);
    yDelta := ROUND(HalfLength*SinTheta);

    {plot the needle}
    SetColor (Random(GetMaxColor+1));  {needle color}
    SetLineStyle (Random(MaxStyles), 0, NormWidth);
    Line (xMid-xDelta,yMid-yDelta, xMid+xDelta,yMid+yDelta);

    IF   toss MOD RescaleCount = 0     {change scaling periodically}
    THEN SetupLines
    ELSE
      IF   toss MOD DrawLineCount = 0  {redraw lines being thrown at}
      THEN DrawLines
  END {ThrowNeedle};

  {$F+}
  PROCEDURE BuffonExit;      {all exits normal/abnormal come here}
    VAR
      Elapsed :  REAL;
      PiApprox:  REAL;
  BEGIN
    TextMode (SaveMode);     {restore CRT attributes}
    TextAttr := SaveAttr;
    ClrScr;
    Elapsed := Timer.Elapsed;
    WRITELN (' ':13,'Buffon''s Needle:  Monte Carlo Method of Calculating ');
    WRITELN (' ':10,'by Earl F. Glynn, Overland Park, KS.  Compuserve 73257,3527.');
    WRITELN;
    WRITELN (' ':13,' Needle Tosses:',toss:9,
      ' ':6,' Elapsed Time:',hhmmss(Timer.Elapsed):9);
    WRITE   (' ':13,'Line Crossings:',cross:9);
    IF   Elapsed > 0.0
    THEN WRITE (' ':6,'Tosses/Second:',toss/Timer.Elapsed:9:0);
    WRITELN;
    WRITELN;
    IF   cross > 0
    THEN BEGIN
      PiApprox := 2.0*toss/cross; {recall needle length, L = 1}
      WRITELN (' ':34,'  ',PiApprox:8:6)
    END
  END {BuffonExit};
  {$F-}

BEGIN
  ExitSave := ExitProc;
  ExitProc := @BuffonExit;   {establish EXIT procedure}

  SaveAttr := CRT.TextAttr;  {save CRT attributes to restore later}
  SaveMode := CRT.LastMode;

  timer.Start (DOSClock);    {elapsed time clock}

  Randomize;                 {seeds of random number generator based on clock}
  toss  := 0;
  cross := 0;
  RescaleCount := 2000;
  IF   ParamCount > 0
  THEN BEGIN
    VAL (ParamStr(1),Ltemp,Itemp);
    IF   (Itemp = 0) AND (Ltemp > 0)
    THEN RescaleCount := Ltemp         {change default if valid number}
  END;
  DrawLineCount := RescaleCount DIV 5;

  DeCodeDriverMode (GetEnv('DRIVER'),GetEnv('MODE'),
                    VideoDriver,VideoMode);
  InitGraph (VideoDriver,VideoMode,'');
  SetViewPort (0,0, GetMaxX,GetMaxY, ClipOn);

  SetupLines;
  REPEAT
    ThrowNeedle
  UNTIL KeyPressed;
  CloseGraph;

  key := Readkey;            {read the key from keyboard buffer}
  IF   key = null            {extended key code?  (e.g., a function key)}
  THEN key := Readkey        {read second character}

END {BuffonsNeedle}.
